/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies whether the configuration recorder excludes certain resource types from being recorded. Use the
 * <code>resourceTypes</code> field to enter a comma-separated list of resource types you want to exclude from
 * recording.
 * </p>
 * <p>
 * By default, when Config adds support for a new resource type in the Region where you set up the configuration
 * recorder, including global resource types, Config starts recording resources of that type automatically.
 * </p>
 * <note>
 * <p>
 * <b>How to use the exclusion recording strategy </b>
 * </p>
 * <p>
 * To use this option, you must set the <code>useOnly</code> field of <a
 * href="https://docs.aws.amazon.com/config/latest/APIReference/API_RecordingStrategy.html">RecordingStrategy</a> to
 * <code>EXCLUSION_BY_RESOURCE_TYPES</code>.
 * </p>
 * <p>
 * Config will then record configuration changes for all supported resource types, except the resource types that you
 * specify to exclude from being recorded.
 * </p>
 * <p>
 * <b>Global resource types and the exclusion recording strategy </b>
 * </p>
 * <p>
 * Unless specifically listed as exclusions, <code>AWS::RDS::GlobalCluster</code> will be recorded automatically in all
 * supported Config Regions were the configuration recorder is enabled.
 * </p>
 * <p>
 * IAM users, groups, roles, and customer managed policies will be recorded in the Region where you set up the
 * configuration recorder if that is a Region where Config was available before February 2022. You cannot be record the
 * global IAM resouce types in Regions supported by Config after February 2022. This list where you cannot record the
 * global IAM resource types includes the following Regions:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Asia Pacific (Hyderabad)
 * </p>
 * </li>
 * <li>
 * <p>
 * Asia Pacific (Melbourne)
 * </p>
 * </li>
 * <li>
 * <p>
 * Canada West (Calgary)
 * </p>
 * </li>
 * <li>
 * <p>
 * Europe (Spain)
 * </p>
 * </li>
 * <li>
 * <p>
 * Europe (Zurich)
 * </p>
 * </li>
 * <li>
 * <p>
 * Israel (Tel Aviv)
 * </p>
 * </li>
 * <li>
 * <p>
 * Middle East (UAE)
 * </p>
 * </li>
 * </ul>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExclusionByResourceTypes implements SdkPojo, Serializable,
        ToCopyableBuilder<ExclusionByResourceTypes.Builder, ExclusionByResourceTypes> {
    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceTypes")
            .getter(getter(ExclusionByResourceTypes::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> resourceTypes;

    private ExclusionByResourceTypes(BuilderImpl builder) {
        this.resourceTypes = builder.resourceTypes;
    }

    /**
     * <p>
     * A comma-separated list of resource types to exclude from recording by the configuration recorder.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return A comma-separated list of resource types to exclude from recording by the configuration recorder.
     */
    public final List<ResourceType> resourceTypes() {
        return ResourceTypeListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A comma-separated list of resource types to exclude from recording by the configuration recorder.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return A comma-separated list of resource types to exclude from recording by the configuration recorder.
     */
    public final List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExclusionByResourceTypes)) {
            return false;
        }
        ExclusionByResourceTypes other = (ExclusionByResourceTypes) obj;
        return hasResourceTypes() == other.hasResourceTypes()
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExclusionByResourceTypes")
                .add("ResourceTypes", hasResourceTypes() ? resourceTypesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ExclusionByResourceTypes, T> g) {
        return obj -> g.apply((ExclusionByResourceTypes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExclusionByResourceTypes> {
        /**
         * <p>
         * A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceType> resourceTypes);

        /**
         * <p>
         * A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list of resource types to exclude from recording by the configuration recorder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceType... resourceTypes);
    }

    static final class BuilderImpl implements Builder {
        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ExclusionByResourceTypes model) {
            resourceTypesWithStrings(model.resourceTypes);
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceType> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceType... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public ExclusionByResourceTypes build() {
            return new ExclusionByResourceTypes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
