/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the types of AWS resource for which AWS Config records configuration changes.
 * </p>
 * <p>
 * In the recording group, you specify whether all supported types or specific types of resources are recorded.
 * </p>
 * <p>
 * By default, AWS Config records configuration changes for all supported types of regional resources that AWS Config
 * discovers in the region in which it is running. Regional resources are tied to a region and can be used only in that
 * region. Examples of regional resources are EC2 instances and EBS volumes.
 * </p>
 * <p>
 * You can also have AWS Config record configuration changes for supported types of global resources (for example, IAM
 * resources). Global resources are not tied to an individual region and can be used in all regions.
 * </p>
 * <important>
 * <p>
 * The configuration details for any global resource are the same in all regions. If you customize AWS Config in
 * multiple regions to record global resources, it will create multiple configuration items each time a global resource
 * changes: one configuration item for each region. These configuration items will contain identical data. To prevent
 * duplicate configuration items, you should consider customizing AWS Config in only one region to record global
 * resources, unless you want the configuration items to be available in multiple regions.
 * </p>
 * </important>
 * <p>
 * If you don't want AWS Config to record all resources, you can specify which types of resources it will record with
 * the <code>resourceTypes</code> parameter.
 * </p>
 * <p>
 * For a list of supported resource types, see <a
 * href="https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
 * >Supported Resource Types</a>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/config/latest/developerguide/select-resources.html">Selecting Which Resources AWS
 * Config Records</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecordingGroup implements SdkPojo, Serializable, ToCopyableBuilder<RecordingGroup.Builder, RecordingGroup> {
    private static final SdkField<Boolean> ALL_SUPPORTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(RecordingGroup::allSupported)).setter(setter(Builder::allSupported))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allSupported").build()).build();

    private static final SdkField<Boolean> INCLUDE_GLOBAL_RESOURCE_TYPES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(RecordingGroup::includeGlobalResourceTypes))
            .setter(setter(Builder::includeGlobalResourceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("includeGlobalResourceTypes").build())
            .build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(RecordingGroup::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALL_SUPPORTED_FIELD,
            INCLUDE_GLOBAL_RESOURCE_TYPES_FIELD, RESOURCE_TYPES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean allSupported;

    private final Boolean includeGlobalResourceTypes;

    private final List<String> resourceTypes;

    private RecordingGroup(BuilderImpl builder) {
        this.allSupported = builder.allSupported;
        this.includeGlobalResourceTypes = builder.includeGlobalResourceTypes;
        this.resourceTypes = builder.resourceTypes;
    }

    /**
     * <p>
     * Specifies whether AWS Config records configuration changes for every supported type of regional resource.
     * </p>
     * <p>
     * If you set this option to <code>true</code>, when AWS Config adds support for a new type of regional resource, it
     * starts recording resources of that type automatically.
     * </p>
     * <p>
     * If you set this option to <code>true</code>, you cannot enumerate a list of <code>resourceTypes</code>.
     * </p>
     * 
     * @return Specifies whether AWS Config records configuration changes for every supported type of regional
     *         resource.</p>
     *         <p>
     *         If you set this option to <code>true</code>, when AWS Config adds support for a new type of regional
     *         resource, it starts recording resources of that type automatically.
     *         </p>
     *         <p>
     *         If you set this option to <code>true</code>, you cannot enumerate a list of <code>resourceTypes</code>.
     */
    public Boolean allSupported() {
        return allSupported;
    }

    /**
     * <p>
     * Specifies whether AWS Config includes all supported types of global resources (for example, IAM resources) with
     * the resources that it records.
     * </p>
     * <p>
     * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
     * <code>true</code>.
     * </p>
     * <p>
     * If you set this option to <code>true</code>, when AWS Config adds support for a new type of global resource, it
     * starts recording resources of that type automatically.
     * </p>
     * <p>
     * The configuration details for any global resource are the same in all regions. To prevent duplicate configuration
     * items, you should consider customizing AWS Config in only one region to record global resources.
     * </p>
     * 
     * @return Specifies whether AWS Config includes all supported types of global resources (for example, IAM
     *         resources) with the resources that it records.</p>
     *         <p>
     *         Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
     *         <code>true</code>.
     *         </p>
     *         <p>
     *         If you set this option to <code>true</code>, when AWS Config adds support for a new type of global
     *         resource, it starts recording resources of that type automatically.
     *         </p>
     *         <p>
     *         The configuration details for any global resource are the same in all regions. To prevent duplicate
     *         configuration items, you should consider customizing AWS Config in only one region to record global
     *         resources.
     */
    public Boolean includeGlobalResourceTypes() {
        return includeGlobalResourceTypes;
    }

    /**
     * <p>
     * A comma-separated list that specifies the types of AWS resources for which AWS Config records configuration
     * changes (for example, <code>AWS::EC2::Instance</code> or <code>AWS::CloudTrail::Trail</code>).
     * </p>
     * <p>
     * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
     * <code>false</code>.
     * </p>
     * <p>
     * If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it will not
     * record resources of that type unless you manually add that type to your recording group.
     * </p>
     * <p>
     * For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a href=
     * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     * >Supported AWS Resource Types</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A comma-separated list that specifies the types of AWS resources for which AWS Config records
     *         configuration changes (for example, <code>AWS::EC2::Instance</code> or
     *         <code>AWS::CloudTrail::Trail</code>).</p>
     *         <p>
     *         Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
     *         <code>false</code>.
     *         </p>
     *         <p>
     *         If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it
     *         will not record resources of that type unless you manually add that type to your recording group.
     *         </p>
     *         <p>
     *         For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a
     *         href=
     *         "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     *         >Supported AWS Resource Types</a>.
     */
    public List<ResourceType> resourceTypes() {
        return ResourceTypeListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * <p>
     * A comma-separated list that specifies the types of AWS resources for which AWS Config records configuration
     * changes (for example, <code>AWS::EC2::Instance</code> or <code>AWS::CloudTrail::Trail</code>).
     * </p>
     * <p>
     * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
     * <code>false</code>.
     * </p>
     * <p>
     * If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it will not
     * record resources of that type unless you manually add that type to your recording group.
     * </p>
     * <p>
     * For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a href=
     * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     * >Supported AWS Resource Types</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A comma-separated list that specifies the types of AWS resources for which AWS Config records
     *         configuration changes (for example, <code>AWS::EC2::Instance</code> or
     *         <code>AWS::CloudTrail::Trail</code>).</p>
     *         <p>
     *         Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
     *         <code>false</code>.
     *         </p>
     *         <p>
     *         If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it
     *         will not record resources of that type unless you manually add that type to your recording group.
     *         </p>
     *         <p>
     *         For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a
     *         href=
     *         "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
     *         >Supported AWS Resource Types</a>.
     */
    public List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allSupported());
        hashCode = 31 * hashCode + Objects.hashCode(includeGlobalResourceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypesAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecordingGroup)) {
            return false;
        }
        RecordingGroup other = (RecordingGroup) obj;
        return Objects.equals(allSupported(), other.allSupported())
                && Objects.equals(includeGlobalResourceTypes(), other.includeGlobalResourceTypes())
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RecordingGroup").add("AllSupported", allSupported())
                .add("IncludeGlobalResourceTypes", includeGlobalResourceTypes()).add("ResourceTypes", resourceTypesAsStrings())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "allSupported":
            return Optional.ofNullable(clazz.cast(allSupported()));
        case "includeGlobalResourceTypes":
            return Optional.ofNullable(clazz.cast(includeGlobalResourceTypes()));
        case "resourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecordingGroup, T> g) {
        return obj -> g.apply((RecordingGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecordingGroup> {
        /**
         * <p>
         * Specifies whether AWS Config records configuration changes for every supported type of regional resource.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, when AWS Config adds support for a new type of regional
         * resource, it starts recording resources of that type automatically.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, you cannot enumerate a list of <code>resourceTypes</code>.
         * </p>
         * 
         * @param allSupported
         *        Specifies whether AWS Config records configuration changes for every supported type of regional
         *        resource.</p>
         *        <p>
         *        If you set this option to <code>true</code>, when AWS Config adds support for a new type of regional
         *        resource, it starts recording resources of that type automatically.
         *        </p>
         *        <p>
         *        If you set this option to <code>true</code>, you cannot enumerate a list of <code>resourceTypes</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allSupported(Boolean allSupported);

        /**
         * <p>
         * Specifies whether AWS Config includes all supported types of global resources (for example, IAM resources)
         * with the resources that it records.
         * </p>
         * <p>
         * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
         * <code>true</code>.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, when AWS Config adds support for a new type of global resource,
         * it starts recording resources of that type automatically.
         * </p>
         * <p>
         * The configuration details for any global resource are the same in all regions. To prevent duplicate
         * configuration items, you should consider customizing AWS Config in only one region to record global
         * resources.
         * </p>
         * 
         * @param includeGlobalResourceTypes
         *        Specifies whether AWS Config includes all supported types of global resources (for example, IAM
         *        resources) with the resources that it records.</p>
         *        <p>
         *        Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option
         *        to <code>true</code>.
         *        </p>
         *        <p>
         *        If you set this option to <code>true</code>, when AWS Config adds support for a new type of global
         *        resource, it starts recording resources of that type automatically.
         *        </p>
         *        <p>
         *        The configuration details for any global resource are the same in all regions. To prevent duplicate
         *        configuration items, you should consider customizing AWS Config in only one region to record global
         *        resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeGlobalResourceTypes(Boolean includeGlobalResourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies the types of AWS resources for which AWS Config records configuration
         * changes (for example, <code>AWS::EC2::Instance</code> or <code>AWS::CloudTrail::Trail</code>).
         * </p>
         * <p>
         * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
         * <code>false</code>.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it will
         * not record resources of that type unless you manually add that type to your recording group.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported AWS Resource Types</a>.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies the types of AWS resources for which AWS Config records
         *        configuration changes (for example, <code>AWS::EC2::Instance</code> or
         *        <code>AWS::CloudTrail::Trail</code>).</p>
         *        <p>
         *        Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option
         *        to <code>false</code>.
         *        </p>
         *        <p>
         *        If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource,
         *        it will not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported AWS Resource Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies the types of AWS resources for which AWS Config records configuration
         * changes (for example, <code>AWS::EC2::Instance</code> or <code>AWS::CloudTrail::Trail</code>).
         * </p>
         * <p>
         * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
         * <code>false</code>.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it will
         * not record resources of that type unless you manually add that type to your recording group.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported AWS Resource Types</a>.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies the types of AWS resources for which AWS Config records
         *        configuration changes (for example, <code>AWS::EC2::Instance</code> or
         *        <code>AWS::CloudTrail::Trail</code>).</p>
         *        <p>
         *        Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option
         *        to <code>false</code>.
         *        </p>
         *        <p>
         *        If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource,
         *        it will not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported AWS Resource Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies the types of AWS resources for which AWS Config records configuration
         * changes (for example, <code>AWS::EC2::Instance</code> or <code>AWS::CloudTrail::Trail</code>).
         * </p>
         * <p>
         * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
         * <code>false</code>.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it will
         * not record resources of that type unless you manually add that type to your recording group.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported AWS Resource Types</a>.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies the types of AWS resources for which AWS Config records
         *        configuration changes (for example, <code>AWS::EC2::Instance</code> or
         *        <code>AWS::CloudTrail::Trail</code>).</p>
         *        <p>
         *        Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option
         *        to <code>false</code>.
         *        </p>
         *        <p>
         *        If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource,
         *        it will not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported AWS Resource Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceType> resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies the types of AWS resources for which AWS Config records configuration
         * changes (for example, <code>AWS::EC2::Instance</code> or <code>AWS::CloudTrail::Trail</code>).
         * </p>
         * <p>
         * Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option to
         * <code>false</code>.
         * </p>
         * <p>
         * If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource, it will
         * not record resources of that type unless you manually add that type to your recording group.
         * </p>
         * <p>
         * For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a href=
         * "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         * >Supported AWS Resource Types</a>.
         * </p>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies the types of AWS resources for which AWS Config records
         *        configuration changes (for example, <code>AWS::EC2::Instance</code> or
         *        <code>AWS::CloudTrail::Trail</code>).</p>
         *        <p>
         *        Before you can set this option to <code>true</code>, you must set the <code>allSupported</code> option
         *        to <code>false</code>.
         *        </p>
         *        <p>
         *        If you set this option to <code>true</code>, when AWS Config adds support for a new type of resource,
         *        it will not record resources of that type unless you manually add that type to your recording group.
         *        </p>
         *        <p>
         *        For a list of valid <code>resourceTypes</code> values, see the <b>resourceType Value</b> column in <a
         *        href=
         *        "https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources"
         *        >Supported AWS Resource Types</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceType... resourceTypes);
    }

    static final class BuilderImpl implements Builder {
        private Boolean allSupported;

        private Boolean includeGlobalResourceTypes;

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RecordingGroup model) {
            allSupported(model.allSupported);
            includeGlobalResourceTypes(model.includeGlobalResourceTypes);
            resourceTypesWithStrings(model.resourceTypes);
        }

        public final Boolean getAllSupported() {
            return allSupported;
        }

        @Override
        public final Builder allSupported(Boolean allSupported) {
            this.allSupported = allSupported;
            return this;
        }

        public final void setAllSupported(Boolean allSupported) {
            this.allSupported = allSupported;
        }

        public final Boolean getIncludeGlobalResourceTypes() {
            return includeGlobalResourceTypes;
        }

        @Override
        public final Builder includeGlobalResourceTypes(Boolean includeGlobalResourceTypes) {
            this.includeGlobalResourceTypes = includeGlobalResourceTypes;
            return this;
        }

        public final void setIncludeGlobalResourceTypes(Boolean includeGlobalResourceTypes) {
            this.includeGlobalResourceTypes = includeGlobalResourceTypes;
        }

        public final Collection<String> getResourceTypesAsStrings() {
            return resourceTypes;
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceType> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceType... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final void setResourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
        }

        @Override
        public RecordingGroup build() {
            return new RecordingGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
