/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connect.model.ConnectException;
import software.amazon.awssdk.services.connect.model.ConnectRequest;
import software.amazon.awssdk.services.connect.model.ContactNotFoundException;
import software.amazon.awssdk.services.connect.model.CreateUserRequest;
import software.amazon.awssdk.services.connect.model.CreateUserResponse;
import software.amazon.awssdk.services.connect.model.DeleteUserRequest;
import software.amazon.awssdk.services.connect.model.DeleteUserResponse;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyGroupRequest;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyGroupResponse;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyStructureRequest;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyStructureResponse;
import software.amazon.awssdk.services.connect.model.DescribeUserRequest;
import software.amazon.awssdk.services.connect.model.DescribeUserResponse;
import software.amazon.awssdk.services.connect.model.DestinationNotAllowedException;
import software.amazon.awssdk.services.connect.model.DuplicateResourceException;
import software.amazon.awssdk.services.connect.model.GetContactAttributesRequest;
import software.amazon.awssdk.services.connect.model.GetContactAttributesResponse;
import software.amazon.awssdk.services.connect.model.GetCurrentMetricDataRequest;
import software.amazon.awssdk.services.connect.model.GetCurrentMetricDataResponse;
import software.amazon.awssdk.services.connect.model.GetFederationTokenRequest;
import software.amazon.awssdk.services.connect.model.GetFederationTokenResponse;
import software.amazon.awssdk.services.connect.model.GetMetricDataRequest;
import software.amazon.awssdk.services.connect.model.GetMetricDataResponse;
import software.amazon.awssdk.services.connect.model.InternalServiceException;
import software.amazon.awssdk.services.connect.model.InvalidParameterException;
import software.amazon.awssdk.services.connect.model.InvalidRequestException;
import software.amazon.awssdk.services.connect.model.LimitExceededException;
import software.amazon.awssdk.services.connect.model.ListContactFlowsRequest;
import software.amazon.awssdk.services.connect.model.ListContactFlowsResponse;
import software.amazon.awssdk.services.connect.model.ListHoursOfOperationsRequest;
import software.amazon.awssdk.services.connect.model.ListHoursOfOperationsResponse;
import software.amazon.awssdk.services.connect.model.ListPhoneNumbersRequest;
import software.amazon.awssdk.services.connect.model.ListPhoneNumbersResponse;
import software.amazon.awssdk.services.connect.model.ListQueuesRequest;
import software.amazon.awssdk.services.connect.model.ListQueuesResponse;
import software.amazon.awssdk.services.connect.model.ListRoutingProfilesRequest;
import software.amazon.awssdk.services.connect.model.ListRoutingProfilesResponse;
import software.amazon.awssdk.services.connect.model.ListSecurityProfilesRequest;
import software.amazon.awssdk.services.connect.model.ListSecurityProfilesResponse;
import software.amazon.awssdk.services.connect.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.connect.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsRequest;
import software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsResponse;
import software.amazon.awssdk.services.connect.model.ListUsersRequest;
import software.amazon.awssdk.services.connect.model.ListUsersResponse;
import software.amazon.awssdk.services.connect.model.OutboundContactNotPermittedException;
import software.amazon.awssdk.services.connect.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connect.model.ResumeContactRecordingRequest;
import software.amazon.awssdk.services.connect.model.ResumeContactRecordingResponse;
import software.amazon.awssdk.services.connect.model.StartChatContactRequest;
import software.amazon.awssdk.services.connect.model.StartChatContactResponse;
import software.amazon.awssdk.services.connect.model.StartContactRecordingRequest;
import software.amazon.awssdk.services.connect.model.StartContactRecordingResponse;
import software.amazon.awssdk.services.connect.model.StartOutboundVoiceContactRequest;
import software.amazon.awssdk.services.connect.model.StartOutboundVoiceContactResponse;
import software.amazon.awssdk.services.connect.model.StopContactRecordingRequest;
import software.amazon.awssdk.services.connect.model.StopContactRecordingResponse;
import software.amazon.awssdk.services.connect.model.StopContactRequest;
import software.amazon.awssdk.services.connect.model.StopContactResponse;
import software.amazon.awssdk.services.connect.model.SuspendContactRecordingRequest;
import software.amazon.awssdk.services.connect.model.SuspendContactRecordingResponse;
import software.amazon.awssdk.services.connect.model.TagResourceRequest;
import software.amazon.awssdk.services.connect.model.TagResourceResponse;
import software.amazon.awssdk.services.connect.model.ThrottlingException;
import software.amazon.awssdk.services.connect.model.UntagResourceRequest;
import software.amazon.awssdk.services.connect.model.UntagResourceResponse;
import software.amazon.awssdk.services.connect.model.UpdateContactAttributesRequest;
import software.amazon.awssdk.services.connect.model.UpdateContactAttributesResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserHierarchyRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserHierarchyResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserIdentityInfoRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserIdentityInfoResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserPhoneConfigRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserPhoneConfigResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserRoutingProfileRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserRoutingProfileResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserSecurityProfilesRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserSecurityProfilesResponse;
import software.amazon.awssdk.services.connect.model.UserNotFoundException;
import software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable;
import software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable;
import software.amazon.awssdk.services.connect.paginators.ListContactFlowsIterable;
import software.amazon.awssdk.services.connect.paginators.ListHoursOfOperationsIterable;
import software.amazon.awssdk.services.connect.paginators.ListPhoneNumbersIterable;
import software.amazon.awssdk.services.connect.paginators.ListQueuesIterable;
import software.amazon.awssdk.services.connect.paginators.ListRoutingProfilesIterable;
import software.amazon.awssdk.services.connect.paginators.ListSecurityProfilesIterable;
import software.amazon.awssdk.services.connect.paginators.ListUserHierarchyGroupsIterable;
import software.amazon.awssdk.services.connect.paginators.ListUsersIterable;
import software.amazon.awssdk.services.connect.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DescribeUserHierarchyGroupRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DescribeUserHierarchyStructureRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetContactAttributesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetCurrentMetricDataRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetFederationTokenRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetMetricDataRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListContactFlowsRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListHoursOfOperationsRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListPhoneNumbersRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListQueuesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListRoutingProfilesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListSecurityProfilesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListUserHierarchyGroupsRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ResumeContactRecordingRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StartChatContactRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StartContactRecordingRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StartOutboundVoiceContactRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StopContactRecordingRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StopContactRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.SuspendContactRecordingRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateContactAttributesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserHierarchyRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserIdentityInfoRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserPhoneConfigRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserRoutingProfileRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserSecurityProfilesRequestMarshaller;

/**
 * Internal implementation of {@link ConnectClient}.
 *
 * @see ConnectClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectClient implements ConnectClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultConnectClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a user account for the specified Amazon Connect instance.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws LimitExceededException
     *         The allowed limit for the resource has been exceeded.
     * @throws DuplicateResourceException
     *         A resource with the specified name already exists.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws InvalidRequestException,
            InvalidParameterException, LimitExceededException, DuplicateResourceException, ResourceNotFoundException,
            ThrottlingException, InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a user account from the specified Amazon Connect instance.
     * </p>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteUserRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified user account. You can find the instance ID in the console (it’s the final part of the
     * ARN). The console does not display the user IDs. Instead, list the users and note the IDs provided in the output.
     * </p>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");

            return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                    .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified hierarchy group.
     * </p>
     *
     * @param describeUserHierarchyGroupRequest
     * @return Result of the DescribeUserHierarchyGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DescribeUserHierarchyGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DescribeUserHierarchyGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeUserHierarchyGroupResponse describeUserHierarchyGroup(
            DescribeUserHierarchyGroupRequest describeUserHierarchyGroupRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserHierarchyGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeUserHierarchyGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserHierarchyGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUserHierarchyGroup");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeUserHierarchyGroupRequest, DescribeUserHierarchyGroupResponse>()
                            .withOperationName("DescribeUserHierarchyGroup").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeUserHierarchyGroupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeUserHierarchyGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the hierarchy structure of the specified Amazon Connect instance.
     * </p>
     *
     * @param describeUserHierarchyStructureRequest
     * @return Result of the DescribeUserHierarchyStructure operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DescribeUserHierarchyStructure
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DescribeUserHierarchyStructure"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeUserHierarchyStructureResponse describeUserHierarchyStructure(
            DescribeUserHierarchyStructureRequest describeUserHierarchyStructureRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserHierarchyStructureResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeUserHierarchyStructureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeUserHierarchyStructureRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUserHierarchyStructure");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeUserHierarchyStructureRequest, DescribeUserHierarchyStructureResponse>()
                            .withOperationName("DescribeUserHierarchyStructure").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeUserHierarchyStructureRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeUserHierarchyStructureRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the contact attributes for the specified contact.
     * </p>
     *
     * @param getContactAttributesRequest
     * @return Result of the GetContactAttributes operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetContactAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetContactAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetContactAttributesResponse getContactAttributes(GetContactAttributesRequest getContactAttributesRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetContactAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getContactAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetContactAttributes");

            return clientHandler.execute(new ClientExecutionParams<GetContactAttributesRequest, GetContactAttributesResponse>()
                    .withOperationName("GetContactAttributes").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getContactAttributesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetContactAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the real-time metric data from the specified Amazon Connect instance.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/real-time-metrics-reports.html">Real-time Metrics
     * Reports</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     *
     * @param getCurrentMetricDataRequest
     * @return Result of the GetCurrentMetricData operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetCurrentMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetCurrentMetricData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCurrentMetricDataResponse getCurrentMetricData(GetCurrentMetricDataRequest getCurrentMetricDataRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCurrentMetricDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCurrentMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCurrentMetricDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCurrentMetricData");

            return clientHandler.execute(new ClientExecutionParams<GetCurrentMetricDataRequest, GetCurrentMetricDataResponse>()
                    .withOperationName("GetCurrentMetricData").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCurrentMetricDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCurrentMetricDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the real-time metric data from the specified Amazon Connect instance.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/real-time-metrics-reports.html">Real-time Metrics
     * Reports</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getCurrentMetricData(software.amazon.awssdk.services.connect.model.GetCurrentMetricDataRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable responses = client.getCurrentMetricDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable responses = client
     *             .getCurrentMetricDataPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.GetCurrentMetricDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable responses = client.getCurrentMetricDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getCurrentMetricData(software.amazon.awssdk.services.connect.model.GetCurrentMetricDataRequest)}
     * operation.</b>
     * </p>
     *
     * @param getCurrentMetricDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetCurrentMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetCurrentMetricData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCurrentMetricDataIterable getCurrentMetricDataPaginator(GetCurrentMetricDataRequest getCurrentMetricDataRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ConnectException {
        return new GetCurrentMetricDataIterable(this, applyPaginatorUserAgent(getCurrentMetricDataRequest));
    }

    /**
     * <p>
     * Retrieves a token for federation.
     * </p>
     *
     * @param getFederationTokenRequest
     * @return Result of the GetFederationToken operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws UserNotFoundException
     *         No user with the specified credentials was found in the Amazon Connect instance.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws DuplicateResourceException
     *         A resource with the specified name already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetFederationToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetFederationToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFederationTokenResponse getFederationToken(GetFederationTokenRequest getFederationTokenRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, UserNotFoundException,
            InternalServiceException, DuplicateResourceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFederationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetFederationTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFederationTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFederationToken");

            return clientHandler.execute(new ClientExecutionParams<GetFederationTokenRequest, GetFederationTokenResponse>()
                    .withOperationName("GetFederationToken").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getFederationTokenRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetFederationTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets historical metric data from the specified Amazon Connect instance.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/historical-metrics.html">Historical Metrics
     * Reports</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     *
     * @param getMetricDataRequest
     * @return Result of the GetMetricData operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataResponse getMetricData(GetMetricDataRequest getMetricDataRequest) throws InvalidRequestException,
            InvalidParameterException, InternalServiceException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMetricDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricData");

            return clientHandler.execute(new ClientExecutionParams<GetMetricDataRequest, GetMetricDataResponse>()
                    .withOperationName("GetMetricData").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMetricDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets historical metric data from the specified Amazon Connect instance.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/historical-metrics.html">Historical Metrics
     * Reports</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getMetricData(software.amazon.awssdk.services.connect.model.GetMetricDataRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.GetMetricDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getMetricData(software.amazon.awssdk.services.connect.model.GetMetricDataRequest)} operation.</b>
     * </p>
     *
     * @param getMetricDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataIterable getMetricDataPaginator(GetMetricDataRequest getMetricDataRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ConnectException {
        return new GetMetricDataIterable(this, applyPaginatorUserAgent(getMetricDataRequest));
    }

    /**
     * <p>
     * Provides information about the contact flows for the specified Amazon Connect instance.
     * </p>
     *
     * @param listContactFlowsRequest
     * @return Result of the ListContactFlows operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListContactFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListContactFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContactFlowsResponse listContactFlows(ListContactFlowsRequest listContactFlowsRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContactFlowsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListContactFlowsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContactFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContactFlows");

            return clientHandler.execute(new ClientExecutionParams<ListContactFlowsRequest, ListContactFlowsResponse>()
                    .withOperationName("ListContactFlows").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listContactFlowsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListContactFlowsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the contact flows for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContactFlows(software.amazon.awssdk.services.connect.model.ListContactFlowsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListContactFlowsIterable responses = client.listContactFlowsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListContactFlowsIterable responses = client
     *             .listContactFlowsPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListContactFlowsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListContactFlowsIterable responses = client.listContactFlowsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContactFlows(software.amazon.awssdk.services.connect.model.ListContactFlowsRequest)} operation.</b>
     * </p>
     *
     * @param listContactFlowsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListContactFlows
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListContactFlows" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListContactFlowsIterable listContactFlowsPaginator(ListContactFlowsRequest listContactFlowsRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        return new ListContactFlowsIterable(this, applyPaginatorUserAgent(listContactFlowsRequest));
    }

    /**
     * <p>
     * Provides information about the hours of operation for the specified Amazon Connect instance.
     * </p>
     *
     * @param listHoursOfOperationsRequest
     * @return Result of the ListHoursOfOperations operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListHoursOfOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListHoursOfOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListHoursOfOperationsResponse listHoursOfOperations(ListHoursOfOperationsRequest listHoursOfOperationsRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHoursOfOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListHoursOfOperationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHoursOfOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHoursOfOperations");

            return clientHandler.execute(new ClientExecutionParams<ListHoursOfOperationsRequest, ListHoursOfOperationsResponse>()
                    .withOperationName("ListHoursOfOperations").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listHoursOfOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListHoursOfOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the hours of operation for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHoursOfOperations(software.amazon.awssdk.services.connect.model.ListHoursOfOperationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListHoursOfOperationsIterable responses = client.listHoursOfOperationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListHoursOfOperationsIterable responses = client
     *             .listHoursOfOperationsPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListHoursOfOperationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListHoursOfOperationsIterable responses = client.listHoursOfOperationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHoursOfOperations(software.amazon.awssdk.services.connect.model.ListHoursOfOperationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listHoursOfOperationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListHoursOfOperations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListHoursOfOperations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListHoursOfOperationsIterable listHoursOfOperationsPaginator(ListHoursOfOperationsRequest listHoursOfOperationsRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        return new ListHoursOfOperationsIterable(this, applyPaginatorUserAgent(listHoursOfOperationsRequest));
    }

    /**
     * <p>
     * Provides information about the phone numbers for the specified Amazon Connect instance.
     * </p>
     *
     * @param listPhoneNumbersRequest
     * @return Result of the ListPhoneNumbers operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListPhoneNumbers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListPhoneNumbers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPhoneNumbersResponse listPhoneNumbers(ListPhoneNumbersRequest listPhoneNumbersRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPhoneNumbersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPhoneNumbersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPhoneNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPhoneNumbers");

            return clientHandler.execute(new ClientExecutionParams<ListPhoneNumbersRequest, ListPhoneNumbersResponse>()
                    .withOperationName("ListPhoneNumbers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPhoneNumbersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPhoneNumbersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the phone numbers for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPhoneNumbers(software.amazon.awssdk.services.connect.model.ListPhoneNumbersRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListPhoneNumbersIterable responses = client.listPhoneNumbersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListPhoneNumbersIterable responses = client
     *             .listPhoneNumbersPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListPhoneNumbersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListPhoneNumbersIterable responses = client.listPhoneNumbersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPhoneNumbers(software.amazon.awssdk.services.connect.model.ListPhoneNumbersRequest)} operation.</b>
     * </p>
     *
     * @param listPhoneNumbersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListPhoneNumbers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListPhoneNumbers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPhoneNumbersIterable listPhoneNumbersPaginator(ListPhoneNumbersRequest listPhoneNumbersRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        return new ListPhoneNumbersIterable(this, applyPaginatorUserAgent(listPhoneNumbersRequest));
    }

    /**
     * <p>
     * Provides information about the queues for the specified Amazon Connect instance.
     * </p>
     *
     * @param listQueuesRequest
     * @return Result of the ListQueues operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueuesResponse listQueues(ListQueuesRequest listQueuesRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQueuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListQueuesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueues");

            return clientHandler
                    .execute(new ClientExecutionParams<ListQueuesRequest, ListQueuesResponse>().withOperationName("ListQueues")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listQueuesRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListQueuesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides information about the queues for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listQueues(software.amazon.awssdk.services.connect.model.ListQueuesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListQueuesIterable responses = client.listQueuesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListQueuesIterable responses = client.listQueuesPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListQueuesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListQueuesIterable responses = client.listQueuesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueues(software.amazon.awssdk.services.connect.model.ListQueuesRequest)} operation.</b>
     * </p>
     *
     * @param listQueuesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListQueues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueuesIterable listQueuesPaginator(ListQueuesRequest listQueuesRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        return new ListQueuesIterable(this, applyPaginatorUserAgent(listQueuesRequest));
    }

    /**
     * <p>
     * Provides summary information about the routing profiles for the specified Amazon Connect instance.
     * </p>
     *
     * @param listRoutingProfilesRequest
     * @return Result of the ListRoutingProfiles operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListRoutingProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListRoutingProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListRoutingProfilesResponse listRoutingProfiles(ListRoutingProfilesRequest listRoutingProfilesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRoutingProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRoutingProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoutingProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoutingProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListRoutingProfilesRequest, ListRoutingProfilesResponse>()
                    .withOperationName("ListRoutingProfiles").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRoutingProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRoutingProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summary information about the routing profiles for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRoutingProfiles(software.amazon.awssdk.services.connect.model.ListRoutingProfilesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListRoutingProfilesIterable responses = client.listRoutingProfilesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListRoutingProfilesIterable responses = client
     *             .listRoutingProfilesPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListRoutingProfilesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListRoutingProfilesIterable responses = client.listRoutingProfilesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRoutingProfiles(software.amazon.awssdk.services.connect.model.ListRoutingProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRoutingProfilesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListRoutingProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListRoutingProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListRoutingProfilesIterable listRoutingProfilesPaginator(ListRoutingProfilesRequest listRoutingProfilesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        return new ListRoutingProfilesIterable(this, applyPaginatorUserAgent(listRoutingProfilesRequest));
    }

    /**
     * <p>
     * Provides summary information about the security profiles for the specified Amazon Connect instance.
     * </p>
     *
     * @param listSecurityProfilesRequest
     * @return Result of the ListSecurityProfiles operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListSecurityProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListSecurityProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSecurityProfilesResponse listSecurityProfiles(ListSecurityProfilesRequest listSecurityProfilesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSecurityProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSecurityProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSecurityProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSecurityProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListSecurityProfilesRequest, ListSecurityProfilesResponse>()
                    .withOperationName("ListSecurityProfiles").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listSecurityProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSecurityProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summary information about the security profiles for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSecurityProfiles(software.amazon.awssdk.services.connect.model.ListSecurityProfilesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListSecurityProfilesIterable responses = client.listSecurityProfilesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListSecurityProfilesIterable responses = client
     *             .listSecurityProfilesPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListSecurityProfilesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListSecurityProfilesIterable responses = client.listSecurityProfilesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSecurityProfiles(software.amazon.awssdk.services.connect.model.ListSecurityProfilesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSecurityProfilesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListSecurityProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListSecurityProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSecurityProfilesIterable listSecurityProfilesPaginator(ListSecurityProfilesRequest listSecurityProfilesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        return new ListSecurityProfilesIterable(this, applyPaginatorUserAgent(listSecurityProfilesRequest));
    }

    /**
     * <p>
     * Lists the tags for the specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summary information about the hierarchy groups for the specified Amazon Connect instance.
     * </p>
     *
     * @param listUserHierarchyGroupsRequest
     * @return Result of the ListUserHierarchyGroups operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListUserHierarchyGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListUserHierarchyGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListUserHierarchyGroupsResponse listUserHierarchyGroups(ListUserHierarchyGroupsRequest listUserHierarchyGroupsRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUserHierarchyGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListUserHierarchyGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUserHierarchyGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUserHierarchyGroups");

            return clientHandler
                    .execute(new ClientExecutionParams<ListUserHierarchyGroupsRequest, ListUserHierarchyGroupsResponse>()
                            .withOperationName("ListUserHierarchyGroups").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listUserHierarchyGroupsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListUserHierarchyGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summary information about the hierarchy groups for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listUserHierarchyGroups(software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListUserHierarchyGroupsIterable responses = client.listUserHierarchyGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListUserHierarchyGroupsIterable responses = client
     *             .listUserHierarchyGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListUserHierarchyGroupsIterable responses = client.listUserHierarchyGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUserHierarchyGroups(software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listUserHierarchyGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListUserHierarchyGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListUserHierarchyGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListUserHierarchyGroupsIterable listUserHierarchyGroupsPaginator(
            ListUserHierarchyGroupsRequest listUserHierarchyGroupsRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        return new ListUserHierarchyGroupsIterable(this, applyPaginatorUserAgent(listUserHierarchyGroupsRequest));
    }

    /**
     * <p>
     * Provides summary information about the users for the specified Amazon Connect instance.
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");

            return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                    .withOperationName("ListUsers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides summary information about the users for the specified Amazon Connect instance.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listUsers(software.amazon.awssdk.services.connect.model.ListUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.ListUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.ListUsersIterable responses = client.listUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listUsers(software.amazon.awssdk.services.connect.model.ListUsersRequest)} operation.</b>
     * </p>
     *
     * @param listUsersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersIterable listUsersPaginator(ListUsersRequest listUsersRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        return new ListUsersIterable(this, applyPaginatorUserAgent(listUsersRequest));
    }

    /**
     * <p>
     * When a contact is being recorded, and the recording has been suspended using SuspendContactRecording, this API
     * resumes recording the call.
     * </p>
     * <p>
     * Only voice recordings are supported at this time.
     * </p>
     *
     * @param resumeContactRecordingRequest
     * @return Result of the ResumeContactRecording operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ResumeContactRecording
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ResumeContactRecording" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ResumeContactRecordingResponse resumeContactRecording(ResumeContactRecordingRequest resumeContactRecordingRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResumeContactRecordingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResumeContactRecordingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resumeContactRecordingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResumeContactRecording");

            return clientHandler
                    .execute(new ClientExecutionParams<ResumeContactRecordingRequest, ResumeContactRecordingResponse>()
                            .withOperationName("ResumeContactRecording").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(resumeContactRecordingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ResumeContactRecordingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Initiates a contact flow to start a new chat for the customer. Response of this API provides a token required to
     * obtain credentials from the <a
     * href="https://docs.aws.amazon.com/connect-participant/latest/APIReference/API_CreateParticipantConnection.html"
     * >CreateParticipantConnection</a> API in the Amazon Connect Participant Service.
     * </p>
     * <p>
     * When a new chat contact is successfully created, clients need to subscribe to the participant’s connection for
     * the created chat within 5 minutes. This is achieved by invoking <a
     * href="https://docs.aws.amazon.com/connect-participant/latest/APIReference/API_CreateParticipantConnection.html"
     * >CreateParticipantConnection</a> with WEBSOCKET and CONNECTION_CREDENTIALS.
     * </p>
     *
     * @param startChatContactRequest
     * @return Result of the StartChatContact operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws LimitExceededException
     *         The allowed limit for the resource has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StartChatContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StartChatContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartChatContactResponse startChatContact(StartChatContactRequest startChatContactRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, InternalServiceException,
            LimitExceededException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartChatContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartChatContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startChatContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartChatContact");

            return clientHandler.execute(new ClientExecutionParams<StartChatContactRequest, StartChatContactResponse>()
                    .withOperationName("StartChatContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startChatContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartChatContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API starts recording the contact when the agent joins the call. StartContactRecording is a one-time action.
     * For example, if you use StopContactRecording to stop recording an ongoing call, you can't use
     * StartContactRecording to restart it. For scenarios where the recording has started and you want to suspend and
     * resume it, such as when collecting sensitive information (for example, a credit card number), use
     * SuspendContactRecording and ResumeContactRecording.
     * </p>
     * <p>
     * You can use this API to override the recording behavior configured in the <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/set-recording-behavior.html">Set recording
     * behavior</a> block.
     * </p>
     * <p>
     * Only voice recordings are supported at this time.
     * </p>
     *
     * @param startContactRecordingRequest
     * @return Result of the StartContactRecording operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StartContactRecording
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StartContactRecording" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartContactRecordingResponse startContactRecording(StartContactRecordingRequest startContactRecordingRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartContactRecordingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartContactRecordingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startContactRecordingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartContactRecording");

            return clientHandler.execute(new ClientExecutionParams<StartContactRecordingRequest, StartContactRecordingResponse>()
                    .withOperationName("StartContactRecording").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startContactRecordingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartContactRecordingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API places an outbound call to a contact, and then initiates the contact flow. It performs the actions in
     * the contact flow that's specified (in <code>ContactFlowId</code>).
     * </p>
     * <p>
     * Agents are not involved in initiating the outbound API (that is, dialing the contact). If the contact flow places
     * an outbound call to a contact, and then puts the contact in queue, that's when the call is routed to the agent,
     * like any other inbound case.
     * </p>
     * <p>
     * There is a 60 second dialing timeout for this operation. If the call is not connected after 60 seconds, it fails.
     * </p>
     *
     * @param startOutboundVoiceContactRequest
     * @return Result of the StartOutboundVoiceContact operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws LimitExceededException
     *         The allowed limit for the resource has been exceeded.
     * @throws DestinationNotAllowedException
     *         Outbound calls to the destination number are not allowed.
     * @throws OutboundContactNotPermittedException
     *         The contact is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StartOutboundVoiceContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StartOutboundVoiceContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartOutboundVoiceContactResponse startOutboundVoiceContact(
            StartOutboundVoiceContactRequest startOutboundVoiceContactRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, InternalServiceException, LimitExceededException,
            DestinationNotAllowedException, OutboundContactNotPermittedException, AwsServiceException, SdkClientException,
            ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartOutboundVoiceContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartOutboundVoiceContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startOutboundVoiceContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartOutboundVoiceContact");

            return clientHandler
                    .execute(new ClientExecutionParams<StartOutboundVoiceContactRequest, StartOutboundVoiceContactResponse>()
                            .withOperationName("StartOutboundVoiceContact").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startOutboundVoiceContactRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartOutboundVoiceContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Ends the specified contact.
     * </p>
     *
     * @param stopContactRequest
     * @return Result of the StopContact operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ContactNotFoundException
     *         The contact with the specified ID is not active or does not exist.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StopContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StopContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopContactResponse stopContact(StopContactRequest stopContactRequest) throws InvalidRequestException,
            ContactNotFoundException, InvalidParameterException, ResourceNotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopContactRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopContact");

            return clientHandler.execute(new ClientExecutionParams<StopContactRequest, StopContactResponse>()
                    .withOperationName("StopContact").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopContactRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopContactRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When a contact is being recorded, this API stops recording the call. StopContactRecording is a one-time action.
     * If you use StopContactRecording to stop recording an ongoing call, you can't use StartContactRecording to restart
     * it. For scenarios where the recording has started and you want to suspend it for sensitive information (for
     * example, to collect a credit card number), and then restart it, use SuspendContactRecording and
     * ResumeContactRecording.
     * </p>
     * <p>
     * Only voice recordings are supported at this time.
     * </p>
     *
     * @param stopContactRecordingRequest
     * @return Result of the StopContactRecording operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StopContactRecording
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StopContactRecording" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopContactRecordingResponse stopContactRecording(StopContactRecordingRequest stopContactRecordingRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopContactRecordingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopContactRecordingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopContactRecordingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopContactRecording");

            return clientHandler.execute(new ClientExecutionParams<StopContactRecordingRequest, StopContactRecordingResponse>()
                    .withOperationName("StopContactRecording").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopContactRecordingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopContactRecordingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When a contact is being recorded, this API suspends recording the call. For example, you might suspend the call
     * recording while collecting sensitive information, such as a credit card number. Then use ResumeContactRecording
     * to restart recording.
     * </p>
     * <p>
     * The period of time that the recording is suspended is filled with silence in the final recording.
     * </p>
     * <p>
     * Only voice recordings are supported at this time.
     * </p>
     *
     * @param suspendContactRecordingRequest
     * @return Result of the SuspendContactRecording operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.SuspendContactRecording
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/SuspendContactRecording"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SuspendContactRecordingResponse suspendContactRecording(SuspendContactRecordingRequest suspendContactRecordingRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SuspendContactRecordingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SuspendContactRecordingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, suspendContactRecordingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SuspendContactRecording");

            return clientHandler
                    .execute(new ClientExecutionParams<SuspendContactRecordingRequest, SuspendContactRecordingResponse>()
                            .withOperationName("SuspendContactRecording").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(suspendContactRecordingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SuspendContactRecordingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified resource.
     * </p>
     * <p>
     * The supported resource type is users.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidRequestException,
            InvalidParameterException, InternalServiceException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidRequestException,
            InvalidParameterException, InternalServiceException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates the contact attributes associated with the specified contact.
     * </p>
     * <p>
     * You can add or update attributes for both ongoing and completed contacts. For example, you can update the
     * customer's name or the reason the customer called while the call is active, or add notes about steps that the
     * agent took during the call that are displayed to the next agent that takes the call. You can also update
     * attributes for a contact using data from your CRM application and save the data with the contact in Amazon
     * Connect. You could also flag calls for additional analysis, such as legal review or identifying abusive callers.
     * </p>
     * <p>
     * Contact attributes are available in Amazon Connect for 24 months, and are then deleted.
     * </p>
     * <p>
     * <b>Important:</b> You cannot use the operation to update attributes for contacts that occurred prior to the
     * release of the API, September 12, 2018. You can update attributes only for contacts that started after the
     * release of the API. If you attempt to update attributes for a contact that occurred prior to the release of the
     * API, a 400 error is returned. This applies also to queued callbacks that were initiated prior to the release of
     * the API but are still active in your instance.
     * </p>
     *
     * @param updateContactAttributesRequest
     * @return Result of the UpdateContactAttributes operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateContactAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateContactAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContactAttributesResponse updateContactAttributes(UpdateContactAttributesRequest updateContactAttributesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContactAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContactAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContactAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContactAttributes");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateContactAttributesRequest, UpdateContactAttributesResponse>()
                            .withOperationName("UpdateContactAttributes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateContactAttributesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateContactAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the specified hierarchy group to the specified user.
     * </p>
     *
     * @param updateUserHierarchyRequest
     * @return Result of the UpdateUserHierarchy operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserHierarchy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserHierarchy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserHierarchyResponse updateUserHierarchy(UpdateUserHierarchyRequest updateUserHierarchyRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserHierarchyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserHierarchyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserHierarchyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserHierarchy");

            return clientHandler.execute(new ClientExecutionParams<UpdateUserHierarchyRequest, UpdateUserHierarchyResponse>()
                    .withOperationName("UpdateUserHierarchy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateUserHierarchyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateUserHierarchyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the identity information for the specified user.
     * </p>
     *
     * @param updateUserIdentityInfoRequest
     * @return Result of the UpdateUserIdentityInfo operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserIdentityInfo
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserIdentityInfo" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserIdentityInfoResponse updateUserIdentityInfo(UpdateUserIdentityInfoRequest updateUserIdentityInfoRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserIdentityInfoResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserIdentityInfoResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserIdentityInfoRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserIdentityInfo");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateUserIdentityInfoRequest, UpdateUserIdentityInfoResponse>()
                            .withOperationName("UpdateUserIdentityInfo").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateUserIdentityInfoRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateUserIdentityInfoRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the phone configuration settings for the specified user.
     * </p>
     *
     * @param updateUserPhoneConfigRequest
     * @return Result of the UpdateUserPhoneConfig operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserPhoneConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserPhoneConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserPhoneConfigResponse updateUserPhoneConfig(UpdateUserPhoneConfigRequest updateUserPhoneConfigRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserPhoneConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserPhoneConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserPhoneConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserPhoneConfig");

            return clientHandler.execute(new ClientExecutionParams<UpdateUserPhoneConfigRequest, UpdateUserPhoneConfigResponse>()
                    .withOperationName("UpdateUserPhoneConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateUserPhoneConfigRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateUserPhoneConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the specified routing profile to the specified user.
     * </p>
     *
     * @param updateUserRoutingProfileRequest
     * @return Result of the UpdateUserRoutingProfile operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserRoutingProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserRoutingProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateUserRoutingProfileResponse updateUserRoutingProfile(
            UpdateUserRoutingProfileRequest updateUserRoutingProfileRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserRoutingProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserRoutingProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRoutingProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserRoutingProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRoutingProfileRequest, UpdateUserRoutingProfileResponse>()
                            .withOperationName("UpdateUserRoutingProfile").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateUserRoutingProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateUserRoutingProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns the specified security profiles to the specified user.
     * </p>
     *
     * @param updateUserSecurityProfilesRequest
     * @return Result of the UpdateUserSecurityProfiles operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the specified parameters are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserSecurityProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserSecurityProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateUserSecurityProfilesResponse updateUserSecurityProfiles(
            UpdateUserSecurityProfilesRequest updateUserSecurityProfilesRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserSecurityProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserSecurityProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserSecurityProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserSecurityProfiles");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateUserSecurityProfilesRequest, UpdateUserSecurityProfilesResponse>()
                            .withOperationName("UpdateUserSecurityProfiles").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateUserSecurityProfilesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateUserSecurityProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OutboundContactNotPermittedException")
                                .exceptionBuilderSupplier(OutboundContactNotPermittedException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContactNotFoundException")
                                .exceptionBuilderSupplier(ContactNotFoundException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateResourceException")
                                .exceptionBuilderSupplier(DuplicateResourceException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserNotFoundException")
                                .exceptionBuilderSupplier(UserNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DestinationNotAllowedException")
                                .exceptionBuilderSupplier(DestinationNotAllowedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ConnectRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
