/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a contact.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Contact implements SdkPojo, Serializable, ToCopyableBuilder<Contact.Builder, Contact> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Contact::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Contact::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> INITIAL_CONTACT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InitialContactId").getter(getter(Contact::initialContactId)).setter(setter(Builder::initialContactId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitialContactId").build()).build();

    private static final SdkField<String> PREVIOUS_CONTACT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreviousContactId").getter(getter(Contact::previousContactId))
            .setter(setter(Builder::previousContactId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreviousContactId").build()).build();

    private static final SdkField<String> INITIATION_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InitiationMethod").getter(getter(Contact::initiationMethodAsString))
            .setter(setter(Builder::initiationMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitiationMethod").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Contact::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Contact::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CHANNEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Channel")
            .getter(getter(Contact::channelAsString)).setter(setter(Builder::channel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Channel").build()).build();

    private static final SdkField<QueueInfo> QUEUE_INFO_FIELD = SdkField.<QueueInfo> builder(MarshallingType.SDK_POJO)
            .memberName("QueueInfo").getter(getter(Contact::queueInfo)).setter(setter(Builder::queueInfo))
            .constructor(QueueInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueueInfo").build()).build();

    private static final SdkField<AgentInfo> AGENT_INFO_FIELD = SdkField.<AgentInfo> builder(MarshallingType.SDK_POJO)
            .memberName("AgentInfo").getter(getter(Contact::agentInfo)).setter(setter(Builder::agentInfo))
            .constructor(AgentInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentInfo").build()).build();

    private static final SdkField<Instant> INITIATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("InitiationTimestamp").getter(getter(Contact::initiationTimestamp))
            .setter(setter(Builder::initiationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitiationTimestamp").build())
            .build();

    private static final SdkField<Instant> DISCONNECT_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DisconnectTimestamp").getter(getter(Contact::disconnectTimestamp))
            .setter(setter(Builder::disconnectTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisconnectTimestamp").build())
            .build();

    private static final SdkField<Instant> LAST_UPDATE_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdateTimestamp").getter(getter(Contact::lastUpdateTimestamp))
            .setter(setter(Builder::lastUpdateTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdateTimestamp").build())
            .build();

    private static final SdkField<Instant> LAST_PAUSED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastPausedTimestamp").getter(getter(Contact::lastPausedTimestamp))
            .setter(setter(Builder::lastPausedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastPausedTimestamp").build())
            .build();

    private static final SdkField<Instant> LAST_RESUMED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastResumedTimestamp").getter(getter(Contact::lastResumedTimestamp))
            .setter(setter(Builder::lastResumedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastResumedTimestamp").build())
            .build();

    private static final SdkField<Integer> TOTAL_PAUSE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalPauseCount").getter(getter(Contact::totalPauseCount)).setter(setter(Builder::totalPauseCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalPauseCount").build()).build();

    private static final SdkField<Integer> TOTAL_PAUSE_DURATION_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalPauseDurationInSeconds")
            .getter(getter(Contact::totalPauseDurationInSeconds))
            .setter(setter(Builder::totalPauseDurationInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalPauseDurationInSeconds")
                    .build()).build();

    private static final SdkField<Instant> SCHEDULED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ScheduledTimestamp").getter(getter(Contact::scheduledTimestamp))
            .setter(setter(Builder::scheduledTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScheduledTimestamp").build())
            .build();

    private static final SdkField<String> RELATED_CONTACT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RelatedContactId").getter(getter(Contact::relatedContactId)).setter(setter(Builder::relatedContactId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedContactId").build()).build();

    private static final SdkField<WisdomInfo> WISDOM_INFO_FIELD = SdkField.<WisdomInfo> builder(MarshallingType.SDK_POJO)
            .memberName("WisdomInfo").getter(getter(Contact::wisdomInfo)).setter(setter(Builder::wisdomInfo))
            .constructor(WisdomInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WisdomInfo").build()).build();

    private static final SdkField<Integer> QUEUE_TIME_ADJUSTMENT_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("QueueTimeAdjustmentSeconds")
            .getter(getter(Contact::queueTimeAdjustmentSeconds))
            .setter(setter(Builder::queueTimeAdjustmentSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueueTimeAdjustmentSeconds").build())
            .build();

    private static final SdkField<Long> QUEUE_PRIORITY_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("QueuePriority").getter(getter(Contact::queuePriority)).setter(setter(Builder::queuePriority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueuePriority").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(Contact::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, ID_FIELD,
            INITIAL_CONTACT_ID_FIELD, PREVIOUS_CONTACT_ID_FIELD, INITIATION_METHOD_FIELD, NAME_FIELD, DESCRIPTION_FIELD,
            CHANNEL_FIELD, QUEUE_INFO_FIELD, AGENT_INFO_FIELD, INITIATION_TIMESTAMP_FIELD, DISCONNECT_TIMESTAMP_FIELD,
            LAST_UPDATE_TIMESTAMP_FIELD, LAST_PAUSED_TIMESTAMP_FIELD, LAST_RESUMED_TIMESTAMP_FIELD, TOTAL_PAUSE_COUNT_FIELD,
            TOTAL_PAUSE_DURATION_IN_SECONDS_FIELD, SCHEDULED_TIMESTAMP_FIELD, RELATED_CONTACT_ID_FIELD, WISDOM_INFO_FIELD,
            QUEUE_TIME_ADJUSTMENT_SECONDS_FIELD, QUEUE_PRIORITY_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String id;

    private final String initialContactId;

    private final String previousContactId;

    private final String initiationMethod;

    private final String name;

    private final String description;

    private final String channel;

    private final QueueInfo queueInfo;

    private final AgentInfo agentInfo;

    private final Instant initiationTimestamp;

    private final Instant disconnectTimestamp;

    private final Instant lastUpdateTimestamp;

    private final Instant lastPausedTimestamp;

    private final Instant lastResumedTimestamp;

    private final Integer totalPauseCount;

    private final Integer totalPauseDurationInSeconds;

    private final Instant scheduledTimestamp;

    private final String relatedContactId;

    private final WisdomInfo wisdomInfo;

    private final Integer queueTimeAdjustmentSeconds;

    private final Long queuePriority;

    private final Map<String, String> tags;

    private Contact(BuilderImpl builder) {
        this.arn = builder.arn;
        this.id = builder.id;
        this.initialContactId = builder.initialContactId;
        this.previousContactId = builder.previousContactId;
        this.initiationMethod = builder.initiationMethod;
        this.name = builder.name;
        this.description = builder.description;
        this.channel = builder.channel;
        this.queueInfo = builder.queueInfo;
        this.agentInfo = builder.agentInfo;
        this.initiationTimestamp = builder.initiationTimestamp;
        this.disconnectTimestamp = builder.disconnectTimestamp;
        this.lastUpdateTimestamp = builder.lastUpdateTimestamp;
        this.lastPausedTimestamp = builder.lastPausedTimestamp;
        this.lastResumedTimestamp = builder.lastResumedTimestamp;
        this.totalPauseCount = builder.totalPauseCount;
        this.totalPauseDurationInSeconds = builder.totalPauseDurationInSeconds;
        this.scheduledTimestamp = builder.scheduledTimestamp;
        this.relatedContactId = builder.relatedContactId;
        this.wisdomInfo = builder.wisdomInfo;
        this.queueTimeAdjustmentSeconds = builder.queueTimeAdjustmentSeconds;
        this.queuePriority = builder.queuePriority;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the contact.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the contact.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The identifier for the contact.
     * </p>
     * 
     * @return The identifier for the contact.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * If this contact is related to other contacts, this is the ID of the initial contact.
     * </p>
     * 
     * @return If this contact is related to other contacts, this is the ID of the initial contact.
     */
    public final String initialContactId() {
        return initialContactId;
    }

    /**
     * <p>
     * If this contact is not the first contact, this is the ID of the previous contact.
     * </p>
     * 
     * @return If this contact is not the first contact, this is the ID of the previous contact.
     */
    public final String previousContactId() {
        return previousContactId;
    }

    /**
     * <p>
     * Indicates how the contact was initiated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #initiationMethod}
     * will return {@link ContactInitiationMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #initiationMethodAsString}.
     * </p>
     * 
     * @return Indicates how the contact was initiated.
     * @see ContactInitiationMethod
     */
    public final ContactInitiationMethod initiationMethod() {
        return ContactInitiationMethod.fromValue(initiationMethod);
    }

    /**
     * <p>
     * Indicates how the contact was initiated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #initiationMethod}
     * will return {@link ContactInitiationMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #initiationMethodAsString}.
     * </p>
     * 
     * @return Indicates how the contact was initiated.
     * @see ContactInitiationMethod
     */
    public final String initiationMethodAsString() {
        return initiationMethod;
    }

    /**
     * <p>
     * The name of the contact.
     * </p>
     * 
     * @return The name of the contact.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the contact.
     * </p>
     * 
     * @return The description of the contact.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * How the contact reached your contact center.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #channel} will
     * return {@link Channel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #channelAsString}.
     * </p>
     * 
     * @return How the contact reached your contact center.
     * @see Channel
     */
    public final Channel channel() {
        return Channel.fromValue(channel);
    }

    /**
     * <p>
     * How the contact reached your contact center.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #channel} will
     * return {@link Channel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #channelAsString}.
     * </p>
     * 
     * @return How the contact reached your contact center.
     * @see Channel
     */
    public final String channelAsString() {
        return channel;
    }

    /**
     * <p>
     * If this contact was queued, this contains information about the queue.
     * </p>
     * 
     * @return If this contact was queued, this contains information about the queue.
     */
    public final QueueInfo queueInfo() {
        return queueInfo;
    }

    /**
     * <p>
     * Information about the agent who accepted the contact.
     * </p>
     * 
     * @return Information about the agent who accepted the contact.
     */
    public final AgentInfo agentInfo() {
        return agentInfo;
    }

    /**
     * <p>
     * The date and time this contact was initiated, in UTC time. For <code>INBOUND</code>, this is when the contact
     * arrived. For <code>OUTBOUND</code>, this is when the agent began dialing. For <code>CALLBACK</code>, this is when
     * the callback contact was created. For <code>TRANSFER</code> and <code>QUEUE_TRANSFER</code>, this is when the
     * transfer was initiated. For <code>API</code>, this is when the request arrived. For
     * <code>EXTERNAL_OUTBOUND</code>, this is when the agent started dialing the external participant. For
     * <code>MONITOR</code>, this is when the supervisor started listening to a contact.
     * </p>
     * 
     * @return The date and time this contact was initiated, in UTC time. For <code>INBOUND</code>, this is when the
     *         contact arrived. For <code>OUTBOUND</code>, this is when the agent began dialing. For
     *         <code>CALLBACK</code>, this is when the callback contact was created. For <code>TRANSFER</code> and
     *         <code>QUEUE_TRANSFER</code>, this is when the transfer was initiated. For <code>API</code>, this is when
     *         the request arrived. For <code>EXTERNAL_OUTBOUND</code>, this is when the agent started dialing the
     *         external participant. For <code>MONITOR</code>, this is when the supervisor started listening to a
     *         contact.
     */
    public final Instant initiationTimestamp() {
        return initiationTimestamp;
    }

    /**
     * <p>
     * The timestamp when the customer endpoint disconnected from Amazon Connect.
     * </p>
     * 
     * @return The timestamp when the customer endpoint disconnected from Amazon Connect.
     */
    public final Instant disconnectTimestamp() {
        return disconnectTimestamp;
    }

    /**
     * <p>
     * The timestamp when contact was last updated.
     * </p>
     * 
     * @return The timestamp when contact was last updated.
     */
    public final Instant lastUpdateTimestamp() {
        return lastUpdateTimestamp;
    }

    /**
     * <p>
     * The timestamp when the contact was last paused.
     * </p>
     * 
     * @return The timestamp when the contact was last paused.
     */
    public final Instant lastPausedTimestamp() {
        return lastPausedTimestamp;
    }

    /**
     * <p>
     * The timestamp when the contact was last resumed.
     * </p>
     * 
     * @return The timestamp when the contact was last resumed.
     */
    public final Instant lastResumedTimestamp() {
        return lastResumedTimestamp;
    }

    /**
     * <p>
     * Total pause count for a contact.
     * </p>
     * 
     * @return Total pause count for a contact.
     */
    public final Integer totalPauseCount() {
        return totalPauseCount;
    }

    /**
     * <p>
     * Total pause duration for a contact in seconds.
     * </p>
     * 
     * @return Total pause duration for a contact in seconds.
     */
    public final Integer totalPauseDurationInSeconds() {
        return totalPauseDurationInSeconds;
    }

    /**
     * <p>
     * The timestamp, in Unix epoch time format, at which to start running the inbound flow.
     * </p>
     * 
     * @return The timestamp, in Unix epoch time format, at which to start running the inbound flow.
     */
    public final Instant scheduledTimestamp() {
        return scheduledTimestamp;
    }

    /**
     * <p>
     * The contactId that is <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-persistence.html#relatedcontactid">related</a>
     * to this contact.
     * </p>
     * 
     * @return The contactId that is <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-persistence.html#relatedcontactid"
     *         >related</a> to this contact.
     */
    public final String relatedContactId() {
        return relatedContactId;
    }

    /**
     * <p>
     * Information about Amazon Connect Wisdom.
     * </p>
     * 
     * @return Information about Amazon Connect Wisdom.
     */
    public final WisdomInfo wisdomInfo() {
        return wisdomInfo;
    }

    /**
     * <p>
     * An integer that represents the queue time adjust to be applied to the contact, in seconds (longer / larger queue
     * time are routed preferentially). Cannot be specified if the QueuePriority is specified. Must be statically
     * defined and a valid integer value.
     * </p>
     * 
     * @return An integer that represents the queue time adjust to be applied to the contact, in seconds (longer /
     *         larger queue time are routed preferentially). Cannot be specified if the QueuePriority is specified. Must
     *         be statically defined and a valid integer value.
     */
    public final Integer queueTimeAdjustmentSeconds() {
        return queueTimeAdjustmentSeconds;
    }

    /**
     * <p>
     * An integer that represents the queue priority to be applied to the contact (lower priorities are routed
     * preferentially). Cannot be specified if the QueueTimeAdjustmentSeconds is specified. Must be statically defined,
     * must be larger than zero, and a valid integer value. Default Value is 5.
     * </p>
     * 
     * @return An integer that represents the queue priority to be applied to the contact (lower priorities are routed
     *         preferentially). Cannot be specified if the QueueTimeAdjustmentSeconds is specified. Must be statically
     *         defined, must be larger than zero, and a valid integer value. Default Value is 5.
     */
    public final Long queuePriority() {
        return queuePriority;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Tags associated with the contact. This contains both Amazon Web Services generated and user-defined tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Tags associated with the contact. This contains both Amazon Web Services generated and user-defined tags.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(initialContactId());
        hashCode = 31 * hashCode + Objects.hashCode(previousContactId());
        hashCode = 31 * hashCode + Objects.hashCode(initiationMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(channelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(queueInfo());
        hashCode = 31 * hashCode + Objects.hashCode(agentInfo());
        hashCode = 31 * hashCode + Objects.hashCode(initiationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(disconnectTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastPausedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastResumedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(totalPauseCount());
        hashCode = 31 * hashCode + Objects.hashCode(totalPauseDurationInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(scheduledTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(relatedContactId());
        hashCode = 31 * hashCode + Objects.hashCode(wisdomInfo());
        hashCode = 31 * hashCode + Objects.hashCode(queueTimeAdjustmentSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(queuePriority());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Contact)) {
            return false;
        }
        Contact other = (Contact) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(id(), other.id())
                && Objects.equals(initialContactId(), other.initialContactId())
                && Objects.equals(previousContactId(), other.previousContactId())
                && Objects.equals(initiationMethodAsString(), other.initiationMethodAsString())
                && Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(channelAsString(), other.channelAsString()) && Objects.equals(queueInfo(), other.queueInfo())
                && Objects.equals(agentInfo(), other.agentInfo())
                && Objects.equals(initiationTimestamp(), other.initiationTimestamp())
                && Objects.equals(disconnectTimestamp(), other.disconnectTimestamp())
                && Objects.equals(lastUpdateTimestamp(), other.lastUpdateTimestamp())
                && Objects.equals(lastPausedTimestamp(), other.lastPausedTimestamp())
                && Objects.equals(lastResumedTimestamp(), other.lastResumedTimestamp())
                && Objects.equals(totalPauseCount(), other.totalPauseCount())
                && Objects.equals(totalPauseDurationInSeconds(), other.totalPauseDurationInSeconds())
                && Objects.equals(scheduledTimestamp(), other.scheduledTimestamp())
                && Objects.equals(relatedContactId(), other.relatedContactId())
                && Objects.equals(wisdomInfo(), other.wisdomInfo())
                && Objects.equals(queueTimeAdjustmentSeconds(), other.queueTimeAdjustmentSeconds())
                && Objects.equals(queuePriority(), other.queuePriority()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Contact").add("Arn", arn()).add("Id", id()).add("InitialContactId", initialContactId())
                .add("PreviousContactId", previousContactId()).add("InitiationMethod", initiationMethodAsString())
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Channel", channelAsString()).add("QueueInfo", queueInfo()).add("AgentInfo", agentInfo())
                .add("InitiationTimestamp", initiationTimestamp()).add("DisconnectTimestamp", disconnectTimestamp())
                .add("LastUpdateTimestamp", lastUpdateTimestamp()).add("LastPausedTimestamp", lastPausedTimestamp())
                .add("LastResumedTimestamp", lastResumedTimestamp()).add("TotalPauseCount", totalPauseCount())
                .add("TotalPauseDurationInSeconds", totalPauseDurationInSeconds())
                .add("ScheduledTimestamp", scheduledTimestamp()).add("RelatedContactId", relatedContactId())
                .add("WisdomInfo", wisdomInfo()).add("QueueTimeAdjustmentSeconds", queueTimeAdjustmentSeconds())
                .add("QueuePriority", queuePriority()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "InitialContactId":
            return Optional.ofNullable(clazz.cast(initialContactId()));
        case "PreviousContactId":
            return Optional.ofNullable(clazz.cast(previousContactId()));
        case "InitiationMethod":
            return Optional.ofNullable(clazz.cast(initiationMethodAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Channel":
            return Optional.ofNullable(clazz.cast(channelAsString()));
        case "QueueInfo":
            return Optional.ofNullable(clazz.cast(queueInfo()));
        case "AgentInfo":
            return Optional.ofNullable(clazz.cast(agentInfo()));
        case "InitiationTimestamp":
            return Optional.ofNullable(clazz.cast(initiationTimestamp()));
        case "DisconnectTimestamp":
            return Optional.ofNullable(clazz.cast(disconnectTimestamp()));
        case "LastUpdateTimestamp":
            return Optional.ofNullable(clazz.cast(lastUpdateTimestamp()));
        case "LastPausedTimestamp":
            return Optional.ofNullable(clazz.cast(lastPausedTimestamp()));
        case "LastResumedTimestamp":
            return Optional.ofNullable(clazz.cast(lastResumedTimestamp()));
        case "TotalPauseCount":
            return Optional.ofNullable(clazz.cast(totalPauseCount()));
        case "TotalPauseDurationInSeconds":
            return Optional.ofNullable(clazz.cast(totalPauseDurationInSeconds()));
        case "ScheduledTimestamp":
            return Optional.ofNullable(clazz.cast(scheduledTimestamp()));
        case "RelatedContactId":
            return Optional.ofNullable(clazz.cast(relatedContactId()));
        case "WisdomInfo":
            return Optional.ofNullable(clazz.cast(wisdomInfo()));
        case "QueueTimeAdjustmentSeconds":
            return Optional.ofNullable(clazz.cast(queueTimeAdjustmentSeconds()));
        case "QueuePriority":
            return Optional.ofNullable(clazz.cast(queuePriority()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Contact, T> g) {
        return obj -> g.apply((Contact) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Contact> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) for the contact.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) for the contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The identifier for the contact.
         * </p>
         * 
         * @param id
         *        The identifier for the contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * If this contact is related to other contacts, this is the ID of the initial contact.
         * </p>
         * 
         * @param initialContactId
         *        If this contact is related to other contacts, this is the ID of the initial contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initialContactId(String initialContactId);

        /**
         * <p>
         * If this contact is not the first contact, this is the ID of the previous contact.
         * </p>
         * 
         * @param previousContactId
         *        If this contact is not the first contact, this is the ID of the previous contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder previousContactId(String previousContactId);

        /**
         * <p>
         * Indicates how the contact was initiated.
         * </p>
         * 
         * @param initiationMethod
         *        Indicates how the contact was initiated.
         * @see ContactInitiationMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContactInitiationMethod
         */
        Builder initiationMethod(String initiationMethod);

        /**
         * <p>
         * Indicates how the contact was initiated.
         * </p>
         * 
         * @param initiationMethod
         *        Indicates how the contact was initiated.
         * @see ContactInitiationMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ContactInitiationMethod
         */
        Builder initiationMethod(ContactInitiationMethod initiationMethod);

        /**
         * <p>
         * The name of the contact.
         * </p>
         * 
         * @param name
         *        The name of the contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the contact.
         * </p>
         * 
         * @param description
         *        The description of the contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * How the contact reached your contact center.
         * </p>
         * 
         * @param channel
         *        How the contact reached your contact center.
         * @see Channel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Channel
         */
        Builder channel(String channel);

        /**
         * <p>
         * How the contact reached your contact center.
         * </p>
         * 
         * @param channel
         *        How the contact reached your contact center.
         * @see Channel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Channel
         */
        Builder channel(Channel channel);

        /**
         * <p>
         * If this contact was queued, this contains information about the queue.
         * </p>
         * 
         * @param queueInfo
         *        If this contact was queued, this contains information about the queue.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queueInfo(QueueInfo queueInfo);

        /**
         * <p>
         * If this contact was queued, this contains information about the queue.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueueInfo.Builder} avoiding the need to
         * create one manually via {@link QueueInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QueueInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #queueInfo(QueueInfo)}.
         * 
         * @param queueInfo
         *        a consumer that will call methods on {@link QueueInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queueInfo(QueueInfo)
         */
        default Builder queueInfo(Consumer<QueueInfo.Builder> queueInfo) {
            return queueInfo(QueueInfo.builder().applyMutation(queueInfo).build());
        }

        /**
         * <p>
         * Information about the agent who accepted the contact.
         * </p>
         * 
         * @param agentInfo
         *        Information about the agent who accepted the contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentInfo(AgentInfo agentInfo);

        /**
         * <p>
         * Information about the agent who accepted the contact.
         * </p>
         * This is a convenience method that creates an instance of the {@link AgentInfo.Builder} avoiding the need to
         * create one manually via {@link AgentInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AgentInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #agentInfo(AgentInfo)}.
         * 
         * @param agentInfo
         *        a consumer that will call methods on {@link AgentInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #agentInfo(AgentInfo)
         */
        default Builder agentInfo(Consumer<AgentInfo.Builder> agentInfo) {
            return agentInfo(AgentInfo.builder().applyMutation(agentInfo).build());
        }

        /**
         * <p>
         * The date and time this contact was initiated, in UTC time. For <code>INBOUND</code>, this is when the contact
         * arrived. For <code>OUTBOUND</code>, this is when the agent began dialing. For <code>CALLBACK</code>, this is
         * when the callback contact was created. For <code>TRANSFER</code> and <code>QUEUE_TRANSFER</code>, this is
         * when the transfer was initiated. For <code>API</code>, this is when the request arrived. For
         * <code>EXTERNAL_OUTBOUND</code>, this is when the agent started dialing the external participant. For
         * <code>MONITOR</code>, this is when the supervisor started listening to a contact.
         * </p>
         * 
         * @param initiationTimestamp
         *        The date and time this contact was initiated, in UTC time. For <code>INBOUND</code>, this is when the
         *        contact arrived. For <code>OUTBOUND</code>, this is when the agent began dialing. For
         *        <code>CALLBACK</code>, this is when the callback contact was created. For <code>TRANSFER</code> and
         *        <code>QUEUE_TRANSFER</code>, this is when the transfer was initiated. For <code>API</code>, this is
         *        when the request arrived. For <code>EXTERNAL_OUTBOUND</code>, this is when the agent started dialing
         *        the external participant. For <code>MONITOR</code>, this is when the supervisor started listening to a
         *        contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initiationTimestamp(Instant initiationTimestamp);

        /**
         * <p>
         * The timestamp when the customer endpoint disconnected from Amazon Connect.
         * </p>
         * 
         * @param disconnectTimestamp
         *        The timestamp when the customer endpoint disconnected from Amazon Connect.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disconnectTimestamp(Instant disconnectTimestamp);

        /**
         * <p>
         * The timestamp when contact was last updated.
         * </p>
         * 
         * @param lastUpdateTimestamp
         *        The timestamp when contact was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTimestamp(Instant lastUpdateTimestamp);

        /**
         * <p>
         * The timestamp when the contact was last paused.
         * </p>
         * 
         * @param lastPausedTimestamp
         *        The timestamp when the contact was last paused.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastPausedTimestamp(Instant lastPausedTimestamp);

        /**
         * <p>
         * The timestamp when the contact was last resumed.
         * </p>
         * 
         * @param lastResumedTimestamp
         *        The timestamp when the contact was last resumed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastResumedTimestamp(Instant lastResumedTimestamp);

        /**
         * <p>
         * Total pause count for a contact.
         * </p>
         * 
         * @param totalPauseCount
         *        Total pause count for a contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalPauseCount(Integer totalPauseCount);

        /**
         * <p>
         * Total pause duration for a contact in seconds.
         * </p>
         * 
         * @param totalPauseDurationInSeconds
         *        Total pause duration for a contact in seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalPauseDurationInSeconds(Integer totalPauseDurationInSeconds);

        /**
         * <p>
         * The timestamp, in Unix epoch time format, at which to start running the inbound flow.
         * </p>
         * 
         * @param scheduledTimestamp
         *        The timestamp, in Unix epoch time format, at which to start running the inbound flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scheduledTimestamp(Instant scheduledTimestamp);

        /**
         * <p>
         * The contactId that is <a
         * href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-persistence.html#relatedcontactid"
         * >related</a> to this contact.
         * </p>
         * 
         * @param relatedContactId
         *        The contactId that is <a
         *        href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-persistence.html#relatedcontactid"
         *        >related</a> to this contact.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedContactId(String relatedContactId);

        /**
         * <p>
         * Information about Amazon Connect Wisdom.
         * </p>
         * 
         * @param wisdomInfo
         *        Information about Amazon Connect Wisdom.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wisdomInfo(WisdomInfo wisdomInfo);

        /**
         * <p>
         * Information about Amazon Connect Wisdom.
         * </p>
         * This is a convenience method that creates an instance of the {@link WisdomInfo.Builder} avoiding the need to
         * create one manually via {@link WisdomInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link WisdomInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #wisdomInfo(WisdomInfo)}.
         * 
         * @param wisdomInfo
         *        a consumer that will call methods on {@link WisdomInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #wisdomInfo(WisdomInfo)
         */
        default Builder wisdomInfo(Consumer<WisdomInfo.Builder> wisdomInfo) {
            return wisdomInfo(WisdomInfo.builder().applyMutation(wisdomInfo).build());
        }

        /**
         * <p>
         * An integer that represents the queue time adjust to be applied to the contact, in seconds (longer / larger
         * queue time are routed preferentially). Cannot be specified if the QueuePriority is specified. Must be
         * statically defined and a valid integer value.
         * </p>
         * 
         * @param queueTimeAdjustmentSeconds
         *        An integer that represents the queue time adjust to be applied to the contact, in seconds (longer /
         *        larger queue time are routed preferentially). Cannot be specified if the QueuePriority is specified.
         *        Must be statically defined and a valid integer value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queueTimeAdjustmentSeconds(Integer queueTimeAdjustmentSeconds);

        /**
         * <p>
         * An integer that represents the queue priority to be applied to the contact (lower priorities are routed
         * preferentially). Cannot be specified if the QueueTimeAdjustmentSeconds is specified. Must be statically
         * defined, must be larger than zero, and a valid integer value. Default Value is 5.
         * </p>
         * 
         * @param queuePriority
         *        An integer that represents the queue priority to be applied to the contact (lower priorities are
         *        routed preferentially). Cannot be specified if the QueueTimeAdjustmentSeconds is specified. Must be
         *        statically defined, must be larger than zero, and a valid integer value. Default Value is 5.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queuePriority(Long queuePriority);

        /**
         * <p>
         * Tags associated with the contact. This contains both Amazon Web Services generated and user-defined tags.
         * </p>
         * 
         * @param tags
         *        Tags associated with the contact. This contains both Amazon Web Services generated and user-defined
         *        tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String id;

        private String initialContactId;

        private String previousContactId;

        private String initiationMethod;

        private String name;

        private String description;

        private String channel;

        private QueueInfo queueInfo;

        private AgentInfo agentInfo;

        private Instant initiationTimestamp;

        private Instant disconnectTimestamp;

        private Instant lastUpdateTimestamp;

        private Instant lastPausedTimestamp;

        private Instant lastResumedTimestamp;

        private Integer totalPauseCount;

        private Integer totalPauseDurationInSeconds;

        private Instant scheduledTimestamp;

        private String relatedContactId;

        private WisdomInfo wisdomInfo;

        private Integer queueTimeAdjustmentSeconds;

        private Long queuePriority;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Contact model) {
            arn(model.arn);
            id(model.id);
            initialContactId(model.initialContactId);
            previousContactId(model.previousContactId);
            initiationMethod(model.initiationMethod);
            name(model.name);
            description(model.description);
            channel(model.channel);
            queueInfo(model.queueInfo);
            agentInfo(model.agentInfo);
            initiationTimestamp(model.initiationTimestamp);
            disconnectTimestamp(model.disconnectTimestamp);
            lastUpdateTimestamp(model.lastUpdateTimestamp);
            lastPausedTimestamp(model.lastPausedTimestamp);
            lastResumedTimestamp(model.lastResumedTimestamp);
            totalPauseCount(model.totalPauseCount);
            totalPauseDurationInSeconds(model.totalPauseDurationInSeconds);
            scheduledTimestamp(model.scheduledTimestamp);
            relatedContactId(model.relatedContactId);
            wisdomInfo(model.wisdomInfo);
            queueTimeAdjustmentSeconds(model.queueTimeAdjustmentSeconds);
            queuePriority(model.queuePriority);
            tags(model.tags);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getInitialContactId() {
            return initialContactId;
        }

        public final void setInitialContactId(String initialContactId) {
            this.initialContactId = initialContactId;
        }

        @Override
        public final Builder initialContactId(String initialContactId) {
            this.initialContactId = initialContactId;
            return this;
        }

        public final String getPreviousContactId() {
            return previousContactId;
        }

        public final void setPreviousContactId(String previousContactId) {
            this.previousContactId = previousContactId;
        }

        @Override
        public final Builder previousContactId(String previousContactId) {
            this.previousContactId = previousContactId;
            return this;
        }

        public final String getInitiationMethod() {
            return initiationMethod;
        }

        public final void setInitiationMethod(String initiationMethod) {
            this.initiationMethod = initiationMethod;
        }

        @Override
        public final Builder initiationMethod(String initiationMethod) {
            this.initiationMethod = initiationMethod;
            return this;
        }

        @Override
        public final Builder initiationMethod(ContactInitiationMethod initiationMethod) {
            this.initiationMethod(initiationMethod == null ? null : initiationMethod.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getChannel() {
            return channel;
        }

        public final void setChannel(String channel) {
            this.channel = channel;
        }

        @Override
        public final Builder channel(String channel) {
            this.channel = channel;
            return this;
        }

        @Override
        public final Builder channel(Channel channel) {
            this.channel(channel == null ? null : channel.toString());
            return this;
        }

        public final QueueInfo.Builder getQueueInfo() {
            return queueInfo != null ? queueInfo.toBuilder() : null;
        }

        public final void setQueueInfo(QueueInfo.BuilderImpl queueInfo) {
            this.queueInfo = queueInfo != null ? queueInfo.build() : null;
        }

        @Override
        public final Builder queueInfo(QueueInfo queueInfo) {
            this.queueInfo = queueInfo;
            return this;
        }

        public final AgentInfo.Builder getAgentInfo() {
            return agentInfo != null ? agentInfo.toBuilder() : null;
        }

        public final void setAgentInfo(AgentInfo.BuilderImpl agentInfo) {
            this.agentInfo = agentInfo != null ? agentInfo.build() : null;
        }

        @Override
        public final Builder agentInfo(AgentInfo agentInfo) {
            this.agentInfo = agentInfo;
            return this;
        }

        public final Instant getInitiationTimestamp() {
            return initiationTimestamp;
        }

        public final void setInitiationTimestamp(Instant initiationTimestamp) {
            this.initiationTimestamp = initiationTimestamp;
        }

        @Override
        public final Builder initiationTimestamp(Instant initiationTimestamp) {
            this.initiationTimestamp = initiationTimestamp;
            return this;
        }

        public final Instant getDisconnectTimestamp() {
            return disconnectTimestamp;
        }

        public final void setDisconnectTimestamp(Instant disconnectTimestamp) {
            this.disconnectTimestamp = disconnectTimestamp;
        }

        @Override
        public final Builder disconnectTimestamp(Instant disconnectTimestamp) {
            this.disconnectTimestamp = disconnectTimestamp;
            return this;
        }

        public final Instant getLastUpdateTimestamp() {
            return lastUpdateTimestamp;
        }

        public final void setLastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
        }

        @Override
        public final Builder lastUpdateTimestamp(Instant lastUpdateTimestamp) {
            this.lastUpdateTimestamp = lastUpdateTimestamp;
            return this;
        }

        public final Instant getLastPausedTimestamp() {
            return lastPausedTimestamp;
        }

        public final void setLastPausedTimestamp(Instant lastPausedTimestamp) {
            this.lastPausedTimestamp = lastPausedTimestamp;
        }

        @Override
        public final Builder lastPausedTimestamp(Instant lastPausedTimestamp) {
            this.lastPausedTimestamp = lastPausedTimestamp;
            return this;
        }

        public final Instant getLastResumedTimestamp() {
            return lastResumedTimestamp;
        }

        public final void setLastResumedTimestamp(Instant lastResumedTimestamp) {
            this.lastResumedTimestamp = lastResumedTimestamp;
        }

        @Override
        public final Builder lastResumedTimestamp(Instant lastResumedTimestamp) {
            this.lastResumedTimestamp = lastResumedTimestamp;
            return this;
        }

        public final Integer getTotalPauseCount() {
            return totalPauseCount;
        }

        public final void setTotalPauseCount(Integer totalPauseCount) {
            this.totalPauseCount = totalPauseCount;
        }

        @Override
        public final Builder totalPauseCount(Integer totalPauseCount) {
            this.totalPauseCount = totalPauseCount;
            return this;
        }

        public final Integer getTotalPauseDurationInSeconds() {
            return totalPauseDurationInSeconds;
        }

        public final void setTotalPauseDurationInSeconds(Integer totalPauseDurationInSeconds) {
            this.totalPauseDurationInSeconds = totalPauseDurationInSeconds;
        }

        @Override
        public final Builder totalPauseDurationInSeconds(Integer totalPauseDurationInSeconds) {
            this.totalPauseDurationInSeconds = totalPauseDurationInSeconds;
            return this;
        }

        public final Instant getScheduledTimestamp() {
            return scheduledTimestamp;
        }

        public final void setScheduledTimestamp(Instant scheduledTimestamp) {
            this.scheduledTimestamp = scheduledTimestamp;
        }

        @Override
        public final Builder scheduledTimestamp(Instant scheduledTimestamp) {
            this.scheduledTimestamp = scheduledTimestamp;
            return this;
        }

        public final String getRelatedContactId() {
            return relatedContactId;
        }

        public final void setRelatedContactId(String relatedContactId) {
            this.relatedContactId = relatedContactId;
        }

        @Override
        public final Builder relatedContactId(String relatedContactId) {
            this.relatedContactId = relatedContactId;
            return this;
        }

        public final WisdomInfo.Builder getWisdomInfo() {
            return wisdomInfo != null ? wisdomInfo.toBuilder() : null;
        }

        public final void setWisdomInfo(WisdomInfo.BuilderImpl wisdomInfo) {
            this.wisdomInfo = wisdomInfo != null ? wisdomInfo.build() : null;
        }

        @Override
        public final Builder wisdomInfo(WisdomInfo wisdomInfo) {
            this.wisdomInfo = wisdomInfo;
            return this;
        }

        public final Integer getQueueTimeAdjustmentSeconds() {
            return queueTimeAdjustmentSeconds;
        }

        public final void setQueueTimeAdjustmentSeconds(Integer queueTimeAdjustmentSeconds) {
            this.queueTimeAdjustmentSeconds = queueTimeAdjustmentSeconds;
        }

        @Override
        public final Builder queueTimeAdjustmentSeconds(Integer queueTimeAdjustmentSeconds) {
            this.queueTimeAdjustmentSeconds = queueTimeAdjustmentSeconds;
            return this;
        }

        public final Long getQueuePriority() {
            return queuePriority;
        }

        public final void setQueuePriority(Long queuePriority) {
            this.queuePriority = queuePriority;
        }

        @Override
        public final Builder queuePriority(Long queuePriority) {
            this.queuePriority = queuePriority;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = ContactTagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = ContactTagMapCopier.copy(tags);
            return this;
        }

        @Override
        public Contact build() {
            return new Contact(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
