/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Fields required when uploading an attached file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UploadUrlMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<UploadUrlMetadata.Builder, UploadUrlMetadata> {
    private static final SdkField<String> URL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Url")
            .getter(getter(UploadUrlMetadata::url)).setter(setter(Builder::url))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Url").build()).build();

    private static final SdkField<String> URL_EXPIRY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UrlExpiry").getter(getter(UploadUrlMetadata::urlExpiry)).setter(setter(Builder::urlExpiry))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UrlExpiry").build()).build();

    private static final SdkField<Map<String, String>> HEADERS_TO_INCLUDE_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("HeadersToInclude")
            .getter(getter(UploadUrlMetadata::headersToInclude))
            .setter(setter(Builder::headersToInclude))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeadersToInclude").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(URL_FIELD, URL_EXPIRY_FIELD,
            HEADERS_TO_INCLUDE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String url;

    private final String urlExpiry;

    private final Map<String, String> headersToInclude;

    private UploadUrlMetadata(BuilderImpl builder) {
        this.url = builder.url;
        this.urlExpiry = builder.urlExpiry;
        this.headersToInclude = builder.headersToInclude;
    }

    /**
     * <p>
     * A pre-signed S3 URL that should be used for uploading the attached file.
     * </p>
     * 
     * @return A pre-signed S3 URL that should be used for uploading the attached file.
     */
    public final String url() {
        return url;
    }

    /**
     * <p>
     * The expiration time of the URL in ISO timestamp. It's specified in ISO 8601 format:
     * <code>yyyy-MM-ddThh:mm:ss.SSSZ</code>. For example, <code>2019-11-08T02:41:28.172Z</code>.
     * </p>
     * 
     * @return The expiration time of the URL in ISO timestamp. It's specified in ISO 8601 format:
     *         <code>yyyy-MM-ddThh:mm:ss.SSSZ</code>. For example, <code>2019-11-08T02:41:28.172Z</code>.
     */
    public final String urlExpiry() {
        return urlExpiry;
    }

    /**
     * For responses, this returns true if the service returned a value for the HeadersToInclude property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasHeadersToInclude() {
        return headersToInclude != null && !(headersToInclude instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of headers that should be provided when uploading the attached file.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasHeadersToInclude} method.
     * </p>
     * 
     * @return A map of headers that should be provided when uploading the attached file.
     */
    public final Map<String, String> headersToInclude() {
        return headersToInclude;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(url());
        hashCode = 31 * hashCode + Objects.hashCode(urlExpiry());
        hashCode = 31 * hashCode + Objects.hashCode(hasHeadersToInclude() ? headersToInclude() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UploadUrlMetadata)) {
            return false;
        }
        UploadUrlMetadata other = (UploadUrlMetadata) obj;
        return Objects.equals(url(), other.url()) && Objects.equals(urlExpiry(), other.urlExpiry())
                && hasHeadersToInclude() == other.hasHeadersToInclude()
                && Objects.equals(headersToInclude(), other.headersToInclude());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UploadUrlMetadata").add("Url", url()).add("UrlExpiry", urlExpiry())
                .add("HeadersToInclude", hasHeadersToInclude() ? headersToInclude() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Url":
            return Optional.ofNullable(clazz.cast(url()));
        case "UrlExpiry":
            return Optional.ofNullable(clazz.cast(urlExpiry()));
        case "HeadersToInclude":
            return Optional.ofNullable(clazz.cast(headersToInclude()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Url", URL_FIELD);
        map.put("UrlExpiry", URL_EXPIRY_FIELD);
        map.put("HeadersToInclude", HEADERS_TO_INCLUDE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UploadUrlMetadata, T> g) {
        return obj -> g.apply((UploadUrlMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UploadUrlMetadata> {
        /**
         * <p>
         * A pre-signed S3 URL that should be used for uploading the attached file.
         * </p>
         * 
         * @param url
         *        A pre-signed S3 URL that should be used for uploading the attached file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder url(String url);

        /**
         * <p>
         * The expiration time of the URL in ISO timestamp. It's specified in ISO 8601 format:
         * <code>yyyy-MM-ddThh:mm:ss.SSSZ</code>. For example, <code>2019-11-08T02:41:28.172Z</code>.
         * </p>
         * 
         * @param urlExpiry
         *        The expiration time of the URL in ISO timestamp. It's specified in ISO 8601 format:
         *        <code>yyyy-MM-ddThh:mm:ss.SSSZ</code>. For example, <code>2019-11-08T02:41:28.172Z</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder urlExpiry(String urlExpiry);

        /**
         * <p>
         * A map of headers that should be provided when uploading the attached file.
         * </p>
         * 
         * @param headersToInclude
         *        A map of headers that should be provided when uploading the attached file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headersToInclude(Map<String, String> headersToInclude);
    }

    static final class BuilderImpl implements Builder {
        private String url;

        private String urlExpiry;

        private Map<String, String> headersToInclude = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UploadUrlMetadata model) {
            url(model.url);
            urlExpiry(model.urlExpiry);
            headersToInclude(model.headersToInclude);
        }

        public final String getUrl() {
            return url;
        }

        public final void setUrl(String url) {
            this.url = url;
        }

        @Override
        public final Builder url(String url) {
            this.url = url;
            return this;
        }

        public final String getUrlExpiry() {
            return urlExpiry;
        }

        public final void setUrlExpiry(String urlExpiry) {
            this.urlExpiry = urlExpiry;
        }

        @Override
        public final Builder urlExpiry(String urlExpiry) {
            this.urlExpiry = urlExpiry;
            return this;
        }

        public final Map<String, String> getHeadersToInclude() {
            if (headersToInclude instanceof SdkAutoConstructMap) {
                return null;
            }
            return headersToInclude;
        }

        public final void setHeadersToInclude(Map<String, String> headersToInclude) {
            this.headersToInclude = UrlMetadataSignedHeadersCopier.copy(headersToInclude);
        }

        @Override
        public final Builder headersToInclude(Map<String, String> headersToInclude) {
            this.headersToInclude = UrlMetadataSignedHeadersCopier.copy(headersToInclude);
            return this;
        }

        @Override
        public UploadUrlMetadata build() {
            return new UploadUrlMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
