/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connect.model.ConnectException;
import software.amazon.awssdk.services.connect.model.ConnectRequest;
import software.amazon.awssdk.services.connect.model.ContactNotFoundException;
import software.amazon.awssdk.services.connect.model.CreateUserRequest;
import software.amazon.awssdk.services.connect.model.CreateUserResponse;
import software.amazon.awssdk.services.connect.model.DeleteUserRequest;
import software.amazon.awssdk.services.connect.model.DeleteUserResponse;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyGroupRequest;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyGroupResponse;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyStructureRequest;
import software.amazon.awssdk.services.connect.model.DescribeUserHierarchyStructureResponse;
import software.amazon.awssdk.services.connect.model.DescribeUserRequest;
import software.amazon.awssdk.services.connect.model.DescribeUserResponse;
import software.amazon.awssdk.services.connect.model.DestinationNotAllowedException;
import software.amazon.awssdk.services.connect.model.DuplicateResourceException;
import software.amazon.awssdk.services.connect.model.GetContactAttributesRequest;
import software.amazon.awssdk.services.connect.model.GetContactAttributesResponse;
import software.amazon.awssdk.services.connect.model.GetCurrentMetricDataRequest;
import software.amazon.awssdk.services.connect.model.GetCurrentMetricDataResponse;
import software.amazon.awssdk.services.connect.model.GetFederationTokenRequest;
import software.amazon.awssdk.services.connect.model.GetFederationTokenResponse;
import software.amazon.awssdk.services.connect.model.GetMetricDataRequest;
import software.amazon.awssdk.services.connect.model.GetMetricDataResponse;
import software.amazon.awssdk.services.connect.model.InternalServiceException;
import software.amazon.awssdk.services.connect.model.InvalidParameterException;
import software.amazon.awssdk.services.connect.model.InvalidRequestException;
import software.amazon.awssdk.services.connect.model.LimitExceededException;
import software.amazon.awssdk.services.connect.model.ListRoutingProfilesRequest;
import software.amazon.awssdk.services.connect.model.ListRoutingProfilesResponse;
import software.amazon.awssdk.services.connect.model.ListSecurityProfilesRequest;
import software.amazon.awssdk.services.connect.model.ListSecurityProfilesResponse;
import software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsRequest;
import software.amazon.awssdk.services.connect.model.ListUserHierarchyGroupsResponse;
import software.amazon.awssdk.services.connect.model.ListUsersRequest;
import software.amazon.awssdk.services.connect.model.ListUsersResponse;
import software.amazon.awssdk.services.connect.model.OutboundContactNotPermittedException;
import software.amazon.awssdk.services.connect.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connect.model.StartOutboundVoiceContactRequest;
import software.amazon.awssdk.services.connect.model.StartOutboundVoiceContactResponse;
import software.amazon.awssdk.services.connect.model.StopContactRequest;
import software.amazon.awssdk.services.connect.model.StopContactResponse;
import software.amazon.awssdk.services.connect.model.ThrottlingException;
import software.amazon.awssdk.services.connect.model.UpdateContactAttributesRequest;
import software.amazon.awssdk.services.connect.model.UpdateContactAttributesResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserHierarchyRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserHierarchyResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserIdentityInfoRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserIdentityInfoResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserPhoneConfigRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserPhoneConfigResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserRoutingProfileRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserRoutingProfileResponse;
import software.amazon.awssdk.services.connect.model.UpdateUserSecurityProfilesRequest;
import software.amazon.awssdk.services.connect.model.UpdateUserSecurityProfilesResponse;
import software.amazon.awssdk.services.connect.model.UserNotFoundException;
import software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable;
import software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable;
import software.amazon.awssdk.services.connect.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DescribeUserHierarchyGroupRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DescribeUserHierarchyStructureRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetContactAttributesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetCurrentMetricDataRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetFederationTokenRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.GetMetricDataRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListRoutingProfilesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListSecurityProfilesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListUserHierarchyGroupsRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StartOutboundVoiceContactRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.StopContactRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateContactAttributesRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserHierarchyRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserIdentityInfoRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserPhoneConfigRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserRoutingProfileRequestMarshaller;
import software.amazon.awssdk.services.connect.transform.UpdateUserSecurityProfilesRequestMarshaller;

/**
 * Internal implementation of {@link ConnectClient}.
 *
 * @see ConnectClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectClient implements ConnectClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultConnectClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new user account in your Amazon Connect instance.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws LimitExceededException
     *         The allowed limit for the resource has been reached.
     * @throws DuplicateResourceException
     *         A resource with that name already exists.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.CreateUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws InvalidRequestException,
            InvalidParameterException, LimitExceededException, DuplicateResourceException, ResourceNotFoundException,
            ThrottlingException, InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>()
                .withOperationName("CreateUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createUserRequest)
                .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a user account from Amazon Connect.
     * </p>
     *
     * @param deleteUserRequest
     * @return Result of the DeleteUser operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DeleteUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteUserResponse deleteUser(DeleteUserRequest deleteUserRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>()
                .withOperationName("DeleteUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteUserRequest)
                .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a <code>User</code> object that contains information about the user account specified by the
     * <code>UserId</code>.
     * </p>
     *
     * @param describeUserRequest
     * @return Result of the DescribeUser operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DescribeUser
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeUserResponse describeUser(DescribeUserRequest describeUserRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                .withOperationName("DescribeUser").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeUserRequest)
                .withMarshaller(new DescribeUserRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a <code>HierarchyGroup</code> object that includes information about a hierarchy group in your instance.
     * </p>
     *
     * @param describeUserHierarchyGroupRequest
     * @return Result of the DescribeUserHierarchyGroup operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DescribeUserHierarchyGroup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DescribeUserHierarchyGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeUserHierarchyGroupResponse describeUserHierarchyGroup(
            DescribeUserHierarchyGroupRequest describeUserHierarchyGroupRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserHierarchyGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeUserHierarchyGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeUserHierarchyGroupRequest, DescribeUserHierarchyGroupResponse>()
                        .withOperationName("DescribeUserHierarchyGroup").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeUserHierarchyGroupRequest)
                        .withMarshaller(new DescribeUserHierarchyGroupRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a <code>HiearchyGroupStructure</code> object, which contains data about the levels in the agent
     * hierarchy.
     * </p>
     *
     * @param describeUserHierarchyStructureRequest
     * @return Result of the DescribeUserHierarchyStructure operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.DescribeUserHierarchyStructure
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/DescribeUserHierarchyStructure"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeUserHierarchyStructureResponse describeUserHierarchyStructure(
            DescribeUserHierarchyStructureRequest describeUserHierarchyStructureRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserHierarchyStructureResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeUserHierarchyStructureResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeUserHierarchyStructureRequest, DescribeUserHierarchyStructureResponse>()
                        .withOperationName("DescribeUserHierarchyStructure").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeUserHierarchyStructureRequest)
                        .withMarshaller(new DescribeUserHierarchyStructureRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the contact attributes associated with a contact.
     * </p>
     *
     * @param getContactAttributesRequest
     * @return Result of the GetContactAttributes operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetContactAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetContactAttributes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetContactAttributesResponse getContactAttributes(GetContactAttributesRequest getContactAttributesRequest)
            throws InvalidRequestException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetContactAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetContactAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetContactAttributesRequest, GetContactAttributesResponse>()
                .withOperationName("GetContactAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getContactAttributesRequest)
                .withMarshaller(new GetContactAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>GetCurrentMetricData</code> operation retrieves current metric data from your Amazon Connect instance.
     * </p>
     * <p>
     * If you are using an IAM account, it must have permission to the <code>connect:GetCurrentMetricData</code> action.
     * </p>
     *
     * @param getCurrentMetricDataRequest
     * @return Result of the GetCurrentMetricData operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetCurrentMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetCurrentMetricData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCurrentMetricDataResponse getCurrentMetricData(GetCurrentMetricDataRequest getCurrentMetricDataRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCurrentMetricDataResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCurrentMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetCurrentMetricDataRequest, GetCurrentMetricDataResponse>()
                .withOperationName("GetCurrentMetricData").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getCurrentMetricDataRequest)
                .withMarshaller(new GetCurrentMetricDataRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>GetCurrentMetricData</code> operation retrieves current metric data from your Amazon Connect instance.
     * </p>
     * <p>
     * If you are using an IAM account, it must have permission to the <code>connect:GetCurrentMetricData</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getCurrentMetricData(software.amazon.awssdk.services.connect.model.GetCurrentMetricDataRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable responses = client.getCurrentMetricDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable responses = client
     *             .getCurrentMetricDataPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.GetCurrentMetricDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetCurrentMetricDataIterable responses = client.getCurrentMetricDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getCurrentMetricData(software.amazon.awssdk.services.connect.model.GetCurrentMetricDataRequest)}
     * operation.</b>
     * </p>
     *
     * @param getCurrentMetricDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetCurrentMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetCurrentMetricData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetCurrentMetricDataIterable getCurrentMetricDataPaginator(GetCurrentMetricDataRequest getCurrentMetricDataRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ConnectException {
        return new GetCurrentMetricDataIterable(this, applyPaginatorUserAgent(getCurrentMetricDataRequest));
    }

    /**
     * <p>
     * Retrieves a token for federation.
     * </p>
     *
     * @param getFederationTokenRequest
     * @return Result of the GetFederationToken operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws UserNotFoundException
     *         No user with the specified credentials was found in the Amazon Connect instance.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws DuplicateResourceException
     *         A resource with that name already exists.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetFederationToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetFederationToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetFederationTokenResponse getFederationToken(GetFederationTokenRequest getFederationTokenRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, UserNotFoundException,
            InternalServiceException, DuplicateResourceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFederationTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetFederationTokenResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetFederationTokenRequest, GetFederationTokenResponse>()
                .withOperationName("GetFederationToken").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getFederationTokenRequest)
                .withMarshaller(new GetFederationTokenRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>GetMetricData</code> operation retrieves historical metrics data from your Amazon Connect instance.
     * </p>
     * <p>
     * If you are using an IAM account, it must have permission to the <code>connect:GetMetricData</code> action.
     * </p>
     *
     * @param getMetricDataRequest
     * @return Result of the GetMetricData operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataResponse getMetricData(GetMetricDataRequest getMetricDataRequest) throws InvalidRequestException,
            InvalidParameterException, InternalServiceException, ThrottlingException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMetricDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetMetricDataRequest, GetMetricDataResponse>()
                .withOperationName("GetMetricData").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMetricDataRequest)
                .withMarshaller(new GetMetricDataRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>GetMetricData</code> operation retrieves historical metrics data from your Amazon Connect instance.
     * </p>
     * <p>
     * If you are using an IAM account, it must have permission to the <code>connect:GetMetricData</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getMetricData(software.amazon.awssdk.services.connect.model.GetMetricDataRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     *     for (software.amazon.awssdk.services.connect.model.GetMetricDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connect.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getMetricData(software.amazon.awssdk.services.connect.model.GetMetricDataRequest)} operation.</b>
     * </p>
     *
     * @param getMetricDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.GetMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataIterable getMetricDataPaginator(GetMetricDataRequest getMetricDataRequest)
            throws InvalidRequestException, InvalidParameterException, InternalServiceException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, ConnectException {
        return new GetMetricDataIterable(this, applyPaginatorUserAgent(getMetricDataRequest));
    }

    /**
     * <p>
     * Returns an array of <code>RoutingProfileSummary</code> objects that includes information about the routing
     * profiles in your instance.
     * </p>
     *
     * @param listRoutingProfilesRequest
     * @return Result of the ListRoutingProfiles operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListRoutingProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListRoutingProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListRoutingProfilesResponse listRoutingProfiles(ListRoutingProfilesRequest listRoutingProfilesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRoutingProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRoutingProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListRoutingProfilesRequest, ListRoutingProfilesResponse>()
                .withOperationName("ListRoutingProfiles").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listRoutingProfilesRequest)
                .withMarshaller(new ListRoutingProfilesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns an array of SecurityProfileSummary objects that contain information about the security profiles in your
     * instance, including the ARN, Id, and Name of the security profile.
     * </p>
     *
     * @param listSecurityProfilesRequest
     * @return Result of the ListSecurityProfiles operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListSecurityProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListSecurityProfiles" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListSecurityProfilesResponse listSecurityProfiles(ListSecurityProfilesRequest listSecurityProfilesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSecurityProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSecurityProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListSecurityProfilesRequest, ListSecurityProfilesResponse>()
                .withOperationName("ListSecurityProfiles").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSecurityProfilesRequest)
                .withMarshaller(new ListSecurityProfilesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a <code>UserHierarchyGroupSummaryList</code>, which is an array of <code>HierarchyGroupSummary</code>
     * objects that contain information about the hierarchy groups in your instance.
     * </p>
     *
     * @param listUserHierarchyGroupsRequest
     * @return Result of the ListUserHierarchyGroups operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListUserHierarchyGroups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListUserHierarchyGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListUserHierarchyGroupsResponse listUserHierarchyGroups(ListUserHierarchyGroupsRequest listUserHierarchyGroupsRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUserHierarchyGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListUserHierarchyGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListUserHierarchyGroupsRequest, ListUserHierarchyGroupsResponse>()
                .withOperationName("ListUserHierarchyGroups").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listUserHierarchyGroupsRequest)
                .withMarshaller(new ListUserHierarchyGroupsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a <code>UserSummaryList</code>, which is an array of <code>UserSummary</code> objects.
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.ListUsers
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                .withOperationName("ListUsers").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listUsersRequest)
                .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>StartOutboundVoiceContact</code> operation initiates a contact flow to place an outbound call to a
     * customer.
     * </p>
     * <p>
     * If you are using an IAM account, it must have permission to the <code>connect:StartOutboundVoiceContact</code>
     * action.
     * </p>
     * <p>
     * There is a 60 second dialing timeout for this operation. If the call is not connected after 60 seconds, the call
     * fails.
     * </p>
     *
     * @param startOutboundVoiceContactRequest
     * @return Result of the StartOutboundVoiceContact operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws LimitExceededException
     *         The allowed limit for the resource has been reached.
     * @throws DestinationNotAllowedException
     *         Outbound calls to the destination number are not allowed.
     * @throws OutboundContactNotPermittedException
     *         The contact is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StartOutboundVoiceContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StartOutboundVoiceContact"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartOutboundVoiceContactResponse startOutboundVoiceContact(
            StartOutboundVoiceContactRequest startOutboundVoiceContactRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, InternalServiceException, LimitExceededException,
            DestinationNotAllowedException, OutboundContactNotPermittedException, AwsServiceException, SdkClientException,
            ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartOutboundVoiceContactResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartOutboundVoiceContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<StartOutboundVoiceContactRequest, StartOutboundVoiceContactResponse>()
                        .withOperationName("StartOutboundVoiceContact").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(startOutboundVoiceContactRequest)
                        .withMarshaller(new StartOutboundVoiceContactRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Ends the contact initiated by the <code>StartOutboundVoiceContact</code> operation.
     * </p>
     * <p>
     * If you are using an IAM account, it must have permission to the <code>connect:StopContact</code> action.
     * </p>
     *
     * @param stopContactRequest
     * @return Result of the StopContact operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws ContactNotFoundException
     *         The contact with the specified ID is not active or does not exist.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.StopContact
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/StopContact" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopContactResponse stopContact(StopContactRequest stopContactRequest) throws InvalidRequestException,
            ContactNotFoundException, InvalidParameterException, ResourceNotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopContactResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopContactResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<StopContactRequest, StopContactResponse>()
                .withOperationName("StopContact").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(stopContactRequest)
                .withMarshaller(new StopContactRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * The <code>UpdateContactAttributes</code> operation lets you programmatically create new, or update existing,
     * contact attributes associated with a contact. You can use the operation to add or update attributes for both
     * ongoing and completed contacts. For example, you can update the customer's name or the reason the customer called
     * while the call is active, or add notes about steps that the agent took during the call that are displayed to the
     * next agent that takes the call. You can also use the <code>UpdateContactAttributes</code> operation to update
     * attributes for a contact using data from your CRM application and save the data with the contact in Amazon
     * Connect. You could also flag calls for additional analysis, such as legal review or identifying abusive callers.
     * </p>
     * <p>
     * Contact attributes are available in Amazon Connect for 24 months, and are then deleted.
     * </p>
     * <p>
     * <i>Important:</i>
     * </p>
     * <p>
     * You cannot use the operation to update attributes for contacts that occurred prior to the release of the API,
     * September 12, 2018. You can update attributes only for contacts that started after the release of the API. If you
     * attempt to update attributes for a contact that occurred prior to the release of the API, a 400 error is
     * returned. This applies also to queued callbacks that were initiated prior to the release of the API but are still
     * active in your instance.
     * </p>
     *
     * @param updateContactAttributesRequest
     * @return Result of the UpdateContactAttributes operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateContactAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateContactAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContactAttributesResponse updateContactAttributes(UpdateContactAttributesRequest updateContactAttributesRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContactAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContactAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateContactAttributesRequest, UpdateContactAttributesResponse>()
                .withOperationName("UpdateContactAttributes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateContactAttributesRequest)
                .withMarshaller(new UpdateContactAttributesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Assigns the specified hierarchy group to the user.
     * </p>
     *
     * @param updateUserHierarchyRequest
     * @return Result of the UpdateUserHierarchy operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserHierarchy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserHierarchy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserHierarchyResponse updateUserHierarchy(UpdateUserHierarchyRequest updateUserHierarchyRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserHierarchyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserHierarchyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateUserHierarchyRequest, UpdateUserHierarchyResponse>()
                .withOperationName("UpdateUserHierarchy").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateUserHierarchyRequest)
                .withMarshaller(new UpdateUserHierarchyRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the identity information for the specified user in a <code>UserIdentityInfo</code> object, including
     * email, first name, and last name.
     * </p>
     *
     * @param updateUserIdentityInfoRequest
     * @return Result of the UpdateUserIdentityInfo operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserIdentityInfo
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserIdentityInfo" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserIdentityInfoResponse updateUserIdentityInfo(UpdateUserIdentityInfoRequest updateUserIdentityInfoRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserIdentityInfoResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserIdentityInfoResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateUserIdentityInfoRequest, UpdateUserIdentityInfoResponse>()
                .withOperationName("UpdateUserIdentityInfo").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateUserIdentityInfoRequest)
                .withMarshaller(new UpdateUserIdentityInfoRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the phone configuration settings in the <code>UserPhoneConfig</code> object for the specified user.
     * </p>
     *
     * @param updateUserPhoneConfigRequest
     * @return Result of the UpdateUserPhoneConfig operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserPhoneConfig
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserPhoneConfig" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserPhoneConfigResponse updateUserPhoneConfig(UpdateUserPhoneConfigRequest updateUserPhoneConfigRequest)
            throws InvalidRequestException, InvalidParameterException, ResourceNotFoundException, ThrottlingException,
            InternalServiceException, AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserPhoneConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserPhoneConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateUserPhoneConfigRequest, UpdateUserPhoneConfigResponse>()
                .withOperationName("UpdateUserPhoneConfig").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateUserPhoneConfigRequest)
                .withMarshaller(new UpdateUserPhoneConfigRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Assigns the specified routing profile to a user.
     * </p>
     *
     * @param updateUserRoutingProfileRequest
     * @return Result of the UpdateUserRoutingProfile operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserRoutingProfile
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserRoutingProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateUserRoutingProfileResponse updateUserRoutingProfile(
            UpdateUserRoutingProfileRequest updateUserRoutingProfileRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserRoutingProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserRoutingProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateUserRoutingProfileRequest, UpdateUserRoutingProfileResponse>()
                        .withOperationName("UpdateUserRoutingProfile").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateUserRoutingProfileRequest)
                        .withMarshaller(new UpdateUserRoutingProfileRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates the security profiles assigned to the user.
     * </p>
     *
     * @param updateUserSecurityProfilesRequest
     * @return Result of the UpdateUserSecurityProfiles operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws InvalidParameterException
     *         One or more of the parameters provided to the operation are not valid.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectClient.UpdateUserSecurityProfiles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connect-2017-08-08/UpdateUserSecurityProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateUserSecurityProfilesResponse updateUserSecurityProfiles(
            UpdateUserSecurityProfilesRequest updateUserSecurityProfilesRequest) throws InvalidRequestException,
            InvalidParameterException, ResourceNotFoundException, ThrottlingException, InternalServiceException,
            AwsServiceException, SdkClientException, ConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserSecurityProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserSecurityProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateUserSecurityProfilesRequest, UpdateUserSecurityProfilesResponse>()
                        .withOperationName("UpdateUserSecurityProfiles").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateUserSecurityProfilesRequest)
                        .withMarshaller(new UpdateUserSecurityProfilesRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserNotFoundException")
                                .exceptionBuilderSupplier(UserNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OutboundContactNotPermittedException")
                                .exceptionBuilderSupplier(OutboundContactNotPermittedException::builder).httpStatusCode(403)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DestinationNotAllowedException")
                                .exceptionBuilderSupplier(DestinationNotAllowedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContactNotFoundException")
                                .exceptionBuilderSupplier(ContactNotFoundException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateResourceException")
                                .exceptionBuilderSupplier(DuplicateResourceException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(429).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ConnectRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
