/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectcontactlens;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connectcontactlens.model.AccessDeniedException;
import software.amazon.awssdk.services.connectcontactlens.model.ConnectContactLensException;
import software.amazon.awssdk.services.connectcontactlens.model.ConnectContactLensRequest;
import software.amazon.awssdk.services.connectcontactlens.model.InternalServiceException;
import software.amazon.awssdk.services.connectcontactlens.model.InvalidRequestException;
import software.amazon.awssdk.services.connectcontactlens.model.ListRealtimeContactAnalysisSegmentsRequest;
import software.amazon.awssdk.services.connectcontactlens.model.ListRealtimeContactAnalysisSegmentsResponse;
import software.amazon.awssdk.services.connectcontactlens.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connectcontactlens.model.ThrottlingException;
import software.amazon.awssdk.services.connectcontactlens.paginators.ListRealtimeContactAnalysisSegmentsIterable;
import software.amazon.awssdk.services.connectcontactlens.transform.ListRealtimeContactAnalysisSegmentsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ConnectContactLensClient}.
 *
 * @see ConnectContactLensClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectContactLensClient implements ConnectContactLensClient {
    private static final Logger log = Logger.loggerFor(DefaultConnectContactLensClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ConnectContactLensServiceClientConfiguration serviceClientConfiguration;

    protected DefaultConnectContactLensClient(ConnectContactLensServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Provides a list of analysis segments for a real-time analysis session.
     * </p>
     *
     * @param listRealtimeContactAnalysisSegmentsRequest
     * @return Result of the ListRealtimeContactAnalysisSegments operation returned by the service.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectContactLensException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectContactLensClient.ListRealtimeContactAnalysisSegments
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/connect-contact-lens-2020-08-21/ListRealtimeContactAnalysisSegments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRealtimeContactAnalysisSegmentsResponse listRealtimeContactAnalysisSegments(
            ListRealtimeContactAnalysisSegmentsRequest listRealtimeContactAnalysisSegmentsRequest)
            throws InvalidRequestException, AccessDeniedException, ResourceNotFoundException, InternalServiceException,
            ThrottlingException, AwsServiceException, SdkClientException, ConnectContactLensException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRealtimeContactAnalysisSegmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRealtimeContactAnalysisSegmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listRealtimeContactAnalysisSegmentsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Connect Contact Lens");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRealtimeContactAnalysisSegments");

            return clientHandler
                    .execute(new ClientExecutionParams<ListRealtimeContactAnalysisSegmentsRequest, ListRealtimeContactAnalysisSegmentsResponse>()
                            .withOperationName("ListRealtimeContactAnalysisSegments").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listRealtimeContactAnalysisSegmentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListRealtimeContactAnalysisSegmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of analysis segments for a real-time analysis session.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRealtimeContactAnalysisSegments(software.amazon.awssdk.services.connectcontactlens.model.ListRealtimeContactAnalysisSegmentsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcontactlens.paginators.ListRealtimeContactAnalysisSegmentsIterable responses = client.listRealtimeContactAnalysisSegmentsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.connectcontactlens.paginators.ListRealtimeContactAnalysisSegmentsIterable responses = client
     *             .listRealtimeContactAnalysisSegmentsPaginator(request);
     *     for (software.amazon.awssdk.services.connectcontactlens.model.ListRealtimeContactAnalysisSegmentsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectcontactlens.paginators.ListRealtimeContactAnalysisSegmentsIterable responses = client.listRealtimeContactAnalysisSegmentsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRealtimeContactAnalysisSegments(software.amazon.awssdk.services.connectcontactlens.model.ListRealtimeContactAnalysisSegmentsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRealtimeContactAnalysisSegmentsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidRequestException
     *         The request is not valid.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The specified resource was not found.
     * @throws InternalServiceException
     *         Request processing failed due to an error or failure with the service.
     * @throws ThrottlingException
     *         The throttling limit has been exceeded.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectContactLensException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectContactLensClient.ListRealtimeContactAnalysisSegments
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/connect-contact-lens-2020-08-21/ListRealtimeContactAnalysisSegments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRealtimeContactAnalysisSegmentsIterable listRealtimeContactAnalysisSegmentsPaginator(
            ListRealtimeContactAnalysisSegmentsRequest listRealtimeContactAnalysisSegmentsRequest)
            throws InvalidRequestException, AccessDeniedException, ResourceNotFoundException, InternalServiceException,
            ThrottlingException, AwsServiceException, SdkClientException, ConnectContactLensException {
        return new ListRealtimeContactAnalysisSegmentsIterable(this,
                applyPaginatorUserAgent(listRealtimeContactAnalysisSegmentsRequest));
    }

    private <T extends ConnectContactLensRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectContactLensException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public final ConnectContactLensServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
