/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectparticipant;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connectparticipant.model.AccessDeniedException;
import software.amazon.awssdk.services.connectparticipant.model.ConnectParticipantException;
import software.amazon.awssdk.services.connectparticipant.model.ConnectParticipantRequest;
import software.amazon.awssdk.services.connectparticipant.model.CreateParticipantConnectionRequest;
import software.amazon.awssdk.services.connectparticipant.model.CreateParticipantConnectionResponse;
import software.amazon.awssdk.services.connectparticipant.model.DisconnectParticipantRequest;
import software.amazon.awssdk.services.connectparticipant.model.DisconnectParticipantResponse;
import software.amazon.awssdk.services.connectparticipant.model.GetTranscriptRequest;
import software.amazon.awssdk.services.connectparticipant.model.GetTranscriptResponse;
import software.amazon.awssdk.services.connectparticipant.model.InternalServerException;
import software.amazon.awssdk.services.connectparticipant.model.SendEventRequest;
import software.amazon.awssdk.services.connectparticipant.model.SendEventResponse;
import software.amazon.awssdk.services.connectparticipant.model.SendMessageRequest;
import software.amazon.awssdk.services.connectparticipant.model.SendMessageResponse;
import software.amazon.awssdk.services.connectparticipant.model.ThrottlingException;
import software.amazon.awssdk.services.connectparticipant.model.ValidationException;
import software.amazon.awssdk.services.connectparticipant.paginators.GetTranscriptPublisher;
import software.amazon.awssdk.services.connectparticipant.transform.CreateParticipantConnectionRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.DisconnectParticipantRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.GetTranscriptRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.SendEventRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.SendMessageRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ConnectParticipantAsyncClient}.
 *
 * @see ConnectParticipantAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectParticipantAsyncClient implements ConnectParticipantAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultConnectParticipantAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultConnectParticipantAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates the participant's connection. Note that ParticipantToken is used for invoking this API instead of
     * ConnectionToken.
     * </p>
     * <p>
     * The participant token is valid for the lifetime of the participant – until the they are part of a contact.
     * </p>
     * <p>
     * The response URL for <code>WEBSOCKET</code> Type has a connect expiry timeout of 100s. Clients must manually
     * connect to the returned websocket URL and subscribe to the desired topic.
     * </p>
     * <p>
     * For chat, you need to publish the following on the established websocket connection:
     * </p>
     * <p>
     * <code>{"topic":"aws/subscribe","content":{"topics":["aws/chat"]}}</code>
     * </p>
     * <p>
     * Upon websocket URL expiry, as specified in the response ConnectionExpiry parameter, clients need to call this API
     * again to obtain a new websocket URL and perform the same steps as before.
     * </p>
     *
     * @param createParticipantConnectionRequest
     * @return A Java Future containing the result of the CreateParticipantConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.CreateParticipantConnection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/CreateParticipantConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParticipantConnectionResponse> createParticipantConnection(
            CreateParticipantConnectionRequest createParticipantConnectionRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParticipantConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParticipantConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParticipantConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateParticipantConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParticipantConnectionRequest, CreateParticipantConnectionResponse>()
                            .withOperationName("CreateParticipantConnection")
                            .withMarshaller(new CreateParticipantConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createParticipantConnectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createParticipantConnectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateParticipantConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    createParticipantConnectionRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disconnects a participant. Note that ConnectionToken is used for invoking this API instead of ParticipantToken.
     * </p>
     *
     * @param disconnectParticipantRequest
     * @return A Java Future containing the result of the DisconnectParticipant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.DisconnectParticipant
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/DisconnectParticipant"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisconnectParticipantResponse> disconnectParticipant(
            DisconnectParticipantRequest disconnectParticipantRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectParticipant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisconnectParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisconnectParticipantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisconnectParticipantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisconnectParticipantRequest, DisconnectParticipantResponse>()
                            .withOperationName("DisconnectParticipant")
                            .withMarshaller(new DisconnectParticipantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disconnectParticipantRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disconnectParticipantRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DisconnectParticipantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectParticipantRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a transcript of the session. Note that ConnectionToken is used for invoking this API instead of
     * ParticipantToken.
     * </p>
     *
     * @param getTranscriptRequest
     * @return A Java Future containing the result of the GetTranscript operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.GetTranscript
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetTranscript"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTranscriptResponse> getTranscript(GetTranscriptRequest getTranscriptRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscript");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTranscriptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTranscriptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTranscriptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTranscriptRequest, GetTranscriptResponse>()
                            .withOperationName("GetTranscript")
                            .withMarshaller(new GetTranscriptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTranscriptRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getTranscriptRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetTranscriptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a transcript of the session. Note that ConnectionToken is used for invoking this API instead of
     * ParticipantToken.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getTranscript(software.amazon.awssdk.services.connectparticipant.model.GetTranscriptRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectparticipant.paginators.GetTranscriptPublisher publisher = client.getTranscriptPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.connectparticipant.paginators.GetTranscriptPublisher publisher = client.getTranscriptPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.connectparticipant.model.GetTranscriptResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.connectparticipant.model.GetTranscriptResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getTranscript(software.amazon.awssdk.services.connectparticipant.model.GetTranscriptRequest)}
     * operation.</b>
     * </p>
     *
     * @param getTranscriptRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.GetTranscript
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetTranscript"
     *      target="_top">AWS API Documentation</a>
     */
    public GetTranscriptPublisher getTranscriptPaginator(GetTranscriptRequest getTranscriptRequest) {
        return new GetTranscriptPublisher(this, applyPaginatorUserAgent(getTranscriptRequest));
    }

    /**
     * <p>
     * Sends an event. Note that ConnectionToken is used for invoking this API instead of ParticipantToken.
     * </p>
     *
     * @param sendEventRequest
     * @return A Java Future containing the result of the SendEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.SendEvent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/SendEvent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SendEventResponse> sendEvent(SendEventRequest sendEventRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendEventResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendEventRequest, SendEventResponse>().withOperationName("SendEvent")
                            .withMarshaller(new SendEventRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendEventRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = sendEventRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SendEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendEventRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a message. Note that ConnectionToken is used for invoking this API instead of ParticipantToken.
     * </p>
     *
     * @param sendMessageRequest
     * @return A Java Future containing the result of the SendMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.SendMessage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/SendMessage" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SendMessageResponse> sendMessage(SendMessageRequest sendMessageRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendMessageRequest, SendMessageResponse>()
                            .withOperationName("SendMessage").withMarshaller(new SendMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendMessageRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = sendMessageRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SendMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendMessageRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectParticipantException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ConnectParticipantRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
