/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectparticipant;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.connectparticipant.internal.ConnectParticipantServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.connectparticipant.model.AccessDeniedException;
import software.amazon.awssdk.services.connectparticipant.model.CompleteAttachmentUploadRequest;
import software.amazon.awssdk.services.connectparticipant.model.CompleteAttachmentUploadResponse;
import software.amazon.awssdk.services.connectparticipant.model.ConflictException;
import software.amazon.awssdk.services.connectparticipant.model.ConnectParticipantException;
import software.amazon.awssdk.services.connectparticipant.model.CreateParticipantConnectionRequest;
import software.amazon.awssdk.services.connectparticipant.model.CreateParticipantConnectionResponse;
import software.amazon.awssdk.services.connectparticipant.model.DescribeViewRequest;
import software.amazon.awssdk.services.connectparticipant.model.DescribeViewResponse;
import software.amazon.awssdk.services.connectparticipant.model.DisconnectParticipantRequest;
import software.amazon.awssdk.services.connectparticipant.model.DisconnectParticipantResponse;
import software.amazon.awssdk.services.connectparticipant.model.GetAttachmentRequest;
import software.amazon.awssdk.services.connectparticipant.model.GetAttachmentResponse;
import software.amazon.awssdk.services.connectparticipant.model.GetTranscriptRequest;
import software.amazon.awssdk.services.connectparticipant.model.GetTranscriptResponse;
import software.amazon.awssdk.services.connectparticipant.model.InternalServerException;
import software.amazon.awssdk.services.connectparticipant.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connectparticipant.model.SendEventRequest;
import software.amazon.awssdk.services.connectparticipant.model.SendEventResponse;
import software.amazon.awssdk.services.connectparticipant.model.SendMessageRequest;
import software.amazon.awssdk.services.connectparticipant.model.SendMessageResponse;
import software.amazon.awssdk.services.connectparticipant.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.connectparticipant.model.StartAttachmentUploadRequest;
import software.amazon.awssdk.services.connectparticipant.model.StartAttachmentUploadResponse;
import software.amazon.awssdk.services.connectparticipant.model.ThrottlingException;
import software.amazon.awssdk.services.connectparticipant.model.ValidationException;
import software.amazon.awssdk.services.connectparticipant.transform.CompleteAttachmentUploadRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.CreateParticipantConnectionRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.DescribeViewRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.DisconnectParticipantRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.GetAttachmentRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.GetTranscriptRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.SendEventRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.SendMessageRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.StartAttachmentUploadRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ConnectParticipantAsyncClient}.
 *
 * @see ConnectParticipantAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectParticipantAsyncClient implements ConnectParticipantAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultConnectParticipantAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ConnectParticipantServiceClientConfiguration serviceClientConfiguration;

    protected DefaultConnectParticipantAsyncClient(ConnectParticipantServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Allows you to confirm that the attachment has been uploaded using the pre-signed URL provided in
     * StartAttachmentUpload API.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param completeAttachmentUploadRequest
     * @return A Java Future containing the result of the CompleteAttachmentUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>ServiceQuotaExceededException The number of attachments per contact exceeds the quota.</li>
     *         <li>ConflictException An attachment with that identifier is already being uploaded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.CompleteAttachmentUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/CompleteAttachmentUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CompleteAttachmentUploadResponse> completeAttachmentUpload(
            CompleteAttachmentUploadRequest completeAttachmentUploadRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeAttachmentUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeAttachmentUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteAttachmentUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CompleteAttachmentUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CompleteAttachmentUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CompleteAttachmentUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CompleteAttachmentUploadRequest, CompleteAttachmentUploadResponse>()
                            .withOperationName("CompleteAttachmentUpload")
                            .withMarshaller(new CompleteAttachmentUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(completeAttachmentUploadRequest));
            CompletableFuture<CompleteAttachmentUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates the participant's connection.
     * </p>
     * <note>
     * <p>
     * <code>ParticipantToken</code> is used for invoking this API instead of <code>ConnectionToken</code>.
     * </p>
     * </note>
     * <p>
     * The participant token is valid for the lifetime of the participant – until they are part of a contact.
     * </p>
     * <p>
     * The response URL for <code>WEBSOCKET</code> Type has a connect expiry timeout of 100s. Clients must manually
     * connect to the returned websocket URL and subscribe to the desired topic.
     * </p>
     * <p>
     * For chat, you need to publish the following on the established websocket connection:
     * </p>
     * <p>
     * <code>{"topic":"aws/subscribe","content":{"topics":["aws/chat"]}}</code>
     * </p>
     * <p>
     * Upon websocket URL expiry, as specified in the response ConnectionExpiry parameter, clients need to call this API
     * again to obtain a new websocket URL and perform the same steps as before.
     * </p>
     * <p>
     * <b>Message streaming support</b>: This API can also be used together with the <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactStreaming.html"
     * >StartContactStreaming</a> API to create a participant connection for chat contacts that are not using a
     * websocket. For more information about message streaming, <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-message-streaming.html">Enable real-time chat
     * message streaming</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <p>
     * <b>Feature specifications</b>: For information about feature specifications, such as the allowed number of open
     * websocket connections per participant, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#feature-limits"
     * >Feature specifications</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     * </note>
     *
     * @param createParticipantConnectionRequest
     * @return A Java Future containing the result of the CreateParticipantConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.CreateParticipantConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/CreateParticipantConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParticipantConnectionResponse> createParticipantConnection(
            CreateParticipantConnectionRequest createParticipantConnectionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParticipantConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParticipantConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParticipantConnection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParticipantConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParticipantConnectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateParticipantConnectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParticipantConnectionRequest, CreateParticipantConnectionResponse>()
                            .withOperationName("CreateParticipantConnection")
                            .withMarshaller(new CreateParticipantConnectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createParticipantConnectionRequest));
            CompletableFuture<CreateParticipantConnectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the view for the specified view token.
     * </p>
     *
     * @param describeViewRequest
     * @return A Java Future containing the result of the DescribeView operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException The resource was not found.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.DescribeView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/DescribeView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeViewResponse> describeView(DescribeViewRequest describeViewRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeView");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeViewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeViewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeViewRequest, DescribeViewResponse>()
                            .withOperationName("DescribeView").withMarshaller(new DescribeViewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeViewRequest));
            CompletableFuture<DescribeViewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disconnects a participant.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param disconnectParticipantRequest
     * @return A Java Future containing the result of the DisconnectParticipant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.DisconnectParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/DisconnectParticipant"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisconnectParticipantResponse> disconnectParticipant(
            DisconnectParticipantRequest disconnectParticipantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disconnectParticipantRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectParticipant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisconnectParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisconnectParticipantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisconnectParticipantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisconnectParticipantRequest, DisconnectParticipantResponse>()
                            .withOperationName("DisconnectParticipant")
                            .withMarshaller(new DisconnectParticipantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disconnectParticipantRequest));
            CompletableFuture<DisconnectParticipantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a pre-signed URL for download of a completed attachment. This is an asynchronous API for use with active
     * contacts.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param getAttachmentRequest
     * @return A Java Future containing the result of the GetAttachment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.GetAttachment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetAttachment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAttachmentResponse> getAttachment(GetAttachmentRequest getAttachmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAttachmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAttachmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAttachment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAttachmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetAttachmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAttachmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAttachmentRequest, GetAttachmentResponse>()
                            .withOperationName("GetAttachment")
                            .withMarshaller(new GetAttachmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAttachmentRequest));
            CompletableFuture<GetAttachmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a transcript of the session, including details about any attachments. For information about accessing
     * past chat contact transcripts for a persistent chat, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-persistence.html">Enable persistent chat</a>.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param getTranscriptRequest
     * @return A Java Future containing the result of the GetTranscript operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.GetTranscript
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetTranscript"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetTranscriptResponse> getTranscript(GetTranscriptRequest getTranscriptRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTranscriptRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscript");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTranscriptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTranscriptResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTranscriptResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTranscriptRequest, GetTranscriptResponse>()
                            .withOperationName("GetTranscript")
                            .withMarshaller(new GetTranscriptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTranscriptRequest));
            CompletableFuture<GetTranscriptResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends an event.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param sendEventRequest
     * @return A Java Future containing the result of the SendEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.SendEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/SendEvent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SendEventResponse> sendEvent(SendEventRequest sendEventRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendEventRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendEvent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendEventResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendEventRequest, SendEventResponse>().withOperationName("SendEvent")
                            .withMarshaller(new SendEventRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendEventRequest));
            CompletableFuture<SendEventResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sends a message.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param sendMessageRequest
     * @return A Java Future containing the result of the SendMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.SendMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/SendMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendMessageResponse> sendMessage(SendMessageRequest sendMessageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendMessageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SendMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendMessageRequest, SendMessageResponse>()
                            .withOperationName("SendMessage").withMarshaller(new SendMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(sendMessageRequest));
            CompletableFuture<SendMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a pre-signed Amazon S3 URL in response for uploading the file directly to S3.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param startAttachmentUploadRequest
     * @return A Java Future containing the result of the StartAttachmentUpload operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InternalServerException This exception occurs when there is an internal failure in the Amazon Connect
     *         service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by Amazon Connect.</li>
     *         <li>ServiceQuotaExceededException The number of attachments per contact exceeds the quota.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ConnectParticipantException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ConnectParticipantAsyncClient.StartAttachmentUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/StartAttachmentUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartAttachmentUploadResponse> startAttachmentUpload(
            StartAttachmentUploadRequest startAttachmentUploadRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startAttachmentUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAttachmentUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartAttachmentUpload");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartAttachmentUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartAttachmentUploadResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartAttachmentUploadResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartAttachmentUploadRequest, StartAttachmentUploadResponse>()
                            .withOperationName("StartAttachmentUpload")
                            .withMarshaller(new StartAttachmentUploadRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(startAttachmentUploadRequest));
            CompletableFuture<StartAttachmentUploadResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ConnectParticipantServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectParticipantException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        ConnectParticipantServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = ConnectParticipantServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
