/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connectparticipant;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.connectparticipant.internal.ConnectParticipantServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.connectparticipant.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.connectparticipant.model.AccessDeniedException;
import software.amazon.awssdk.services.connectparticipant.model.CancelParticipantAuthenticationRequest;
import software.amazon.awssdk.services.connectparticipant.model.CancelParticipantAuthenticationResponse;
import software.amazon.awssdk.services.connectparticipant.model.CompleteAttachmentUploadRequest;
import software.amazon.awssdk.services.connectparticipant.model.CompleteAttachmentUploadResponse;
import software.amazon.awssdk.services.connectparticipant.model.ConflictException;
import software.amazon.awssdk.services.connectparticipant.model.ConnectParticipantException;
import software.amazon.awssdk.services.connectparticipant.model.CreateParticipantConnectionRequest;
import software.amazon.awssdk.services.connectparticipant.model.CreateParticipantConnectionResponse;
import software.amazon.awssdk.services.connectparticipant.model.DescribeViewRequest;
import software.amazon.awssdk.services.connectparticipant.model.DescribeViewResponse;
import software.amazon.awssdk.services.connectparticipant.model.DisconnectParticipantRequest;
import software.amazon.awssdk.services.connectparticipant.model.DisconnectParticipantResponse;
import software.amazon.awssdk.services.connectparticipant.model.GetAttachmentRequest;
import software.amazon.awssdk.services.connectparticipant.model.GetAttachmentResponse;
import software.amazon.awssdk.services.connectparticipant.model.GetAuthenticationUrlRequest;
import software.amazon.awssdk.services.connectparticipant.model.GetAuthenticationUrlResponse;
import software.amazon.awssdk.services.connectparticipant.model.GetTranscriptRequest;
import software.amazon.awssdk.services.connectparticipant.model.GetTranscriptResponse;
import software.amazon.awssdk.services.connectparticipant.model.InternalServerException;
import software.amazon.awssdk.services.connectparticipant.model.ResourceNotFoundException;
import software.amazon.awssdk.services.connectparticipant.model.SendEventRequest;
import software.amazon.awssdk.services.connectparticipant.model.SendEventResponse;
import software.amazon.awssdk.services.connectparticipant.model.SendMessageRequest;
import software.amazon.awssdk.services.connectparticipant.model.SendMessageResponse;
import software.amazon.awssdk.services.connectparticipant.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.connectparticipant.model.StartAttachmentUploadRequest;
import software.amazon.awssdk.services.connectparticipant.model.StartAttachmentUploadResponse;
import software.amazon.awssdk.services.connectparticipant.model.ThrottlingException;
import software.amazon.awssdk.services.connectparticipant.model.ValidationException;
import software.amazon.awssdk.services.connectparticipant.transform.CancelParticipantAuthenticationRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.CompleteAttachmentUploadRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.CreateParticipantConnectionRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.DescribeViewRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.DisconnectParticipantRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.GetAttachmentRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.GetAuthenticationUrlRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.GetTranscriptRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.SendEventRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.SendMessageRequestMarshaller;
import software.amazon.awssdk.services.connectparticipant.transform.StartAttachmentUploadRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ConnectParticipantClient}.
 *
 * @see ConnectParticipantClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultConnectParticipantClient implements ConnectParticipantClient {
    private static final Logger log = Logger.loggerFor(DefaultConnectParticipantClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultConnectParticipantClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "ConnectParticipant" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels the authentication session. The opted out branch of the Authenticate Customer flow block will be taken.
     * </p>
     * <note>
     * <p>
     * The current supported channel is chat. This API is not supported for Apple Messages for Business, WhatsApp, or
     * SMS chats.
     * </p>
     * </note> <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param cancelParticipantAuthenticationRequest
     * @return Result of the CancelParticipantAuthentication operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.CancelParticipantAuthentication
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/CancelParticipantAuthentication"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelParticipantAuthenticationResponse cancelParticipantAuthentication(
            CancelParticipantAuthenticationRequest cancelParticipantAuthenticationRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelParticipantAuthenticationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelParticipantAuthenticationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelParticipantAuthenticationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                cancelParticipantAuthenticationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelParticipantAuthentication");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelParticipantAuthenticationRequest, CancelParticipantAuthenticationResponse>()
                            .withOperationName("CancelParticipantAuthentication").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(cancelParticipantAuthenticationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelParticipantAuthenticationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Allows you to confirm that the attachment has been uploaded using the pre-signed URL provided in
     * StartAttachmentUpload API. A conflict exception is thrown when an attachment with that identifier is already
     * being uploaded.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param completeAttachmentUploadRequest
     * @return Result of the CompleteAttachmentUpload operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws ServiceQuotaExceededException
     *         The number of attachments per contact exceeds the quota.
     * @throws ConflictException
     *         The requested operation conflicts with the current state of a service resource associated with the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.CompleteAttachmentUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/CompleteAttachmentUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompleteAttachmentUploadResponse completeAttachmentUpload(
            CompleteAttachmentUploadRequest completeAttachmentUploadRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, ServiceQuotaExceededException, ConflictException,
            AwsServiceException, SdkClientException, ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CompleteAttachmentUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CompleteAttachmentUploadResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(completeAttachmentUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, completeAttachmentUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CompleteAttachmentUpload");

            return clientHandler
                    .execute(new ClientExecutionParams<CompleteAttachmentUploadRequest, CompleteAttachmentUploadResponse>()
                            .withOperationName("CompleteAttachmentUpload").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(completeAttachmentUploadRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CompleteAttachmentUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates the participant's connection.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <p>
     * For WebRTC security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-webrtc-security"
     * >Amazon Connect WebRTC security best practices</a>.
     * </p>
     * <note>
     * <p>
     * <code>ParticipantToken</code> is used for invoking this API instead of <code>ConnectionToken</code>.
     * </p>
     * </note>
     * <p>
     * The participant token is valid for the lifetime of the participant – until they are part of a contact. For WebRTC
     * participants, if they leave or are disconnected for 60 seconds, a new participant needs to be created using the
     * <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateParticipant.html">CreateParticipant</a>
     * API.
     * </p>
     * <p>
     * <b>For <code>WEBSOCKET</code> Type</b>:
     * </p>
     * <p>
     * The response URL for has a connect expiry timeout of 100s. Clients must manually connect to the returned
     * websocket URL and subscribe to the desired topic.
     * </p>
     * <p>
     * For chat, you need to publish the following on the established websocket connection:
     * </p>
     * <p>
     * <code>{"topic":"aws/subscribe","content":{"topics":["aws/chat"]}}</code>
     * </p>
     * <p>
     * Upon websocket URL expiry, as specified in the response ConnectionExpiry parameter, clients need to call this API
     * again to obtain a new websocket URL and perform the same steps as before.
     * </p>
     * <p>
     * The expiry time for the connection token is different than the <code>ChatDurationInMinutes</code>. Expiry time
     * for the connection token is 1 day.
     * </p>
     * <p>
     * <b>For <code>WEBRTC_CONNECTION</code> Type</b>:
     * </p>
     * <p>
     * The response includes connection data required for the client application to join the call using the Amazon Chime
     * SDK client libraries. The WebRTCConnection response contains Meeting and Attendee information needed to establish
     * the media connection.
     * </p>
     * <p>
     * The attendee join token in WebRTCConnection response is valid for the lifetime of the participant in the call. If
     * a participant leaves or is disconnected for 60 seconds, their participant credentials will no longer be valid,
     * and a new participant will need to be created to rejoin the call.
     * </p>
     * <p>
     * <b>Message streaming support</b>: This API can also be used together with the <a
     * href="https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactStreaming.html"
     * >StartContactStreaming</a> API to create a participant connection for chat contacts that are not using a
     * websocket. For more information about message streaming, <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-message-streaming.html">Enable real-time chat
     * message streaming</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <p>
     * <b>Multi-user web, in-app, video calling support</b>:
     * </p>
     * <p>
     * For WebRTC calls, this API is used in conjunction with the CreateParticipant API to enable multi-party calling.
     * The StartWebRTCContact API creates the initial contact and routes it to an agent, while CreateParticipant adds
     * additional participants to the ongoing call. For more information about multi-party WebRTC calls, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/enable-multiuser-inapp.html">Enable multi-user web,
     * in-app, and video calling</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <p>
     * <b>Feature specifications</b>: For information about feature specifications, such as the allowed number of open
     * websocket connections per participant or maximum number of WebRTC participants, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-service-limits.html#feature-limits"
     * >Feature specifications</a> in the <i>Amazon Connect Administrator Guide</i>.
     * </p>
     * <note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     * </note>
     *
     * @param createParticipantConnectionRequest
     * @return Result of the CreateParticipantConnection operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.CreateParticipantConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/CreateParticipantConnection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateParticipantConnectionResponse createParticipantConnection(
            CreateParticipantConnectionRequest createParticipantConnectionRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateParticipantConnectionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateParticipantConnectionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParticipantConnectionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParticipantConnectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParticipantConnection");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateParticipantConnectionRequest, CreateParticipantConnectionResponse>()
                            .withOperationName("CreateParticipantConnection").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createParticipantConnectionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateParticipantConnectionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the view for the specified view token.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     *
     * @param describeViewRequest
     * @return Result of the DescribeView operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The resource was not found.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.DescribeView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/DescribeView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeViewResponse describeView(DescribeViewRequest describeViewRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ResourceNotFoundException, ValidationException, AwsServiceException,
            SdkClientException, ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeViewResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeView");

            return clientHandler.execute(new ClientExecutionParams<DescribeViewRequest, DescribeViewResponse>()
                    .withOperationName("DescribeView").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeViewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disconnects a participant.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param disconnectParticipantRequest
     * @return Result of the DisconnectParticipant operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.DisconnectParticipant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/DisconnectParticipant"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisconnectParticipantResponse disconnectParticipant(DisconnectParticipantRequest disconnectParticipantRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisconnectParticipantResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisconnectParticipantResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disconnectParticipantRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disconnectParticipantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisconnectParticipant");

            return clientHandler.execute(new ClientExecutionParams<DisconnectParticipantRequest, DisconnectParticipantResponse>()
                    .withOperationName("DisconnectParticipant").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disconnectParticipantRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisconnectParticipantRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a pre-signed URL for download of a completed attachment. This is an asynchronous API for use with active
     * contacts.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * The participant role <code>CUSTOM_BOT</code> is not permitted to access attachments customers may upload. An
     * <code>AccessDeniedException</code> can indicate that the participant may be a CUSTOM_BOT, and it doesn't have
     * access to attachments.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param getAttachmentRequest
     * @return Result of the GetAttachment operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.GetAttachment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetAttachment"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAttachmentResponse getAttachment(GetAttachmentRequest getAttachmentRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAttachmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAttachmentResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAttachmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAttachmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAttachment");

            return clientHandler.execute(new ClientExecutionParams<GetAttachmentRequest, GetAttachmentResponse>()
                    .withOperationName("GetAttachment").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAttachmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAttachmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the AuthenticationUrl for the current authentication session for the AuthenticateCustomer flow block.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This API can only be called within one minute of receiving the authenticationInitiated event.
     * </p>
     * </li>
     * <li>
     * <p>
     * The current supported channel is chat. This API is not supported for Apple Messages for Business, WhatsApp, or
     * SMS chats.
     * </p>
     * </li>
     * </ul>
     * </note> <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param getAuthenticationUrlRequest
     * @return Result of the GetAuthenticationUrl operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.GetAuthenticationUrl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetAuthenticationUrl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAuthenticationUrlResponse getAuthenticationUrl(GetAuthenticationUrlRequest getAuthenticationUrlRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException, AwsServiceException,
            SdkClientException, ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAuthenticationUrlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAuthenticationUrlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAuthenticationUrlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAuthenticationUrlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAuthenticationUrl");

            return clientHandler.execute(new ClientExecutionParams<GetAuthenticationUrlRequest, GetAuthenticationUrlResponse>()
                    .withOperationName("GetAuthenticationUrl").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAuthenticationUrlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAuthenticationUrlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves a transcript of the session, including details about any attachments. For information about accessing
     * past chat contact transcripts for a persistent chat, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/chat-persistence.html">Enable persistent chat</a>.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <p>
     * If you have a process that consumes events in the transcript of an chat that has ended, note that chat
     * transcripts contain the following event content types if the event has occurred during the chat session:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>application/vnd.amazonaws.connect.event.participant.invited</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application/vnd.amazonaws.connect.event.participant.joined</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application/vnd.amazonaws.connect.event.participant.left</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application/vnd.amazonaws.connect.event.chat.ended</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application/vnd.amazonaws.connect.event.transfer.succeeded</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>application/vnd.amazonaws.connect.event.transfer.failed</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param getTranscriptRequest
     * @return Result of the GetTranscript operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.GetTranscript
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/GetTranscript"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetTranscriptResponse getTranscript(GetTranscriptRequest getTranscriptRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTranscriptResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTranscriptResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getTranscriptRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTranscriptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTranscript");

            return clientHandler.execute(new ClientExecutionParams<GetTranscriptRequest, GetTranscriptResponse>()
                    .withOperationName("GetTranscript").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getTranscriptRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTranscriptRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <note>
     * <p>
     * The <code>application/vnd.amazonaws.connect.event.connection.acknowledged</code> ContentType is no longer
     * maintained since December 31, 2024. This event has been migrated to the <a
     * href="https://docs.aws.amazon.com/connect-participant/latest/APIReference/API_CreateParticipantConnection.html"
     * >CreateParticipantConnection</a> API using the <code>ConnectParticipant</code> field.
     * </p>
     * </note>
     * <p>
     * Sends an event. Message receipts are not supported when there are more than two active participants in the chat.
     * Using the SendEvent API for message receipts when a supervisor is barged-in will result in a conflict exception.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param sendEventRequest
     * @return Result of the SendEvent operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws ConflictException
     *         The requested operation conflicts with the current state of a service resource associated with the
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.SendEvent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/SendEvent" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public SendEventResponse sendEvent(SendEventRequest sendEventRequest) throws AccessDeniedException, InternalServerException,
            ThrottlingException, ValidationException, ConflictException, AwsServiceException, SdkClientException,
            ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendEventResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendEventRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendEventRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendEvent");

            return clientHandler.execute(new ClientExecutionParams<SendEventRequest, SendEventResponse>()
                    .withOperationName("SendEvent").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(sendEventRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendEventRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends a message.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param sendMessageRequest
     * @return Result of the SendMessage operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.SendMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/SendMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SendMessageResponse sendMessage(SendMessageRequest sendMessageRequest) throws AccessDeniedException,
            InternalServerException, ThrottlingException, ValidationException, AwsServiceException, SdkClientException,
            ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendMessageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(sendMessageRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendMessage");

            return clientHandler.execute(new ClientExecutionParams<SendMessageRequest, SendMessageResponse>()
                    .withOperationName("SendMessage").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(sendMessageRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SendMessageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a pre-signed Amazon S3 URL in response for uploading the file directly to S3.
     * </p>
     * <p>
     * For security recommendations, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/security-best-practices.html#bp-security-chat">Amazon
     * Connect Chat security best practices</a>.
     * </p>
     * <note>
     * <p>
     * <code>ConnectionToken</code> is used for invoking this API instead of <code>ParticipantToken</code>.
     * </p>
     * </note>
     * <p>
     * The Amazon Connect Participant Service APIs do not use <a
     * href="https://docs.aws.amazon.com/general/latest/gr/signature-version-4.html">Signature Version 4
     * authentication</a>.
     * </p>
     *
     * @param startAttachmentUploadRequest
     * @return Result of the StartAttachmentUpload operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         This exception occurs when there is an internal failure in the Amazon Connect service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by Amazon Connect.
     * @throws ServiceQuotaExceededException
     *         The number of attachments per contact exceeds the quota.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ConnectParticipantException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ConnectParticipantClient.StartAttachmentUpload
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/connectparticipant-2018-09-07/StartAttachmentUpload"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartAttachmentUploadResponse startAttachmentUpload(StartAttachmentUploadRequest startAttachmentUploadRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, ConnectParticipantException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartAttachmentUploadResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartAttachmentUploadResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startAttachmentUploadRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startAttachmentUploadRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ConnectParticipant");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartAttachmentUpload");

            return clientHandler.execute(new ClientExecutionParams<StartAttachmentUploadRequest, StartAttachmentUploadResponse>()
                    .withOperationName("StartAttachmentUpload").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(startAttachmentUploadRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartAttachmentUploadRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ConnectParticipantServiceClientConfigurationBuilder serviceConfigBuilder = new ConnectParticipantServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ConnectParticipantException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    @Override
    public final ConnectParticipantServiceClientConfiguration serviceClientConfiguration() {
        return new ConnectParticipantServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
