/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.controlcatalog;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.controlcatalog.model.AccessDeniedException;
import software.amazon.awssdk.services.controlcatalog.model.ControlCatalogException;
import software.amazon.awssdk.services.controlcatalog.model.InternalServerException;
import software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse;
import software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse;
import software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest;
import software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse;
import software.amazon.awssdk.services.controlcatalog.model.ThrottlingException;
import software.amazon.awssdk.services.controlcatalog.model.ValidationException;
import software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable;
import software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable;
import software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable;

/**
 * Service client for accessing AWS Control Catalog. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Welcome to the Amazon Web Services Control Catalog API reference. This guide is for developers who need detailed
 * information about how to programmatically identify and filter the common controls and related metadata that are
 * available to Amazon Web Services customers. This API reference provides descriptions, syntax, and usage examples for
 * each of the actions and data types that are supported by Amazon Web Services Control Catalog.
 * </p>
 * <p>
 * Use the following links to get started with the Amazon Web Services Control Catalog API:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_Operations.html">Actions</a>: An
 * alphabetical list of all Control Catalog API operations.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_Types.html">Data types</a>: An
 * alphabetical list of all Control Catalog data types.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/CommonParameters.html">Common parameters</a>:
 * Parameters that all operations can use.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/controlcatalog/latest/APIReference/CommonErrors.html">Common errors</a>: Client
 * and server errors that all operations can return.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ControlCatalogClient extends AwsClient {
    String SERVICE_NAME = "controlcatalog";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "controlcatalog";

    /**
     * <p>
     * Returns a paginated list of common controls from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see common controls that have a specific objective. If you don’t provide a
     * filter, the operation returns all common controls.
     * </p>
     *
     * @param listCommonControlsRequest
     * @return Result of the ListCommonControls operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCommonControlsResponse listCommonControls(ListCommonControlsRequest listCommonControlsRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of common controls from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see common controls that have a specific objective. If you don’t provide a
     * filter, the operation returns all common controls.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListCommonControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListCommonControlsRequest#builder()}
     * </p>
     *
     * @param listCommonControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest.Builder} to create a
     *        request.
     * @return Result of the ListCommonControls operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCommonControlsResponse listCommonControls(Consumer<ListCommonControlsRequest.Builder> listCommonControlsRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return listCommonControls(ListCommonControlsRequest.builder().applyMutation(listCommonControlsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable responses = client.listCommonControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable responses = client
     *             .listCommonControlsPaginator(request);
     *     for (software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable responses = client.listCommonControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCommonControlsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCommonControlsIterable listCommonControlsPaginator(ListCommonControlsRequest listCommonControlsRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return new ListCommonControlsIterable(this, listCommonControlsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable responses = client.listCommonControlsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable responses = client
     *             .listCommonControlsPaginator(request);
     *     for (software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListCommonControlsIterable responses = client.listCommonControlsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCommonControls(software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListCommonControlsRequest.Builder} avoiding the
     * need to create one manually via {@link ListCommonControlsRequest#builder()}
     * </p>
     *
     * @param listCommonControlsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListCommonControlsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListCommonControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListCommonControls"
     *      target="_top">AWS API Documentation</a>
     */
    default ListCommonControlsIterable listCommonControlsPaginator(
            Consumer<ListCommonControlsRequest.Builder> listCommonControlsRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlCatalogException {
        return listCommonControlsPaginator(ListCommonControlsRequest.builder().applyMutation(listCommonControlsRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of domains from the Amazon Web Services Control Catalog.
     * </p>
     *
     * @param listDomainsRequest
     * @return Result of the ListDomains operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDomainsResponse listDomains(ListDomainsRequest listDomainsRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlCatalogException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of domains from the Amazon Web Services Control Catalog.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDomainsRequest.Builder} avoiding the need to
     * create one manually via {@link ListDomainsRequest#builder()}
     * </p>
     *
     * @param listDomainsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest.Builder} to create a
     *        request.
     * @return Result of the ListDomains operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDomainsResponse listDomains(Consumer<ListDomainsRequest.Builder> listDomainsRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return listDomains(ListDomainsRequest.builder().applyMutation(listDomainsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable responses = client
     *             .listDomainsPaginator(request);
     *     for (software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation.</b>
     * </p>
     *
     * @param listDomainsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDomainsIterable listDomainsPaginator(ListDomainsRequest listDomainsRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlCatalogException {
        return new ListDomainsIterable(this, listDomainsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable responses = client
     *             .listDomainsPaginator(request);
     *     for (software.amazon.awssdk.services.controlcatalog.model.ListDomainsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListDomainsIterable responses = client.listDomainsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDomains(software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDomainsRequest.Builder} avoiding the need to
     * create one manually via {@link ListDomainsRequest#builder()}
     * </p>
     *
     * @param listDomainsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListDomainsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDomainsIterable listDomainsPaginator(Consumer<ListDomainsRequest.Builder> listDomainsRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return listDomainsPaginator(ListDomainsRequest.builder().applyMutation(listDomainsRequest).build());
    }

    /**
     * <p>
     * Returns a paginated list of objectives from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see the objectives that belong to a specific domain. If you don’t provide a
     * filter, the operation returns all objectives.
     * </p>
     *
     * @param listObjectivesRequest
     * @return Result of the ListObjectives operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default ListObjectivesResponse listObjectives(ListObjectivesRequest listObjectivesRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlCatalogException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a paginated list of objectives from the Amazon Web Services Control Catalog.
     * </p>
     * <p>
     * You can apply an optional filter to see the objectives that belong to a specific domain. If you don’t provide a
     * filter, the operation returns all objectives.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectivesRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectivesRequest#builder()}
     * </p>
     *
     * @param listObjectivesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest.Builder} to create a
     *        request.
     * @return Result of the ListObjectives operation returned by the service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default ListObjectivesResponse listObjectives(Consumer<ListObjectivesRequest.Builder> listObjectivesRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return listObjectives(ListObjectivesRequest.builder().applyMutation(listObjectivesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable responses = client.listObjectivesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable responses = client
     *             .listObjectivesPaginator(request);
     *     for (software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable responses = client.listObjectivesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listObjectivesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default ListObjectivesIterable listObjectivesPaginator(ListObjectivesRequest listObjectivesRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return new ListObjectivesIterable(this, listObjectivesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable responses = client.listObjectivesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable responses = client
     *             .listObjectivesPaginator(request);
     *     for (software.amazon.awssdk.services.controlcatalog.model.ListObjectivesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.controlcatalog.paginators.ListObjectivesIterable responses = client.listObjectivesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listObjectives(software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListObjectivesRequest.Builder} avoiding the need to
     * create one manually via {@link ListObjectivesRequest#builder()}
     * </p>
     *
     * @param listObjectivesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.controlcatalog.model.ListObjectivesRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InternalServerException
     *         An internal service error occurred during the processing of your request. Try again later.
     * @throws ValidationException
     *         The request has invalid or missing parameters.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlCatalogException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlCatalogClient.ListObjectives
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controlcatalog-2018-05-10/ListObjectives" target="_top">AWS
     *      API Documentation</a>
     */
    default ListObjectivesIterable listObjectivesPaginator(Consumer<ListObjectivesRequest.Builder> listObjectivesRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlCatalogException {
        return listObjectivesPaginator(ListObjectivesRequest.builder().applyMutation(listObjectivesRequest).build());
    }

    /**
     * Create a {@link ControlCatalogClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ControlCatalogClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ControlCatalogClient}.
     */
    static ControlCatalogClientBuilder builder() {
        return new DefaultControlCatalogClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ControlCatalogServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
