/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.controlcatalog.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information about the control, including the scope of the control, if enabled, and the Regions in which the
 * control is available for deployment. For more information about scope, see <a
 * href="https://docs.aws.amazon.com/whitepapers/latest/aws-fault-isolation-boundaries/global-services.html">Global
 * services</a>.
 * </p>
 * <p>
 * If you are applying controls through an Amazon Web Services Control Tower landing zone environment, remember that the
 * values returned in the <code>RegionConfiguration</code> API operation are not related to the governed Regions in your
 * landing zone. For example, if you are governing Regions <code>A</code>,<code>B</code>,and <code>C</code> while the
 * control is available in Regions <code>A</code>, <code>B</code>, C<code>,</code> and <code>D</code>, you'd see a
 * response with <code>DeployableRegions</code> of <code>A</code>, <code>B</code>, <code>C</code>, and <code>D</code>
 * for a control with <code>REGIONAL</code> scope, even though you may not intend to deploy the control in Region
 * <code>D</code>, because you do not govern it through your landing zone.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RegionConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<RegionConfiguration.Builder, RegionConfiguration> {
    private static final SdkField<String> SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Scope")
            .getter(getter(RegionConfiguration::scopeAsString)).setter(setter(Builder::scope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scope").build()).build();

    private static final SdkField<List<String>> DEPLOYABLE_REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DeployableRegions")
            .getter(getter(RegionConfiguration::deployableRegions))
            .setter(setter(Builder::deployableRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeployableRegions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCOPE_FIELD,
            DEPLOYABLE_REGIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String scope;

    private final List<String> deployableRegions;

    private RegionConfiguration(BuilderImpl builder) {
        this.scope = builder.scope;
        this.deployableRegions = builder.deployableRegions;
    }

    /**
     * <p>
     * The coverage of the control, if deployed. Scope is an enumerated type, with value <code>Regional</code>, or
     * <code>Global</code>. A control with Global scope is effective in all Amazon Web Services Regions, regardless of
     * the Region from which it is enabled, or to which it is deployed. A control implemented by an SCP is usually
     * Global in scope. A control with Regional scope has operations that are restricted specifically to the Region from
     * which it is enabled and to which it is deployed. Controls implemented by Config rules and CloudFormation hooks
     * usually are Regional in scope. Security Hub controls usually are Regional in scope.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scope} will return
     * {@link ControlScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scopeAsString}.
     * </p>
     * 
     * @return The coverage of the control, if deployed. Scope is an enumerated type, with value <code>Regional</code>,
     *         or <code>Global</code>. A control with Global scope is effective in all Amazon Web Services Regions,
     *         regardless of the Region from which it is enabled, or to which it is deployed. A control implemented by
     *         an SCP is usually Global in scope. A control with Regional scope has operations that are restricted
     *         specifically to the Region from which it is enabled and to which it is deployed. Controls implemented by
     *         Config rules and CloudFormation hooks usually are Regional in scope. Security Hub controls usually are
     *         Regional in scope.
     * @see ControlScope
     */
    public final ControlScope scope() {
        return ControlScope.fromValue(scope);
    }

    /**
     * <p>
     * The coverage of the control, if deployed. Scope is an enumerated type, with value <code>Regional</code>, or
     * <code>Global</code>. A control with Global scope is effective in all Amazon Web Services Regions, regardless of
     * the Region from which it is enabled, or to which it is deployed. A control implemented by an SCP is usually
     * Global in scope. A control with Regional scope has operations that are restricted specifically to the Region from
     * which it is enabled and to which it is deployed. Controls implemented by Config rules and CloudFormation hooks
     * usually are Regional in scope. Security Hub controls usually are Regional in scope.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scope} will return
     * {@link ControlScope#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scopeAsString}.
     * </p>
     * 
     * @return The coverage of the control, if deployed. Scope is an enumerated type, with value <code>Regional</code>,
     *         or <code>Global</code>. A control with Global scope is effective in all Amazon Web Services Regions,
     *         regardless of the Region from which it is enabled, or to which it is deployed. A control implemented by
     *         an SCP is usually Global in scope. A control with Regional scope has operations that are restricted
     *         specifically to the Region from which it is enabled and to which it is deployed. Controls implemented by
     *         Config rules and CloudFormation hooks usually are Regional in scope. Security Hub controls usually are
     *         Regional in scope.
     * @see ControlScope
     */
    public final String scopeAsString() {
        return scope;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeployableRegions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeployableRegions() {
        return deployableRegions != null && !(deployableRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Regions in which the control is available to be deployed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeployableRegions} method.
     * </p>
     * 
     * @return Regions in which the control is available to be deployed.
     */
    public final List<String> deployableRegions() {
        return deployableRegions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(scopeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDeployableRegions() ? deployableRegions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RegionConfiguration)) {
            return false;
        }
        RegionConfiguration other = (RegionConfiguration) obj;
        return Objects.equals(scopeAsString(), other.scopeAsString()) && hasDeployableRegions() == other.hasDeployableRegions()
                && Objects.equals(deployableRegions(), other.deployableRegions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RegionConfiguration").add("Scope", scopeAsString())
                .add("DeployableRegions", hasDeployableRegions() ? deployableRegions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Scope":
            return Optional.ofNullable(clazz.cast(scopeAsString()));
        case "DeployableRegions":
            return Optional.ofNullable(clazz.cast(deployableRegions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Scope", SCOPE_FIELD);
        map.put("DeployableRegions", DEPLOYABLE_REGIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RegionConfiguration, T> g) {
        return obj -> g.apply((RegionConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RegionConfiguration> {
        /**
         * <p>
         * The coverage of the control, if deployed. Scope is an enumerated type, with value <code>Regional</code>, or
         * <code>Global</code>. A control with Global scope is effective in all Amazon Web Services Regions, regardless
         * of the Region from which it is enabled, or to which it is deployed. A control implemented by an SCP is
         * usually Global in scope. A control with Regional scope has operations that are restricted specifically to the
         * Region from which it is enabled and to which it is deployed. Controls implemented by Config rules and
         * CloudFormation hooks usually are Regional in scope. Security Hub controls usually are Regional in scope.
         * </p>
         * 
         * @param scope
         *        The coverage of the control, if deployed. Scope is an enumerated type, with value
         *        <code>Regional</code>, or <code>Global</code>. A control with Global scope is effective in all Amazon
         *        Web Services Regions, regardless of the Region from which it is enabled, or to which it is deployed. A
         *        control implemented by an SCP is usually Global in scope. A control with Regional scope has operations
         *        that are restricted specifically to the Region from which it is enabled and to which it is deployed.
         *        Controls implemented by Config rules and CloudFormation hooks usually are Regional in scope. Security
         *        Hub controls usually are Regional in scope.
         * @see ControlScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlScope
         */
        Builder scope(String scope);

        /**
         * <p>
         * The coverage of the control, if deployed. Scope is an enumerated type, with value <code>Regional</code>, or
         * <code>Global</code>. A control with Global scope is effective in all Amazon Web Services Regions, regardless
         * of the Region from which it is enabled, or to which it is deployed. A control implemented by an SCP is
         * usually Global in scope. A control with Regional scope has operations that are restricted specifically to the
         * Region from which it is enabled and to which it is deployed. Controls implemented by Config rules and
         * CloudFormation hooks usually are Regional in scope. Security Hub controls usually are Regional in scope.
         * </p>
         * 
         * @param scope
         *        The coverage of the control, if deployed. Scope is an enumerated type, with value
         *        <code>Regional</code>, or <code>Global</code>. A control with Global scope is effective in all Amazon
         *        Web Services Regions, regardless of the Region from which it is enabled, or to which it is deployed. A
         *        control implemented by an SCP is usually Global in scope. A control with Regional scope has operations
         *        that are restricted specifically to the Region from which it is enabled and to which it is deployed.
         *        Controls implemented by Config rules and CloudFormation hooks usually are Regional in scope. Security
         *        Hub controls usually are Regional in scope.
         * @see ControlScope
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ControlScope
         */
        Builder scope(ControlScope scope);

        /**
         * <p>
         * Regions in which the control is available to be deployed.
         * </p>
         * 
         * @param deployableRegions
         *        Regions in which the control is available to be deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deployableRegions(Collection<String> deployableRegions);

        /**
         * <p>
         * Regions in which the control is available to be deployed.
         * </p>
         * 
         * @param deployableRegions
         *        Regions in which the control is available to be deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deployableRegions(String... deployableRegions);
    }

    static final class BuilderImpl implements Builder {
        private String scope;

        private List<String> deployableRegions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RegionConfiguration model) {
            scope(model.scope);
            deployableRegions(model.deployableRegions);
        }

        public final String getScope() {
            return scope;
        }

        public final void setScope(String scope) {
            this.scope = scope;
        }

        @Override
        public final Builder scope(String scope) {
            this.scope = scope;
            return this;
        }

        @Override
        public final Builder scope(ControlScope scope) {
            this.scope(scope == null ? null : scope.toString());
            return this;
        }

        public final Collection<String> getDeployableRegions() {
            if (deployableRegions instanceof SdkAutoConstructList) {
                return null;
            }
            return deployableRegions;
        }

        public final void setDeployableRegions(Collection<String> deployableRegions) {
            this.deployableRegions = DeployableRegionsCopier.copy(deployableRegions);
        }

        @Override
        public final Builder deployableRegions(Collection<String> deployableRegions) {
            this.deployableRegions = DeployableRegionsCopier.copy(deployableRegions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder deployableRegions(String... deployableRegions) {
            deployableRegions(Arrays.asList(deployableRegions));
            return this;
        }

        @Override
        public RegionConfiguration build() {
            return new RegionConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
