/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.controltower;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.controltower.internal.ControlTowerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.controltower.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.controltower.model.AccessDeniedException;
import software.amazon.awssdk.services.controltower.model.ConflictException;
import software.amazon.awssdk.services.controltower.model.ControlTowerException;
import software.amazon.awssdk.services.controltower.model.CreateLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.CreateLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.DeleteLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.DeleteLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.DisableBaselineRequest;
import software.amazon.awssdk.services.controltower.model.DisableBaselineResponse;
import software.amazon.awssdk.services.controltower.model.DisableControlRequest;
import software.amazon.awssdk.services.controltower.model.DisableControlResponse;
import software.amazon.awssdk.services.controltower.model.EnableBaselineRequest;
import software.amazon.awssdk.services.controltower.model.EnableBaselineResponse;
import software.amazon.awssdk.services.controltower.model.EnableControlRequest;
import software.amazon.awssdk.services.controltower.model.EnableControlResponse;
import software.amazon.awssdk.services.controltower.model.GetBaselineOperationRequest;
import software.amazon.awssdk.services.controltower.model.GetBaselineOperationResponse;
import software.amazon.awssdk.services.controltower.model.GetBaselineRequest;
import software.amazon.awssdk.services.controltower.model.GetBaselineResponse;
import software.amazon.awssdk.services.controltower.model.GetControlOperationRequest;
import software.amazon.awssdk.services.controltower.model.GetControlOperationResponse;
import software.amazon.awssdk.services.controltower.model.GetEnabledBaselineRequest;
import software.amazon.awssdk.services.controltower.model.GetEnabledBaselineResponse;
import software.amazon.awssdk.services.controltower.model.GetEnabledControlRequest;
import software.amazon.awssdk.services.controltower.model.GetEnabledControlResponse;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneOperationRequest;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneOperationResponse;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.GetLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.InternalServerException;
import software.amazon.awssdk.services.controltower.model.ListBaselinesRequest;
import software.amazon.awssdk.services.controltower.model.ListBaselinesResponse;
import software.amazon.awssdk.services.controltower.model.ListControlOperationsRequest;
import software.amazon.awssdk.services.controltower.model.ListControlOperationsResponse;
import software.amazon.awssdk.services.controltower.model.ListEnabledBaselinesRequest;
import software.amazon.awssdk.services.controltower.model.ListEnabledBaselinesResponse;
import software.amazon.awssdk.services.controltower.model.ListEnabledControlsRequest;
import software.amazon.awssdk.services.controltower.model.ListEnabledControlsResponse;
import software.amazon.awssdk.services.controltower.model.ListLandingZoneOperationsRequest;
import software.amazon.awssdk.services.controltower.model.ListLandingZoneOperationsResponse;
import software.amazon.awssdk.services.controltower.model.ListLandingZonesRequest;
import software.amazon.awssdk.services.controltower.model.ListLandingZonesResponse;
import software.amazon.awssdk.services.controltower.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.controltower.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.controltower.model.ResetEnabledBaselineRequest;
import software.amazon.awssdk.services.controltower.model.ResetEnabledBaselineResponse;
import software.amazon.awssdk.services.controltower.model.ResetEnabledControlRequest;
import software.amazon.awssdk.services.controltower.model.ResetEnabledControlResponse;
import software.amazon.awssdk.services.controltower.model.ResetLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.ResetLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.ResourceNotFoundException;
import software.amazon.awssdk.services.controltower.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.controltower.model.TagResourceRequest;
import software.amazon.awssdk.services.controltower.model.TagResourceResponse;
import software.amazon.awssdk.services.controltower.model.ThrottlingException;
import software.amazon.awssdk.services.controltower.model.UntagResourceRequest;
import software.amazon.awssdk.services.controltower.model.UntagResourceResponse;
import software.amazon.awssdk.services.controltower.model.UpdateEnabledBaselineRequest;
import software.amazon.awssdk.services.controltower.model.UpdateEnabledBaselineResponse;
import software.amazon.awssdk.services.controltower.model.UpdateEnabledControlRequest;
import software.amazon.awssdk.services.controltower.model.UpdateEnabledControlResponse;
import software.amazon.awssdk.services.controltower.model.UpdateLandingZoneRequest;
import software.amazon.awssdk.services.controltower.model.UpdateLandingZoneResponse;
import software.amazon.awssdk.services.controltower.model.ValidationException;
import software.amazon.awssdk.services.controltower.transform.CreateLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.DeleteLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.DisableBaselineRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.DisableControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.EnableBaselineRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.EnableControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetBaselineOperationRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetBaselineRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetControlOperationRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetEnabledBaselineRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetEnabledControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetLandingZoneOperationRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.GetLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListBaselinesRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListControlOperationsRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListEnabledBaselinesRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListEnabledControlsRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListLandingZoneOperationsRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListLandingZonesRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ResetEnabledBaselineRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ResetEnabledControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.ResetLandingZoneRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.UpdateEnabledBaselineRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.UpdateEnabledControlRequestMarshaller;
import software.amazon.awssdk.services.controltower.transform.UpdateLandingZoneRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ControlTowerClient}.
 *
 * @see ControlTowerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultControlTowerClient implements ControlTowerClient {
    private static final Logger log = Logger.loggerFor(DefaultControlTowerClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultControlTowerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "ControlTower" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new landing zone. This API call starts an asynchronous operation that creates and configures a landing
     * zone, based on the parameters specified in the manifest JSON file.
     * </p>
     *
     * @param createLandingZoneRequest
     * @return Result of the CreateLandingZone operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.CreateLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/CreateLandingZone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateLandingZoneResponse createLandingZone(CreateLandingZoneRequest createLandingZoneRequest)
            throws ConflictException, ValidationException, InternalServerException, AccessDeniedException, ThrottlingException,
            AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLandingZoneResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLandingZone");

            return clientHandler.execute(new ClientExecutionParams<CreateLandingZoneRequest, CreateLandingZoneResponse>()
                    .withOperationName("CreateLandingZone").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createLandingZoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLandingZoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Decommissions a landing zone. This API call starts an asynchronous operation that deletes Amazon Web Services
     * Control Tower resources deployed in accounts managed by Amazon Web Services Control Tower.
     * </p>
     * <p>
     * Decommissioning a landing zone is a process with significant consequences, and it cannot be undone. We strongly
     * recommend that you perform this decommissioning process only if you intend to stop using your landing zone.
     * </p>
     *
     * @param deleteLandingZoneRequest
     * @return Result of the DeleteLandingZone operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.DeleteLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DeleteLandingZone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteLandingZoneResponse deleteLandingZone(DeleteLandingZoneRequest deleteLandingZoneRequest)
            throws ConflictException, ValidationException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLandingZoneResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLandingZone");

            return clientHandler.execute(new ClientExecutionParams<DeleteLandingZoneRequest, DeleteLandingZoneResponse>()
                    .withOperationName("DeleteLandingZone").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteLandingZoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLandingZoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disable an <code>EnabledBaseline</code> resource on the specified Target. This API starts an asynchronous
     * operation to remove all resources deployed as part of the baseline enablement. The resource will vary depending
     * on the enabled baseline. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html"> <i>the Amazon Web
     * Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param disableBaselineRequest
     * @return Result of the DisableBaseline operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.DisableBaseline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DisableBaseline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisableBaselineResponse disableBaseline(DisableBaselineRequest disableBaselineRequest) throws ConflictException,
            ValidationException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableBaselineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisableBaselineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableBaselineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableBaselineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableBaseline");

            return clientHandler.execute(new ClientExecutionParams<DisableBaselineRequest, DisableBaselineResponse>()
                    .withOperationName("DisableBaseline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableBaselineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableBaselineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API call turns off a control. It starts an asynchronous operation that deletes Amazon Web Services resources
     * on the specified organizational unit and the accounts it contains. The resources will vary according to the
     * control that you specify. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param disableControlRequest
     * @return Result of the DisableControl operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.DisableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/DisableControl" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisableControlResponse disableControl(DisableControlRequest disableControlRequest) throws ConflictException,
            ValidationException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableControlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DisableControlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableControlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableControl");

            return clientHandler.execute(new ClientExecutionParams<DisableControlRequest, DisableControlResponse>()
                    .withOperationName("DisableControl").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(disableControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enable (apply) a <code>Baseline</code> to a Target. This API starts an asynchronous operation to deploy resources
     * specified by the <code>Baseline</code> to the specified Target. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html"> <i>the Amazon Web
     * Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param enableBaselineRequest
     * @return Result of the EnableBaseline operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.EnableBaseline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/EnableBaseline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableBaselineResponse enableBaseline(EnableBaselineRequest enableBaselineRequest) throws ConflictException,
            ValidationException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableBaselineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableBaselineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableBaselineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableBaselineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableBaseline");

            return clientHandler.execute(new ClientExecutionParams<EnableBaselineRequest, EnableBaselineResponse>()
                    .withOperationName("EnableBaseline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableBaselineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableBaselineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API call activates a control. It starts an asynchronous operation that creates Amazon Web Services resources
     * on the specified organizational unit and the accounts it contains. The resources created will vary according to
     * the control that you specify. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param enableControlRequest
     * @return Result of the EnableControl operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.EnableControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/EnableControl" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableControlResponse enableControl(EnableControlRequest enableControlRequest) throws ConflictException,
            ValidationException, ServiceQuotaExceededException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableControlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EnableControlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableControlRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableControl");

            return clientHandler.execute(new ClientExecutionParams<EnableControlRequest, EnableControlResponse>()
                    .withOperationName("EnableControl").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(enableControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve details about an existing <code>Baseline</code> resource by specifying its identifier. For usage
     * examples, see <a href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html">
     * <i>the Amazon Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param getBaselineRequest
     * @return Result of the GetBaseline operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetBaseline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetBaseline" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBaselineResponse getBaseline(GetBaselineRequest getBaselineRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBaselineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBaselineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBaselineRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBaselineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBaseline");

            return clientHandler.execute(new ClientExecutionParams<GetBaselineRequest, GetBaselineResponse>()
                    .withOperationName("GetBaseline").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBaselineRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBaselineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the details of an asynchronous baseline operation, as initiated by any of these APIs:
     * <code>EnableBaseline</code>, <code>DisableBaseline</code>, <code>UpdateEnabledBaseline</code>,
     * <code>ResetEnabledBaseline</code>. A status message is displayed in case of operation failure. For usage
     * examples, see <a href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html">
     * <i>the Amazon Web Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param getBaselineOperationRequest
     * @return Result of the GetBaselineOperation operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetBaselineOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetBaselineOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBaselineOperationResponse getBaselineOperation(GetBaselineOperationRequest getBaselineOperationRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBaselineOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetBaselineOperationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBaselineOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBaselineOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBaselineOperation");

            return clientHandler.execute(new ClientExecutionParams<GetBaselineOperationRequest, GetBaselineOperationResponse>()
                    .withOperationName("GetBaselineOperation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getBaselineOperationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBaselineOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the status of a particular <code>EnableControl</code> or <code>DisableControl</code> operation. Displays
     * a message in case of error. Details for an operation are available for 90 days. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param getControlOperationRequest
     * @return Result of the GetControlOperation operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetControlOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetControlOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetControlOperationResponse getControlOperation(GetControlOperationRequest getControlOperationRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetControlOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetControlOperationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getControlOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getControlOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetControlOperation");

            return clientHandler.execute(new ClientExecutionParams<GetControlOperationRequest, GetControlOperationResponse>()
                    .withOperationName("GetControlOperation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getControlOperationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetControlOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieve details of an <code>EnabledBaseline</code> resource by specifying its identifier.
     * </p>
     *
     * @param getEnabledBaselineRequest
     * @return Result of the GetEnabledBaseline operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetEnabledBaseline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetEnabledBaseline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEnabledBaselineResponse getEnabledBaseline(GetEnabledBaselineRequest getEnabledBaselineRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnabledBaselineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEnabledBaselineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnabledBaselineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnabledBaselineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnabledBaseline");

            return clientHandler.execute(new ClientExecutionParams<GetEnabledBaselineRequest, GetEnabledBaselineResponse>()
                    .withOperationName("GetEnabledBaseline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getEnabledBaselineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnabledBaselineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details about an enabled control. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param getEnabledControlRequest
     * @return Result of the GetEnabledControl operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetEnabledControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetEnabledControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEnabledControlResponse getEnabledControl(GetEnabledControlRequest getEnabledControlRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnabledControlResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEnabledControlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnabledControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnabledControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnabledControl");

            return clientHandler.execute(new ClientExecutionParams<GetEnabledControlRequest, GetEnabledControlResponse>()
                    .withOperationName("GetEnabledControl").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getEnabledControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnabledControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns details about the landing zone. Displays a message in case of error.
     * </p>
     *
     * @param getLandingZoneRequest
     * @return Result of the GetLandingZone operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetLandingZone" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetLandingZoneResponse getLandingZone(GetLandingZoneRequest getLandingZoneRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetLandingZoneResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLandingZoneRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLandingZone");

            return clientHandler.execute(new ClientExecutionParams<GetLandingZoneRequest, GetLandingZoneResponse>()
                    .withOperationName("GetLandingZone").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getLandingZoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetLandingZoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the status of the specified landing zone operation. Details for an operation are available for 90 days.
     * </p>
     *
     * @param getLandingZoneOperationRequest
     * @return Result of the GetLandingZoneOperation operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.GetLandingZoneOperation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/GetLandingZoneOperation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetLandingZoneOperationResponse getLandingZoneOperation(GetLandingZoneOperationRequest getLandingZoneOperationRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLandingZoneOperationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLandingZoneOperationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLandingZoneOperationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLandingZoneOperationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLandingZoneOperation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetLandingZoneOperationRequest, GetLandingZoneOperationResponse>()
                            .withOperationName("GetLandingZoneOperation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getLandingZoneOperationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetLandingZoneOperationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a summary list of all available baselines. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html"> <i>the Amazon Web
     * Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param listBaselinesRequest
     * @return Result of the ListBaselines operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListBaselines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListBaselines" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListBaselinesResponse listBaselines(ListBaselinesRequest listBaselinesRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBaselinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBaselinesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBaselinesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBaselinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBaselines");

            return clientHandler.execute(new ClientExecutionParams<ListBaselinesRequest, ListBaselinesResponse>()
                    .withOperationName("ListBaselines").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listBaselinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBaselinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Provides a list of operations in progress or queued. For usage examples, see <a href=
     * "https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html#list-control-operations-api-examples"
     * >ListControlOperation examples</a>.
     * </p>
     *
     * @param listControlOperationsRequest
     * @return Result of the ListControlOperations operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListControlOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListControlOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListControlOperationsResponse listControlOperations(ListControlOperationsRequest listControlOperationsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListControlOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListControlOperationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listControlOperationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listControlOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListControlOperations");

            return clientHandler.execute(new ClientExecutionParams<ListControlOperationsRequest, ListControlOperationsResponse>()
                    .withOperationName("ListControlOperations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listControlOperationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListControlOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of summaries describing <code>EnabledBaseline</code> resources. You can filter the list by the
     * corresponding <code>Baseline</code> or <code>Target</code> of the <code>EnabledBaseline</code> resources. For
     * usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html"> <i>the Amazon Web
     * Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param listEnabledBaselinesRequest
     * @return Result of the ListEnabledBaselines operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListEnabledBaselines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledBaselines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEnabledBaselinesResponse listEnabledBaselines(ListEnabledBaselinesRequest listEnabledBaselinesRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnabledBaselinesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEnabledBaselinesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnabledBaselinesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnabledBaselinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnabledBaselines");

            return clientHandler.execute(new ClientExecutionParams<ListEnabledBaselinesRequest, ListEnabledBaselinesResponse>()
                    .withOperationName("ListEnabledBaselines").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEnabledBaselinesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnabledBaselinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the controls enabled by Amazon Web Services Control Tower on the specified organizational unit and the
     * accounts it contains. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param listEnabledControlsRequest
     * @return Result of the ListEnabledControls operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListEnabledControls
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListEnabledControls"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEnabledControlsResponse listEnabledControls(ListEnabledControlsRequest listEnabledControlsRequest)
            throws ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnabledControlsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEnabledControlsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnabledControlsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnabledControlsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnabledControls");

            return clientHandler.execute(new ClientExecutionParams<ListEnabledControlsRequest, ListEnabledControlsResponse>()
                    .withOperationName("ListEnabledControls").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listEnabledControlsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnabledControlsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all landing zone operations from the past 90 days. Results are sorted by time, with the most recent
     * operation first.
     * </p>
     *
     * @param listLandingZoneOperationsRequest
     * @return Result of the ListLandingZoneOperations operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListLandingZoneOperations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListLandingZoneOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListLandingZoneOperationsResponse listLandingZoneOperations(
            ListLandingZoneOperationsRequest listLandingZoneOperationsRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLandingZoneOperationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListLandingZoneOperationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLandingZoneOperationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLandingZoneOperationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLandingZoneOperations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListLandingZoneOperationsRequest, ListLandingZoneOperationsResponse>()
                            .withOperationName("ListLandingZoneOperations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listLandingZoneOperationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListLandingZoneOperationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the landing zone ARN for the landing zone deployed in your managed account. This API also creates an ARN
     * for existing accounts that do not yet have a landing zone ARN.
     * </p>
     * <p>
     * Returns one landing zone ARN.
     * </p>
     *
     * @param listLandingZonesRequest
     * @return Result of the ListLandingZones operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListLandingZones
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListLandingZones" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListLandingZonesResponse listLandingZones(ListLandingZonesRequest listLandingZonesRequest) throws ValidationException,
            InternalServerException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLandingZonesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLandingZonesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLandingZonesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLandingZonesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLandingZones");

            return clientHandler.execute(new ClientExecutionParams<ListLandingZonesRequest, ListLandingZonesResponse>()
                    .withOperationName("ListLandingZones").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listLandingZonesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLandingZonesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags associated with the resource. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Re-enables an <code>EnabledBaseline</code> resource. For example, this API can re-apply the existing
     * <code>Baseline</code> after a new member account is moved to the target OU. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html"> <i>the Amazon Web
     * Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param resetEnabledBaselineRequest
     * @return Result of the ResetEnabledBaseline operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ResetEnabledBaseline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ResetEnabledBaseline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResetEnabledBaselineResponse resetEnabledBaseline(ResetEnabledBaselineRequest resetEnabledBaselineRequest)
            throws ConflictException, ValidationException, ServiceQuotaExceededException, InternalServerException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetEnabledBaselineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResetEnabledBaselineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetEnabledBaselineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetEnabledBaselineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetEnabledBaseline");

            return clientHandler.execute(new ClientExecutionParams<ResetEnabledBaselineRequest, ResetEnabledBaselineResponse>()
                    .withOperationName("ResetEnabledBaseline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(resetEnabledBaselineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResetEnabledBaselineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resets an enabled control. Does not work for controls implemented with SCPs.
     * </p>
     *
     * @param resetEnabledControlRequest
     * @return Result of the ResetEnabledControl operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ResetEnabledControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ResetEnabledControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ResetEnabledControlResponse resetEnabledControl(ResetEnabledControlRequest resetEnabledControlRequest)
            throws ConflictException, ValidationException, ServiceQuotaExceededException, InternalServerException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetEnabledControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ResetEnabledControlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetEnabledControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetEnabledControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetEnabledControl");

            return clientHandler.execute(new ClientExecutionParams<ResetEnabledControlRequest, ResetEnabledControlResponse>()
                    .withOperationName("ResetEnabledControl").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(resetEnabledControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResetEnabledControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API call resets a landing zone. It starts an asynchronous operation that resets the landing zone to the
     * parameters specified in the original configuration, which you specified in the manifest file. Nothing in the
     * manifest file's original landing zone configuration is changed during the reset process, by default. This API is
     * not the same as a rollback of a landing zone version, which is not a supported operation.
     * </p>
     *
     * @param resetLandingZoneRequest
     * @return Result of the ResetLandingZone operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.ResetLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/ResetLandingZone" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ResetLandingZoneResponse resetLandingZone(ResetLandingZoneRequest resetLandingZoneRequest) throws ConflictException,
            ValidationException, InternalServerException, AccessDeniedException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResetLandingZoneResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetLandingZone");

            return clientHandler.execute(new ClientExecutionParams<ResetLandingZoneRequest, ResetLandingZoneResponse>()
                    .withOperationName("ResetLandingZone").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(resetLandingZoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResetLandingZoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Applies tags to a resource. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a resource. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ResourceNotFoundException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an <code>EnabledBaseline</code> resource's applied parameters or version. For usage examples, see <a
     * href="https://docs.aws.amazon.com/controltower/latest/userguide/baseline-api-examples.html"> <i>the Amazon Web
     * Services Control Tower User Guide</i> </a>.
     * </p>
     *
     * @param updateEnabledBaselineRequest
     * @return Result of the UpdateEnabledBaseline operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.UpdateEnabledBaseline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/UpdateEnabledBaseline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEnabledBaselineResponse updateEnabledBaseline(UpdateEnabledBaselineRequest updateEnabledBaselineRequest)
            throws ConflictException, ValidationException, ServiceQuotaExceededException, InternalServerException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnabledBaselineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEnabledBaselineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnabledBaselineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnabledBaselineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnabledBaseline");

            return clientHandler.execute(new ClientExecutionParams<UpdateEnabledBaselineRequest, UpdateEnabledBaselineResponse>()
                    .withOperationName("UpdateEnabledBaseline").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateEnabledBaselineRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnabledBaselineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an already enabled control.
     * </p>
     * <p>
     * If the enabled control shows an <code>EnablementStatus</code> of SUCCEEDED, supply parameters that are different
     * from the currently configured parameters. Otherwise, Amazon Web Services Control Tower will not accept the
     * request.
     * </p>
     * <p>
     * If the enabled control shows an <code>EnablementStatus</code> of FAILED, Amazon Web Services Control Tower
     * updates the control to match any valid parameters that you supply.
     * </p>
     * <p>
     * If the <code>DriftSummary</code> status for the control shows as <code>DRIFTED</code>, you cannot call this API.
     * Instead, you can update the control by calling the <code>ResetEnabledControl</code> API. Alternatively, you can
     * call <code>DisableControl</code> and then call <code>EnableControl</code> again. Also, you can run an extending
     * governance operation to repair drift. For usage examples, see the <a
     * href="https://docs.aws.amazon.com/controltower/latest/controlreference/control-api-examples-short.html">
     * <i>Controls Reference Guide</i> </a>.
     * </p>
     *
     * @param updateEnabledControlRequest
     * @return Result of the UpdateEnabledControl operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The request would cause a service quota to be exceeded. See <a
     *         href="https://docs.aws.amazon.com/controltower/latest/userguide/request-an-increase.html">Service
     *         quotas</a>.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.UpdateEnabledControl
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/UpdateEnabledControl"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEnabledControlResponse updateEnabledControl(UpdateEnabledControlRequest updateEnabledControlRequest)
            throws ConflictException, ValidationException, ServiceQuotaExceededException, InternalServerException,
            AccessDeniedException, ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException,
            ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnabledControlResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEnabledControlResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnabledControlRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnabledControlRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnabledControl");

            return clientHandler.execute(new ClientExecutionParams<UpdateEnabledControlRequest, UpdateEnabledControlResponse>()
                    .withOperationName("UpdateEnabledControl").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateEnabledControlRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnabledControlRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This API call updates the landing zone. It starts an asynchronous operation that updates the landing zone based
     * on the new landing zone version, or on the changed parameters specified in the updated manifest file.
     * </p>
     *
     * @param updateLandingZoneRequest
     * @return Result of the UpdateLandingZone operation returned by the service.
     * @throws ConflictException
     *         Updating or deleting the resource can cause an inconsistent state.
     * @throws ValidationException
     *         The input does not satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         An unexpected error occurred during processing of a request.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws ResourceNotFoundException
     *         The request references a resource that does not exist.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ControlTowerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ControlTowerClient.UpdateLandingZone
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/controltower-2018-05-10/UpdateLandingZone"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateLandingZoneResponse updateLandingZone(UpdateLandingZoneRequest updateLandingZoneRequest)
            throws ConflictException, ValidationException, InternalServerException, AccessDeniedException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, ControlTowerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateLandingZoneResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateLandingZoneResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLandingZoneRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLandingZoneRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ControlTower");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLandingZone");

            return clientHandler.execute(new ClientExecutionParams<UpdateLandingZoneRequest, UpdateLandingZoneResponse>()
                    .withOperationName("UpdateLandingZone").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateLandingZoneRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateLandingZoneRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ControlTowerServiceClientConfigurationBuilder serviceConfigBuilder = new ControlTowerServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ControlTowerException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final ControlTowerServiceClientConfiguration serviceClientConfiguration() {
        return new ControlTowerServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
