/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costexplorer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An unusual cost pattern. This consists of the detailed metadata and the current status of the anomaly object.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Anomaly implements SdkPojo, Serializable, ToCopyableBuilder<Anomaly.Builder, Anomaly> {
    private static final SdkField<String> ANOMALY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AnomalyId").getter(getter(Anomaly::anomalyId)).setter(setter(Builder::anomalyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyId").build()).build();

    private static final SdkField<String> ANOMALY_START_DATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AnomalyStartDate").getter(getter(Anomaly::anomalyStartDate)).setter(setter(Builder::anomalyStartDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyStartDate").build()).build();

    private static final SdkField<String> ANOMALY_END_DATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AnomalyEndDate").getter(getter(Anomaly::anomalyEndDate)).setter(setter(Builder::anomalyEndDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyEndDate").build()).build();

    private static final SdkField<String> DIMENSION_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DimensionValue").getter(getter(Anomaly::dimensionValue)).setter(setter(Builder::dimensionValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DimensionValue").build()).build();

    private static final SdkField<List<RootCause>> ROOT_CAUSES_FIELD = SdkField
            .<List<RootCause>> builder(MarshallingType.LIST)
            .memberName("RootCauses")
            .getter(getter(Anomaly::rootCauses))
            .setter(setter(Builder::rootCauses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootCauses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RootCause> builder(MarshallingType.SDK_POJO)
                                            .constructor(RootCause::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AnomalyScore> ANOMALY_SCORE_FIELD = SdkField.<AnomalyScore> builder(MarshallingType.SDK_POJO)
            .memberName("AnomalyScore").getter(getter(Anomaly::anomalyScore)).setter(setter(Builder::anomalyScore))
            .constructor(AnomalyScore::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyScore").build()).build();

    private static final SdkField<Impact> IMPACT_FIELD = SdkField.<Impact> builder(MarshallingType.SDK_POJO).memberName("Impact")
            .getter(getter(Anomaly::impact)).setter(setter(Builder::impact)).constructor(Impact::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Impact").build()).build();

    private static final SdkField<String> MONITOR_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MonitorArn").getter(getter(Anomaly::monitorArn)).setter(setter(Builder::monitorArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MonitorArn").build()).build();

    private static final SdkField<String> FEEDBACK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Feedback").getter(getter(Anomaly::feedbackAsString)).setter(setter(Builder::feedback))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Feedback").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ANOMALY_ID_FIELD,
            ANOMALY_START_DATE_FIELD, ANOMALY_END_DATE_FIELD, DIMENSION_VALUE_FIELD, ROOT_CAUSES_FIELD, ANOMALY_SCORE_FIELD,
            IMPACT_FIELD, MONITOR_ARN_FIELD, FEEDBACK_FIELD));

    private static final long serialVersionUID = 1L;

    private final String anomalyId;

    private final String anomalyStartDate;

    private final String anomalyEndDate;

    private final String dimensionValue;

    private final List<RootCause> rootCauses;

    private final AnomalyScore anomalyScore;

    private final Impact impact;

    private final String monitorArn;

    private final String feedback;

    private Anomaly(BuilderImpl builder) {
        this.anomalyId = builder.anomalyId;
        this.anomalyStartDate = builder.anomalyStartDate;
        this.anomalyEndDate = builder.anomalyEndDate;
        this.dimensionValue = builder.dimensionValue;
        this.rootCauses = builder.rootCauses;
        this.anomalyScore = builder.anomalyScore;
        this.impact = builder.impact;
        this.monitorArn = builder.monitorArn;
        this.feedback = builder.feedback;
    }

    /**
     * <p>
     * The unique identifier for the anomaly.
     * </p>
     * 
     * @return The unique identifier for the anomaly.
     */
    public final String anomalyId() {
        return anomalyId;
    }

    /**
     * <p>
     * The first day the anomaly is detected.
     * </p>
     * 
     * @return The first day the anomaly is detected.
     */
    public final String anomalyStartDate() {
        return anomalyStartDate;
    }

    /**
     * <p>
     * The last day the anomaly is detected.
     * </p>
     * 
     * @return The last day the anomaly is detected.
     */
    public final String anomalyEndDate() {
        return anomalyEndDate;
    }

    /**
     * <p>
     * The dimension for the anomaly (for example, an Amazon Web Services service in a service monitor).
     * </p>
     * 
     * @return The dimension for the anomaly (for example, an Amazon Web Services service in a service monitor).
     */
    public final String dimensionValue() {
        return dimensionValue;
    }

    /**
     * For responses, this returns true if the service returned a value for the RootCauses property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRootCauses() {
        return rootCauses != null && !(rootCauses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of identified root causes for the anomaly.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRootCauses} method.
     * </p>
     * 
     * @return The list of identified root causes for the anomaly.
     */
    public final List<RootCause> rootCauses() {
        return rootCauses;
    }

    /**
     * <p>
     * The latest and maximum score for the anomaly.
     * </p>
     * 
     * @return The latest and maximum score for the anomaly.
     */
    public final AnomalyScore anomalyScore() {
        return anomalyScore;
    }

    /**
     * <p>
     * The dollar impact for the anomaly.
     * </p>
     * 
     * @return The dollar impact for the anomaly.
     */
    public final Impact impact() {
        return impact;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the cost monitor that generated this anomaly.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the cost monitor that generated this anomaly.
     */
    public final String monitorArn() {
        return monitorArn;
    }

    /**
     * <p>
     * The feedback value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #feedback} will
     * return {@link AnomalyFeedbackType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #feedbackAsString}.
     * </p>
     * 
     * @return The feedback value.
     * @see AnomalyFeedbackType
     */
    public final AnomalyFeedbackType feedback() {
        return AnomalyFeedbackType.fromValue(feedback);
    }

    /**
     * <p>
     * The feedback value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #feedback} will
     * return {@link AnomalyFeedbackType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #feedbackAsString}.
     * </p>
     * 
     * @return The feedback value.
     * @see AnomalyFeedbackType
     */
    public final String feedbackAsString() {
        return feedback;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(anomalyId());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyStartDate());
        hashCode = 31 * hashCode + Objects.hashCode(anomalyEndDate());
        hashCode = 31 * hashCode + Objects.hashCode(dimensionValue());
        hashCode = 31 * hashCode + Objects.hashCode(hasRootCauses() ? rootCauses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(anomalyScore());
        hashCode = 31 * hashCode + Objects.hashCode(impact());
        hashCode = 31 * hashCode + Objects.hashCode(monitorArn());
        hashCode = 31 * hashCode + Objects.hashCode(feedbackAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Anomaly)) {
            return false;
        }
        Anomaly other = (Anomaly) obj;
        return Objects.equals(anomalyId(), other.anomalyId()) && Objects.equals(anomalyStartDate(), other.anomalyStartDate())
                && Objects.equals(anomalyEndDate(), other.anomalyEndDate())
                && Objects.equals(dimensionValue(), other.dimensionValue()) && hasRootCauses() == other.hasRootCauses()
                && Objects.equals(rootCauses(), other.rootCauses()) && Objects.equals(anomalyScore(), other.anomalyScore())
                && Objects.equals(impact(), other.impact()) && Objects.equals(monitorArn(), other.monitorArn())
                && Objects.equals(feedbackAsString(), other.feedbackAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Anomaly").add("AnomalyId", anomalyId()).add("AnomalyStartDate", anomalyStartDate())
                .add("AnomalyEndDate", anomalyEndDate()).add("DimensionValue", dimensionValue())
                .add("RootCauses", hasRootCauses() ? rootCauses() : null).add("AnomalyScore", anomalyScore())
                .add("Impact", impact()).add("MonitorArn", monitorArn()).add("Feedback", feedbackAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AnomalyId":
            return Optional.ofNullable(clazz.cast(anomalyId()));
        case "AnomalyStartDate":
            return Optional.ofNullable(clazz.cast(anomalyStartDate()));
        case "AnomalyEndDate":
            return Optional.ofNullable(clazz.cast(anomalyEndDate()));
        case "DimensionValue":
            return Optional.ofNullable(clazz.cast(dimensionValue()));
        case "RootCauses":
            return Optional.ofNullable(clazz.cast(rootCauses()));
        case "AnomalyScore":
            return Optional.ofNullable(clazz.cast(anomalyScore()));
        case "Impact":
            return Optional.ofNullable(clazz.cast(impact()));
        case "MonitorArn":
            return Optional.ofNullable(clazz.cast(monitorArn()));
        case "Feedback":
            return Optional.ofNullable(clazz.cast(feedbackAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Anomaly, T> g) {
        return obj -> g.apply((Anomaly) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Anomaly> {
        /**
         * <p>
         * The unique identifier for the anomaly.
         * </p>
         * 
         * @param anomalyId
         *        The unique identifier for the anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyId(String anomalyId);

        /**
         * <p>
         * The first day the anomaly is detected.
         * </p>
         * 
         * @param anomalyStartDate
         *        The first day the anomaly is detected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyStartDate(String anomalyStartDate);

        /**
         * <p>
         * The last day the anomaly is detected.
         * </p>
         * 
         * @param anomalyEndDate
         *        The last day the anomaly is detected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyEndDate(String anomalyEndDate);

        /**
         * <p>
         * The dimension for the anomaly (for example, an Amazon Web Services service in a service monitor).
         * </p>
         * 
         * @param dimensionValue
         *        The dimension for the anomaly (for example, an Amazon Web Services service in a service monitor).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensionValue(String dimensionValue);

        /**
         * <p>
         * The list of identified root causes for the anomaly.
         * </p>
         * 
         * @param rootCauses
         *        The list of identified root causes for the anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootCauses(Collection<RootCause> rootCauses);

        /**
         * <p>
         * The list of identified root causes for the anomaly.
         * </p>
         * 
         * @param rootCauses
         *        The list of identified root causes for the anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootCauses(RootCause... rootCauses);

        /**
         * <p>
         * The list of identified root causes for the anomaly.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<RootCause>.Builder} avoiding the
         * need to create one manually via {@link List<RootCause>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RootCause>.Builder#build()} is called immediately and its
         * result is passed to {@link #rootCauses(List<RootCause>)}.
         * 
         * @param rootCauses
         *        a consumer that will call methods on {@link List<RootCause>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rootCauses(List<RootCause>)
         */
        Builder rootCauses(Consumer<RootCause.Builder>... rootCauses);

        /**
         * <p>
         * The latest and maximum score for the anomaly.
         * </p>
         * 
         * @param anomalyScore
         *        The latest and maximum score for the anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyScore(AnomalyScore anomalyScore);

        /**
         * <p>
         * The latest and maximum score for the anomaly.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnomalyScore.Builder} avoiding the need
         * to create one manually via {@link AnomalyScore#builder()}.
         *
         * When the {@link Consumer} completes, {@link AnomalyScore.Builder#build()} is called immediately and its
         * result is passed to {@link #anomalyScore(AnomalyScore)}.
         * 
         * @param anomalyScore
         *        a consumer that will call methods on {@link AnomalyScore.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalyScore(AnomalyScore)
         */
        default Builder anomalyScore(Consumer<AnomalyScore.Builder> anomalyScore) {
            return anomalyScore(AnomalyScore.builder().applyMutation(anomalyScore).build());
        }

        /**
         * <p>
         * The dollar impact for the anomaly.
         * </p>
         * 
         * @param impact
         *        The dollar impact for the anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder impact(Impact impact);

        /**
         * <p>
         * The dollar impact for the anomaly.
         * </p>
         * This is a convenience method that creates an instance of the {@link Impact.Builder} avoiding the need to
         * create one manually via {@link Impact#builder()}.
         *
         * When the {@link Consumer} completes, {@link Impact.Builder#build()} is called immediately and its result is
         * passed to {@link #impact(Impact)}.
         * 
         * @param impact
         *        a consumer that will call methods on {@link Impact.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #impact(Impact)
         */
        default Builder impact(Consumer<Impact.Builder> impact) {
            return impact(Impact.builder().applyMutation(impact).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the cost monitor that generated this anomaly.
         * </p>
         * 
         * @param monitorArn
         *        The Amazon Resource Name (ARN) for the cost monitor that generated this anomaly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitorArn(String monitorArn);

        /**
         * <p>
         * The feedback value.
         * </p>
         * 
         * @param feedback
         *        The feedback value.
         * @see AnomalyFeedbackType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyFeedbackType
         */
        Builder feedback(String feedback);

        /**
         * <p>
         * The feedback value.
         * </p>
         * 
         * @param feedback
         *        The feedback value.
         * @see AnomalyFeedbackType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyFeedbackType
         */
        Builder feedback(AnomalyFeedbackType feedback);
    }

    static final class BuilderImpl implements Builder {
        private String anomalyId;

        private String anomalyStartDate;

        private String anomalyEndDate;

        private String dimensionValue;

        private List<RootCause> rootCauses = DefaultSdkAutoConstructList.getInstance();

        private AnomalyScore anomalyScore;

        private Impact impact;

        private String monitorArn;

        private String feedback;

        private BuilderImpl() {
        }

        private BuilderImpl(Anomaly model) {
            anomalyId(model.anomalyId);
            anomalyStartDate(model.anomalyStartDate);
            anomalyEndDate(model.anomalyEndDate);
            dimensionValue(model.dimensionValue);
            rootCauses(model.rootCauses);
            anomalyScore(model.anomalyScore);
            impact(model.impact);
            monitorArn(model.monitorArn);
            feedback(model.feedback);
        }

        public final String getAnomalyId() {
            return anomalyId;
        }

        public final void setAnomalyId(String anomalyId) {
            this.anomalyId = anomalyId;
        }

        @Override
        public final Builder anomalyId(String anomalyId) {
            this.anomalyId = anomalyId;
            return this;
        }

        public final String getAnomalyStartDate() {
            return anomalyStartDate;
        }

        public final void setAnomalyStartDate(String anomalyStartDate) {
            this.anomalyStartDate = anomalyStartDate;
        }

        @Override
        public final Builder anomalyStartDate(String anomalyStartDate) {
            this.anomalyStartDate = anomalyStartDate;
            return this;
        }

        public final String getAnomalyEndDate() {
            return anomalyEndDate;
        }

        public final void setAnomalyEndDate(String anomalyEndDate) {
            this.anomalyEndDate = anomalyEndDate;
        }

        @Override
        public final Builder anomalyEndDate(String anomalyEndDate) {
            this.anomalyEndDate = anomalyEndDate;
            return this;
        }

        public final String getDimensionValue() {
            return dimensionValue;
        }

        public final void setDimensionValue(String dimensionValue) {
            this.dimensionValue = dimensionValue;
        }

        @Override
        public final Builder dimensionValue(String dimensionValue) {
            this.dimensionValue = dimensionValue;
            return this;
        }

        public final List<RootCause.Builder> getRootCauses() {
            List<RootCause.Builder> result = RootCausesCopier.copyToBuilder(this.rootCauses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRootCauses(Collection<RootCause.BuilderImpl> rootCauses) {
            this.rootCauses = RootCausesCopier.copyFromBuilder(rootCauses);
        }

        @Override
        public final Builder rootCauses(Collection<RootCause> rootCauses) {
            this.rootCauses = RootCausesCopier.copy(rootCauses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rootCauses(RootCause... rootCauses) {
            rootCauses(Arrays.asList(rootCauses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rootCauses(Consumer<RootCause.Builder>... rootCauses) {
            rootCauses(Stream.of(rootCauses).map(c -> RootCause.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final AnomalyScore.Builder getAnomalyScore() {
            return anomalyScore != null ? anomalyScore.toBuilder() : null;
        }

        public final void setAnomalyScore(AnomalyScore.BuilderImpl anomalyScore) {
            this.anomalyScore = anomalyScore != null ? anomalyScore.build() : null;
        }

        @Override
        public final Builder anomalyScore(AnomalyScore anomalyScore) {
            this.anomalyScore = anomalyScore;
            return this;
        }

        public final Impact.Builder getImpact() {
            return impact != null ? impact.toBuilder() : null;
        }

        public final void setImpact(Impact.BuilderImpl impact) {
            this.impact = impact != null ? impact.build() : null;
        }

        @Override
        public final Builder impact(Impact impact) {
            this.impact = impact;
            return this;
        }

        public final String getMonitorArn() {
            return monitorArn;
        }

        public final void setMonitorArn(String monitorArn) {
            this.monitorArn = monitorArn;
        }

        @Override
        public final Builder monitorArn(String monitorArn) {
            this.monitorArn = monitorArn;
            return this;
        }

        public final String getFeedback() {
            return feedback;
        }

        public final void setFeedback(String feedback) {
            this.feedback = feedback;
        }

        @Override
        public final Builder feedback(String feedback) {
            this.feedback = feedback;
            return this;
        }

        @Override
        public final Builder feedback(AnomalyFeedbackType feedback) {
            this.feedback(feedback == null ? null : feedback.toString());
            return this;
        }

        @Override
        public Anomaly build() {
            return new Anomaly(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
