/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costexplorer.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetCostAndUsageWithResourcesRequest extends CostExplorerRequest implements
        ToCopyableBuilder<GetCostAndUsageWithResourcesRequest.Builder, GetCostAndUsageWithResourcesRequest> {
    private static final SdkField<DateInterval> TIME_PERIOD_FIELD = SdkField.<DateInterval> builder(MarshallingType.SDK_POJO)
            .memberName("TimePeriod").getter(getter(GetCostAndUsageWithResourcesRequest::timePeriod))
            .setter(setter(Builder::timePeriod)).constructor(DateInterval::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimePeriod").build()).build();

    private static final SdkField<String> GRANULARITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Granularity").getter(getter(GetCostAndUsageWithResourcesRequest::granularityAsString))
            .setter(setter(Builder::granularity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Granularity").build()).build();

    private static final SdkField<Expression> FILTER_FIELD = SdkField.<Expression> builder(MarshallingType.SDK_POJO)
            .memberName("Filter").getter(getter(GetCostAndUsageWithResourcesRequest::filter)).setter(setter(Builder::filter))
            .constructor(Expression::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter").build()).build();

    private static final SdkField<List<String>> METRICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Metrics")
            .getter(getter(GetCostAndUsageWithResourcesRequest::metrics))
            .setter(setter(Builder::metrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<GroupDefinition>> GROUP_BY_FIELD = SdkField
            .<List<GroupDefinition>> builder(MarshallingType.LIST)
            .memberName("GroupBy")
            .getter(getter(GetCostAndUsageWithResourcesRequest::groupBy))
            .setter(setter(Builder::groupBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupBy").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<GroupDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(GroupDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_PAGE_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextPageToken").getter(getter(GetCostAndUsageWithResourcesRequest::nextPageToken))
            .setter(setter(Builder::nextPageToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextPageToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIME_PERIOD_FIELD,
            GRANULARITY_FIELD, FILTER_FIELD, METRICS_FIELD, GROUP_BY_FIELD, NEXT_PAGE_TOKEN_FIELD));

    private final DateInterval timePeriod;

    private final String granularity;

    private final Expression filter;

    private final List<String> metrics;

    private final List<GroupDefinition> groupBy;

    private final String nextPageToken;

    private GetCostAndUsageWithResourcesRequest(BuilderImpl builder) {
        super(builder);
        this.timePeriod = builder.timePeriod;
        this.granularity = builder.granularity;
        this.filter = builder.filter;
        this.metrics = builder.metrics;
        this.groupBy = builder.groupBy;
        this.nextPageToken = builder.nextPageToken;
    }

    /**
     * <p>
     * Sets the start and end dates for retrieving Amazon Web Services costs. The range must be within the last 14 days
     * (the start date cannot be earlier than 14 days ago). The start date is inclusive, but the end date is exclusive.
     * For example, if <code>start</code> is <code>2017-01-01</code> and <code>end</code> is <code>2017-05-01</code>,
     * then the cost and usage data is retrieved from <code>2017-01-01</code> up to and including
     * <code>2017-04-30</code> but not including <code>2017-05-01</code>.
     * </p>
     * 
     * @return Sets the start and end dates for retrieving Amazon Web Services costs. The range must be within the last
     *         14 days (the start date cannot be earlier than 14 days ago). The start date is inclusive, but the end
     *         date is exclusive. For example, if <code>start</code> is <code>2017-01-01</code> and <code>end</code> is
     *         <code>2017-05-01</code>, then the cost and usage data is retrieved from <code>2017-01-01</code> up to and
     *         including <code>2017-04-30</code> but not including <code>2017-05-01</code>.
     */
    public final DateInterval timePeriod() {
        return timePeriod;
    }

    /**
     * <p>
     * Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>
     * . If <code>Granularity</code> isn't set, the response object doesn't include the <code>Granularity</code>,
     * <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #granularity} will
     * return {@link Granularity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #granularityAsString}.
     * </p>
     * 
     * @return Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or
     *         <code>HOURLY</code>. If <code>Granularity</code> isn't set, the response object doesn't include the
     *         <code>Granularity</code>, <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
     * @see Granularity
     */
    public final Granularity granularity() {
        return Granularity.fromValue(granularity);
    }

    /**
     * <p>
     * Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>
     * . If <code>Granularity</code> isn't set, the response object doesn't include the <code>Granularity</code>,
     * <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #granularity} will
     * return {@link Granularity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #granularityAsString}.
     * </p>
     * 
     * @return Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or
     *         <code>HOURLY</code>. If <code>Granularity</code> isn't set, the response object doesn't include the
     *         <code>Granularity</code>, <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
     * @see Granularity
     */
    public final String granularityAsString() {
        return granularity;
    }

    /**
     * <p>
     * Filters Amazon Web Services costs by different dimensions. For example, you can specify <code>SERVICE</code> and
     * <code>LINKED_ACCOUNT</code> and get the costs that are associated with that account's usage of that service. You
     * can nest <code>Expression</code> objects to define any combination of dimension filters. For more information,
     * see <a
     * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html">Expression</a>.
     * </p>
     * <p>
     * The <code>GetCostAndUsageWithResources</code> operation requires that you either group by or filter by a
     * <code>ResourceId</code>. It requires the <a
     * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html">Expression</a>
     * <code>"SERVICE = Amazon Elastic Compute Cloud - Compute"</code> in the filter.
     * </p>
     * <p>
     * Valid values for <code>MatchOptions</code> for <code>Dimensions</code> are <code>EQUALS</code> and
     * <code>CASE_SENSITIVE</code>.
     * </p>
     * <p>
     * Valid values for <code>MatchOptions</code> for <code>CostCategories</code> and <code>Tags</code> are
     * <code>EQUALS</code>, <code>ABSENT</code>, and <code>CASE_SENSITIVE</code>. Default values are <code>EQUALS</code>
     * and <code>CASE_SENSITIVE</code>.
     * </p>
     * 
     * @return Filters Amazon Web Services costs by different dimensions. For example, you can specify
     *         <code>SERVICE</code> and <code>LINKED_ACCOUNT</code> and get the costs that are associated with that
     *         account's usage of that service. You can nest <code>Expression</code> objects to define any combination
     *         of dimension filters. For more information, see <a
     *         href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html"
     *         >Expression</a>. </p>
     *         <p>
     *         The <code>GetCostAndUsageWithResources</code> operation requires that you either group by or filter by a
     *         <code>ResourceId</code>. It requires the <a
     *         href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html"
     *         >Expression</a> <code>"SERVICE = Amazon Elastic Compute Cloud - Compute"</code> in the filter.
     *         </p>
     *         <p>
     *         Valid values for <code>MatchOptions</code> for <code>Dimensions</code> are <code>EQUALS</code> and
     *         <code>CASE_SENSITIVE</code>.
     *         </p>
     *         <p>
     *         Valid values for <code>MatchOptions</code> for <code>CostCategories</code> and <code>Tags</code> are
     *         <code>EQUALS</code>, <code>ABSENT</code>, and <code>CASE_SENSITIVE</code>. Default values are
     *         <code>EQUALS</code> and <code>CASE_SENSITIVE</code>.
     */
    public final Expression filter() {
        return filter;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metrics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetrics() {
        return metrics != null && !(metrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Which metrics are returned in the query. For more information about blended and unblended rates, see <a
     * href="http://aws.amazon.com/premiumsupport/knowledge-center/blended-rates-intro/">Why does the "blended"
     * annotation appear on some line items in my bill?</a>.
     * </p>
     * <p>
     * Valid values are <code>AmortizedCost</code>, <code>BlendedCost</code>, <code>NetAmortizedCost</code>,
     * <code>NetUnblendedCost</code>, <code>NormalizedUsageAmount</code>, <code>UnblendedCost</code>, and
     * <code>UsageQuantity</code>.
     * </p>
     * <note>
     * <p>
     * If you return the <code>UsageQuantity</code> metric, the service aggregates all usage numbers without taking the
     * units into account. For example, if you aggregate <code>usageQuantity</code> across all of Amazon EC2, the
     * results aren't meaningful because Amazon EC2 compute hours and data transfer are measured in different units (for
     * example, hours vs. GB). To get more meaningful <code>UsageQuantity</code> metrics, filter by
     * <code>UsageType</code> or <code>UsageTypeGroups</code>.
     * </p>
     * </note>
     * <p>
     * <code>Metrics</code> is required for <code>GetCostAndUsageWithResources</code> requests.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetrics} method.
     * </p>
     * 
     * @return Which metrics are returned in the query. For more information about blended and unblended rates, see <a
     *         href="http://aws.amazon.com/premiumsupport/knowledge-center/blended-rates-intro/">Why does the "blended"
     *         annotation appear on some line items in my bill?</a>. </p>
     *         <p>
     *         Valid values are <code>AmortizedCost</code>, <code>BlendedCost</code>, <code>NetAmortizedCost</code>,
     *         <code>NetUnblendedCost</code>, <code>NormalizedUsageAmount</code>, <code>UnblendedCost</code>, and
     *         <code>UsageQuantity</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If you return the <code>UsageQuantity</code> metric, the service aggregates all usage numbers without
     *         taking the units into account. For example, if you aggregate <code>usageQuantity</code> across all of
     *         Amazon EC2, the results aren't meaningful because Amazon EC2 compute hours and data transfer are measured
     *         in different units (for example, hours vs. GB). To get more meaningful <code>UsageQuantity</code>
     *         metrics, filter by <code>UsageType</code> or <code>UsageTypeGroups</code>.
     *         </p>
     *         </note>
     *         <p>
     *         <code>Metrics</code> is required for <code>GetCostAndUsageWithResources</code> requests.
     */
    public final List<String> metrics() {
        return metrics;
    }

    /**
     * For responses, this returns true if the service returned a value for the GroupBy property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGroupBy() {
        return groupBy != null && !(groupBy instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
     * <code>TAG</code>, <code>COST_CATEGORY</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGroupBy} method.
     * </p>
     * 
     * @return You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
     *         <code>TAG</code>, <code>COST_CATEGORY</code>.
     */
    public final List<GroupDefinition> groupBy() {
        return groupBy;
    }

    /**
     * <p>
     * The token to retrieve the next set of results. Amazon Web Services provides the token when the response from a
     * previous call has more results than the maximum page size.
     * </p>
     * 
     * @return The token to retrieve the next set of results. Amazon Web Services provides the token when the response
     *         from a previous call has more results than the maximum page size.
     */
    public final String nextPageToken() {
        return nextPageToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(timePeriod());
        hashCode = 31 * hashCode + Objects.hashCode(granularityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(filter());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetrics() ? metrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasGroupBy() ? groupBy() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextPageToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetCostAndUsageWithResourcesRequest)) {
            return false;
        }
        GetCostAndUsageWithResourcesRequest other = (GetCostAndUsageWithResourcesRequest) obj;
        return Objects.equals(timePeriod(), other.timePeriod())
                && Objects.equals(granularityAsString(), other.granularityAsString()) && Objects.equals(filter(), other.filter())
                && hasMetrics() == other.hasMetrics() && Objects.equals(metrics(), other.metrics())
                && hasGroupBy() == other.hasGroupBy() && Objects.equals(groupBy(), other.groupBy())
                && Objects.equals(nextPageToken(), other.nextPageToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetCostAndUsageWithResourcesRequest").add("TimePeriod", timePeriod())
                .add("Granularity", granularityAsString()).add("Filter", filter())
                .add("Metrics", hasMetrics() ? metrics() : null).add("GroupBy", hasGroupBy() ? groupBy() : null)
                .add("NextPageToken", nextPageToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TimePeriod":
            return Optional.ofNullable(clazz.cast(timePeriod()));
        case "Granularity":
            return Optional.ofNullable(clazz.cast(granularityAsString()));
        case "Filter":
            return Optional.ofNullable(clazz.cast(filter()));
        case "Metrics":
            return Optional.ofNullable(clazz.cast(metrics()));
        case "GroupBy":
            return Optional.ofNullable(clazz.cast(groupBy()));
        case "NextPageToken":
            return Optional.ofNullable(clazz.cast(nextPageToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetCostAndUsageWithResourcesRequest, T> g) {
        return obj -> g.apply((GetCostAndUsageWithResourcesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CostExplorerRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetCostAndUsageWithResourcesRequest> {
        /**
         * <p>
         * Sets the start and end dates for retrieving Amazon Web Services costs. The range must be within the last 14
         * days (the start date cannot be earlier than 14 days ago). The start date is inclusive, but the end date is
         * exclusive. For example, if <code>start</code> is <code>2017-01-01</code> and <code>end</code> is
         * <code>2017-05-01</code>, then the cost and usage data is retrieved from <code>2017-01-01</code> up to and
         * including <code>2017-04-30</code> but not including <code>2017-05-01</code>.
         * </p>
         * 
         * @param timePeriod
         *        Sets the start and end dates for retrieving Amazon Web Services costs. The range must be within the
         *        last 14 days (the start date cannot be earlier than 14 days ago). The start date is inclusive, but the
         *        end date is exclusive. For example, if <code>start</code> is <code>2017-01-01</code> and
         *        <code>end</code> is <code>2017-05-01</code>, then the cost and usage data is retrieved from
         *        <code>2017-01-01</code> up to and including <code>2017-04-30</code> but not including
         *        <code>2017-05-01</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timePeriod(DateInterval timePeriod);

        /**
         * <p>
         * Sets the start and end dates for retrieving Amazon Web Services costs. The range must be within the last 14
         * days (the start date cannot be earlier than 14 days ago). The start date is inclusive, but the end date is
         * exclusive. For example, if <code>start</code> is <code>2017-01-01</code> and <code>end</code> is
         * <code>2017-05-01</code>, then the cost and usage data is retrieved from <code>2017-01-01</code> up to and
         * including <code>2017-04-30</code> but not including <code>2017-05-01</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link DateInterval.Builder} avoiding the need
         * to create one manually via {@link DateInterval#builder()}.
         *
         * When the {@link Consumer} completes, {@link DateInterval.Builder#build()} is called immediately and its
         * result is passed to {@link #timePeriod(DateInterval)}.
         * 
         * @param timePeriod
         *        a consumer that will call methods on {@link DateInterval.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timePeriod(DateInterval)
         */
        default Builder timePeriod(Consumer<DateInterval.Builder> timePeriod) {
            return timePeriod(DateInterval.builder().applyMutation(timePeriod).build());
        }

        /**
         * <p>
         * Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or
         * <code>HOURLY</code>. If <code>Granularity</code> isn't set, the response object doesn't include the
         * <code>Granularity</code>, <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
         * </p>
         * 
         * @param granularity
         *        Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or
         *        <code>HOURLY</code>. If <code>Granularity</code> isn't set, the response object doesn't include the
         *        <code>Granularity</code>, <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
         * @see Granularity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Granularity
         */
        Builder granularity(String granularity);

        /**
         * <p>
         * Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or
         * <code>HOURLY</code>. If <code>Granularity</code> isn't set, the response object doesn't include the
         * <code>Granularity</code>, <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
         * </p>
         * 
         * @param granularity
         *        Sets the Amazon Web Services cost granularity to <code>MONTHLY</code>, <code>DAILY</code>, or
         *        <code>HOURLY</code>. If <code>Granularity</code> isn't set, the response object doesn't include the
         *        <code>Granularity</code>, <code>MONTHLY</code>, <code>DAILY</code>, or <code>HOURLY</code>.
         * @see Granularity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Granularity
         */
        Builder granularity(Granularity granularity);

        /**
         * <p>
         * Filters Amazon Web Services costs by different dimensions. For example, you can specify <code>SERVICE</code>
         * and <code>LINKED_ACCOUNT</code> and get the costs that are associated with that account's usage of that
         * service. You can nest <code>Expression</code> objects to define any combination of dimension filters. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html"
         * >Expression</a>.
         * </p>
         * <p>
         * The <code>GetCostAndUsageWithResources</code> operation requires that you either group by or filter by a
         * <code>ResourceId</code>. It requires the <a
         * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html">Expression</a>
         * <code>"SERVICE = Amazon Elastic Compute Cloud - Compute"</code> in the filter.
         * </p>
         * <p>
         * Valid values for <code>MatchOptions</code> for <code>Dimensions</code> are <code>EQUALS</code> and
         * <code>CASE_SENSITIVE</code>.
         * </p>
         * <p>
         * Valid values for <code>MatchOptions</code> for <code>CostCategories</code> and <code>Tags</code> are
         * <code>EQUALS</code>, <code>ABSENT</code>, and <code>CASE_SENSITIVE</code>. Default values are
         * <code>EQUALS</code> and <code>CASE_SENSITIVE</code>.
         * </p>
         * 
         * @param filter
         *        Filters Amazon Web Services costs by different dimensions. For example, you can specify
         *        <code>SERVICE</code> and <code>LINKED_ACCOUNT</code> and get the costs that are associated with that
         *        account's usage of that service. You can nest <code>Expression</code> objects to define any
         *        combination of dimension filters. For more information, see <a
         *        href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html"
         *        >Expression</a>. </p>
         *        <p>
         *        The <code>GetCostAndUsageWithResources</code> operation requires that you either group by or filter by
         *        a <code>ResourceId</code>. It requires the <a
         *        href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html"
         *        >Expression</a> <code>"SERVICE = Amazon Elastic Compute Cloud - Compute"</code> in the filter.
         *        </p>
         *        <p>
         *        Valid values for <code>MatchOptions</code> for <code>Dimensions</code> are <code>EQUALS</code> and
         *        <code>CASE_SENSITIVE</code>.
         *        </p>
         *        <p>
         *        Valid values for <code>MatchOptions</code> for <code>CostCategories</code> and <code>Tags</code> are
         *        <code>EQUALS</code>, <code>ABSENT</code>, and <code>CASE_SENSITIVE</code>. Default values are
         *        <code>EQUALS</code> and <code>CASE_SENSITIVE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filter(Expression filter);

        /**
         * <p>
         * Filters Amazon Web Services costs by different dimensions. For example, you can specify <code>SERVICE</code>
         * and <code>LINKED_ACCOUNT</code> and get the costs that are associated with that account's usage of that
         * service. You can nest <code>Expression</code> objects to define any combination of dimension filters. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html"
         * >Expression</a>.
         * </p>
         * <p>
         * The <code>GetCostAndUsageWithResources</code> operation requires that you either group by or filter by a
         * <code>ResourceId</code>. It requires the <a
         * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_Expression.html">Expression</a>
         * <code>"SERVICE = Amazon Elastic Compute Cloud - Compute"</code> in the filter.
         * </p>
         * <p>
         * Valid values for <code>MatchOptions</code> for <code>Dimensions</code> are <code>EQUALS</code> and
         * <code>CASE_SENSITIVE</code>.
         * </p>
         * <p>
         * Valid values for <code>MatchOptions</code> for <code>CostCategories</code> and <code>Tags</code> are
         * <code>EQUALS</code>, <code>ABSENT</code>, and <code>CASE_SENSITIVE</code>. Default values are
         * <code>EQUALS</code> and <code>CASE_SENSITIVE</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link Expression.Builder} avoiding the need to
         * create one manually via {@link Expression#builder()}.
         *
         * When the {@link Consumer} completes, {@link Expression.Builder#build()} is called immediately and its result
         * is passed to {@link #filter(Expression)}.
         * 
         * @param filter
         *        a consumer that will call methods on {@link Expression.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filter(Expression)
         */
        default Builder filter(Consumer<Expression.Builder> filter) {
            return filter(Expression.builder().applyMutation(filter).build());
        }

        /**
         * <p>
         * Which metrics are returned in the query. For more information about blended and unblended rates, see <a
         * href="http://aws.amazon.com/premiumsupport/knowledge-center/blended-rates-intro/">Why does the "blended"
         * annotation appear on some line items in my bill?</a>.
         * </p>
         * <p>
         * Valid values are <code>AmortizedCost</code>, <code>BlendedCost</code>, <code>NetAmortizedCost</code>,
         * <code>NetUnblendedCost</code>, <code>NormalizedUsageAmount</code>, <code>UnblendedCost</code>, and
         * <code>UsageQuantity</code>.
         * </p>
         * <note>
         * <p>
         * If you return the <code>UsageQuantity</code> metric, the service aggregates all usage numbers without taking
         * the units into account. For example, if you aggregate <code>usageQuantity</code> across all of Amazon EC2,
         * the results aren't meaningful because Amazon EC2 compute hours and data transfer are measured in different
         * units (for example, hours vs. GB). To get more meaningful <code>UsageQuantity</code> metrics, filter by
         * <code>UsageType</code> or <code>UsageTypeGroups</code>.
         * </p>
         * </note>
         * <p>
         * <code>Metrics</code> is required for <code>GetCostAndUsageWithResources</code> requests.
         * </p>
         * 
         * @param metrics
         *        Which metrics are returned in the query. For more information about blended and unblended rates, see
         *        <a href="http://aws.amazon.com/premiumsupport/knowledge-center/blended-rates-intro/">Why does the
         *        "blended" annotation appear on some line items in my bill?</a>. </p>
         *        <p>
         *        Valid values are <code>AmortizedCost</code>, <code>BlendedCost</code>, <code>NetAmortizedCost</code>,
         *        <code>NetUnblendedCost</code>, <code>NormalizedUsageAmount</code>, <code>UnblendedCost</code>, and
         *        <code>UsageQuantity</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If you return the <code>UsageQuantity</code> metric, the service aggregates all usage numbers without
         *        taking the units into account. For example, if you aggregate <code>usageQuantity</code> across all of
         *        Amazon EC2, the results aren't meaningful because Amazon EC2 compute hours and data transfer are
         *        measured in different units (for example, hours vs. GB). To get more meaningful
         *        <code>UsageQuantity</code> metrics, filter by <code>UsageType</code> or <code>UsageTypeGroups</code>.
         *        </p>
         *        </note>
         *        <p>
         *        <code>Metrics</code> is required for <code>GetCostAndUsageWithResources</code> requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Collection<String> metrics);

        /**
         * <p>
         * Which metrics are returned in the query. For more information about blended and unblended rates, see <a
         * href="http://aws.amazon.com/premiumsupport/knowledge-center/blended-rates-intro/">Why does the "blended"
         * annotation appear on some line items in my bill?</a>.
         * </p>
         * <p>
         * Valid values are <code>AmortizedCost</code>, <code>BlendedCost</code>, <code>NetAmortizedCost</code>,
         * <code>NetUnblendedCost</code>, <code>NormalizedUsageAmount</code>, <code>UnblendedCost</code>, and
         * <code>UsageQuantity</code>.
         * </p>
         * <note>
         * <p>
         * If you return the <code>UsageQuantity</code> metric, the service aggregates all usage numbers without taking
         * the units into account. For example, if you aggregate <code>usageQuantity</code> across all of Amazon EC2,
         * the results aren't meaningful because Amazon EC2 compute hours and data transfer are measured in different
         * units (for example, hours vs. GB). To get more meaningful <code>UsageQuantity</code> metrics, filter by
         * <code>UsageType</code> or <code>UsageTypeGroups</code>.
         * </p>
         * </note>
         * <p>
         * <code>Metrics</code> is required for <code>GetCostAndUsageWithResources</code> requests.
         * </p>
         * 
         * @param metrics
         *        Which metrics are returned in the query. For more information about blended and unblended rates, see
         *        <a href="http://aws.amazon.com/premiumsupport/knowledge-center/blended-rates-intro/">Why does the
         *        "blended" annotation appear on some line items in my bill?</a>. </p>
         *        <p>
         *        Valid values are <code>AmortizedCost</code>, <code>BlendedCost</code>, <code>NetAmortizedCost</code>,
         *        <code>NetUnblendedCost</code>, <code>NormalizedUsageAmount</code>, <code>UnblendedCost</code>, and
         *        <code>UsageQuantity</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If you return the <code>UsageQuantity</code> metric, the service aggregates all usage numbers without
         *        taking the units into account. For example, if you aggregate <code>usageQuantity</code> across all of
         *        Amazon EC2, the results aren't meaningful because Amazon EC2 compute hours and data transfer are
         *        measured in different units (for example, hours vs. GB). To get more meaningful
         *        <code>UsageQuantity</code> metrics, filter by <code>UsageType</code> or <code>UsageTypeGroups</code>.
         *        </p>
         *        </note>
         *        <p>
         *        <code>Metrics</code> is required for <code>GetCostAndUsageWithResources</code> requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(String... metrics);

        /**
         * <p>
         * You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
         * <code>TAG</code>, <code>COST_CATEGORY</code>.
         * </p>
         * 
         * @param groupBy
         *        You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
         *        <code>TAG</code>, <code>COST_CATEGORY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupBy(Collection<GroupDefinition> groupBy);

        /**
         * <p>
         * You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
         * <code>TAG</code>, <code>COST_CATEGORY</code>.
         * </p>
         * 
         * @param groupBy
         *        You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
         *        <code>TAG</code>, <code>COST_CATEGORY</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupBy(GroupDefinition... groupBy);

        /**
         * <p>
         * You can group Amazon Web Services costs using up to two different groups: <code>DIMENSION</code>,
         * <code>TAG</code>, <code>COST_CATEGORY</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<GroupDefinition>.Builder} avoiding
         * the need to create one manually via {@link List<GroupDefinition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<GroupDefinition>.Builder#build()} is called immediately and
         * its result is passed to {@link #groupBy(List<GroupDefinition>)}.
         * 
         * @param groupBy
         *        a consumer that will call methods on {@link List<GroupDefinition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #groupBy(List<GroupDefinition>)
         */
        Builder groupBy(Consumer<GroupDefinition.Builder>... groupBy);

        /**
         * <p>
         * The token to retrieve the next set of results. Amazon Web Services provides the token when the response from
         * a previous call has more results than the maximum page size.
         * </p>
         * 
         * @param nextPageToken
         *        The token to retrieve the next set of results. Amazon Web Services provides the token when the
         *        response from a previous call has more results than the maximum page size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextPageToken(String nextPageToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CostExplorerRequest.BuilderImpl implements Builder {
        private DateInterval timePeriod;

        private String granularity;

        private Expression filter;

        private List<String> metrics = DefaultSdkAutoConstructList.getInstance();

        private List<GroupDefinition> groupBy = DefaultSdkAutoConstructList.getInstance();

        private String nextPageToken;

        private BuilderImpl() {
        }

        private BuilderImpl(GetCostAndUsageWithResourcesRequest model) {
            super(model);
            timePeriod(model.timePeriod);
            granularity(model.granularity);
            filter(model.filter);
            metrics(model.metrics);
            groupBy(model.groupBy);
            nextPageToken(model.nextPageToken);
        }

        public final DateInterval.Builder getTimePeriod() {
            return timePeriod != null ? timePeriod.toBuilder() : null;
        }

        public final void setTimePeriod(DateInterval.BuilderImpl timePeriod) {
            this.timePeriod = timePeriod != null ? timePeriod.build() : null;
        }

        @Override
        public final Builder timePeriod(DateInterval timePeriod) {
            this.timePeriod = timePeriod;
            return this;
        }

        public final String getGranularity() {
            return granularity;
        }

        public final void setGranularity(String granularity) {
            this.granularity = granularity;
        }

        @Override
        public final Builder granularity(String granularity) {
            this.granularity = granularity;
            return this;
        }

        @Override
        public final Builder granularity(Granularity granularity) {
            this.granularity(granularity == null ? null : granularity.toString());
            return this;
        }

        public final Expression.Builder getFilter() {
            return filter != null ? filter.toBuilder() : null;
        }

        public final void setFilter(Expression.BuilderImpl filter) {
            this.filter = filter != null ? filter.build() : null;
        }

        @Override
        public final Builder filter(Expression filter) {
            this.filter = filter;
            return this;
        }

        public final Collection<String> getMetrics() {
            if (metrics instanceof SdkAutoConstructList) {
                return null;
            }
            return metrics;
        }

        public final void setMetrics(Collection<String> metrics) {
            this.metrics = MetricNamesCopier.copy(metrics);
        }

        @Override
        public final Builder metrics(Collection<String> metrics) {
            this.metrics = MetricNamesCopier.copy(metrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metrics(String... metrics) {
            metrics(Arrays.asList(metrics));
            return this;
        }

        public final List<GroupDefinition.Builder> getGroupBy() {
            List<GroupDefinition.Builder> result = GroupDefinitionsCopier.copyToBuilder(this.groupBy);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGroupBy(Collection<GroupDefinition.BuilderImpl> groupBy) {
            this.groupBy = GroupDefinitionsCopier.copyFromBuilder(groupBy);
        }

        @Override
        public final Builder groupBy(Collection<GroupDefinition> groupBy) {
            this.groupBy = GroupDefinitionsCopier.copy(groupBy);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groupBy(GroupDefinition... groupBy) {
            groupBy(Arrays.asList(groupBy));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groupBy(Consumer<GroupDefinition.Builder>... groupBy) {
            groupBy(Stream.of(groupBy).map(c -> GroupDefinition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getNextPageToken() {
            return nextPageToken;
        }

        public final void setNextPageToken(String nextPageToken) {
            this.nextPageToken = nextPageToken;
        }

        @Override
        public final Builder nextPageToken(String nextPageToken) {
            this.nextPageToken = nextPageToken;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetCostAndUsageWithResourcesRequest build() {
            return new GetCostAndUsageWithResourcesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
