/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costexplorer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use the split charge rule to split the cost of one Cost Category value across several other target values.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CostCategorySplitChargeRule implements SdkPojo, Serializable,
        ToCopyableBuilder<CostCategorySplitChargeRule.Builder, CostCategorySplitChargeRule> {
    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(CostCategorySplitChargeRule::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<List<String>> TARGETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Targets")
            .getter(getter(CostCategorySplitChargeRule::targets))
            .setter(setter(Builder::targets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Method")
            .getter(getter(CostCategorySplitChargeRule::methodAsString)).setter(setter(Builder::method))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Method").build()).build();

    private static final SdkField<List<CostCategorySplitChargeRuleParameter>> PARAMETERS_FIELD = SdkField
            .<List<CostCategorySplitChargeRuleParameter>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(CostCategorySplitChargeRule::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CostCategorySplitChargeRuleParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(CostCategorySplitChargeRuleParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD, TARGETS_FIELD,
            METHOD_FIELD, PARAMETERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String source;

    private final List<String> targets;

    private final String method;

    private final List<CostCategorySplitChargeRuleParameter> parameters;

    private CostCategorySplitChargeRule(BuilderImpl builder) {
        this.source = builder.source;
        this.targets = builder.targets;
        this.method = builder.method;
        this.parameters = builder.parameters;
    }

    /**
     * <p>
     * The Cost Category value that you want to split. That value can't be used as a source or a target in other split
     * charge rules. To indicate uncategorized costs, you can use an empty string as the source.
     * </p>
     * 
     * @return The Cost Category value that you want to split. That value can't be used as a source or a target in other
     *         split charge rules. To indicate uncategorized costs, you can use an empty string as the source.
     */
    public final String source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the Targets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTargets() {
        return targets != null && !(targets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Cost Category values that you want to split costs across. These values can't be used as a source in other
     * split charge rules.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTargets} method.
     * </p>
     * 
     * @return The Cost Category values that you want to split costs across. These values can't be used as a source in
     *         other split charge rules.
     */
    public final List<String> targets() {
        return targets;
    }

    /**
     * <p>
     * The method that's used to define how to split your source costs across your targets.
     * </p>
     * <p>
     * <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted cost of each
     * target.
     * </p>
     * <p>
     * <code>Fixed</code> - Allocates charges across your targets based on your defined allocation percentage.
     * </p>
     * <p>
     * &gt;<code>Even</code> - Allocates costs evenly across all targets.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #method} will
     * return {@link CostCategorySplitChargeMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #methodAsString}.
     * </p>
     * 
     * @return The method that's used to define how to split your source costs across your targets. </p>
     *         <p>
     *         <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted cost
     *         of each target.
     *         </p>
     *         <p>
     *         <code>Fixed</code> - Allocates charges across your targets based on your defined allocation percentage.
     *         </p>
     *         <p>
     *         &gt;<code>Even</code> - Allocates costs evenly across all targets.
     * @see CostCategorySplitChargeMethod
     */
    public final CostCategorySplitChargeMethod method() {
        return CostCategorySplitChargeMethod.fromValue(method);
    }

    /**
     * <p>
     * The method that's used to define how to split your source costs across your targets.
     * </p>
     * <p>
     * <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted cost of each
     * target.
     * </p>
     * <p>
     * <code>Fixed</code> - Allocates charges across your targets based on your defined allocation percentage.
     * </p>
     * <p>
     * &gt;<code>Even</code> - Allocates costs evenly across all targets.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #method} will
     * return {@link CostCategorySplitChargeMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #methodAsString}.
     * </p>
     * 
     * @return The method that's used to define how to split your source costs across your targets. </p>
     *         <p>
     *         <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted cost
     *         of each target.
     *         </p>
     *         <p>
     *         <code>Fixed</code> - Allocates charges across your targets based on your defined allocation percentage.
     *         </p>
     *         <p>
     *         &gt;<code>Even</code> - Allocates costs evenly across all targets.
     * @see CostCategorySplitChargeMethod
     */
    public final String methodAsString() {
        return method;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
     */
    public final List<CostCategorySplitChargeRuleParameter> parameters() {
        return parameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargets() ? targets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(methodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CostCategorySplitChargeRule)) {
            return false;
        }
        CostCategorySplitChargeRule other = (CostCategorySplitChargeRule) obj;
        return Objects.equals(source(), other.source()) && hasTargets() == other.hasTargets()
                && Objects.equals(targets(), other.targets()) && Objects.equals(methodAsString(), other.methodAsString())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CostCategorySplitChargeRule").add("Source", source())
                .add("Targets", hasTargets() ? targets() : null).add("Method", methodAsString())
                .add("Parameters", hasParameters() ? parameters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "Method":
            return Optional.ofNullable(clazz.cast(methodAsString()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Source", SOURCE_FIELD);
        map.put("Targets", TARGETS_FIELD);
        map.put("Method", METHOD_FIELD);
        map.put("Parameters", PARAMETERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CostCategorySplitChargeRule, T> g) {
        return obj -> g.apply((CostCategorySplitChargeRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CostCategorySplitChargeRule> {
        /**
         * <p>
         * The Cost Category value that you want to split. That value can't be used as a source or a target in other
         * split charge rules. To indicate uncategorized costs, you can use an empty string as the source.
         * </p>
         * 
         * @param source
         *        The Cost Category value that you want to split. That value can't be used as a source or a target in
         *        other split charge rules. To indicate uncategorized costs, you can use an empty string as the source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * The Cost Category values that you want to split costs across. These values can't be used as a source in other
         * split charge rules.
         * </p>
         * 
         * @param targets
         *        The Cost Category values that you want to split costs across. These values can't be used as a source
         *        in other split charge rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(Collection<String> targets);

        /**
         * <p>
         * The Cost Category values that you want to split costs across. These values can't be used as a source in other
         * split charge rules.
         * </p>
         * 
         * @param targets
         *        The Cost Category values that you want to split costs across. These values can't be used as a source
         *        in other split charge rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(String... targets);

        /**
         * <p>
         * The method that's used to define how to split your source costs across your targets.
         * </p>
         * <p>
         * <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted cost of
         * each target.
         * </p>
         * <p>
         * <code>Fixed</code> - Allocates charges across your targets based on your defined allocation percentage.
         * </p>
         * <p>
         * &gt;<code>Even</code> - Allocates costs evenly across all targets.
         * </p>
         * 
         * @param method
         *        The method that's used to define how to split your source costs across your targets. </p>
         *        <p>
         *        <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted
         *        cost of each target.
         *        </p>
         *        <p>
         *        <code>Fixed</code> - Allocates charges across your targets based on your defined allocation
         *        percentage.
         *        </p>
         *        <p>
         *        &gt;<code>Even</code> - Allocates costs evenly across all targets.
         * @see CostCategorySplitChargeMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CostCategorySplitChargeMethod
         */
        Builder method(String method);

        /**
         * <p>
         * The method that's used to define how to split your source costs across your targets.
         * </p>
         * <p>
         * <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted cost of
         * each target.
         * </p>
         * <p>
         * <code>Fixed</code> - Allocates charges across your targets based on your defined allocation percentage.
         * </p>
         * <p>
         * &gt;<code>Even</code> - Allocates costs evenly across all targets.
         * </p>
         * 
         * @param method
         *        The method that's used to define how to split your source costs across your targets. </p>
         *        <p>
         *        <code>Proportional</code> - Allocates charges across your targets based on the proportional weighted
         *        cost of each target.
         *        </p>
         *        <p>
         *        <code>Fixed</code> - Allocates charges across your targets based on your defined allocation
         *        percentage.
         *        </p>
         *        <p>
         *        &gt;<code>Even</code> - Allocates costs evenly across all targets.
         * @see CostCategorySplitChargeMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CostCategorySplitChargeMethod
         */
        Builder method(CostCategorySplitChargeMethod method);

        /**
         * <p>
         * The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
         * </p>
         * 
         * @param parameters
         *        The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<CostCategorySplitChargeRuleParameter> parameters);

        /**
         * <p>
         * The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
         * </p>
         * 
         * @param parameters
         *        The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(CostCategorySplitChargeRuleParameter... parameters);

        /**
         * <p>
         * The parameters for a split charge method. This is only required for the <code>FIXED</code> method.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.costexplorer.model.CostCategorySplitChargeRuleParameter.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.costexplorer.model.CostCategorySplitChargeRuleParameter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.costexplorer.model.CostCategorySplitChargeRuleParameter.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #parameters(List<CostCategorySplitChargeRuleParameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.costexplorer.model.CostCategorySplitChargeRuleParameter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(java.util.Collection<CostCategorySplitChargeRuleParameter>)
         */
        Builder parameters(Consumer<CostCategorySplitChargeRuleParameter.Builder>... parameters);
    }

    static final class BuilderImpl implements Builder {
        private String source;

        private List<String> targets = DefaultSdkAutoConstructList.getInstance();

        private String method;

        private List<CostCategorySplitChargeRuleParameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CostCategorySplitChargeRule model) {
            source(model.source);
            targets(model.targets);
            method(model.method);
            parameters(model.parameters);
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final Collection<String> getTargets() {
            if (targets instanceof SdkAutoConstructList) {
                return null;
            }
            return targets;
        }

        public final void setTargets(Collection<String> targets) {
            this.targets = CostCategorySplitChargeRuleTargetsListCopier.copy(targets);
        }

        @Override
        public final Builder targets(Collection<String> targets) {
            this.targets = CostCategorySplitChargeRuleTargetsListCopier.copy(targets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targets(String... targets) {
            targets(Arrays.asList(targets));
            return this;
        }

        public final String getMethod() {
            return method;
        }

        public final void setMethod(String method) {
            this.method = method;
        }

        @Override
        public final Builder method(String method) {
            this.method = method;
            return this;
        }

        @Override
        public final Builder method(CostCategorySplitChargeMethod method) {
            this.method(method == null ? null : method.toString());
            return this;
        }

        public final List<CostCategorySplitChargeRuleParameter.Builder> getParameters() {
            List<CostCategorySplitChargeRuleParameter.Builder> result = CostCategorySplitChargeRuleParametersListCopier
                    .copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParameters(Collection<CostCategorySplitChargeRuleParameter.BuilderImpl> parameters) {
            this.parameters = CostCategorySplitChargeRuleParametersListCopier.copyFromBuilder(parameters);
        }

        @Override
        public final Builder parameters(Collection<CostCategorySplitChargeRuleParameter> parameters) {
            this.parameters = CostCategorySplitChargeRuleParametersListCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(CostCategorySplitChargeRuleParameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<CostCategorySplitChargeRuleParameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> CostCategorySplitChargeRuleParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CostCategorySplitChargeRule build() {
            return new CostCategorySplitChargeRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
