/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costexplorer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents factors that contribute to cost variations between the baseline and comparison time periods, including the
 * type of driver, an identifier of the driver, and associated metrics.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CostDriver implements SdkPojo, Serializable, ToCopyableBuilder<CostDriver.Builder, CostDriver> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(CostDriver::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(CostDriver::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<Map<String, ComparisonMetricValue>> METRICS_FIELD = SdkField
            .<Map<String, ComparisonMetricValue>> builder(MarshallingType.MAP)
            .memberName("Metrics")
            .getter(getter(CostDriver::metrics))
            .setter(setter(Builder::metrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metrics").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ComparisonMetricValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(ComparisonMetricValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, NAME_FIELD,
            METRICS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String name;

    private final Map<String, ComparisonMetricValue> metrics;

    private CostDriver(BuilderImpl builder) {
        this.type = builder.type;
        this.name = builder.name;
        this.metrics = builder.metrics;
    }

    /**
     * <p>
     * The category or classification of the cost driver.
     * </p>
     * <p>
     * Values include: BUNDLED_DISCOUNT, CREDIT, OUT_OF_CYCLE_CHARGE, REFUND, RECURRING_RESERVATION_FEE,
     * RESERVATION_USAGE, RI_VOLUME_DISCOUNT, SAVINGS_PLAN_USAGE, SAVINGS_PLAN_RECURRING_FEE, SUPPORT_FEE, TAX,
     * UPFRONT_RESERVATION_FEE, USAGE_CHANGE, COMMITMENT
     * </p>
     * 
     * @return The category or classification of the cost driver.</p>
     *         <p>
     *         Values include: BUNDLED_DISCOUNT, CREDIT, OUT_OF_CYCLE_CHARGE, REFUND, RECURRING_RESERVATION_FEE,
     *         RESERVATION_USAGE, RI_VOLUME_DISCOUNT, SAVINGS_PLAN_USAGE, SAVINGS_PLAN_RECURRING_FEE, SUPPORT_FEE, TAX,
     *         UPFRONT_RESERVATION_FEE, USAGE_CHANGE, COMMITMENT
     */
    public final String type() {
        return type;
    }

    /**
     * <p>
     * The specific identifier of the cost driver.
     * </p>
     * 
     * @return The specific identifier of the cost driver.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metrics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetrics() {
        return metrics != null && !(metrics instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A mapping of metric names to their comparison values, measuring the impact of this cost driver.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetrics} method.
     * </p>
     * 
     * @return A mapping of metric names to their comparison values, measuring the impact of this cost driver.
     */
    public final Map<String, ComparisonMetricValue> metrics() {
        return metrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetrics() ? metrics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CostDriver)) {
            return false;
        }
        CostDriver other = (CostDriver) obj;
        return Objects.equals(type(), other.type()) && Objects.equals(name(), other.name()) && hasMetrics() == other.hasMetrics()
                && Objects.equals(metrics(), other.metrics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CostDriver").add("Type", type()).add("Name", name())
                .add("Metrics", hasMetrics() ? metrics() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Metrics":
            return Optional.ofNullable(clazz.cast(metrics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("Metrics", METRICS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CostDriver, T> g) {
        return obj -> g.apply((CostDriver) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CostDriver> {
        /**
         * <p>
         * The category or classification of the cost driver.
         * </p>
         * <p>
         * Values include: BUNDLED_DISCOUNT, CREDIT, OUT_OF_CYCLE_CHARGE, REFUND, RECURRING_RESERVATION_FEE,
         * RESERVATION_USAGE, RI_VOLUME_DISCOUNT, SAVINGS_PLAN_USAGE, SAVINGS_PLAN_RECURRING_FEE, SUPPORT_FEE, TAX,
         * UPFRONT_RESERVATION_FEE, USAGE_CHANGE, COMMITMENT
         * </p>
         * 
         * @param type
         *        The category or classification of the cost driver.</p>
         *        <p>
         *        Values include: BUNDLED_DISCOUNT, CREDIT, OUT_OF_CYCLE_CHARGE, REFUND, RECURRING_RESERVATION_FEE,
         *        RESERVATION_USAGE, RI_VOLUME_DISCOUNT, SAVINGS_PLAN_USAGE, SAVINGS_PLAN_RECURRING_FEE, SUPPORT_FEE,
         *        TAX, UPFRONT_RESERVATION_FEE, USAGE_CHANGE, COMMITMENT
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * The specific identifier of the cost driver.
         * </p>
         * 
         * @param name
         *        The specific identifier of the cost driver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A mapping of metric names to their comparison values, measuring the impact of this cost driver.
         * </p>
         * 
         * @param metrics
         *        A mapping of metric names to their comparison values, measuring the impact of this cost driver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Map<String, ComparisonMetricValue> metrics);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String name;

        private Map<String, ComparisonMetricValue> metrics = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CostDriver model) {
            type(model.type);
            name(model.name);
            metrics(model.metrics);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, ComparisonMetricValue.Builder> getMetrics() {
            Map<String, ComparisonMetricValue.Builder> result = ComparisonMetricsCopier.copyToBuilder(this.metrics);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setMetrics(Map<String, ComparisonMetricValue.BuilderImpl> metrics) {
            this.metrics = ComparisonMetricsCopier.copyFromBuilder(metrics);
        }

        @Override
        public final Builder metrics(Map<String, ComparisonMetricValue> metrics) {
            this.metrics = ComparisonMetricsCopier.copy(metrics);
            return this;
        }

        @Override
        public CostDriver build() {
            return new CostDriver(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
