/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costoptimizationhub;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.costoptimizationhub.internal.CostOptimizationHubServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.costoptimizationhub.model.AccessDeniedException;
import software.amazon.awssdk.services.costoptimizationhub.model.CostOptimizationHubException;
import software.amazon.awssdk.services.costoptimizationhub.model.GetPreferencesRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.GetPreferencesResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.GetRecommendationRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.GetRecommendationResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.InternalServerException;
import software.amazon.awssdk.services.costoptimizationhub.model.ListEnrollmentStatusesRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.ListEnrollmentStatusesResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.ListRecommendationSummariesRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.ListRecommendationSummariesResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.ResourceNotFoundException;
import software.amazon.awssdk.services.costoptimizationhub.model.ThrottlingException;
import software.amazon.awssdk.services.costoptimizationhub.model.UpdateEnrollmentStatusRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.UpdateEnrollmentStatusResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.UpdatePreferencesRequest;
import software.amazon.awssdk.services.costoptimizationhub.model.UpdatePreferencesResponse;
import software.amazon.awssdk.services.costoptimizationhub.model.ValidationException;
import software.amazon.awssdk.services.costoptimizationhub.transform.GetPreferencesRequestMarshaller;
import software.amazon.awssdk.services.costoptimizationhub.transform.GetRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costoptimizationhub.transform.ListEnrollmentStatusesRequestMarshaller;
import software.amazon.awssdk.services.costoptimizationhub.transform.ListRecommendationSummariesRequestMarshaller;
import software.amazon.awssdk.services.costoptimizationhub.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.costoptimizationhub.transform.UpdateEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.services.costoptimizationhub.transform.UpdatePreferencesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CostOptimizationHubAsyncClient}.
 *
 * @see CostOptimizationHubAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCostOptimizationHubAsyncClient implements CostOptimizationHubAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCostOptimizationHubAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCostOptimizationHubAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns a set of preferences for an account in order to add account-specific preferences into the service. These
     * preferences impact how the savings associated with recommendations are presented—estimated savings after
     * discounts or estimated savings before discounts, for example.
     * </p>
     *
     * @param getPreferencesRequest
     * @return A Java Future containing the result of the GetPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.GetPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/GetPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPreferencesResponse> getPreferences(GetPreferencesRequest getPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPreferencesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPreferencesRequest, GetPreferencesResponse>()
                            .withOperationName("GetPreferences").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPreferencesRequest));
            CompletableFuture<GetPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns both the current and recommended resource configuration and the estimated cost impact for a
     * recommendation.
     * </p>
     * <p>
     * The <code>recommendationId</code> is only valid for up to a maximum of 24 hours as recommendations are refreshed
     * daily. To retrieve the <code>recommendationId</code>, use the <code>ListRecommendations</code> API.
     * </p>
     *
     * @param getRecommendationRequest
     * @return A Java Future containing the result of the GetRecommendation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ResourceNotFoundException The specified Amazon Resource Name (ARN) in the request doesn't exist.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.GetRecommendation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/GetRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationResponse> getRecommendation(GetRecommendationRequest getRecommendationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecommendationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationRequest, GetRecommendationResponse>()
                            .withOperationName("GetRecommendation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecommendationRequest));
            CompletableFuture<GetRecommendationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the enrollment status for an account. It can also return the list of accounts that are enrolled under
     * the organization.
     * </p>
     *
     * @param listEnrollmentStatusesRequest
     * @return A Java Future containing the result of the ListEnrollmentStatuses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.ListEnrollmentStatuses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/ListEnrollmentStatuses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnrollmentStatusesResponse> listEnrollmentStatuses(
            ListEnrollmentStatusesRequest listEnrollmentStatusesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnrollmentStatusesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnrollmentStatusesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnrollmentStatuses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnrollmentStatusesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnrollmentStatusesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnrollmentStatusesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnrollmentStatusesRequest, ListEnrollmentStatusesResponse>()
                            .withOperationName("ListEnrollmentStatuses").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnrollmentStatusesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnrollmentStatusesRequest));
            CompletableFuture<ListEnrollmentStatusesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a concise representation of savings estimates for resources. Also returns de-duped savings across
     * different types of recommendations.
     * </p>
     * <note>
     * <p>
     * The following filters are not supported for this API: <code>recommendationIds</code>, <code>resourceArns</code>,
     * and <code>resourceIds</code>.
     * </p>
     * </note>
     *
     * @param listRecommendationSummariesRequest
     * @return A Java Future containing the result of the ListRecommendationSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.ListRecommendationSummaries
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/ListRecommendationSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecommendationSummariesResponse> listRecommendationSummaries(
            ListRecommendationSummariesRequest listRecommendationSummariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecommendationSummariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendationSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecommendationSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecommendationSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecommendationSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecommendationSummariesRequest, ListRecommendationSummariesResponse>()
                            .withOperationName("ListRecommendationSummaries").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRecommendationSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRecommendationSummariesRequest));
            CompletableFuture<ListRecommendationSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of recommendations.
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A Java Future containing the result of the ListRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.ListRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecommendationsResponse> listRecommendations(
            ListRecommendationsRequest listRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                            .withOperationName("ListRecommendations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listRecommendationsRequest));
            CompletableFuture<ListRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the enrollment (opt in and opt out) status of an account to the Cost Optimization Hub service.
     * </p>
     * <p>
     * If the account is a management account of an organization, this action can also be used to enroll member accounts
     * of the organization.
     * </p>
     * <p>
     * You must have the appropriate permissions to opt in to Cost Optimization Hub and to view its recommendations.
     * When you opt in, Cost Optimization Hub automatically creates a service-linked role in your account to access its
     * data.
     * </p>
     *
     * @param updateEnrollmentStatusRequest
     * @return A Java Future containing the result of the UpdateEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.UpdateEnrollmentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/UpdateEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnrollmentStatusResponse> updateEnrollmentStatus(
            UpdateEnrollmentStatusRequest updateEnrollmentStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnrollmentStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnrollmentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnrollmentStatusRequest, UpdateEnrollmentStatusResponse>()
                            .withOperationName("UpdateEnrollmentStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEnrollmentStatusRequest));
            CompletableFuture<UpdateEnrollmentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a set of preferences for an account in order to add account-specific preferences into the service. These
     * preferences impact how the savings associated with recommendations are presented.
     * </p>
     *
     * @param updatePreferencesRequest
     * @return A Java Future containing the result of the UpdatePreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an Amazon Web Services
     *         service.</li>
     *         <li>InternalServerException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>AccessDeniedException You are not authorized to use this operation with the given parameters.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostOptimizationHubException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostOptimizationHubAsyncClient.UpdatePreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cost-optimization-hub-2022-07-26/UpdatePreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePreferencesResponse> updatePreferences(UpdatePreferencesRequest updatePreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost Optimization Hub");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePreferencesRequest, UpdatePreferencesResponse>()
                            .withOperationName("UpdatePreferences").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdatePreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePreferencesRequest));
            CompletableFuture<UpdatePreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CostOptimizationHubServiceClientConfiguration serviceClientConfiguration() {
        return new CostOptimizationHubServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CostOptimizationHubException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        CostOptimizationHubServiceClientConfigurationBuilder serviceConfigBuilder = new CostOptimizationHubServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
