/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costoptimizationhub.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a filter that returns a more specific list of recommendations. Filters recommendations by different
 * dimensions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Filter implements SdkPojo, Serializable, ToCopyableBuilder<Filter.Builder, Filter> {
    private static final SdkField<Boolean> RESTART_NEEDED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("restartNeeded").getter(getter(Filter::restartNeeded)).setter(setter(Builder::restartNeeded))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("restartNeeded").build()).build();

    private static final SdkField<Boolean> ROLLBACK_POSSIBLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("rollbackPossible").getter(getter(Filter::rollbackPossible)).setter(setter(Builder::rollbackPossible))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rollbackPossible").build()).build();

    private static final SdkField<List<String>> IMPLEMENTATION_EFFORTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("implementationEfforts")
            .getter(getter(Filter::implementationEffortsAsStrings))
            .setter(setter(Builder::implementationEffortsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("implementationEfforts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ACCOUNT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("accountIds")
            .getter(getter(Filter::accountIds))
            .setter(setter(Builder::accountIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("regions")
            .getter(getter(Filter::regions))
            .setter(setter(Builder::regions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("regions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceTypes")
            .getter(getter(Filter::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ACTION_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("actionTypes")
            .getter(getter(Filter::actionTypesAsStrings))
            .setter(setter(Builder::actionTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(Filter::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceIds")
            .getter(getter(Filter::resourceIds))
            .setter(setter(Builder::resourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceArns")
            .getter(getter(Filter::resourceArns))
            .setter(setter(Builder::resourceArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RECOMMENDATION_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendationIds")
            .getter(getter(Filter::recommendationIds))
            .setter(setter(Builder::recommendationIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESTART_NEEDED_FIELD,
            ROLLBACK_POSSIBLE_FIELD, IMPLEMENTATION_EFFORTS_FIELD, ACCOUNT_IDS_FIELD, REGIONS_FIELD, RESOURCE_TYPES_FIELD,
            ACTION_TYPES_FIELD, TAGS_FIELD, RESOURCE_IDS_FIELD, RESOURCE_ARNS_FIELD, RECOMMENDATION_IDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean restartNeeded;

    private final Boolean rollbackPossible;

    private final List<String> implementationEfforts;

    private final List<String> accountIds;

    private final List<String> regions;

    private final List<String> resourceTypes;

    private final List<String> actionTypes;

    private final List<Tag> tags;

    private final List<String> resourceIds;

    private final List<String> resourceArns;

    private final List<String> recommendationIds;

    private Filter(BuilderImpl builder) {
        this.restartNeeded = builder.restartNeeded;
        this.rollbackPossible = builder.rollbackPossible;
        this.implementationEfforts = builder.implementationEfforts;
        this.accountIds = builder.accountIds;
        this.regions = builder.regions;
        this.resourceTypes = builder.resourceTypes;
        this.actionTypes = builder.actionTypes;
        this.tags = builder.tags;
        this.resourceIds = builder.resourceIds;
        this.resourceArns = builder.resourceArns;
        this.recommendationIds = builder.recommendationIds;
    }

    /**
     * <p>
     * Whether or not implementing the recommendation requires a restart.
     * </p>
     * 
     * @return Whether or not implementing the recommendation requires a restart.
     */
    public final Boolean restartNeeded() {
        return restartNeeded;
    }

    /**
     * <p>
     * Whether or not implementing the recommendation can be rolled back.
     * </p>
     * 
     * @return Whether or not implementing the recommendation can be rolled back.
     */
    public final Boolean rollbackPossible() {
        return rollbackPossible;
    }

    /**
     * <p>
     * The effort required to implement the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImplementationEfforts} method.
     * </p>
     * 
     * @return The effort required to implement the recommendation.
     */
    public final List<ImplementationEffort> implementationEfforts() {
        return ImplementationEffortListCopier.copyStringToEnum(implementationEfforts);
    }

    /**
     * For responses, this returns true if the service returned a value for the ImplementationEfforts property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasImplementationEfforts() {
        return implementationEfforts != null && !(implementationEfforts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The effort required to implement the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImplementationEfforts} method.
     * </p>
     * 
     * @return The effort required to implement the recommendation.
     */
    public final List<String> implementationEffortsAsStrings() {
        return implementationEfforts;
    }

    /**
     * For responses, this returns true if the service returned a value for the AccountIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAccountIds() {
        return accountIds != null && !(accountIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The account that the recommendation is for.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccountIds} method.
     * </p>
     * 
     * @return The account that the recommendation is for.
     */
    public final List<String> accountIds() {
        return accountIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the Regions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRegions() {
        return regions != null && !(regions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services Region of the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRegions} method.
     * </p>
     * 
     * @return The Amazon Web Services Region of the resource.
     */
    public final List<String> regions() {
        return regions;
    }

    /**
     * <p>
     * The resource type of the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return The resource type of the recommendation.
     */
    public final List<ResourceType> resourceTypes() {
        return ResourceTypeListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource type of the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return The resource type of the recommendation.
     */
    public final List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    /**
     * <p>
     * The type of action you can take by adopting the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActionTypes} method.
     * </p>
     * 
     * @return The type of action you can take by adopting the recommendation.
     */
    public final List<ActionType> actionTypes() {
        return ActionTypeListCopier.copyStringToEnum(actionTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ActionTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasActionTypes() {
        return actionTypes != null && !(actionTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of action you can take by adopting the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasActionTypes} method.
     * </p>
     * 
     * @return The type of action you can take by adopting the recommendation.
     */
    public final List<String> actionTypesAsStrings() {
        return actionTypes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags assigned to the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of tags assigned to the recommendation.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceIds() {
        return resourceIds != null && !(resourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource ID of the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceIds} method.
     * </p>
     * 
     * @return The resource ID of the recommendation.
     */
    public final List<String> resourceIds() {
        return resourceIds;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceArns property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceArns() {
        return resourceArns != null && !(resourceArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceArns} method.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the recommendation.
     */
    public final List<String> resourceArns() {
        return resourceArns;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendationIds() {
        return recommendationIds != null && !(recommendationIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs for the recommendations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationIds} method.
     * </p>
     * 
     * @return The IDs for the recommendations.
     */
    public final List<String> recommendationIds() {
        return recommendationIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(restartNeeded());
        hashCode = 31 * hashCode + Objects.hashCode(rollbackPossible());
        hashCode = 31 * hashCode + Objects.hashCode(hasImplementationEfforts() ? implementationEffortsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAccountIds() ? accountIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRegions() ? regions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasActionTypes() ? actionTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIds() ? resourceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceArns() ? resourceArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationIds() ? recommendationIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Filter)) {
            return false;
        }
        Filter other = (Filter) obj;
        return Objects.equals(restartNeeded(), other.restartNeeded())
                && Objects.equals(rollbackPossible(), other.rollbackPossible())
                && hasImplementationEfforts() == other.hasImplementationEfforts()
                && Objects.equals(implementationEffortsAsStrings(), other.implementationEffortsAsStrings())
                && hasAccountIds() == other.hasAccountIds() && Objects.equals(accountIds(), other.accountIds())
                && hasRegions() == other.hasRegions() && Objects.equals(regions(), other.regions())
                && hasResourceTypes() == other.hasResourceTypes()
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings())
                && hasActionTypes() == other.hasActionTypes()
                && Objects.equals(actionTypesAsStrings(), other.actionTypesAsStrings()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasResourceIds() == other.hasResourceIds()
                && Objects.equals(resourceIds(), other.resourceIds()) && hasResourceArns() == other.hasResourceArns()
                && Objects.equals(resourceArns(), other.resourceArns()) && hasRecommendationIds() == other.hasRecommendationIds()
                && Objects.equals(recommendationIds(), other.recommendationIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Filter").add("RestartNeeded", restartNeeded()).add("RollbackPossible", rollbackPossible())
                .add("ImplementationEfforts", hasImplementationEfforts() ? implementationEffortsAsStrings() : null)
                .add("AccountIds", hasAccountIds() ? accountIds() : null).add("Regions", hasRegions() ? regions() : null)
                .add("ResourceTypes", hasResourceTypes() ? resourceTypesAsStrings() : null)
                .add("ActionTypes", hasActionTypes() ? actionTypesAsStrings() : null).add("Tags", hasTags() ? tags() : null)
                .add("ResourceIds", hasResourceIds() ? resourceIds() : null)
                .add("ResourceArns", hasResourceArns() ? resourceArns() : null)
                .add("RecommendationIds", hasRecommendationIds() ? recommendationIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "restartNeeded":
            return Optional.ofNullable(clazz.cast(restartNeeded()));
        case "rollbackPossible":
            return Optional.ofNullable(clazz.cast(rollbackPossible()));
        case "implementationEfforts":
            return Optional.ofNullable(clazz.cast(implementationEffortsAsStrings()));
        case "accountIds":
            return Optional.ofNullable(clazz.cast(accountIds()));
        case "regions":
            return Optional.ofNullable(clazz.cast(regions()));
        case "resourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        case "actionTypes":
            return Optional.ofNullable(clazz.cast(actionTypesAsStrings()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "resourceIds":
            return Optional.ofNullable(clazz.cast(resourceIds()));
        case "resourceArns":
            return Optional.ofNullable(clazz.cast(resourceArns()));
        case "recommendationIds":
            return Optional.ofNullable(clazz.cast(recommendationIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("restartNeeded", RESTART_NEEDED_FIELD);
        map.put("rollbackPossible", ROLLBACK_POSSIBLE_FIELD);
        map.put("implementationEfforts", IMPLEMENTATION_EFFORTS_FIELD);
        map.put("accountIds", ACCOUNT_IDS_FIELD);
        map.put("regions", REGIONS_FIELD);
        map.put("resourceTypes", RESOURCE_TYPES_FIELD);
        map.put("actionTypes", ACTION_TYPES_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("resourceIds", RESOURCE_IDS_FIELD);
        map.put("resourceArns", RESOURCE_ARNS_FIELD);
        map.put("recommendationIds", RECOMMENDATION_IDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Filter, T> g) {
        return obj -> g.apply((Filter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Filter> {
        /**
         * <p>
         * Whether or not implementing the recommendation requires a restart.
         * </p>
         * 
         * @param restartNeeded
         *        Whether or not implementing the recommendation requires a restart.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restartNeeded(Boolean restartNeeded);

        /**
         * <p>
         * Whether or not implementing the recommendation can be rolled back.
         * </p>
         * 
         * @param rollbackPossible
         *        Whether or not implementing the recommendation can be rolled back.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackPossible(Boolean rollbackPossible);

        /**
         * <p>
         * The effort required to implement the recommendation.
         * </p>
         * 
         * @param implementationEfforts
         *        The effort required to implement the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder implementationEffortsWithStrings(Collection<String> implementationEfforts);

        /**
         * <p>
         * The effort required to implement the recommendation.
         * </p>
         * 
         * @param implementationEfforts
         *        The effort required to implement the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder implementationEffortsWithStrings(String... implementationEfforts);

        /**
         * <p>
         * The effort required to implement the recommendation.
         * </p>
         * 
         * @param implementationEfforts
         *        The effort required to implement the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder implementationEfforts(Collection<ImplementationEffort> implementationEfforts);

        /**
         * <p>
         * The effort required to implement the recommendation.
         * </p>
         * 
         * @param implementationEfforts
         *        The effort required to implement the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder implementationEfforts(ImplementationEffort... implementationEfforts);

        /**
         * <p>
         * The account that the recommendation is for.
         * </p>
         * 
         * @param accountIds
         *        The account that the recommendation is for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIds(Collection<String> accountIds);

        /**
         * <p>
         * The account that the recommendation is for.
         * </p>
         * 
         * @param accountIds
         *        The account that the recommendation is for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountIds(String... accountIds);

        /**
         * <p>
         * The Amazon Web Services Region of the resource.
         * </p>
         * 
         * @param regions
         *        The Amazon Web Services Region of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<String> regions);

        /**
         * <p>
         * The Amazon Web Services Region of the resource.
         * </p>
         * 
         * @param regions
         *        The Amazon Web Services Region of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(String... regions);

        /**
         * <p>
         * The resource type of the recommendation.
         * </p>
         * 
         * @param resourceTypes
         *        The resource type of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * The resource type of the recommendation.
         * </p>
         * 
         * @param resourceTypes
         *        The resource type of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * The resource type of the recommendation.
         * </p>
         * 
         * @param resourceTypes
         *        The resource type of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceType> resourceTypes);

        /**
         * <p>
         * The resource type of the recommendation.
         * </p>
         * 
         * @param resourceTypes
         *        The resource type of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceType... resourceTypes);

        /**
         * <p>
         * The type of action you can take by adopting the recommendation.
         * </p>
         * 
         * @param actionTypes
         *        The type of action you can take by adopting the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionTypesWithStrings(Collection<String> actionTypes);

        /**
         * <p>
         * The type of action you can take by adopting the recommendation.
         * </p>
         * 
         * @param actionTypes
         *        The type of action you can take by adopting the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionTypesWithStrings(String... actionTypes);

        /**
         * <p>
         * The type of action you can take by adopting the recommendation.
         * </p>
         * 
         * @param actionTypes
         *        The type of action you can take by adopting the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionTypes(Collection<ActionType> actionTypes);

        /**
         * <p>
         * The type of action you can take by adopting the recommendation.
         * </p>
         * 
         * @param actionTypes
         *        The type of action you can take by adopting the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionTypes(ActionType... actionTypes);

        /**
         * <p>
         * A list of tags assigned to the recommendation.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags assigned to the recommendation.
         * </p>
         * 
         * @param tags
         *        A list of tags assigned to the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags assigned to the recommendation.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.costoptimizationhub.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.costoptimizationhub.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.costoptimizationhub.model.Tag.Builder#build()} is called immediately
         * and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.costoptimizationhub.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The resource ID of the recommendation.
         * </p>
         * 
         * @param resourceIds
         *        The resource ID of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(Collection<String> resourceIds);

        /**
         * <p>
         * The resource ID of the recommendation.
         * </p>
         * 
         * @param resourceIds
         *        The resource ID of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIds(String... resourceIds);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the recommendation.
         * </p>
         * 
         * @param resourceArns
         *        The Amazon Resource Name (ARN) of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArns(Collection<String> resourceArns);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the recommendation.
         * </p>
         * 
         * @param resourceArns
         *        The Amazon Resource Name (ARN) of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArns(String... resourceArns);

        /**
         * <p>
         * The IDs for the recommendations.
         * </p>
         * 
         * @param recommendationIds
         *        The IDs for the recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationIds(Collection<String> recommendationIds);

        /**
         * <p>
         * The IDs for the recommendations.
         * </p>
         * 
         * @param recommendationIds
         *        The IDs for the recommendations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationIds(String... recommendationIds);
    }

    static final class BuilderImpl implements Builder {
        private Boolean restartNeeded;

        private Boolean rollbackPossible;

        private List<String> implementationEfforts = DefaultSdkAutoConstructList.getInstance();

        private List<String> accountIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> regions = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> actionTypes = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourceIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourceArns = DefaultSdkAutoConstructList.getInstance();

        private List<String> recommendationIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Filter model) {
            restartNeeded(model.restartNeeded);
            rollbackPossible(model.rollbackPossible);
            implementationEffortsWithStrings(model.implementationEfforts);
            accountIds(model.accountIds);
            regions(model.regions);
            resourceTypesWithStrings(model.resourceTypes);
            actionTypesWithStrings(model.actionTypes);
            tags(model.tags);
            resourceIds(model.resourceIds);
            resourceArns(model.resourceArns);
            recommendationIds(model.recommendationIds);
        }

        public final Boolean getRestartNeeded() {
            return restartNeeded;
        }

        public final void setRestartNeeded(Boolean restartNeeded) {
            this.restartNeeded = restartNeeded;
        }

        @Override
        public final Builder restartNeeded(Boolean restartNeeded) {
            this.restartNeeded = restartNeeded;
            return this;
        }

        public final Boolean getRollbackPossible() {
            return rollbackPossible;
        }

        public final void setRollbackPossible(Boolean rollbackPossible) {
            this.rollbackPossible = rollbackPossible;
        }

        @Override
        public final Builder rollbackPossible(Boolean rollbackPossible) {
            this.rollbackPossible = rollbackPossible;
            return this;
        }

        public final Collection<String> getImplementationEfforts() {
            if (implementationEfforts instanceof SdkAutoConstructList) {
                return null;
            }
            return implementationEfforts;
        }

        public final void setImplementationEfforts(Collection<String> implementationEfforts) {
            this.implementationEfforts = ImplementationEffortListCopier.copy(implementationEfforts);
        }

        @Override
        public final Builder implementationEffortsWithStrings(Collection<String> implementationEfforts) {
            this.implementationEfforts = ImplementationEffortListCopier.copy(implementationEfforts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder implementationEffortsWithStrings(String... implementationEfforts) {
            implementationEffortsWithStrings(Arrays.asList(implementationEfforts));
            return this;
        }

        @Override
        public final Builder implementationEfforts(Collection<ImplementationEffort> implementationEfforts) {
            this.implementationEfforts = ImplementationEffortListCopier.copyEnumToString(implementationEfforts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder implementationEfforts(ImplementationEffort... implementationEfforts) {
            implementationEfforts(Arrays.asList(implementationEfforts));
            return this;
        }

        public final Collection<String> getAccountIds() {
            if (accountIds instanceof SdkAutoConstructList) {
                return null;
            }
            return accountIds;
        }

        public final void setAccountIds(Collection<String> accountIds) {
            this.accountIds = AccountIdListCopier.copy(accountIds);
        }

        @Override
        public final Builder accountIds(Collection<String> accountIds) {
            this.accountIds = AccountIdListCopier.copy(accountIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accountIds(String... accountIds) {
            accountIds(Arrays.asList(accountIds));
            return this;
        }

        public final Collection<String> getRegions() {
            if (regions instanceof SdkAutoConstructList) {
                return null;
            }
            return regions;
        }

        public final void setRegions(Collection<String> regions) {
            this.regions = RegionListCopier.copy(regions);
        }

        @Override
        public final Builder regions(Collection<String> regions) {
            this.regions = RegionListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(String... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceType> resourceTypes) {
            this.resourceTypes = ResourceTypeListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceType... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final Collection<String> getActionTypes() {
            if (actionTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return actionTypes;
        }

        public final void setActionTypes(Collection<String> actionTypes) {
            this.actionTypes = ActionTypeListCopier.copy(actionTypes);
        }

        @Override
        public final Builder actionTypesWithStrings(Collection<String> actionTypes) {
            this.actionTypes = ActionTypeListCopier.copy(actionTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actionTypesWithStrings(String... actionTypes) {
            actionTypesWithStrings(Arrays.asList(actionTypes));
            return this;
        }

        @Override
        public final Builder actionTypes(Collection<ActionType> actionTypes) {
            this.actionTypes = ActionTypeListCopier.copyEnumToString(actionTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder actionTypes(ActionType... actionTypes) {
            actionTypes(Arrays.asList(actionTypes));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getResourceIds() {
            if (resourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceIds;
        }

        public final void setResourceIds(Collection<String> resourceIds) {
            this.resourceIds = ResourceIdListCopier.copy(resourceIds);
        }

        @Override
        public final Builder resourceIds(Collection<String> resourceIds) {
            this.resourceIds = ResourceIdListCopier.copy(resourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceIds(String... resourceIds) {
            resourceIds(Arrays.asList(resourceIds));
            return this;
        }

        public final Collection<String> getResourceArns() {
            if (resourceArns instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceArns;
        }

        public final void setResourceArns(Collection<String> resourceArns) {
            this.resourceArns = ResourceArnListCopier.copy(resourceArns);
        }

        @Override
        public final Builder resourceArns(Collection<String> resourceArns) {
            this.resourceArns = ResourceArnListCopier.copy(resourceArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceArns(String... resourceArns) {
            resourceArns(Arrays.asList(resourceArns));
            return this;
        }

        public final Collection<String> getRecommendationIds() {
            if (recommendationIds instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendationIds;
        }

        public final void setRecommendationIds(Collection<String> recommendationIds) {
            this.recommendationIds = RecommendationIdListCopier.copy(recommendationIds);
        }

        @Override
        public final Builder recommendationIds(Collection<String> recommendationIds) {
            this.recommendationIds = RecommendationIdListCopier.copy(recommendationIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationIds(String... recommendationIds) {
            recommendationIds(Arrays.asList(recommendationIds));
            return this;
        }

        @Override
        public Filter build() {
            return new Filter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
