/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.customerprofiles;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.customerprofiles.model.AccessDeniedException;
import software.amazon.awssdk.services.customerprofiles.model.AddProfileKeyRequest;
import software.amazon.awssdk.services.customerprofiles.model.AddProfileKeyResponse;
import software.amazon.awssdk.services.customerprofiles.model.BadRequestException;
import software.amazon.awssdk.services.customerprofiles.model.CreateDomainRequest;
import software.amazon.awssdk.services.customerprofiles.model.CreateDomainResponse;
import software.amazon.awssdk.services.customerprofiles.model.CreateProfileRequest;
import software.amazon.awssdk.services.customerprofiles.model.CreateProfileResponse;
import software.amazon.awssdk.services.customerprofiles.model.CustomerProfilesException;
import software.amazon.awssdk.services.customerprofiles.model.DeleteDomainRequest;
import software.amazon.awssdk.services.customerprofiles.model.DeleteDomainResponse;
import software.amazon.awssdk.services.customerprofiles.model.DeleteIntegrationRequest;
import software.amazon.awssdk.services.customerprofiles.model.DeleteIntegrationResponse;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileKeyRequest;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileKeyResponse;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileObjectRequest;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileObjectResponse;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileObjectTypeRequest;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileObjectTypeResponse;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileRequest;
import software.amazon.awssdk.services.customerprofiles.model.DeleteProfileResponse;
import software.amazon.awssdk.services.customerprofiles.model.GetDomainRequest;
import software.amazon.awssdk.services.customerprofiles.model.GetDomainResponse;
import software.amazon.awssdk.services.customerprofiles.model.GetIntegrationRequest;
import software.amazon.awssdk.services.customerprofiles.model.GetIntegrationResponse;
import software.amazon.awssdk.services.customerprofiles.model.GetMatchesRequest;
import software.amazon.awssdk.services.customerprofiles.model.GetMatchesResponse;
import software.amazon.awssdk.services.customerprofiles.model.GetProfileObjectTypeRequest;
import software.amazon.awssdk.services.customerprofiles.model.GetProfileObjectTypeResponse;
import software.amazon.awssdk.services.customerprofiles.model.GetProfileObjectTypeTemplateRequest;
import software.amazon.awssdk.services.customerprofiles.model.GetProfileObjectTypeTemplateResponse;
import software.amazon.awssdk.services.customerprofiles.model.InternalServerException;
import software.amazon.awssdk.services.customerprofiles.model.ListAccountIntegrationsRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListAccountIntegrationsResponse;
import software.amazon.awssdk.services.customerprofiles.model.ListDomainsRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListDomainsResponse;
import software.amazon.awssdk.services.customerprofiles.model.ListIntegrationsRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListIntegrationsResponse;
import software.amazon.awssdk.services.customerprofiles.model.ListProfileObjectTypeTemplatesRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListProfileObjectTypeTemplatesResponse;
import software.amazon.awssdk.services.customerprofiles.model.ListProfileObjectTypesRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListProfileObjectTypesResponse;
import software.amazon.awssdk.services.customerprofiles.model.ListProfileObjectsRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListProfileObjectsResponse;
import software.amazon.awssdk.services.customerprofiles.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.customerprofiles.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.customerprofiles.model.MergeProfilesRequest;
import software.amazon.awssdk.services.customerprofiles.model.MergeProfilesResponse;
import software.amazon.awssdk.services.customerprofiles.model.PutIntegrationRequest;
import software.amazon.awssdk.services.customerprofiles.model.PutIntegrationResponse;
import software.amazon.awssdk.services.customerprofiles.model.PutProfileObjectRequest;
import software.amazon.awssdk.services.customerprofiles.model.PutProfileObjectResponse;
import software.amazon.awssdk.services.customerprofiles.model.PutProfileObjectTypeRequest;
import software.amazon.awssdk.services.customerprofiles.model.PutProfileObjectTypeResponse;
import software.amazon.awssdk.services.customerprofiles.model.ResourceNotFoundException;
import software.amazon.awssdk.services.customerprofiles.model.SearchProfilesRequest;
import software.amazon.awssdk.services.customerprofiles.model.SearchProfilesResponse;
import software.amazon.awssdk.services.customerprofiles.model.TagResourceRequest;
import software.amazon.awssdk.services.customerprofiles.model.TagResourceResponse;
import software.amazon.awssdk.services.customerprofiles.model.ThrottlingException;
import software.amazon.awssdk.services.customerprofiles.model.UntagResourceRequest;
import software.amazon.awssdk.services.customerprofiles.model.UntagResourceResponse;
import software.amazon.awssdk.services.customerprofiles.model.UpdateDomainRequest;
import software.amazon.awssdk.services.customerprofiles.model.UpdateDomainResponse;
import software.amazon.awssdk.services.customerprofiles.model.UpdateProfileRequest;
import software.amazon.awssdk.services.customerprofiles.model.UpdateProfileResponse;
import software.amazon.awssdk.services.customerprofiles.transform.AddProfileKeyRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.CreateDomainRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.CreateProfileRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.DeleteDomainRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.DeleteIntegrationRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.DeleteProfileKeyRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.DeleteProfileObjectRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.DeleteProfileObjectTypeRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.DeleteProfileRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.GetDomainRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.GetIntegrationRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.GetMatchesRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.GetProfileObjectTypeRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.GetProfileObjectTypeTemplateRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListAccountIntegrationsRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListDomainsRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListIntegrationsRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListProfileObjectTypeTemplatesRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListProfileObjectTypesRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListProfileObjectsRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.MergeProfilesRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.PutIntegrationRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.PutProfileObjectRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.PutProfileObjectTypeRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.SearchProfilesRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.UpdateDomainRequestMarshaller;
import software.amazon.awssdk.services.customerprofiles.transform.UpdateProfileRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CustomerProfilesAsyncClient}.
 *
 * @see CustomerProfilesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCustomerProfilesAsyncClient implements CustomerProfilesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCustomerProfilesAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCustomerProfilesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates a new key value with a specific profile, such as a Contact Trace Record (CTR) ContactId.
     * </p>
     * <p>
     * A profile object can have a single unique key and any number of additional keys that can be used to identify the
     * profile that it belongs to.
     * </p>
     *
     * @param addProfileKeyRequest
     * @return A Java Future containing the result of the AddProfileKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.AddProfileKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/AddProfileKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddProfileKeyResponse> addProfileKey(AddProfileKeyRequest addProfileKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addProfileKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddProfileKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddProfileKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddProfileKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddProfileKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddProfileKeyRequest, AddProfileKeyResponse>()
                            .withOperationName("AddProfileKey")
                            .withMarshaller(new AddProfileKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addProfileKeyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addProfileKeyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<AddProfileKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a domain, which is a container for all customer data, such as customer profile attributes, object types,
     * profile keys, and encryption keys. You can create multiple domains, and each domain can have multiple third-party
     * integrations.
     * </p>
     * <p>
     * Each Amazon Connect instance can be associated with only one domain. Multiple Amazon Connect instances can be
     * associated with one domain.
     * </p>
     *
     * @param createDomainRequest
     * @return A Java Future containing the result of the CreateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.CreateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/CreateDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDomainResponse> createDomain(CreateDomainRequest createDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDomainRequest, CreateDomainResponse>()
                            .withOperationName("CreateDomain").withMarshaller(new CreateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a standard profile.
     * </p>
     * <p>
     * A standard profile represents the following attributes for a customer profile in a domain.
     * </p>
     *
     * @param createProfileRequest
     * @return A Java Future containing the result of the CreateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.CreateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/CreateProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProfileResponse> createProfile(CreateProfileRequest createProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProfileRequest, CreateProfileResponse>()
                            .withOperationName("CreateProfile")
                            .withMarshaller(new CreateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createProfileRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specific domain and all of its customer data, such as customer profile attributes and their related
     * objects.
     * </p>
     *
     * @param deleteDomainRequest
     * @return A Java Future containing the result of the DeleteDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.DeleteDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/DeleteDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDomainResponse> deleteDomain(DeleteDomainRequest deleteDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDomainRequest, DeleteDomainResponse>()
                            .withOperationName("DeleteDomain").withMarshaller(new DeleteDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an integration from a specific domain.
     * </p>
     *
     * @param deleteIntegrationRequest
     * @return A Java Future containing the result of the DeleteIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.DeleteIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/DeleteIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIntegrationResponse> deleteIntegration(DeleteIntegrationRequest deleteIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIntegrationRequest, DeleteIntegrationResponse>()
                            .withOperationName("DeleteIntegration")
                            .withMarshaller(new DeleteIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteIntegrationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the standard customer profile and all data pertaining to the profile.
     * </p>
     *
     * @param deleteProfileRequest
     * @return A Java Future containing the result of the DeleteProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.DeleteProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/DeleteProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileResponse> deleteProfile(DeleteProfileRequest deleteProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileRequest, DeleteProfileResponse>()
                            .withOperationName("DeleteProfile")
                            .withMarshaller(new DeleteProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteProfileRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a searchable key from a customer profile.
     * </p>
     *
     * @param deleteProfileKeyRequest
     * @return A Java Future containing the result of the DeleteProfileKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.DeleteProfileKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/DeleteProfileKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileKeyResponse> deleteProfileKey(DeleteProfileKeyRequest deleteProfileKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfileKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileKeyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProfileKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileKeyRequest, DeleteProfileKeyResponse>()
                            .withOperationName("DeleteProfileKey")
                            .withMarshaller(new DeleteProfileKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProfileKeyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteProfileKeyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteProfileKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an object associated with a profile of a given ProfileObjectType.
     * </p>
     *
     * @param deleteProfileObjectRequest
     * @return A Java Future containing the result of the DeleteProfileObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.DeleteProfileObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/DeleteProfileObject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileObjectResponse> deleteProfileObject(
            DeleteProfileObjectRequest deleteProfileObjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfileObject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileObjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProfileObjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileObjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileObjectRequest, DeleteProfileObjectResponse>()
                            .withOperationName("DeleteProfileObject")
                            .withMarshaller(new DeleteProfileObjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProfileObjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteProfileObjectRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DeleteProfileObjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a ProfileObjectType from a specific domain as well as removes all the ProfileObjects of that type. It
     * also disables integrations from this specific ProfileObjectType. In addition, it scrubs all of the fields of the
     * standard profile that were populated from this ProfileObjectType.
     * </p>
     *
     * @param deleteProfileObjectTypeRequest
     * @return A Java Future containing the result of the DeleteProfileObjectType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.DeleteProfileObjectType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/DeleteProfileObjectType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProfileObjectTypeResponse> deleteProfileObjectType(
            DeleteProfileObjectTypeRequest deleteProfileObjectTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProfileObjectTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProfileObjectType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProfileObjectTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProfileObjectTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProfileObjectTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProfileObjectTypeRequest, DeleteProfileObjectTypeResponse>()
                            .withOperationName("DeleteProfileObjectType")
                            .withMarshaller(new DeleteProfileObjectTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProfileObjectTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteProfileObjectTypeRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteProfileObjectTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific domain.
     * </p>
     *
     * @param getDomainRequest
     * @return A Java Future containing the result of the GetDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.GetDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/GetDomain" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDomainResponse> getDomain(GetDomainRequest getDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDomainRequest, GetDomainResponse>().withOperationName("GetDomain")
                            .withMarshaller(new GetDomainRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an integration for a domain.
     * </p>
     *
     * @param getIntegrationRequest
     * @return A Java Future containing the result of the GetIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.GetIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/GetIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIntegrationResponse> getIntegration(GetIntegrationRequest getIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIntegrationRequest, GetIntegrationResponse>()
                            .withOperationName("GetIntegration")
                            .withMarshaller(new GetIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIntegrationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API is in preview release for Amazon Connect and subject to change.
     * </p>
     * <p>
     * Before calling this API, use <a
     * href="https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateDomain.html">CreateDomain</a> or
     * <a href="https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateDomain.html">UpdateDomain</a>
     * to enable identity resolution: set <code>Matching</code> to true.
     * </p>
     * <p>
     * GetMatches returns potentially matching profiles, based on the results of the latest run of a machine learning
     * process.
     * </p>
     * <important>
     * <p>
     * Amazon Connect runs a batch process every Saturday at 12AM UTC to identify matching profiles. The results are
     * returned up to seven days after the Saturday run.
     * </p>
     * </important>
     * <p>
     * Amazon Connect uses the following profile attributes to identify matches:
     * </p>
     * <ul>
     * <li>
     * <p>
     * PhoneNumber
     * </p>
     * </li>
     * <li>
     * <p>
     * HomePhoneNumber
     * </p>
     * </li>
     * <li>
     * <p>
     * BusinessPhoneNumber
     * </p>
     * </li>
     * <li>
     * <p>
     * MobilePhoneNumber
     * </p>
     * </li>
     * <li>
     * <p>
     * EmailAddress
     * </p>
     * </li>
     * <li>
     * <p>
     * PersonalEmailAddress
     * </p>
     * </li>
     * <li>
     * <p>
     * BusinessEmailAddress
     * </p>
     * </li>
     * <li>
     * <p>
     * FullName
     * </p>
     * </li>
     * <li>
     * <p>
     * BusinessName
     * </p>
     * </li>
     * </ul>
     *
     * @param getMatchesRequest
     * @return A Java Future containing the result of the GetMatches operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.GetMatches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/GetMatches" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMatchesResponse> getMatches(GetMatchesRequest getMatchesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMatchesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMatches");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMatchesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMatchesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMatchesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMatchesRequest, GetMatchesResponse>().withOperationName("GetMatches")
                            .withMarshaller(new GetMatchesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMatchesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMatchesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetMatchesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the object types for a specific domain.
     * </p>
     *
     * @param getProfileObjectTypeRequest
     * @return A Java Future containing the result of the GetProfileObjectType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.GetProfileObjectType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/GetProfileObjectType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileObjectTypeResponse> getProfileObjectType(
            GetProfileObjectTypeRequest getProfileObjectTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileObjectTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfileObjectType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileObjectTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProfileObjectTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileObjectTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileObjectTypeRequest, GetProfileObjectTypeResponse>()
                            .withOperationName("GetProfileObjectType")
                            .withMarshaller(new GetProfileObjectTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProfileObjectTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getProfileObjectTypeRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetProfileObjectTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the template information for a specific object type.
     * </p>
     * <p>
     * A template is a predefined ProfileObjectType, such as “Salesforce-Account” or “Salesforce-Contact.” When a user
     * sends a ProfileObject, using the PutProfileObject API, with an ObjectTypeName that matches one of the
     * TemplateIds, it uses the mappings from the template.
     * </p>
     *
     * @param getProfileObjectTypeTemplateRequest
     * @return A Java Future containing the result of the GetProfileObjectTypeTemplate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.GetProfileObjectTypeTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/GetProfileObjectTypeTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProfileObjectTypeTemplateResponse> getProfileObjectTypeTemplate(
            GetProfileObjectTypeTemplateRequest getProfileObjectTypeTemplateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProfileObjectTypeTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProfileObjectTypeTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProfileObjectTypeTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProfileObjectTypeTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProfileObjectTypeTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProfileObjectTypeTemplateRequest, GetProfileObjectTypeTemplateResponse>()
                            .withOperationName("GetProfileObjectTypeTemplate")
                            .withMarshaller(new GetProfileObjectTypeTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProfileObjectTypeTemplateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getProfileObjectTypeTemplateRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetProfileObjectTypeTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the integrations associated to a specific URI in the AWS account.
     * </p>
     *
     * @param listAccountIntegrationsRequest
     * @return A Java Future containing the result of the ListAccountIntegrations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListAccountIntegrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListAccountIntegrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAccountIntegrationsResponse> listAccountIntegrations(
            ListAccountIntegrationsRequest listAccountIntegrationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountIntegrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccountIntegrations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAccountIntegrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAccountIntegrationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAccountIntegrationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAccountIntegrationsRequest, ListAccountIntegrationsResponse>()
                            .withOperationName("ListAccountIntegrations")
                            .withMarshaller(new ListAccountIntegrationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAccountIntegrationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAccountIntegrationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListAccountIntegrationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the domains for an AWS account that have been created.
     * </p>
     *
     * @param listDomainsRequest
     * @return A Java Future containing the result of the ListDomains operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListDomains
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListDomains" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDomainsResponse> listDomains(ListDomainsRequest listDomainsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDomainsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDomains");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDomainsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDomainsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDomainsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDomainsRequest, ListDomainsResponse>()
                            .withOperationName("ListDomains").withMarshaller(new ListDomainsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDomainsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listDomainsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListDomainsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the integrations in your domain.
     * </p>
     *
     * @param listIntegrationsRequest
     * @return A Java Future containing the result of the ListIntegrations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListIntegrations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListIntegrations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIntegrationsResponse> listIntegrations(ListIntegrationsRequest listIntegrationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIntegrationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIntegrations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIntegrationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIntegrationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIntegrationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIntegrationsRequest, ListIntegrationsResponse>()
                            .withOperationName("ListIntegrations")
                            .withMarshaller(new ListIntegrationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIntegrationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listIntegrationsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListIntegrationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the template information for object types.
     * </p>
     *
     * @param listProfileObjectTypeTemplatesRequest
     * @return A Java Future containing the result of the ListProfileObjectTypeTemplates operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListProfileObjectTypeTemplates
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListProfileObjectTypeTemplates"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfileObjectTypeTemplatesResponse> listProfileObjectTypeTemplates(
            ListProfileObjectTypeTemplatesRequest listProfileObjectTypeTemplatesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listProfileObjectTypeTemplatesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileObjectTypeTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfileObjectTypeTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfileObjectTypeTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfileObjectTypeTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfileObjectTypeTemplatesRequest, ListProfileObjectTypeTemplatesResponse>()
                            .withOperationName("ListProfileObjectTypeTemplates")
                            .withMarshaller(new ListProfileObjectTypeTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProfileObjectTypeTemplatesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listProfileObjectTypeTemplatesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListProfileObjectTypeTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the templates available within the service.
     * </p>
     *
     * @param listProfileObjectTypesRequest
     * @return A Java Future containing the result of the ListProfileObjectTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListProfileObjectTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListProfileObjectTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfileObjectTypesResponse> listProfileObjectTypes(
            ListProfileObjectTypesRequest listProfileObjectTypesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfileObjectTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileObjectTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfileObjectTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfileObjectTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfileObjectTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfileObjectTypesRequest, ListProfileObjectTypesResponse>()
                            .withOperationName("ListProfileObjectTypes")
                            .withMarshaller(new ListProfileObjectTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProfileObjectTypesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listProfileObjectTypesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListProfileObjectTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of objects associated with a profile of a given ProfileObjectType.
     * </p>
     *
     * @param listProfileObjectsRequest
     * @return A Java Future containing the result of the ListProfileObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListProfileObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListProfileObjects"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProfileObjectsResponse> listProfileObjects(ListProfileObjectsRequest listProfileObjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProfileObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProfileObjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProfileObjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProfileObjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProfileObjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProfileObjectsRequest, ListProfileObjectsResponse>()
                            .withOperationName("ListProfileObjects")
                            .withMarshaller(new ListProfileObjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProfileObjectsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listProfileObjectsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListProfileObjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Displays the tags associated with an Amazon Connect Customer Profiles resource. In Connect Customer Profiles,
     * domains, profile object types, and integrations can be tagged.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This API is in preview release for Amazon Connect and subject to change.
     * </p>
     * <p>
     * Runs an AWS Lambda job that does the following:
     * </p>
     * <ol>
     * <li>
     * <p>
     * All the profileKeys in the <code>ProfileToBeMerged</code> will be moved to the main profile.
     * </p>
     * </li>
     * <li>
     * <p>
     * All the objects in the <code>ProfileToBeMerged</code> will be moved to the main profile.
     * </p>
     * </li>
     * <li>
     * <p>
     * All the <code>ProfileToBeMerged</code> will be deleted at the end.
     * </p>
     * </li>
     * <li>
     * <p>
     * All the profileKeys in the <code>ProfileIdsToBeMerged</code> will be moved to the main profile.
     * </p>
     * </li>
     * <li>
     * <p>
     * Standard fields are merged as follows:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Fields are always "union"-ed if there are no conflicts in standard fields or attributeKeys.
     * </p>
     * </li>
     * <li>
     * <p>
     * When there are conflicting fields:
     * </p>
     * <ol>
     * <li>
     * <p>
     * If no <code>SourceProfileIds</code> entry is specified, the main Profile value is always taken.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a <code>SourceProfileIds</code> entry is specified, the specified profileId is always taken, even if it is a
     * NULL value.
     * </p>
     * </li>
     * </ol>
     * </li>
     * </ol>
     * </li>
     * </ol>
     * <p>
     * You can use MergeProfiles together with <a
     * href="https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetMatches.html">GetMatches</a>, which
     * returns potentially matching profiles, or use it with the results of another matching system. After profiles have
     * been merged, they cannot be separated (unmerged).
     * </p>
     *
     * @param mergeProfilesRequest
     * @return A Java Future containing the result of the MergeProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.MergeProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/MergeProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergeProfilesResponse> mergeProfiles(MergeProfilesRequest mergeProfilesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergeProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    MergeProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergeProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergeProfilesRequest, MergeProfilesResponse>()
                            .withOperationName("MergeProfiles")
                            .withMarshaller(new MergeProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(mergeProfilesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = mergeProfilesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<MergeProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds an integration between the service and a third-party service, which includes Amazon AppFlow and Amazon
     * Connect.
     * </p>
     * <p>
     * An integration can belong to only one domain.
     * </p>
     *
     * @param putIntegrationRequest
     * @return A Java Future containing the result of the PutIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.PutIntegration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/PutIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutIntegrationResponse> putIntegration(PutIntegrationRequest putIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutIntegrationRequest, PutIntegrationResponse>()
                            .withOperationName("PutIntegration")
                            .withMarshaller(new PutIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putIntegrationRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds additional objects to customer profiles of a given ObjectType.
     * </p>
     * <p>
     * When adding a specific profile object, like a Contact Trace Record (CTR), an inferred profile can get created if
     * it is not mapped to an existing profile. The resulting profile will only have a phone number populated in the
     * standard ProfileObject. Any additional CTRs with the same phone number will be mapped to the same inferred
     * profile.
     * </p>
     * <p>
     * When a ProfileObject is created and if a ProfileObjectType already exists for the ProfileObject, it will provide
     * data to a standard profile depending on the ProfileObjectType definition.
     * </p>
     * <p>
     * PutProfileObject needs an ObjectType, which can be created using PutProfileObjectType.
     * </p>
     *
     * @param putProfileObjectRequest
     * @return A Java Future containing the result of the PutProfileObject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.PutProfileObject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/PutProfileObject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutProfileObjectResponse> putProfileObject(PutProfileObjectRequest putProfileObjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProfileObjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProfileObject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutProfileObjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutProfileObjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutProfileObjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutProfileObjectRequest, PutProfileObjectResponse>()
                            .withOperationName("PutProfileObject")
                            .withMarshaller(new PutProfileObjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putProfileObjectRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putProfileObjectRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutProfileObjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Defines a ProfileObjectType.
     * </p>
     *
     * @param putProfileObjectTypeRequest
     * @return A Java Future containing the result of the PutProfileObjectType operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.PutProfileObjectType
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/PutProfileObjectType"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutProfileObjectTypeResponse> putProfileObjectType(
            PutProfileObjectTypeRequest putProfileObjectTypeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putProfileObjectTypeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutProfileObjectType");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutProfileObjectTypeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutProfileObjectTypeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutProfileObjectTypeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutProfileObjectTypeRequest, PutProfileObjectTypeResponse>()
                            .withOperationName("PutProfileObjectType")
                            .withMarshaller(new PutProfileObjectTypeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putProfileObjectTypeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putProfileObjectTypeRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<PutProfileObjectTypeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for profiles within a specific domain name using name, phone number, email address, account number, or a
     * custom defined index.
     * </p>
     *
     * @param searchProfilesRequest
     * @return A Java Future containing the result of the SearchProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.SearchProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/SearchProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchProfilesResponse> searchProfiles(SearchProfilesRequest searchProfilesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchProfilesRequest, SearchProfilesResponse>()
                            .withOperationName("SearchProfiles")
                            .withMarshaller(new SearchProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchProfilesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchProfilesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SearchProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified Amazon Connect Customer Profiles resource. Tags can
     * help you organize and categorize your resources. You can also use them to scope user permissions by granting a
     * user permission to access or change only resources with certain tag values. In Connect Customer Profiles,
     * domains, profile object types, and integrations can be tagged.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to AWS and are interpreted strictly as strings of characters.
     * </p>
     * <p>
     * You can use the TagResource action with a resource that already has tags. If you specify a new tag key, this tag
     * is appended to the list of tags associated with the resource. If you specify a tag key that is already associated
     * with the resource, the new tag value that you specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified Amazon Connect Customer Profiles resource. In Connect Customer
     * Profiles, domains, profile object types, and integrations can be tagged.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the properties of a domain, including creating or selecting a dead letter queue or an encryption key.
     * </p>
     * <p>
     * After a domain is created, the name can’t be changed.
     * </p>
     *
     * @param updateDomainRequest
     * @return A Java Future containing the result of the UpdateDomain operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.UpdateDomain
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/UpdateDomain"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDomainResponse> updateDomain(UpdateDomainRequest updateDomainRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDomainRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDomain");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDomainResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateDomainResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDomainResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDomainRequest, UpdateDomainResponse>()
                            .withOperationName("UpdateDomain").withMarshaller(new UpdateDomainRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDomainRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateDomainRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateDomainResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the properties of a profile. The ProfileId is required for updating a customer profile.
     * </p>
     * <p>
     * When calling the UpdateProfile API, specifying an empty string value means that any existing value will be
     * removed. Not specifying a string value means that any value already there will be kept.
     * </p>
     *
     * @param updateProfileRequest
     * @return A Java Future containing the result of the UpdateProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input you provided is invalid.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException The requested resource does not exist, or access was denied.</li>
     *         <li>ThrottlingException You exceeded the maximum number of requests.</li>
     *         <li>InternalServerException An internal service error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CustomerProfilesException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CustomerProfilesAsyncClient.UpdateProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/customer-profiles-2020-08-15/UpdateProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProfileResponse> updateProfile(UpdateProfileRequest updateProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Customer Profiles");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProfileRequest, UpdateProfileResponse>()
                            .withOperationName("UpdateProfile")
                            .withMarshaller(new UpdateProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProfileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateProfileRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CustomerProfilesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
