/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.databasemigration.DatabaseMigrationAsyncClient;
import software.amazon.awssdk.services.databasemigration.model.DescribeConnectionsRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeConnectionsResponse;
import software.amazon.awssdk.services.databasemigration.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationInstancesRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationInstancesResponse;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationTasksRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationTasksResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface DatabaseMigrationAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeEndpoints} API until the desired condition
     * {@code EndpointDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeEndpointsRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointsResponse>> waitUntilEndpointDeleted(
            DescribeEndpointsRequest describeEndpointsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeEndpoints} API until the desired condition
     * {@code EndpointDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeEndpointsRequest#builder()}
     *
     * @param describeEndpointsRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointsResponse>> waitUntilEndpointDeleted(
            Consumer<DescribeEndpointsRequest.Builder> describeEndpointsRequest) {
        return waitUntilEndpointDeleted(DescribeEndpointsRequest.builder().applyMutation(describeEndpointsRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeEndpoints} API until the desired condition
     * {@code EndpointDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeEndpointsRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointsResponse>> waitUntilEndpointDeleted(
            DescribeEndpointsRequest describeEndpointsRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeEndpoints} API until the desired condition
     * {@code EndpointDeleted} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeEndpointsRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeEndpointsResponse>> waitUntilEndpointDeleted(
            Consumer<DescribeEndpointsRequest.Builder> describeEndpointsRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilEndpointDeleted(DescribeEndpointsRequest.builder().applyMutation(describeEndpointsRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceAvailable} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeReplicationInstancesRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceAvailable(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceAvailable} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationInstancesRequest#builder()}
     *
     * @param describeReplicationInstancesRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceAvailable(
            Consumer<DescribeReplicationInstancesRequest.Builder> describeReplicationInstancesRequest) {
        return waitUntilReplicationInstanceAvailable(DescribeReplicationInstancesRequest.builder()
                .applyMutation(describeReplicationInstancesRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceAvailable} is met, or until it is determined that the resource will never enter into
     * the desired state
     *
     * @param describeReplicationInstancesRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceAvailable(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceAvailable} is met, or until it is determined that the resource will never enter into
     * the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationInstancesRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceAvailable(
            Consumer<DescribeReplicationInstancesRequest.Builder> describeReplicationInstancesRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationInstanceAvailable(
                DescribeReplicationInstancesRequest.builder().applyMutation(describeReplicationInstancesRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationInstancesRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceDeleted(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationInstancesRequest#builder()}
     *
     * @param describeReplicationInstancesRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceDeleted(
            Consumer<DescribeReplicationInstancesRequest.Builder> describeReplicationInstancesRequest) {
        return waitUntilReplicationInstanceDeleted(DescribeReplicationInstancesRequest.builder()
                .applyMutation(describeReplicationInstancesRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationInstancesRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceDeleted(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationInstances} API until the desired condition
     * {@code ReplicationInstanceDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationInstancesRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceDeleted(
            Consumer<DescribeReplicationInstancesRequest.Builder> describeReplicationInstancesRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationInstanceDeleted(
                DescribeReplicationInstancesRequest.builder().applyMutation(describeReplicationInstancesRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskDeleted(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationTasksRequest#builder()}
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskDeleted(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest) {
        return waitUntilReplicationTaskDeleted(DescribeReplicationTasksRequest.builder()
                .applyMutation(describeReplicationTasksRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskDeleted(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskDeleted(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationTaskDeleted(
                DescribeReplicationTasksRequest.builder().applyMutation(describeReplicationTasksRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskReady} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskReady(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskReady} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationTasksRequest#builder()}
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskReady(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest) {
        return waitUntilReplicationTaskReady(DescribeReplicationTasksRequest.builder()
                .applyMutation(describeReplicationTasksRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskReady} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskReady(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskReady} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskReady(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationTaskReady(
                DescribeReplicationTasksRequest.builder().applyMutation(describeReplicationTasksRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskRunning} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskRunning(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskRunning} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationTasksRequest#builder()}
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskRunning(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest) {
        return waitUntilReplicationTaskRunning(DescribeReplicationTasksRequest.builder()
                .applyMutation(describeReplicationTasksRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskRunning} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskRunning(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskRunning} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskRunning(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationTaskRunning(
                DescribeReplicationTasksRequest.builder().applyMutation(describeReplicationTasksRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskStopped} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskStopped(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskStopped} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationTasksRequest#builder()}
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskStopped(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest) {
        return waitUntilReplicationTaskStopped(DescribeReplicationTasksRequest.builder()
                .applyMutation(describeReplicationTasksRequest).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskStopped} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationTasksRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskStopped(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeReplicationTasks} API until the desired condition
     * {@code ReplicationTaskStopped} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationTasksRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskStopped(
            Consumer<DescribeReplicationTasksRequest.Builder> describeReplicationTasksRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationTaskStopped(
                DescribeReplicationTasksRequest.builder().applyMutation(describeReplicationTasksRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeConnections} API until the desired condition
     * {@code TestConnectionSucceeds} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeConnectionsRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeConnectionsResponse>> waitUntilTestConnectionSucceeds(
            DescribeConnectionsRequest describeConnectionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeConnections} API until the desired condition
     * {@code TestConnectionSucceeds} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeConnectionsRequest#builder()}
     *
     * @param describeConnectionsRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeConnectionsResponse>> waitUntilTestConnectionSucceeds(
            Consumer<DescribeConnectionsRequest.Builder> describeConnectionsRequest) {
        return waitUntilTestConnectionSucceeds(DescribeConnectionsRequest.builder().applyMutation(describeConnectionsRequest)
                .build());
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeConnections} API until the desired condition
     * {@code TestConnectionSucceeds} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeConnectionsRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeConnectionsResponse>> waitUntilTestConnectionSucceeds(
            DescribeConnectionsRequest describeConnectionsRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DatabaseMigrationAsyncClient#describeConnections} API until the desired condition
     * {@code TestConnectionSucceeds} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeConnectionsRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeConnectionsResponse>> waitUntilTestConnectionSucceeds(
            Consumer<DescribeConnectionsRequest.Builder> describeConnectionsRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTestConnectionSucceeds(DescribeConnectionsRequest.builder().applyMutation(describeConnectionsRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link DatabaseMigrationAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultDatabaseMigrationAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link DatabaseMigrationAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link DatabaseMigrationAsyncClient} will be created to poll resources. It is recommended to share a
     * single instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link DatabaseMigrationAsyncWaiter}
     */
    static DatabaseMigrationAsyncWaiter create() {
        return DefaultDatabaseMigrationAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link DatabaseMigrationAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(DatabaseMigrationAsyncClient client);

        /**
         * Builds an instance of {@link DatabaseMigrationAsyncWaiter} based on the configurations supplied to this
         * builder
         *
         * @return An initialized {@link DatabaseMigrationAsyncWaiter}
         */
        DatabaseMigrationAsyncWaiter build();
    }
}
