/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This object provides information about a DMS data migration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataMigration implements SdkPojo, Serializable, ToCopyableBuilder<DataMigration.Builder, DataMigration> {
    private static final SdkField<String> DATA_MIGRATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataMigrationName").getter(getter(DataMigration::dataMigrationName))
            .setter(setter(Builder::dataMigrationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationName").build()).build();

    private static final SdkField<String> DATA_MIGRATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataMigrationArn").getter(getter(DataMigration::dataMigrationArn))
            .setter(setter(Builder::dataMigrationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationArn").build()).build();

    private static final SdkField<Instant> DATA_MIGRATION_CREATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DataMigrationCreateTime")
            .getter(getter(DataMigration::dataMigrationCreateTime))
            .setter(setter(Builder::dataMigrationCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationCreateTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> DATA_MIGRATION_START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DataMigrationStartTime")
            .getter(getter(DataMigration::dataMigrationStartTime))
            .setter(setter(Builder::dataMigrationStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationStartTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> DATA_MIGRATION_END_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DataMigrationEndTime")
            .getter(getter(DataMigration::dataMigrationEndTime))
            .setter(setter(Builder::dataMigrationEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationEndTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> SERVICE_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceAccessRoleArn").getter(getter(DataMigration::serviceAccessRoleArn))
            .setter(setter(Builder::serviceAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceAccessRoleArn").build())
            .build();

    private static final SdkField<String> MIGRATION_PROJECT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MigrationProjectArn").getter(getter(DataMigration::migrationProjectArn))
            .setter(setter(Builder::migrationProjectArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MigrationProjectArn").build())
            .build();

    private static final SdkField<String> DATA_MIGRATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataMigrationType").getter(getter(DataMigration::dataMigrationTypeAsString))
            .setter(setter(Builder::dataMigrationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationType").build()).build();

    private static final SdkField<DataMigrationSettings> DATA_MIGRATION_SETTINGS_FIELD = SdkField
            .<DataMigrationSettings> builder(MarshallingType.SDK_POJO).memberName("DataMigrationSettings")
            .getter(getter(DataMigration::dataMigrationSettings)).setter(setter(Builder::dataMigrationSettings))
            .constructor(DataMigrationSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationSettings").build())
            .build();

    private static final SdkField<List<SourceDataSetting>> SOURCE_DATA_SETTINGS_FIELD = SdkField
            .<List<SourceDataSetting>> builder(MarshallingType.LIST)
            .memberName("SourceDataSettings")
            .getter(getter(DataMigration::sourceDataSettings))
            .setter(setter(Builder::sourceDataSettings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDataSettings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SourceDataSetting> builder(MarshallingType.SDK_POJO)
                                            .constructor(SourceDataSetting::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataMigrationStatistics> DATA_MIGRATION_STATISTICS_FIELD = SdkField
            .<DataMigrationStatistics> builder(MarshallingType.SDK_POJO).memberName("DataMigrationStatistics")
            .getter(getter(DataMigration::dataMigrationStatistics)).setter(setter(Builder::dataMigrationStatistics))
            .constructor(DataMigrationStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationStatistics").build())
            .build();

    private static final SdkField<String> DATA_MIGRATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataMigrationStatus").getter(getter(DataMigration::dataMigrationStatus))
            .setter(setter(Builder::dataMigrationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataMigrationStatus").build())
            .build();

    private static final SdkField<List<String>> PUBLIC_IP_ADDRESSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PublicIpAddresses")
            .getter(getter(DataMigration::publicIpAddresses))
            .setter(setter(Builder::publicIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicIpAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LAST_FAILURE_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastFailureMessage").getter(getter(DataMigration::lastFailureMessage))
            .setter(setter(Builder::lastFailureMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastFailureMessage").build())
            .build();

    private static final SdkField<String> STOP_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StopReason").getter(getter(DataMigration::stopReason)).setter(setter(Builder::stopReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StopReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_MIGRATION_NAME_FIELD,
            DATA_MIGRATION_ARN_FIELD, DATA_MIGRATION_CREATE_TIME_FIELD, DATA_MIGRATION_START_TIME_FIELD,
            DATA_MIGRATION_END_TIME_FIELD, SERVICE_ACCESS_ROLE_ARN_FIELD, MIGRATION_PROJECT_ARN_FIELD, DATA_MIGRATION_TYPE_FIELD,
            DATA_MIGRATION_SETTINGS_FIELD, SOURCE_DATA_SETTINGS_FIELD, DATA_MIGRATION_STATISTICS_FIELD,
            DATA_MIGRATION_STATUS_FIELD, PUBLIC_IP_ADDRESSES_FIELD, LAST_FAILURE_MESSAGE_FIELD, STOP_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dataMigrationName;

    private final String dataMigrationArn;

    private final Instant dataMigrationCreateTime;

    private final Instant dataMigrationStartTime;

    private final Instant dataMigrationEndTime;

    private final String serviceAccessRoleArn;

    private final String migrationProjectArn;

    private final String dataMigrationType;

    private final DataMigrationSettings dataMigrationSettings;

    private final List<SourceDataSetting> sourceDataSettings;

    private final DataMigrationStatistics dataMigrationStatistics;

    private final String dataMigrationStatus;

    private final List<String> publicIpAddresses;

    private final String lastFailureMessage;

    private final String stopReason;

    private DataMigration(BuilderImpl builder) {
        this.dataMigrationName = builder.dataMigrationName;
        this.dataMigrationArn = builder.dataMigrationArn;
        this.dataMigrationCreateTime = builder.dataMigrationCreateTime;
        this.dataMigrationStartTime = builder.dataMigrationStartTime;
        this.dataMigrationEndTime = builder.dataMigrationEndTime;
        this.serviceAccessRoleArn = builder.serviceAccessRoleArn;
        this.migrationProjectArn = builder.migrationProjectArn;
        this.dataMigrationType = builder.dataMigrationType;
        this.dataMigrationSettings = builder.dataMigrationSettings;
        this.sourceDataSettings = builder.sourceDataSettings;
        this.dataMigrationStatistics = builder.dataMigrationStatistics;
        this.dataMigrationStatus = builder.dataMigrationStatus;
        this.publicIpAddresses = builder.publicIpAddresses;
        this.lastFailureMessage = builder.lastFailureMessage;
        this.stopReason = builder.stopReason;
    }

    /**
     * <p>
     * The user-friendly name for the data migration.
     * </p>
     * 
     * @return The user-friendly name for the data migration.
     */
    public final String dataMigrationName() {
        return dataMigrationName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that identifies this replication.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that identifies this replication.
     */
    public final String dataMigrationArn() {
        return dataMigrationArn;
    }

    /**
     * <p>
     * The UTC time when DMS created the data migration.
     * </p>
     * 
     * @return The UTC time when DMS created the data migration.
     */
    public final Instant dataMigrationCreateTime() {
        return dataMigrationCreateTime;
    }

    /**
     * <p>
     * The UTC time when DMS started the data migration.
     * </p>
     * 
     * @return The UTC time when DMS started the data migration.
     */
    public final Instant dataMigrationStartTime() {
        return dataMigrationStartTime;
    }

    /**
     * <p>
     * The UTC time when data migration ended.
     * </p>
     * 
     * @return The UTC time when data migration ended.
     */
    public final Instant dataMigrationEndTime() {
        return dataMigrationEndTime;
    }

    /**
     * <p>
     * The IAM role that the data migration uses to access Amazon Web Services resources.
     * </p>
     * 
     * @return The IAM role that the data migration uses to access Amazon Web Services resources.
     */
    public final String serviceAccessRoleArn() {
        return serviceAccessRoleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the data migration's associated migration project.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the data migration's associated migration project.
     */
    public final String migrationProjectArn() {
        return migrationProjectArn;
    }

    /**
     * <p>
     * Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load and CDC.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataMigrationType}
     * will return {@link MigrationTypeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #dataMigrationTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load and
     *         CDC.
     * @see MigrationTypeValue
     */
    public final MigrationTypeValue dataMigrationType() {
        return MigrationTypeValue.fromValue(dataMigrationType);
    }

    /**
     * <p>
     * Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load and CDC.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataMigrationType}
     * will return {@link MigrationTypeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #dataMigrationTypeAsString}.
     * </p>
     * 
     * @return Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load and
     *         CDC.
     * @see MigrationTypeValue
     */
    public final String dataMigrationTypeAsString() {
        return dataMigrationType;
    }

    /**
     * <p>
     * Specifies CloudWatch settings and selection rules for the data migration.
     * </p>
     * 
     * @return Specifies CloudWatch settings and selection rules for the data migration.
     */
    public final DataMigrationSettings dataMigrationSettings() {
        return dataMigrationSettings;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceDataSettings property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceDataSettings() {
        return sourceDataSettings != null && !(sourceDataSettings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies information about the data migration's source data provider.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceDataSettings} method.
     * </p>
     * 
     * @return Specifies information about the data migration's source data provider.
     */
    public final List<SourceDataSetting> sourceDataSettings() {
        return sourceDataSettings;
    }

    /**
     * <p>
     * Provides information about the data migration's run, including start and stop time, latency, and data migration
     * progress.
     * </p>
     * 
     * @return Provides information about the data migration's run, including start and stop time, latency, and data
     *         migration progress.
     */
    public final DataMigrationStatistics dataMigrationStatistics() {
        return dataMigrationStatistics;
    }

    /**
     * <p>
     * The current status of the data migration.
     * </p>
     * 
     * @return The current status of the data migration.
     */
    public final String dataMigrationStatus() {
        return dataMigrationStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the PublicIpAddresses property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPublicIpAddresses() {
        return publicIpAddresses != null && !(publicIpAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IP addresses of the endpoints for the data migration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPublicIpAddresses} method.
     * </p>
     * 
     * @return The IP addresses of the endpoints for the data migration.
     */
    public final List<String> publicIpAddresses() {
        return publicIpAddresses;
    }

    /**
     * <p>
     * Information about the data migration's most recent error or failure.
     * </p>
     * 
     * @return Information about the data migration's most recent error or failure.
     */
    public final String lastFailureMessage() {
        return lastFailureMessage;
    }

    /**
     * <p>
     * The reason the data migration last stopped.
     * </p>
     * 
     * @return The reason the data migration last stopped.
     */
    public final String stopReason() {
        return stopReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationName());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationArn());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(migrationProjectArn());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationSettings());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceDataSettings() ? sourceDataSettings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationStatistics());
        hashCode = 31 * hashCode + Objects.hashCode(dataMigrationStatus());
        hashCode = 31 * hashCode + Objects.hashCode(hasPublicIpAddresses() ? publicIpAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastFailureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(stopReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataMigration)) {
            return false;
        }
        DataMigration other = (DataMigration) obj;
        return Objects.equals(dataMigrationName(), other.dataMigrationName())
                && Objects.equals(dataMigrationArn(), other.dataMigrationArn())
                && Objects.equals(dataMigrationCreateTime(), other.dataMigrationCreateTime())
                && Objects.equals(dataMigrationStartTime(), other.dataMigrationStartTime())
                && Objects.equals(dataMigrationEndTime(), other.dataMigrationEndTime())
                && Objects.equals(serviceAccessRoleArn(), other.serviceAccessRoleArn())
                && Objects.equals(migrationProjectArn(), other.migrationProjectArn())
                && Objects.equals(dataMigrationTypeAsString(), other.dataMigrationTypeAsString())
                && Objects.equals(dataMigrationSettings(), other.dataMigrationSettings())
                && hasSourceDataSettings() == other.hasSourceDataSettings()
                && Objects.equals(sourceDataSettings(), other.sourceDataSettings())
                && Objects.equals(dataMigrationStatistics(), other.dataMigrationStatistics())
                && Objects.equals(dataMigrationStatus(), other.dataMigrationStatus())
                && hasPublicIpAddresses() == other.hasPublicIpAddresses()
                && Objects.equals(publicIpAddresses(), other.publicIpAddresses())
                && Objects.equals(lastFailureMessage(), other.lastFailureMessage())
                && Objects.equals(stopReason(), other.stopReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataMigration").add("DataMigrationName", dataMigrationName())
                .add("DataMigrationArn", dataMigrationArn()).add("DataMigrationCreateTime", dataMigrationCreateTime())
                .add("DataMigrationStartTime", dataMigrationStartTime()).add("DataMigrationEndTime", dataMigrationEndTime())
                .add("ServiceAccessRoleArn", serviceAccessRoleArn()).add("MigrationProjectArn", migrationProjectArn())
                .add("DataMigrationType", dataMigrationTypeAsString()).add("DataMigrationSettings", dataMigrationSettings())
                .add("SourceDataSettings", hasSourceDataSettings() ? sourceDataSettings() : null)
                .add("DataMigrationStatistics", dataMigrationStatistics()).add("DataMigrationStatus", dataMigrationStatus())
                .add("PublicIpAddresses", publicIpAddresses() == null ? null : "*** Sensitive Data Redacted ***")
                .add("LastFailureMessage", lastFailureMessage()).add("StopReason", stopReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataMigrationName":
            return Optional.ofNullable(clazz.cast(dataMigrationName()));
        case "DataMigrationArn":
            return Optional.ofNullable(clazz.cast(dataMigrationArn()));
        case "DataMigrationCreateTime":
            return Optional.ofNullable(clazz.cast(dataMigrationCreateTime()));
        case "DataMigrationStartTime":
            return Optional.ofNullable(clazz.cast(dataMigrationStartTime()));
        case "DataMigrationEndTime":
            return Optional.ofNullable(clazz.cast(dataMigrationEndTime()));
        case "ServiceAccessRoleArn":
            return Optional.ofNullable(clazz.cast(serviceAccessRoleArn()));
        case "MigrationProjectArn":
            return Optional.ofNullable(clazz.cast(migrationProjectArn()));
        case "DataMigrationType":
            return Optional.ofNullable(clazz.cast(dataMigrationTypeAsString()));
        case "DataMigrationSettings":
            return Optional.ofNullable(clazz.cast(dataMigrationSettings()));
        case "SourceDataSettings":
            return Optional.ofNullable(clazz.cast(sourceDataSettings()));
        case "DataMigrationStatistics":
            return Optional.ofNullable(clazz.cast(dataMigrationStatistics()));
        case "DataMigrationStatus":
            return Optional.ofNullable(clazz.cast(dataMigrationStatus()));
        case "PublicIpAddresses":
            return Optional.ofNullable(clazz.cast(publicIpAddresses()));
        case "LastFailureMessage":
            return Optional.ofNullable(clazz.cast(lastFailureMessage()));
        case "StopReason":
            return Optional.ofNullable(clazz.cast(stopReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataMigration, T> g) {
        return obj -> g.apply((DataMigration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataMigration> {
        /**
         * <p>
         * The user-friendly name for the data migration.
         * </p>
         * 
         * @param dataMigrationName
         *        The user-friendly name for the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationName(String dataMigrationName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that identifies this replication.
         * </p>
         * 
         * @param dataMigrationArn
         *        The Amazon Resource Name (ARN) that identifies this replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationArn(String dataMigrationArn);

        /**
         * <p>
         * The UTC time when DMS created the data migration.
         * </p>
         * 
         * @param dataMigrationCreateTime
         *        The UTC time when DMS created the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationCreateTime(Instant dataMigrationCreateTime);

        /**
         * <p>
         * The UTC time when DMS started the data migration.
         * </p>
         * 
         * @param dataMigrationStartTime
         *        The UTC time when DMS started the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationStartTime(Instant dataMigrationStartTime);

        /**
         * <p>
         * The UTC time when data migration ended.
         * </p>
         * 
         * @param dataMigrationEndTime
         *        The UTC time when data migration ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationEndTime(Instant dataMigrationEndTime);

        /**
         * <p>
         * The IAM role that the data migration uses to access Amazon Web Services resources.
         * </p>
         * 
         * @param serviceAccessRoleArn
         *        The IAM role that the data migration uses to access Amazon Web Services resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessRoleArn(String serviceAccessRoleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the data migration's associated migration project.
         * </p>
         * 
         * @param migrationProjectArn
         *        The Amazon Resource Name (ARN) of the data migration's associated migration project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder migrationProjectArn(String migrationProjectArn);

        /**
         * <p>
         * Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load and CDC.
         * </p>
         * 
         * @param dataMigrationType
         *        Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load
         *        and CDC.
         * @see MigrationTypeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationTypeValue
         */
        Builder dataMigrationType(String dataMigrationType);

        /**
         * <p>
         * Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load and CDC.
         * </p>
         * 
         * @param dataMigrationType
         *        Specifies whether the data migration is full-load only, change data capture (CDC) only, or full-load
         *        and CDC.
         * @see MigrationTypeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationTypeValue
         */
        Builder dataMigrationType(MigrationTypeValue dataMigrationType);

        /**
         * <p>
         * Specifies CloudWatch settings and selection rules for the data migration.
         * </p>
         * 
         * @param dataMigrationSettings
         *        Specifies CloudWatch settings and selection rules for the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationSettings(DataMigrationSettings dataMigrationSettings);

        /**
         * <p>
         * Specifies CloudWatch settings and selection rules for the data migration.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataMigrationSettings.Builder} avoiding
         * the need to create one manually via {@link DataMigrationSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataMigrationSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #dataMigrationSettings(DataMigrationSettings)}.
         * 
         * @param dataMigrationSettings
         *        a consumer that will call methods on {@link DataMigrationSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataMigrationSettings(DataMigrationSettings)
         */
        default Builder dataMigrationSettings(Consumer<DataMigrationSettings.Builder> dataMigrationSettings) {
            return dataMigrationSettings(DataMigrationSettings.builder().applyMutation(dataMigrationSettings).build());
        }

        /**
         * <p>
         * Specifies information about the data migration's source data provider.
         * </p>
         * 
         * @param sourceDataSettings
         *        Specifies information about the data migration's source data provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDataSettings(Collection<SourceDataSetting> sourceDataSettings);

        /**
         * <p>
         * Specifies information about the data migration's source data provider.
         * </p>
         * 
         * @param sourceDataSettings
         *        Specifies information about the data migration's source data provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDataSettings(SourceDataSetting... sourceDataSettings);

        /**
         * <p>
         * Specifies information about the data migration's source data provider.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.databasemigration.model.SourceDataSetting.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.databasemigration.model.SourceDataSetting#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.databasemigration.model.SourceDataSetting.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceDataSettings(List<SourceDataSetting>)}.
         * 
         * @param sourceDataSettings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.databasemigration.model.SourceDataSetting.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceDataSettings(java.util.Collection<SourceDataSetting>)
         */
        Builder sourceDataSettings(Consumer<SourceDataSetting.Builder>... sourceDataSettings);

        /**
         * <p>
         * Provides information about the data migration's run, including start and stop time, latency, and data
         * migration progress.
         * </p>
         * 
         * @param dataMigrationStatistics
         *        Provides information about the data migration's run, including start and stop time, latency, and data
         *        migration progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationStatistics(DataMigrationStatistics dataMigrationStatistics);

        /**
         * <p>
         * Provides information about the data migration's run, including start and stop time, latency, and data
         * migration progress.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataMigrationStatistics.Builder} avoiding
         * the need to create one manually via {@link DataMigrationStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataMigrationStatistics.Builder#build()} is called immediately
         * and its result is passed to {@link #dataMigrationStatistics(DataMigrationStatistics)}.
         * 
         * @param dataMigrationStatistics
         *        a consumer that will call methods on {@link DataMigrationStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataMigrationStatistics(DataMigrationStatistics)
         */
        default Builder dataMigrationStatistics(Consumer<DataMigrationStatistics.Builder> dataMigrationStatistics) {
            return dataMigrationStatistics(DataMigrationStatistics.builder().applyMutation(dataMigrationStatistics).build());
        }

        /**
         * <p>
         * The current status of the data migration.
         * </p>
         * 
         * @param dataMigrationStatus
         *        The current status of the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataMigrationStatus(String dataMigrationStatus);

        /**
         * <p>
         * The IP addresses of the endpoints for the data migration.
         * </p>
         * 
         * @param publicIpAddresses
         *        The IP addresses of the endpoints for the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicIpAddresses(Collection<String> publicIpAddresses);

        /**
         * <p>
         * The IP addresses of the endpoints for the data migration.
         * </p>
         * 
         * @param publicIpAddresses
         *        The IP addresses of the endpoints for the data migration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicIpAddresses(String... publicIpAddresses);

        /**
         * <p>
         * Information about the data migration's most recent error or failure.
         * </p>
         * 
         * @param lastFailureMessage
         *        Information about the data migration's most recent error or failure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastFailureMessage(String lastFailureMessage);

        /**
         * <p>
         * The reason the data migration last stopped.
         * </p>
         * 
         * @param stopReason
         *        The reason the data migration last stopped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopReason(String stopReason);
    }

    static final class BuilderImpl implements Builder {
        private String dataMigrationName;

        private String dataMigrationArn;

        private Instant dataMigrationCreateTime;

        private Instant dataMigrationStartTime;

        private Instant dataMigrationEndTime;

        private String serviceAccessRoleArn;

        private String migrationProjectArn;

        private String dataMigrationType;

        private DataMigrationSettings dataMigrationSettings;

        private List<SourceDataSetting> sourceDataSettings = DefaultSdkAutoConstructList.getInstance();

        private DataMigrationStatistics dataMigrationStatistics;

        private String dataMigrationStatus;

        private List<String> publicIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private String lastFailureMessage;

        private String stopReason;

        private BuilderImpl() {
        }

        private BuilderImpl(DataMigration model) {
            dataMigrationName(model.dataMigrationName);
            dataMigrationArn(model.dataMigrationArn);
            dataMigrationCreateTime(model.dataMigrationCreateTime);
            dataMigrationStartTime(model.dataMigrationStartTime);
            dataMigrationEndTime(model.dataMigrationEndTime);
            serviceAccessRoleArn(model.serviceAccessRoleArn);
            migrationProjectArn(model.migrationProjectArn);
            dataMigrationType(model.dataMigrationType);
            dataMigrationSettings(model.dataMigrationSettings);
            sourceDataSettings(model.sourceDataSettings);
            dataMigrationStatistics(model.dataMigrationStatistics);
            dataMigrationStatus(model.dataMigrationStatus);
            publicIpAddresses(model.publicIpAddresses);
            lastFailureMessage(model.lastFailureMessage);
            stopReason(model.stopReason);
        }

        public final String getDataMigrationName() {
            return dataMigrationName;
        }

        public final void setDataMigrationName(String dataMigrationName) {
            this.dataMigrationName = dataMigrationName;
        }

        @Override
        public final Builder dataMigrationName(String dataMigrationName) {
            this.dataMigrationName = dataMigrationName;
            return this;
        }

        public final String getDataMigrationArn() {
            return dataMigrationArn;
        }

        public final void setDataMigrationArn(String dataMigrationArn) {
            this.dataMigrationArn = dataMigrationArn;
        }

        @Override
        public final Builder dataMigrationArn(String dataMigrationArn) {
            this.dataMigrationArn = dataMigrationArn;
            return this;
        }

        public final Instant getDataMigrationCreateTime() {
            return dataMigrationCreateTime;
        }

        public final void setDataMigrationCreateTime(Instant dataMigrationCreateTime) {
            this.dataMigrationCreateTime = dataMigrationCreateTime;
        }

        @Override
        public final Builder dataMigrationCreateTime(Instant dataMigrationCreateTime) {
            this.dataMigrationCreateTime = dataMigrationCreateTime;
            return this;
        }

        public final Instant getDataMigrationStartTime() {
            return dataMigrationStartTime;
        }

        public final void setDataMigrationStartTime(Instant dataMigrationStartTime) {
            this.dataMigrationStartTime = dataMigrationStartTime;
        }

        @Override
        public final Builder dataMigrationStartTime(Instant dataMigrationStartTime) {
            this.dataMigrationStartTime = dataMigrationStartTime;
            return this;
        }

        public final Instant getDataMigrationEndTime() {
            return dataMigrationEndTime;
        }

        public final void setDataMigrationEndTime(Instant dataMigrationEndTime) {
            this.dataMigrationEndTime = dataMigrationEndTime;
        }

        @Override
        public final Builder dataMigrationEndTime(Instant dataMigrationEndTime) {
            this.dataMigrationEndTime = dataMigrationEndTime;
            return this;
        }

        public final String getServiceAccessRoleArn() {
            return serviceAccessRoleArn;
        }

        public final void setServiceAccessRoleArn(String serviceAccessRoleArn) {
            this.serviceAccessRoleArn = serviceAccessRoleArn;
        }

        @Override
        public final Builder serviceAccessRoleArn(String serviceAccessRoleArn) {
            this.serviceAccessRoleArn = serviceAccessRoleArn;
            return this;
        }

        public final String getMigrationProjectArn() {
            return migrationProjectArn;
        }

        public final void setMigrationProjectArn(String migrationProjectArn) {
            this.migrationProjectArn = migrationProjectArn;
        }

        @Override
        public final Builder migrationProjectArn(String migrationProjectArn) {
            this.migrationProjectArn = migrationProjectArn;
            return this;
        }

        public final String getDataMigrationType() {
            return dataMigrationType;
        }

        public final void setDataMigrationType(String dataMigrationType) {
            this.dataMigrationType = dataMigrationType;
        }

        @Override
        public final Builder dataMigrationType(String dataMigrationType) {
            this.dataMigrationType = dataMigrationType;
            return this;
        }

        @Override
        public final Builder dataMigrationType(MigrationTypeValue dataMigrationType) {
            this.dataMigrationType(dataMigrationType == null ? null : dataMigrationType.toString());
            return this;
        }

        public final DataMigrationSettings.Builder getDataMigrationSettings() {
            return dataMigrationSettings != null ? dataMigrationSettings.toBuilder() : null;
        }

        public final void setDataMigrationSettings(DataMigrationSettings.BuilderImpl dataMigrationSettings) {
            this.dataMigrationSettings = dataMigrationSettings != null ? dataMigrationSettings.build() : null;
        }

        @Override
        public final Builder dataMigrationSettings(DataMigrationSettings dataMigrationSettings) {
            this.dataMigrationSettings = dataMigrationSettings;
            return this;
        }

        public final List<SourceDataSetting.Builder> getSourceDataSettings() {
            List<SourceDataSetting.Builder> result = SourceDataSettingsCopier.copyToBuilder(this.sourceDataSettings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceDataSettings(Collection<SourceDataSetting.BuilderImpl> sourceDataSettings) {
            this.sourceDataSettings = SourceDataSettingsCopier.copyFromBuilder(sourceDataSettings);
        }

        @Override
        public final Builder sourceDataSettings(Collection<SourceDataSetting> sourceDataSettings) {
            this.sourceDataSettings = SourceDataSettingsCopier.copy(sourceDataSettings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceDataSettings(SourceDataSetting... sourceDataSettings) {
            sourceDataSettings(Arrays.asList(sourceDataSettings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceDataSettings(Consumer<SourceDataSetting.Builder>... sourceDataSettings) {
            sourceDataSettings(Stream.of(sourceDataSettings).map(c -> SourceDataSetting.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final DataMigrationStatistics.Builder getDataMigrationStatistics() {
            return dataMigrationStatistics != null ? dataMigrationStatistics.toBuilder() : null;
        }

        public final void setDataMigrationStatistics(DataMigrationStatistics.BuilderImpl dataMigrationStatistics) {
            this.dataMigrationStatistics = dataMigrationStatistics != null ? dataMigrationStatistics.build() : null;
        }

        @Override
        public final Builder dataMigrationStatistics(DataMigrationStatistics dataMigrationStatistics) {
            this.dataMigrationStatistics = dataMigrationStatistics;
            return this;
        }

        public final String getDataMigrationStatus() {
            return dataMigrationStatus;
        }

        public final void setDataMigrationStatus(String dataMigrationStatus) {
            this.dataMigrationStatus = dataMigrationStatus;
        }

        @Override
        public final Builder dataMigrationStatus(String dataMigrationStatus) {
            this.dataMigrationStatus = dataMigrationStatus;
            return this;
        }

        public final Collection<String> getPublicIpAddresses() {
            if (publicIpAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return publicIpAddresses;
        }

        public final void setPublicIpAddresses(Collection<String> publicIpAddresses) {
            this.publicIpAddresses = PublicIpAddressListCopier.copy(publicIpAddresses);
        }

        @Override
        public final Builder publicIpAddresses(Collection<String> publicIpAddresses) {
            this.publicIpAddresses = PublicIpAddressListCopier.copy(publicIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder publicIpAddresses(String... publicIpAddresses) {
            publicIpAddresses(Arrays.asList(publicIpAddresses));
            return this;
        }

        public final String getLastFailureMessage() {
            return lastFailureMessage;
        }

        public final void setLastFailureMessage(String lastFailureMessage) {
            this.lastFailureMessage = lastFailureMessage;
        }

        @Override
        public final Builder lastFailureMessage(String lastFailureMessage) {
            this.lastFailureMessage = lastFailureMessage;
            return this;
        }

        public final String getStopReason() {
            return stopReason;
        }

        public final void setStopReason(String stopReason) {
            this.stopReason = stopReason;
        }

        @Override
        public final Builder stopReason(String stopReason) {
            this.stopReason = stopReason;
            return this;
        }

        @Override
        public DataMigration build() {
            return new DataMigration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
