/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about a replication instance version.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EngineVersion implements SdkPojo, Serializable, ToCopyableBuilder<EngineVersion.Builder, EngineVersion> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(EngineVersion::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(EngineVersion::lifecycle)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<String> RELEASE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReleaseStatus").getter(getter(EngineVersion::releaseStatusAsString))
            .setter(setter(Builder::releaseStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseStatus").build()).build();

    private static final SdkField<Instant> LAUNCH_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LaunchDate").getter(getter(EngineVersion::launchDate)).setter(setter(Builder::launchDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchDate").build()).build();

    private static final SdkField<Instant> AUTO_UPGRADE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("AutoUpgradeDate").getter(getter(EngineVersion::autoUpgradeDate))
            .setter(setter(Builder::autoUpgradeDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoUpgradeDate").build()).build();

    private static final SdkField<Instant> DEPRECATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeprecationDate").getter(getter(EngineVersion::deprecationDate))
            .setter(setter(Builder::deprecationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeprecationDate").build()).build();

    private static final SdkField<Instant> FORCE_UPGRADE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ForceUpgradeDate").getter(getter(EngineVersion::forceUpgradeDate))
            .setter(setter(Builder::forceUpgradeDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForceUpgradeDate").build()).build();

    private static final SdkField<List<String>> AVAILABLE_UPGRADES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailableUpgrades")
            .getter(getter(EngineVersion::availableUpgrades))
            .setter(setter(Builder::availableUpgrades))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailableUpgrades").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD,
            LIFECYCLE_FIELD, RELEASE_STATUS_FIELD, LAUNCH_DATE_FIELD, AUTO_UPGRADE_DATE_FIELD, DEPRECATION_DATE_FIELD,
            FORCE_UPGRADE_DATE_FIELD, AVAILABLE_UPGRADES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String version;

    private final String lifecycle;

    private final String releaseStatus;

    private final Instant launchDate;

    private final Instant autoUpgradeDate;

    private final Instant deprecationDate;

    private final Instant forceUpgradeDate;

    private final List<String> availableUpgrades;

    private EngineVersion(BuilderImpl builder) {
        this.version = builder.version;
        this.lifecycle = builder.lifecycle;
        this.releaseStatus = builder.releaseStatus;
        this.launchDate = builder.launchDate;
        this.autoUpgradeDate = builder.autoUpgradeDate;
        this.deprecationDate = builder.deprecationDate;
        this.forceUpgradeDate = builder.forceUpgradeDate;
        this.availableUpgrades = builder.availableUpgrades;
    }

    /**
     * <p>
     * The version number of the replication instance.
     * </p>
     * 
     * @return The version number of the replication instance.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The lifecycle status of the replication instance version. Valid values are <code>DEPRECATED</code>,
     * <code>DEFAULT_VERSION</code>, and <code>ACTIVE</code>.
     * </p>
     * 
     * @return The lifecycle status of the replication instance version. Valid values are <code>DEPRECATED</code>,
     *         <code>DEFAULT_VERSION</code>, and <code>ACTIVE</code>.
     */
    public final String lifecycle() {
        return lifecycle;
    }

    /**
     * <p>
     * The release status of the replication instance version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #releaseStatus}
     * will return {@link ReleaseStatusValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #releaseStatusAsString}.
     * </p>
     * 
     * @return The release status of the replication instance version.
     * @see ReleaseStatusValues
     */
    public final ReleaseStatusValues releaseStatus() {
        return ReleaseStatusValues.fromValue(releaseStatus);
    }

    /**
     * <p>
     * The release status of the replication instance version.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #releaseStatus}
     * will return {@link ReleaseStatusValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #releaseStatusAsString}.
     * </p>
     * 
     * @return The release status of the replication instance version.
     * @see ReleaseStatusValues
     */
    public final String releaseStatusAsString() {
        return releaseStatus;
    }

    /**
     * <p>
     * The date when the replication instance version became publicly available.
     * </p>
     * 
     * @return The date when the replication instance version became publicly available.
     */
    public final Instant launchDate() {
        return launchDate;
    }

    /**
     * <p>
     * The date when the replication instance will be automatically upgraded. This setting only applies if the
     * <code>auto-minor-version</code> setting is enabled.
     * </p>
     * 
     * @return The date when the replication instance will be automatically upgraded. This setting only applies if the
     *         <code>auto-minor-version</code> setting is enabled.
     */
    public final Instant autoUpgradeDate() {
        return autoUpgradeDate;
    }

    /**
     * <p>
     * The date when the replication instance version will be deprecated and can no longer be requested.
     * </p>
     * 
     * @return The date when the replication instance version will be deprecated and can no longer be requested.
     */
    public final Instant deprecationDate() {
        return deprecationDate;
    }

    /**
     * <p>
     * The date when the replication instance will have a version upgrade forced.
     * </p>
     * 
     * @return The date when the replication instance will have a version upgrade forced.
     */
    public final Instant forceUpgradeDate() {
        return forceUpgradeDate;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailableUpgrades property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailableUpgrades() {
        return availableUpgrades != null && !(availableUpgrades instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of valid replication instance versions that you can upgrade to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailableUpgrades} method.
     * </p>
     * 
     * @return The list of valid replication instance versions that you can upgrade to.
     */
    public final List<String> availableUpgrades() {
        return availableUpgrades;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycle());
        hashCode = 31 * hashCode + Objects.hashCode(releaseStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(launchDate());
        hashCode = 31 * hashCode + Objects.hashCode(autoUpgradeDate());
        hashCode = 31 * hashCode + Objects.hashCode(deprecationDate());
        hashCode = 31 * hashCode + Objects.hashCode(forceUpgradeDate());
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailableUpgrades() ? availableUpgrades() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EngineVersion)) {
            return false;
        }
        EngineVersion other = (EngineVersion) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(lifecycle(), other.lifecycle())
                && Objects.equals(releaseStatusAsString(), other.releaseStatusAsString())
                && Objects.equals(launchDate(), other.launchDate()) && Objects.equals(autoUpgradeDate(), other.autoUpgradeDate())
                && Objects.equals(deprecationDate(), other.deprecationDate())
                && Objects.equals(forceUpgradeDate(), other.forceUpgradeDate())
                && hasAvailableUpgrades() == other.hasAvailableUpgrades()
                && Objects.equals(availableUpgrades(), other.availableUpgrades());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EngineVersion").add("Version", version()).add("Lifecycle", lifecycle())
                .add("ReleaseStatus", releaseStatusAsString()).add("LaunchDate", launchDate())
                .add("AutoUpgradeDate", autoUpgradeDate()).add("DeprecationDate", deprecationDate())
                .add("ForceUpgradeDate", forceUpgradeDate())
                .add("AvailableUpgrades", hasAvailableUpgrades() ? availableUpgrades() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycle()));
        case "ReleaseStatus":
            return Optional.ofNullable(clazz.cast(releaseStatusAsString()));
        case "LaunchDate":
            return Optional.ofNullable(clazz.cast(launchDate()));
        case "AutoUpgradeDate":
            return Optional.ofNullable(clazz.cast(autoUpgradeDate()));
        case "DeprecationDate":
            return Optional.ofNullable(clazz.cast(deprecationDate()));
        case "ForceUpgradeDate":
            return Optional.ofNullable(clazz.cast(forceUpgradeDate()));
        case "AvailableUpgrades":
            return Optional.ofNullable(clazz.cast(availableUpgrades()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Version", VERSION_FIELD);
        map.put("Lifecycle", LIFECYCLE_FIELD);
        map.put("ReleaseStatus", RELEASE_STATUS_FIELD);
        map.put("LaunchDate", LAUNCH_DATE_FIELD);
        map.put("AutoUpgradeDate", AUTO_UPGRADE_DATE_FIELD);
        map.put("DeprecationDate", DEPRECATION_DATE_FIELD);
        map.put("ForceUpgradeDate", FORCE_UPGRADE_DATE_FIELD);
        map.put("AvailableUpgrades", AVAILABLE_UPGRADES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EngineVersion, T> g) {
        return obj -> g.apply((EngineVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EngineVersion> {
        /**
         * <p>
         * The version number of the replication instance.
         * </p>
         * 
         * @param version
         *        The version number of the replication instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The lifecycle status of the replication instance version. Valid values are <code>DEPRECATED</code>,
         * <code>DEFAULT_VERSION</code>, and <code>ACTIVE</code>.
         * </p>
         * 
         * @param lifecycle
         *        The lifecycle status of the replication instance version. Valid values are <code>DEPRECATED</code>,
         *        <code>DEFAULT_VERSION</code>, and <code>ACTIVE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * The release status of the replication instance version.
         * </p>
         * 
         * @param releaseStatus
         *        The release status of the replication instance version.
         * @see ReleaseStatusValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReleaseStatusValues
         */
        Builder releaseStatus(String releaseStatus);

        /**
         * <p>
         * The release status of the replication instance version.
         * </p>
         * 
         * @param releaseStatus
         *        The release status of the replication instance version.
         * @see ReleaseStatusValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReleaseStatusValues
         */
        Builder releaseStatus(ReleaseStatusValues releaseStatus);

        /**
         * <p>
         * The date when the replication instance version became publicly available.
         * </p>
         * 
         * @param launchDate
         *        The date when the replication instance version became publicly available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchDate(Instant launchDate);

        /**
         * <p>
         * The date when the replication instance will be automatically upgraded. This setting only applies if the
         * <code>auto-minor-version</code> setting is enabled.
         * </p>
         * 
         * @param autoUpgradeDate
         *        The date when the replication instance will be automatically upgraded. This setting only applies if
         *        the <code>auto-minor-version</code> setting is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoUpgradeDate(Instant autoUpgradeDate);

        /**
         * <p>
         * The date when the replication instance version will be deprecated and can no longer be requested.
         * </p>
         * 
         * @param deprecationDate
         *        The date when the replication instance version will be deprecated and can no longer be requested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deprecationDate(Instant deprecationDate);

        /**
         * <p>
         * The date when the replication instance will have a version upgrade forced.
         * </p>
         * 
         * @param forceUpgradeDate
         *        The date when the replication instance will have a version upgrade forced.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceUpgradeDate(Instant forceUpgradeDate);

        /**
         * <p>
         * The list of valid replication instance versions that you can upgrade to.
         * </p>
         * 
         * @param availableUpgrades
         *        The list of valid replication instance versions that you can upgrade to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableUpgrades(Collection<String> availableUpgrades);

        /**
         * <p>
         * The list of valid replication instance versions that you can upgrade to.
         * </p>
         * 
         * @param availableUpgrades
         *        The list of valid replication instance versions that you can upgrade to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableUpgrades(String... availableUpgrades);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private String lifecycle;

        private String releaseStatus;

        private Instant launchDate;

        private Instant autoUpgradeDate;

        private Instant deprecationDate;

        private Instant forceUpgradeDate;

        private List<String> availableUpgrades = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EngineVersion model) {
            version(model.version);
            lifecycle(model.lifecycle);
            releaseStatus(model.releaseStatus);
            launchDate(model.launchDate);
            autoUpgradeDate(model.autoUpgradeDate);
            deprecationDate(model.deprecationDate);
            forceUpgradeDate(model.forceUpgradeDate);
            availableUpgrades(model.availableUpgrades);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        public final String getReleaseStatus() {
            return releaseStatus;
        }

        public final void setReleaseStatus(String releaseStatus) {
            this.releaseStatus = releaseStatus;
        }

        @Override
        public final Builder releaseStatus(String releaseStatus) {
            this.releaseStatus = releaseStatus;
            return this;
        }

        @Override
        public final Builder releaseStatus(ReleaseStatusValues releaseStatus) {
            this.releaseStatus(releaseStatus == null ? null : releaseStatus.toString());
            return this;
        }

        public final Instant getLaunchDate() {
            return launchDate;
        }

        public final void setLaunchDate(Instant launchDate) {
            this.launchDate = launchDate;
        }

        @Override
        public final Builder launchDate(Instant launchDate) {
            this.launchDate = launchDate;
            return this;
        }

        public final Instant getAutoUpgradeDate() {
            return autoUpgradeDate;
        }

        public final void setAutoUpgradeDate(Instant autoUpgradeDate) {
            this.autoUpgradeDate = autoUpgradeDate;
        }

        @Override
        public final Builder autoUpgradeDate(Instant autoUpgradeDate) {
            this.autoUpgradeDate = autoUpgradeDate;
            return this;
        }

        public final Instant getDeprecationDate() {
            return deprecationDate;
        }

        public final void setDeprecationDate(Instant deprecationDate) {
            this.deprecationDate = deprecationDate;
        }

        @Override
        public final Builder deprecationDate(Instant deprecationDate) {
            this.deprecationDate = deprecationDate;
            return this;
        }

        public final Instant getForceUpgradeDate() {
            return forceUpgradeDate;
        }

        public final void setForceUpgradeDate(Instant forceUpgradeDate) {
            this.forceUpgradeDate = forceUpgradeDate;
        }

        @Override
        public final Builder forceUpgradeDate(Instant forceUpgradeDate) {
            this.forceUpgradeDate = forceUpgradeDate;
            return this;
        }

        public final Collection<String> getAvailableUpgrades() {
            if (availableUpgrades instanceof SdkAutoConstructList) {
                return null;
            }
            return availableUpgrades;
        }

        public final void setAvailableUpgrades(Collection<String> availableUpgrades) {
            this.availableUpgrades = AvailableUpgradesListCopier.copy(availableUpgrades);
        }

        @Override
        public final Builder availableUpgrades(Collection<String> availableUpgrades) {
            this.availableUpgrades = AvailableUpgradesListCopier.copy(availableUpgrades);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availableUpgrades(String... availableUpgrades) {
            availableUpgrades(Arrays.asList(availableUpgrades));
            return this;
        }

        @Override
        public EngineVersion build() {
            return new EngineVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
