/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.databasemigration.DatabaseMigrationAsyncClient;
import software.amazon.awssdk.services.databasemigration.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.databasemigration.model.DatabaseMigrationRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeConnectionsRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeConnectionsResponse;
import software.amazon.awssdk.services.databasemigration.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationInstancesRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationInstancesResponse;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationTasksRequest;
import software.amazon.awssdk.services.databasemigration.model.DescribeReplicationTasksResponse;
import software.amazon.awssdk.services.databasemigration.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultDatabaseMigrationAsyncWaiter implements DatabaseMigrationAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final DatabaseMigrationAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeConnectionsResponse> testConnectionSucceedsWaiter;

    private final AsyncWaiter<DescribeEndpointsResponse> endpointDeletedWaiter;

    private final AsyncWaiter<DescribeReplicationInstancesResponse> replicationInstanceAvailableWaiter;

    private final AsyncWaiter<DescribeReplicationInstancesResponse> replicationInstanceDeletedWaiter;

    private final AsyncWaiter<DescribeReplicationTasksResponse> replicationTaskReadyWaiter;

    private final AsyncWaiter<DescribeReplicationTasksResponse> replicationTaskStoppedWaiter;

    private final AsyncWaiter<DescribeReplicationTasksResponse> replicationTaskRunningWaiter;

    private final AsyncWaiter<DescribeReplicationTasksResponse> replicationTaskDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultDatabaseMigrationAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = DatabaseMigrationAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.testConnectionSucceedsWaiter = AsyncWaiter.builder(DescribeConnectionsResponse.class)
                .acceptors(testConnectionSucceedsWaiterAcceptors())
                .overrideConfiguration(testConnectionSucceedsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.endpointDeletedWaiter = AsyncWaiter.builder(DescribeEndpointsResponse.class)
                .acceptors(endpointDeletedWaiterAcceptors())
                .overrideConfiguration(endpointDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationInstanceAvailableWaiter = AsyncWaiter.builder(DescribeReplicationInstancesResponse.class)
                .acceptors(replicationInstanceAvailableWaiterAcceptors())
                .overrideConfiguration(replicationInstanceAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationInstanceDeletedWaiter = AsyncWaiter.builder(DescribeReplicationInstancesResponse.class)
                .acceptors(replicationInstanceDeletedWaiterAcceptors())
                .overrideConfiguration(replicationInstanceDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationTaskReadyWaiter = AsyncWaiter.builder(DescribeReplicationTasksResponse.class)
                .acceptors(replicationTaskReadyWaiterAcceptors())
                .overrideConfiguration(replicationTaskReadyWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationTaskStoppedWaiter = AsyncWaiter.builder(DescribeReplicationTasksResponse.class)
                .acceptors(replicationTaskStoppedWaiterAcceptors())
                .overrideConfiguration(replicationTaskStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationTaskRunningWaiter = AsyncWaiter.builder(DescribeReplicationTasksResponse.class)
                .acceptors(replicationTaskRunningWaiterAcceptors())
                .overrideConfiguration(replicationTaskRunningWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationTaskDeletedWaiter = AsyncWaiter.builder(DescribeReplicationTasksResponse.class)
                .acceptors(replicationTaskDeletedWaiterAcceptors())
                .overrideConfiguration(replicationTaskDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEndpointsResponse>> waitUntilEndpointDeleted(
            DescribeEndpointsRequest describeEndpointsRequest) {
        return endpointDeletedWaiter.runAsync(() -> client.describeEndpoints(applyWaitersUserAgent(describeEndpointsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeEndpointsResponse>> waitUntilEndpointDeleted(
            DescribeEndpointsRequest describeEndpointsRequest, WaiterOverrideConfiguration overrideConfig) {
        return endpointDeletedWaiter.runAsync(() -> client.describeEndpoints(applyWaitersUserAgent(describeEndpointsRequest)),
                endpointDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceAvailable(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest) {
        return replicationInstanceAvailableWaiter.runAsync(() -> client
                .describeReplicationInstances(applyWaitersUserAgent(describeReplicationInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceAvailable(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationInstanceAvailableWaiter.runAsync(
                () -> client.describeReplicationInstances(applyWaitersUserAgent(describeReplicationInstancesRequest)),
                replicationInstanceAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceDeleted(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest) {
        return replicationInstanceDeletedWaiter.runAsync(() -> client
                .describeReplicationInstances(applyWaitersUserAgent(describeReplicationInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationInstancesResponse>> waitUntilReplicationInstanceDeleted(
            DescribeReplicationInstancesRequest describeReplicationInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationInstanceDeletedWaiter.runAsync(
                () -> client.describeReplicationInstances(applyWaitersUserAgent(describeReplicationInstancesRequest)),
                replicationInstanceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskDeleted(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        return replicationTaskDeletedWaiter.runAsync(() -> client
                .describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskDeleted(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationTaskDeletedWaiter.runAsync(
                () -> client.describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)),
                replicationTaskDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskReady(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        return replicationTaskReadyWaiter.runAsync(() -> client
                .describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskReady(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationTaskReadyWaiter.runAsync(
                () -> client.describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)),
                replicationTaskReadyWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskRunning(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        return replicationTaskRunningWaiter.runAsync(() -> client
                .describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskRunning(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationTaskRunningWaiter.runAsync(
                () -> client.describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)),
                replicationTaskRunningWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskStopped(
            DescribeReplicationTasksRequest describeReplicationTasksRequest) {
        return replicationTaskStoppedWaiter.runAsync(() -> client
                .describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationTasksResponse>> waitUntilReplicationTaskStopped(
            DescribeReplicationTasksRequest describeReplicationTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationTaskStoppedWaiter.runAsync(
                () -> client.describeReplicationTasks(applyWaitersUserAgent(describeReplicationTasksRequest)),
                replicationTaskStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConnectionsResponse>> waitUntilTestConnectionSucceeds(
            DescribeConnectionsRequest describeConnectionsRequest) {
        return testConnectionSucceedsWaiter.runAsync(() -> client
                .describeConnections(applyWaitersUserAgent(describeConnectionsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConnectionsResponse>> waitUntilTestConnectionSucceeds(
            DescribeConnectionsRequest describeConnectionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return testConnectionSucceedsWaiter.runAsync(
                () -> client.describeConnections(applyWaitersUserAgent(describeConnectionsRequest)),
                testConnectionSucceedsWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeConnectionsResponse>> testConnectionSucceedsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeConnectionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Connections").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "successful"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Connections").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Connections[].Status=failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeEndpointsResponse>> endpointDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeEndpointsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ResourceNotFoundFault")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Endpoints").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "active"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Endpoints[].Status=active) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Endpoints").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Endpoints[].Status=creating) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationInstancesResponse>> replicationInstanceAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor
                .successOnResponseAcceptor(response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationInstances").flatten().field("ReplicationInstanceStatus")
                            .values();
                    return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
                }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationInstances").flatten().field("ReplicationInstanceStatus")
                            .values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationInstances[].ReplicationInstanceStatus=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationInstances").flatten().field("ReplicationInstanceStatus")
                            .values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationInstances[].ReplicationInstanceStatus=incompatible-credentials) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationInstances").flatten().field("ReplicationInstanceStatus")
                            .values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-network"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationInstances[].ReplicationInstanceStatus=incompatible-network) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationInstances").flatten().field("ReplicationInstanceStatus")
                            .values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "inaccessible-encryption-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationInstances[].ReplicationInstanceStatus=inaccessible-encryption-credentials) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationInstancesResponse>> replicationInstanceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationInstances").flatten().field("ReplicationInstanceStatus")
                            .values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "available"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationInstances[].ReplicationInstanceStatus=available) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ResourceNotFoundFault")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> replicationTaskReadyWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "ready"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "starting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=starting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "running"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=running) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=stopping) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=stopped) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "testing"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=testing) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=deleting) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> replicationTaskStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "stopped"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ready"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=ready) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "starting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=starting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "testing"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=testing) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=deleting) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> replicationTaskRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "running"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ready"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=ready) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=stopping) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=stopped) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "testing"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=testing) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=deleting) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> replicationTaskDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ready"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=ready) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopped"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=stopped) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "running"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=running) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationTasks").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationTasks[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ResourceNotFoundFault")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration testConnectionSucceedsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration endpointDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationInstanceAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(60)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationInstanceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationTaskReadyWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationTaskStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationTaskRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationTaskDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static DatabaseMigrationAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends DatabaseMigrationRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements DatabaseMigrationAsyncWaiter.Builder {
        private DatabaseMigrationAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public DatabaseMigrationAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public DatabaseMigrationAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public DatabaseMigrationAsyncWaiter.Builder client(DatabaseMigrationAsyncClient client) {
            this.client = client;
            return this;
        }

        public DatabaseMigrationAsyncWaiter build() {
            return new DefaultDatabaseMigrationAsyncWaiter(this);
        }
    }
}
