/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a structure for defining parameter conditions. Supported conditions are described here: <a
 * href="https://docs-aws.amazon.com/databrew/latest/dg/datasets.multiple-files.html#conditions.for.dynamic.datasets"
 * >Supported conditions for dynamic datasets</a> in the <i>Glue DataBrew Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FilterExpression implements SdkPojo, Serializable,
        ToCopyableBuilder<FilterExpression.Builder, FilterExpression> {
    private static final SdkField<String> EXPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Expression").getter(getter(FilterExpression::expression)).setter(setter(Builder::expression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Expression").build()).build();

    private static final SdkField<Map<String, String>> VALUES_MAP_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ValuesMap")
            .getter(getter(FilterExpression::valuesMap))
            .setter(setter(Builder::valuesMap))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValuesMap").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPRESSION_FIELD,
            VALUES_MAP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String expression;

    private final Map<String, String> valuesMap;

    private FilterExpression(BuilderImpl builder) {
        this.expression = builder.expression;
        this.valuesMap = builder.valuesMap;
    }

    /**
     * <p>
     * The expression which includes condition names followed by substitution variables, possibly grouped and combined
     * with other conditions. For example,
     * "(starts_with :prefix1 or starts_with :prefix2) and (ends_with :suffix1 or ends_with :suffix2)". Substitution
     * variables should start with ':' symbol.
     * </p>
     * 
     * @return The expression which includes condition names followed by substitution variables, possibly grouped and
     *         combined with other conditions. For example,
     *         "(starts_with :prefix1 or starts_with :prefix2) and (ends_with :suffix1 or ends_with :suffix2)".
     *         Substitution variables should start with ':' symbol.
     */
    public final String expression() {
        return expression;
    }

    /**
     * Returns true if the ValuesMap property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValuesMap() {
        return valuesMap != null && !(valuesMap instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The map of substitution variable names to their values used in this filter expression.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValuesMap()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The map of substitution variable names to their values used in this filter expression.
     */
    public final Map<String, String> valuesMap() {
        return valuesMap;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(expression());
        hashCode = 31 * hashCode + Objects.hashCode(hasValuesMap() ? valuesMap() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FilterExpression)) {
            return false;
        }
        FilterExpression other = (FilterExpression) obj;
        return Objects.equals(expression(), other.expression()) && hasValuesMap() == other.hasValuesMap()
                && Objects.equals(valuesMap(), other.valuesMap());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FilterExpression").add("Expression", expression())
                .add("ValuesMap", hasValuesMap() ? valuesMap() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Expression":
            return Optional.ofNullable(clazz.cast(expression()));
        case "ValuesMap":
            return Optional.ofNullable(clazz.cast(valuesMap()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FilterExpression, T> g) {
        return obj -> g.apply((FilterExpression) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FilterExpression> {
        /**
         * <p>
         * The expression which includes condition names followed by substitution variables, possibly grouped and
         * combined with other conditions. For example,
         * "(starts_with :prefix1 or starts_with :prefix2) and (ends_with :suffix1 or ends_with :suffix2)". Substitution
         * variables should start with ':' symbol.
         * </p>
         * 
         * @param expression
         *        The expression which includes condition names followed by substitution variables, possibly grouped and
         *        combined with other conditions. For example,
         *        "(starts_with :prefix1 or starts_with :prefix2) and (ends_with :suffix1 or ends_with :suffix2)".
         *        Substitution variables should start with ':' symbol.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expression(String expression);

        /**
         * <p>
         * The map of substitution variable names to their values used in this filter expression.
         * </p>
         * 
         * @param valuesMap
         *        The map of substitution variable names to their values used in this filter expression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder valuesMap(Map<String, String> valuesMap);
    }

    static final class BuilderImpl implements Builder {
        private String expression;

        private Map<String, String> valuesMap = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FilterExpression model) {
            expression(model.expression);
            valuesMap(model.valuesMap);
        }

        public final String getExpression() {
            return expression;
        }

        @Override
        public final Builder expression(String expression) {
            this.expression = expression;
            return this;
        }

        public final void setExpression(String expression) {
            this.expression = expression;
        }

        public final Map<String, String> getValuesMap() {
            if (valuesMap instanceof SdkAutoConstructMap) {
                return null;
            }
            return valuesMap;
        }

        @Override
        public final Builder valuesMap(Map<String, String> valuesMap) {
            this.valuesMap = ValuesMapCopier.copy(valuesMap);
            return this;
        }

        public final void setValuesMap(Map<String, String> valuesMap) {
            this.valuesMap = ValuesMapCopier.copy(valuesMap);
        }

        @Override
        public FilterExpression build() {
            return new FilterExpression(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
