/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents options that specify how and where DataBrew writes the output generated by recipe jobs or profile jobs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Output implements SdkPojo, Serializable, ToCopyableBuilder<Output.Builder, Output> {
    private static final SdkField<String> COMPRESSION_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompressionFormat").getter(getter(Output::compressionFormatAsString))
            .setter(setter(Builder::compressionFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompressionFormat").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Format")
            .getter(getter(Output::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Format").build()).build();

    private static final SdkField<List<String>> PARTITION_COLUMNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("PartitionColumns")
            .getter(getter(Output::partitionColumns))
            .setter(setter(Builder::partitionColumns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartitionColumns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<S3Location> LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .memberName("Location").getter(getter(Output::location)).setter(setter(Builder::location))
            .constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final SdkField<Boolean> OVERWRITE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Overwrite").getter(getter(Output::overwrite)).setter(setter(Builder::overwrite))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Overwrite").build()).build();

    private static final SdkField<OutputFormatOptions> FORMAT_OPTIONS_FIELD = SdkField
            .<OutputFormatOptions> builder(MarshallingType.SDK_POJO).memberName("FormatOptions")
            .getter(getter(Output::formatOptions)).setter(setter(Builder::formatOptions))
            .constructor(OutputFormatOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FormatOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPRESSION_FORMAT_FIELD,
            FORMAT_FIELD, PARTITION_COLUMNS_FIELD, LOCATION_FIELD, OVERWRITE_FIELD, FORMAT_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String compressionFormat;

    private final String format;

    private final List<String> partitionColumns;

    private final S3Location location;

    private final Boolean overwrite;

    private final OutputFormatOptions formatOptions;

    private Output(BuilderImpl builder) {
        this.compressionFormat = builder.compressionFormat;
        this.format = builder.format;
        this.partitionColumns = builder.partitionColumns;
        this.location = builder.location;
        this.overwrite = builder.overwrite;
        this.formatOptions = builder.formatOptions;
    }

    /**
     * <p>
     * The compression algorithm used to compress the output text of the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionFormat}
     * will return {@link CompressionFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #compressionFormatAsString}.
     * </p>
     * 
     * @return The compression algorithm used to compress the output text of the job.
     * @see CompressionFormat
     */
    public final CompressionFormat compressionFormat() {
        return CompressionFormat.fromValue(compressionFormat);
    }

    /**
     * <p>
     * The compression algorithm used to compress the output text of the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compressionFormat}
     * will return {@link CompressionFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #compressionFormatAsString}.
     * </p>
     * 
     * @return The compression algorithm used to compress the output text of the job.
     * @see CompressionFormat
     */
    public final String compressionFormatAsString() {
        return compressionFormat;
    }

    /**
     * <p>
     * The data format of the output of the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link OutputFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The data format of the output of the job.
     * @see OutputFormat
     */
    public final OutputFormat format() {
        return OutputFormat.fromValue(format);
    }

    /**
     * <p>
     * The data format of the output of the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link OutputFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The data format of the output of the job.
     * @see OutputFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * Returns true if the PartitionColumns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasPartitionColumns() {
        return partitionColumns != null && !(partitionColumns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of one or more partition columns for the output of the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPartitionColumns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The names of one or more partition columns for the output of the job.
     */
    public final List<String> partitionColumns() {
        return partitionColumns;
    }

    /**
     * <p>
     * The location in Amazon S3 where the job writes its output.
     * </p>
     * 
     * @return The location in Amazon S3 where the job writes its output.
     */
    public final S3Location location() {
        return location;
    }

    /**
     * <p>
     * A value that, if true, means that any data in the location specified for output is overwritten with new output.
     * </p>
     * 
     * @return A value that, if true, means that any data in the location specified for output is overwritten with new
     *         output.
     */
    public final Boolean overwrite() {
        return overwrite;
    }

    /**
     * <p>
     * Represents options that define how DataBrew formats job output files.
     * </p>
     * 
     * @return Represents options that define how DataBrew formats job output files.
     */
    public final OutputFormatOptions formatOptions() {
        return formatOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(compressionFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasPartitionColumns() ? partitionColumns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(overwrite());
        hashCode = 31 * hashCode + Objects.hashCode(formatOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Output)) {
            return false;
        }
        Output other = (Output) obj;
        return Objects.equals(compressionFormatAsString(), other.compressionFormatAsString())
                && Objects.equals(formatAsString(), other.formatAsString())
                && hasPartitionColumns() == other.hasPartitionColumns()
                && Objects.equals(partitionColumns(), other.partitionColumns()) && Objects.equals(location(), other.location())
                && Objects.equals(overwrite(), other.overwrite()) && Objects.equals(formatOptions(), other.formatOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Output").add("CompressionFormat", compressionFormatAsString()).add("Format", formatAsString())
                .add("PartitionColumns", hasPartitionColumns() ? partitionColumns() : null).add("Location", location())
                .add("Overwrite", overwrite()).add("FormatOptions", formatOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CompressionFormat":
            return Optional.ofNullable(clazz.cast(compressionFormatAsString()));
        case "Format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "PartitionColumns":
            return Optional.ofNullable(clazz.cast(partitionColumns()));
        case "Location":
            return Optional.ofNullable(clazz.cast(location()));
        case "Overwrite":
            return Optional.ofNullable(clazz.cast(overwrite()));
        case "FormatOptions":
            return Optional.ofNullable(clazz.cast(formatOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Output, T> g) {
        return obj -> g.apply((Output) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Output> {
        /**
         * <p>
         * The compression algorithm used to compress the output text of the job.
         * </p>
         * 
         * @param compressionFormat
         *        The compression algorithm used to compress the output text of the job.
         * @see CompressionFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionFormat
         */
        Builder compressionFormat(String compressionFormat);

        /**
         * <p>
         * The compression algorithm used to compress the output text of the job.
         * </p>
         * 
         * @param compressionFormat
         *        The compression algorithm used to compress the output text of the job.
         * @see CompressionFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CompressionFormat
         */
        Builder compressionFormat(CompressionFormat compressionFormat);

        /**
         * <p>
         * The data format of the output of the job.
         * </p>
         * 
         * @param format
         *        The data format of the output of the job.
         * @see OutputFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The data format of the output of the job.
         * </p>
         * 
         * @param format
         *        The data format of the output of the job.
         * @see OutputFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OutputFormat
         */
        Builder format(OutputFormat format);

        /**
         * <p>
         * The names of one or more partition columns for the output of the job.
         * </p>
         * 
         * @param partitionColumns
         *        The names of one or more partition columns for the output of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionColumns(Collection<String> partitionColumns);

        /**
         * <p>
         * The names of one or more partition columns for the output of the job.
         * </p>
         * 
         * @param partitionColumns
         *        The names of one or more partition columns for the output of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder partitionColumns(String... partitionColumns);

        /**
         * <p>
         * The location in Amazon S3 where the job writes its output.
         * </p>
         * 
         * @param location
         *        The location in Amazon S3 where the job writes its output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(S3Location location);

        /**
         * <p>
         * The location in Amazon S3 where the job writes its output.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Location.Builder} avoiding the need to create
         * one manually via {@link S3Location#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #location(S3Location)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(S3Location)
         */
        default Builder location(Consumer<S3Location.Builder> location) {
            return location(S3Location.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * A value that, if true, means that any data in the location specified for output is overwritten with new
         * output.
         * </p>
         * 
         * @param overwrite
         *        A value that, if true, means that any data in the location specified for output is overwritten with
         *        new output.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overwrite(Boolean overwrite);

        /**
         * <p>
         * Represents options that define how DataBrew formats job output files.
         * </p>
         * 
         * @param formatOptions
         *        Represents options that define how DataBrew formats job output files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formatOptions(OutputFormatOptions formatOptions);

        /**
         * <p>
         * Represents options that define how DataBrew formats job output files.
         * </p>
         * This is a convenience that creates an instance of the {@link OutputFormatOptions.Builder} avoiding the need
         * to create one manually via {@link OutputFormatOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link OutputFormatOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #formatOptions(OutputFormatOptions)}.
         * 
         * @param formatOptions
         *        a consumer that will call methods on {@link OutputFormatOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #formatOptions(OutputFormatOptions)
         */
        default Builder formatOptions(Consumer<OutputFormatOptions.Builder> formatOptions) {
            return formatOptions(OutputFormatOptions.builder().applyMutation(formatOptions).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String compressionFormat;

        private String format;

        private List<String> partitionColumns = DefaultSdkAutoConstructList.getInstance();

        private S3Location location;

        private Boolean overwrite;

        private OutputFormatOptions formatOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(Output model) {
            compressionFormat(model.compressionFormat);
            format(model.format);
            partitionColumns(model.partitionColumns);
            location(model.location);
            overwrite(model.overwrite);
            formatOptions(model.formatOptions);
        }

        public final String getCompressionFormat() {
            return compressionFormat;
        }

        @Override
        public final Builder compressionFormat(String compressionFormat) {
            this.compressionFormat = compressionFormat;
            return this;
        }

        @Override
        public final Builder compressionFormat(CompressionFormat compressionFormat) {
            this.compressionFormat(compressionFormat == null ? null : compressionFormat.toString());
            return this;
        }

        public final void setCompressionFormat(String compressionFormat) {
            this.compressionFormat = compressionFormat;
        }

        public final String getFormat() {
            return format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(OutputFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        public final Collection<String> getPartitionColumns() {
            if (partitionColumns instanceof SdkAutoConstructList) {
                return null;
            }
            return partitionColumns;
        }

        @Override
        public final Builder partitionColumns(Collection<String> partitionColumns) {
            this.partitionColumns = ColumnNameListCopier.copy(partitionColumns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder partitionColumns(String... partitionColumns) {
            partitionColumns(Arrays.asList(partitionColumns));
            return this;
        }

        public final void setPartitionColumns(Collection<String> partitionColumns) {
            this.partitionColumns = ColumnNameListCopier.copy(partitionColumns);
        }

        public final S3Location.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(S3Location location) {
            this.location = location;
            return this;
        }

        public final void setLocation(S3Location.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final Boolean getOverwrite() {
            return overwrite;
        }

        @Override
        public final Builder overwrite(Boolean overwrite) {
            this.overwrite = overwrite;
            return this;
        }

        public final void setOverwrite(Boolean overwrite) {
            this.overwrite = overwrite;
        }

        public final OutputFormatOptions.Builder getFormatOptions() {
            return formatOptions != null ? formatOptions.toBuilder() : null;
        }

        @Override
        public final Builder formatOptions(OutputFormatOptions formatOptions) {
            this.formatOptions = formatOptions;
            return this;
        }

        public final void setFormatOptions(OutputFormatOptions.BuilderImpl formatOptions) {
            this.formatOptions = formatOptions != null ? formatOptions.build() : null;
        }

        @Override
        public Output build() {
            return new Output(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
