/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a dataset parameter that defines type and conditions for a parameter in the Amazon S3 path of the dataset.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DatasetParameter implements SdkPojo, Serializable,
        ToCopyableBuilder<DatasetParameter.Builder, DatasetParameter> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DatasetParameter::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DatasetParameter::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<DatetimeOptions> DATETIME_OPTIONS_FIELD = SdkField
            .<DatetimeOptions> builder(MarshallingType.SDK_POJO).memberName("DatetimeOptions")
            .getter(getter(DatasetParameter::datetimeOptions)).setter(setter(Builder::datetimeOptions))
            .constructor(DatetimeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatetimeOptions").build()).build();

    private static final SdkField<Boolean> CREATE_COLUMN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CreateColumn").getter(getter(DatasetParameter::createColumn)).setter(setter(Builder::createColumn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateColumn").build()).build();

    private static final SdkField<FilterExpression> FILTER_FIELD = SdkField.<FilterExpression> builder(MarshallingType.SDK_POJO)
            .memberName("Filter").getter(getter(DatasetParameter::filter)).setter(setter(Builder::filter))
            .constructor(FilterExpression::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            DATETIME_OPTIONS_FIELD, CREATE_COLUMN_FIELD, FILTER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String type;

    private final DatetimeOptions datetimeOptions;

    private final Boolean createColumn;

    private final FilterExpression filter;

    private DatasetParameter(BuilderImpl builder) {
        this.name = builder.name;
        this.type = builder.type;
        this.datetimeOptions = builder.datetimeOptions;
        this.createColumn = builder.createColumn;
        this.filter = builder.filter;
    }

    /**
     * <p>
     * The name of the parameter that is used in the dataset's Amazon S3 path.
     * </p>
     * 
     * @return The name of the parameter that is used in the dataset's Amazon S3 path.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
     * @see ParameterType
     */
    public final ParameterType type() {
        return ParameterType.fromValue(type);
    }

    /**
     * <p>
     * The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ParameterType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
     * @see ParameterType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Additional parameter options such as a format and a timezone. Required for datetime parameters.
     * </p>
     * 
     * @return Additional parameter options such as a format and a timezone. Required for datetime parameters.
     */
    public final DatetimeOptions datetimeOptions() {
        return datetimeOptions;
    }

    /**
     * <p>
     * Optional boolean value that defines whether the captured value of this parameter should be used to create a new
     * column in a dataset.
     * </p>
     * 
     * @return Optional boolean value that defines whether the captured value of this parameter should be used to create
     *         a new column in a dataset.
     */
    public final Boolean createColumn() {
        return createColumn;
    }

    /**
     * <p>
     * The optional filter expression structure to apply additional matching criteria to the parameter.
     * </p>
     * 
     * @return The optional filter expression structure to apply additional matching criteria to the parameter.
     */
    public final FilterExpression filter() {
        return filter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(datetimeOptions());
        hashCode = 31 * hashCode + Objects.hashCode(createColumn());
        hashCode = 31 * hashCode + Objects.hashCode(filter());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DatasetParameter)) {
            return false;
        }
        DatasetParameter other = (DatasetParameter) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(datetimeOptions(), other.datetimeOptions())
                && Objects.equals(createColumn(), other.createColumn()) && Objects.equals(filter(), other.filter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DatasetParameter").add("Name", name()).add("Type", typeAsString())
                .add("DatetimeOptions", datetimeOptions()).add("CreateColumn", createColumn()).add("Filter", filter()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "DatetimeOptions":
            return Optional.ofNullable(clazz.cast(datetimeOptions()));
        case "CreateColumn":
            return Optional.ofNullable(clazz.cast(createColumn()));
        case "Filter":
            return Optional.ofNullable(clazz.cast(filter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DatasetParameter, T> g) {
        return obj -> g.apply((DatasetParameter) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DatasetParameter> {
        /**
         * <p>
         * The name of the parameter that is used in the dataset's Amazon S3 path.
         * </p>
         * 
         * @param name
         *        The name of the parameter that is used in the dataset's Amazon S3 path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
         * </p>
         * 
         * @param type
         *        The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
         * </p>
         * 
         * @param type
         *        The type of the dataset parameter, can be one of a 'String', 'Number' or 'Datetime'.
         * @see ParameterType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParameterType
         */
        Builder type(ParameterType type);

        /**
         * <p>
         * Additional parameter options such as a format and a timezone. Required for datetime parameters.
         * </p>
         * 
         * @param datetimeOptions
         *        Additional parameter options such as a format and a timezone. Required for datetime parameters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datetimeOptions(DatetimeOptions datetimeOptions);

        /**
         * <p>
         * Additional parameter options such as a format and a timezone. Required for datetime parameters.
         * </p>
         * This is a convenience method that creates an instance of the {@link DatetimeOptions.Builder} avoiding the
         * need to create one manually via {@link DatetimeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DatetimeOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #datetimeOptions(DatetimeOptions)}.
         * 
         * @param datetimeOptions
         *        a consumer that will call methods on {@link DatetimeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #datetimeOptions(DatetimeOptions)
         */
        default Builder datetimeOptions(Consumer<DatetimeOptions.Builder> datetimeOptions) {
            return datetimeOptions(DatetimeOptions.builder().applyMutation(datetimeOptions).build());
        }

        /**
         * <p>
         * Optional boolean value that defines whether the captured value of this parameter should be used to create a
         * new column in a dataset.
         * </p>
         * 
         * @param createColumn
         *        Optional boolean value that defines whether the captured value of this parameter should be used to
         *        create a new column in a dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createColumn(Boolean createColumn);

        /**
         * <p>
         * The optional filter expression structure to apply additional matching criteria to the parameter.
         * </p>
         * 
         * @param filter
         *        The optional filter expression structure to apply additional matching criteria to the parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filter(FilterExpression filter);

        /**
         * <p>
         * The optional filter expression structure to apply additional matching criteria to the parameter.
         * </p>
         * This is a convenience method that creates an instance of the {@link FilterExpression.Builder} avoiding the
         * need to create one manually via {@link FilterExpression#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FilterExpression.Builder#build()} is called immediately and its
         * result is passed to {@link #filter(FilterExpression)}.
         * 
         * @param filter
         *        a consumer that will call methods on {@link FilterExpression.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filter(FilterExpression)
         */
        default Builder filter(Consumer<FilterExpression.Builder> filter) {
            return filter(FilterExpression.builder().applyMutation(filter).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String type;

        private DatetimeOptions datetimeOptions;

        private Boolean createColumn;

        private FilterExpression filter;

        private BuilderImpl() {
        }

        private BuilderImpl(DatasetParameter model) {
            name(model.name);
            type(model.type);
            datetimeOptions(model.datetimeOptions);
            createColumn(model.createColumn);
            filter(model.filter);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ParameterType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final DatetimeOptions.Builder getDatetimeOptions() {
            return datetimeOptions != null ? datetimeOptions.toBuilder() : null;
        }

        public final void setDatetimeOptions(DatetimeOptions.BuilderImpl datetimeOptions) {
            this.datetimeOptions = datetimeOptions != null ? datetimeOptions.build() : null;
        }

        @Override
        public final Builder datetimeOptions(DatetimeOptions datetimeOptions) {
            this.datetimeOptions = datetimeOptions;
            return this;
        }

        public final Boolean getCreateColumn() {
            return createColumn;
        }

        public final void setCreateColumn(Boolean createColumn) {
            this.createColumn = createColumn;
        }

        @Override
        public final Builder createColumn(Boolean createColumn) {
            this.createColumn = createColumn;
            return this;
        }

        public final FilterExpression.Builder getFilter() {
            return filter != null ? filter.toBuilder() : null;
        }

        public final void setFilter(FilterExpression.BuilderImpl filter) {
            this.filter = filter != null ? filter.build() : null;
        }

        @Override
        public final Builder filter(FilterExpression filter) {
            this.filter = filter;
            return this;
        }

        @Override
        public DatasetParameter build() {
            return new DatasetParameter(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
