/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents all of the attributes of a DataBrew job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Job implements SdkPojo, Serializable, ToCopyableBuilder<Job.Builder, Job> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(Job::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatedBy").getter(getter(Job::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<Instant> CREATE_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateDate").getter(getter(Job::createDate)).setter(setter(Builder::createDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateDate").build()).build();

    private static final SdkField<String> DATASET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatasetName").getter(getter(Job::datasetName)).setter(setter(Builder::datasetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatasetName").build()).build();

    private static final SdkField<String> ENCRYPTION_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionKeyArn").getter(getter(Job::encryptionKeyArn)).setter(setter(Builder::encryptionKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionKeyArn").build()).build();

    private static final SdkField<String> ENCRYPTION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionMode").getter(getter(Job::encryptionModeAsString)).setter(setter(Builder::encryptionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionMode").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Job::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(Job::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> LAST_MODIFIED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastModifiedBy").getter(getter(Job::lastModifiedBy)).setter(setter(Builder::lastModifiedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedBy").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedDate").getter(getter(Job::lastModifiedDate)).setter(setter(Builder::lastModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedDate").build()).build();

    private static final SdkField<String> LOG_SUBSCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogSubscription").getter(getter(Job::logSubscriptionAsString)).setter(setter(Builder::logSubscription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogSubscription").build()).build();

    private static final SdkField<Integer> MAX_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxCapacity").getter(getter(Job::maxCapacity)).setter(setter(Builder::maxCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxCapacity").build()).build();

    private static final SdkField<Integer> MAX_RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxRetries").getter(getter(Job::maxRetries)).setter(setter(Builder::maxRetries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRetries").build()).build();

    private static final SdkField<List<Output>> OUTPUTS_FIELD = SdkField
            .<List<Output>> builder(MarshallingType.LIST)
            .memberName("Outputs")
            .getter(getter(Job::outputs))
            .setter(setter(Builder::outputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Outputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Output> builder(MarshallingType.SDK_POJO)
                                            .constructor(Output::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DataCatalogOutput>> DATA_CATALOG_OUTPUTS_FIELD = SdkField
            .<List<DataCatalogOutput>> builder(MarshallingType.LIST)
            .memberName("DataCatalogOutputs")
            .getter(getter(Job::dataCatalogOutputs))
            .setter(setter(Builder::dataCatalogOutputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCatalogOutputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataCatalogOutput> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataCatalogOutput::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<DatabaseOutput>> DATABASE_OUTPUTS_FIELD = SdkField
            .<List<DatabaseOutput>> builder(MarshallingType.LIST)
            .memberName("DatabaseOutputs")
            .getter(getter(Job::databaseOutputs))
            .setter(setter(Builder::databaseOutputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseOutputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DatabaseOutput> builder(MarshallingType.SDK_POJO)
                                            .constructor(DatabaseOutput::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROJECT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProjectName").getter(getter(Job::projectName)).setter(setter(Builder::projectName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProjectName").build()).build();

    private static final SdkField<RecipeReference> RECIPE_REFERENCE_FIELD = SdkField
            .<RecipeReference> builder(MarshallingType.SDK_POJO).memberName("RecipeReference")
            .getter(getter(Job::recipeReference)).setter(setter(Builder::recipeReference)).constructor(RecipeReference::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecipeReference").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(Job::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(Job::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<Integer> TIMEOUT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Timeout").getter(getter(Job::timeout)).setter(setter(Builder::timeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timeout").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(Job::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<JobSample> JOB_SAMPLE_FIELD = SdkField.<JobSample> builder(MarshallingType.SDK_POJO)
            .memberName("JobSample").getter(getter(Job::jobSample)).setter(setter(Builder::jobSample))
            .constructor(JobSample::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobSample").build()).build();

    private static final SdkField<List<ValidationConfiguration>> VALIDATION_CONFIGURATIONS_FIELD = SdkField
            .<List<ValidationConfiguration>> builder(MarshallingType.LIST)
            .memberName("ValidationConfigurations")
            .getter(getter(Job::validationConfigurations))
            .setter(setter(Builder::validationConfigurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationConfigurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ValidationConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(ValidationConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            CREATED_BY_FIELD, CREATE_DATE_FIELD, DATASET_NAME_FIELD, ENCRYPTION_KEY_ARN_FIELD, ENCRYPTION_MODE_FIELD, NAME_FIELD,
            TYPE_FIELD, LAST_MODIFIED_BY_FIELD, LAST_MODIFIED_DATE_FIELD, LOG_SUBSCRIPTION_FIELD, MAX_CAPACITY_FIELD,
            MAX_RETRIES_FIELD, OUTPUTS_FIELD, DATA_CATALOG_OUTPUTS_FIELD, DATABASE_OUTPUTS_FIELD, PROJECT_NAME_FIELD,
            RECIPE_REFERENCE_FIELD, RESOURCE_ARN_FIELD, ROLE_ARN_FIELD, TIMEOUT_FIELD, TAGS_FIELD, JOB_SAMPLE_FIELD,
            VALIDATION_CONFIGURATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String createdBy;

    private final Instant createDate;

    private final String datasetName;

    private final String encryptionKeyArn;

    private final String encryptionMode;

    private final String name;

    private final String type;

    private final String lastModifiedBy;

    private final Instant lastModifiedDate;

    private final String logSubscription;

    private final Integer maxCapacity;

    private final Integer maxRetries;

    private final List<Output> outputs;

    private final List<DataCatalogOutput> dataCatalogOutputs;

    private final List<DatabaseOutput> databaseOutputs;

    private final String projectName;

    private final RecipeReference recipeReference;

    private final String resourceArn;

    private final String roleArn;

    private final Integer timeout;

    private final Map<String, String> tags;

    private final JobSample jobSample;

    private final List<ValidationConfiguration> validationConfigurations;

    private Job(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.createdBy = builder.createdBy;
        this.createDate = builder.createDate;
        this.datasetName = builder.datasetName;
        this.encryptionKeyArn = builder.encryptionKeyArn;
        this.encryptionMode = builder.encryptionMode;
        this.name = builder.name;
        this.type = builder.type;
        this.lastModifiedBy = builder.lastModifiedBy;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.logSubscription = builder.logSubscription;
        this.maxCapacity = builder.maxCapacity;
        this.maxRetries = builder.maxRetries;
        this.outputs = builder.outputs;
        this.dataCatalogOutputs = builder.dataCatalogOutputs;
        this.databaseOutputs = builder.databaseOutputs;
        this.projectName = builder.projectName;
        this.recipeReference = builder.recipeReference;
        this.resourceArn = builder.resourceArn;
        this.roleArn = builder.roleArn;
        this.timeout = builder.timeout;
        this.tags = builder.tags;
        this.jobSample = builder.jobSample;
        this.validationConfigurations = builder.validationConfigurations;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that owns the job.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that owns the job.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the user who created the job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the user who created the job.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The date and time that the job was created.
     * </p>
     * 
     * @return The date and time that the job was created.
     */
    public final Instant createDate() {
        return createDate;
    }

    /**
     * <p>
     * A dataset that the job is to process.
     * </p>
     * 
     * @return A dataset that the job is to process.
     */
    public final String datasetName() {
        return datasetName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an encryption key that is used to protect the job output. For more information,
     * see <a href="https://docs.aws.amazon.com/databrew/latest/dg/encryption-security-configuration.html">Encrypting
     * data written by DataBrew jobs</a>
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an encryption key that is used to protect the job output. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/databrew/latest/dg/encryption-security-configuration.html">Encrypting
     *         data written by DataBrew jobs</a>
     */
    public final String encryptionKeyArn() {
        return encryptionKeyArn;
    }

    /**
     * <p>
     * The encryption mode for the job, which can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMode}
     * will return {@link EncryptionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionModeAsString}.
     * </p>
     * 
     * @return The encryption mode for the job, which can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
     *         </p>
     *         </li>
     * @see EncryptionMode
     */
    public final EncryptionMode encryptionMode() {
        return EncryptionMode.fromValue(encryptionMode);
    }

    /**
     * <p>
     * The encryption mode for the job, which can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionMode}
     * will return {@link EncryptionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionModeAsString}.
     * </p>
     * 
     * @return The encryption mode for the job, which can be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
     *         </p>
     *         </li>
     * @see EncryptionMode
     */
    public final String encryptionModeAsString() {
        return encryptionMode;
    }

    /**
     * <p>
     * The unique name of the job.
     * </p>
     * 
     * @return The unique name of the job.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The job type of the job, which must be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data distribution, and
     * more.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The job type of the job, which must be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data distribution,
     *         and more.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
     *         </p>
     *         </li>
     * @see JobType
     */
    public final JobType type() {
        return JobType.fromValue(type);
    }

    /**
     * <p>
     * The job type of the job, which must be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data distribution, and
     * more.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The job type of the job, which must be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data distribution,
     *         and more.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
     *         </p>
     *         </li>
     * @see JobType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the user who last modified the job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the user who last modified the job.
     */
    public final String lastModifiedBy() {
        return lastModifiedBy;
    }

    /**
     * <p>
     * The modification date and time of the job.
     * </p>
     * 
     * @return The modification date and time of the job.
     */
    public final Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * The current status of Amazon CloudWatch logging for the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logSubscription}
     * will return {@link LogSubscription#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #logSubscriptionAsString}.
     * </p>
     * 
     * @return The current status of Amazon CloudWatch logging for the job.
     * @see LogSubscription
     */
    public final LogSubscription logSubscription() {
        return LogSubscription.fromValue(logSubscription);
    }

    /**
     * <p>
     * The current status of Amazon CloudWatch logging for the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logSubscription}
     * will return {@link LogSubscription#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #logSubscriptionAsString}.
     * </p>
     * 
     * @return The current status of Amazon CloudWatch logging for the job.
     * @see LogSubscription
     */
    public final String logSubscriptionAsString() {
        return logSubscription;
    }

    /**
     * <p>
     * The maximum number of nodes that can be consumed when the job processes data.
     * </p>
     * 
     * @return The maximum number of nodes that can be consumed when the job processes data.
     */
    public final Integer maxCapacity() {
        return maxCapacity;
    }

    /**
     * <p>
     * The maximum number of times to retry the job after a job run fails.
     * </p>
     * 
     * @return The maximum number of times to retry the job after a job run fails.
     */
    public final Integer maxRetries() {
        return maxRetries;
    }

    /**
     * For responses, this returns true if the service returned a value for the Outputs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOutputs() {
        return outputs != null && !(outputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more artifacts that represent output from running the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOutputs} method.
     * </p>
     * 
     * @return One or more artifacts that represent output from running the job.
     */
    public final List<Output> outputs() {
        return outputs;
    }

    /**
     * For responses, this returns true if the service returned a value for the DataCatalogOutputs property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDataCatalogOutputs() {
        return dataCatalogOutputs != null && !(dataCatalogOutputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more artifacts that represent the Glue Data Catalog output from running the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDataCatalogOutputs} method.
     * </p>
     * 
     * @return One or more artifacts that represent the Glue Data Catalog output from running the job.
     */
    public final List<DataCatalogOutput> dataCatalogOutputs() {
        return dataCatalogOutputs;
    }

    /**
     * For responses, this returns true if the service returned a value for the DatabaseOutputs property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDatabaseOutputs() {
        return databaseOutputs != null && !(databaseOutputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe job
     * to write into.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDatabaseOutputs} method.
     * </p>
     * 
     * @return Represents a list of JDBC database output objects which defines the output destination for a DataBrew
     *         recipe job to write into.
     */
    public final List<DatabaseOutput> databaseOutputs() {
        return databaseOutputs;
    }

    /**
     * <p>
     * The name of the project that the job is associated with.
     * </p>
     * 
     * @return The name of the project that the job is associated with.
     */
    public final String projectName() {
        return projectName;
    }

    /**
     * <p>
     * A set of steps that the job runs.
     * </p>
     * 
     * @return A set of steps that the job runs.
     */
    public final RecipeReference recipeReference() {
        return recipeReference;
    }

    /**
     * <p>
     * The unique Amazon Resource Name (ARN) for the job.
     * </p>
     * 
     * @return The unique Amazon Resource Name (ARN) for the job.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the role to be assumed for this job.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the role to be assumed for this job.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The job's timeout in minutes. A job that attempts to run longer than this timeout period ends with a status of
     * <code>TIMEOUT</code>.
     * </p>
     * 
     * @return The job's timeout in minutes. A job that attempts to run longer than this timeout period ends with a
     *         status of <code>TIMEOUT</code>.
     */
    public final Integer timeout() {
        return timeout;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Metadata tags that have been applied to the job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Metadata tags that have been applied to the job.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * A sample configuration for profile jobs only, which determines the number of rows on which the profile job is
     * run. If a <code>JobSample</code> value isn't provided, the default value is used. The default value is
     * CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
     * </p>
     * 
     * @return A sample configuration for profile jobs only, which determines the number of rows on which the profile
     *         job is run. If a <code>JobSample</code> value isn't provided, the default value is used. The default
     *         value is CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
     */
    public final JobSample jobSample() {
        return jobSample;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidationConfigurations property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasValidationConfigurations() {
        return validationConfigurations != null && !(validationConfigurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of validation configurations that are applied to the profile job.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidationConfigurations} method.
     * </p>
     * 
     * @return List of validation configurations that are applied to the profile job.
     */
    public final List<ValidationConfiguration> validationConfigurations() {
        return validationConfigurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(createDate());
        hashCode = 31 * hashCode + Objects.hashCode(datasetName());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedBy());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(logSubscriptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(maxRetries());
        hashCode = 31 * hashCode + Objects.hashCode(hasOutputs() ? outputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDataCatalogOutputs() ? dataCatalogOutputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDatabaseOutputs() ? databaseOutputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(projectName());
        hashCode = 31 * hashCode + Objects.hashCode(recipeReference());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(timeout());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(jobSample());
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationConfigurations() ? validationConfigurations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Job)) {
            return false;
        }
        Job other = (Job) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(createDate(), other.createDate()) && Objects.equals(datasetName(), other.datasetName())
                && Objects.equals(encryptionKeyArn(), other.encryptionKeyArn())
                && Objects.equals(encryptionModeAsString(), other.encryptionModeAsString())
                && Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(lastModifiedBy(), other.lastModifiedBy())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(logSubscriptionAsString(), other.logSubscriptionAsString())
                && Objects.equals(maxCapacity(), other.maxCapacity()) && Objects.equals(maxRetries(), other.maxRetries())
                && hasOutputs() == other.hasOutputs() && Objects.equals(outputs(), other.outputs())
                && hasDataCatalogOutputs() == other.hasDataCatalogOutputs()
                && Objects.equals(dataCatalogOutputs(), other.dataCatalogOutputs())
                && hasDatabaseOutputs() == other.hasDatabaseOutputs()
                && Objects.equals(databaseOutputs(), other.databaseOutputs())
                && Objects.equals(projectName(), other.projectName())
                && Objects.equals(recipeReference(), other.recipeReference())
                && Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(timeout(), other.timeout()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(jobSample(), other.jobSample())
                && hasValidationConfigurations() == other.hasValidationConfigurations()
                && Objects.equals(validationConfigurations(), other.validationConfigurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Job").add("AccountId", accountId()).add("CreatedBy", createdBy())
                .add("CreateDate", createDate()).add("DatasetName", datasetName()).add("EncryptionKeyArn", encryptionKeyArn())
                .add("EncryptionMode", encryptionModeAsString()).add("Name", name()).add("Type", typeAsString())
                .add("LastModifiedBy", lastModifiedBy()).add("LastModifiedDate", lastModifiedDate())
                .add("LogSubscription", logSubscriptionAsString()).add("MaxCapacity", maxCapacity())
                .add("MaxRetries", maxRetries()).add("Outputs", hasOutputs() ? outputs() : null)
                .add("DataCatalogOutputs", hasDataCatalogOutputs() ? dataCatalogOutputs() : null)
                .add("DatabaseOutputs", hasDatabaseOutputs() ? databaseOutputs() : null).add("ProjectName", projectName())
                .add("RecipeReference", recipeReference()).add("ResourceArn", resourceArn()).add("RoleArn", roleArn())
                .add("Timeout", timeout()).add("Tags", hasTags() ? tags() : null).add("JobSample", jobSample())
                .add("ValidationConfigurations", hasValidationConfigurations() ? validationConfigurations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "CreateDate":
            return Optional.ofNullable(clazz.cast(createDate()));
        case "DatasetName":
            return Optional.ofNullable(clazz.cast(datasetName()));
        case "EncryptionKeyArn":
            return Optional.ofNullable(clazz.cast(encryptionKeyArn()));
        case "EncryptionMode":
            return Optional.ofNullable(clazz.cast(encryptionModeAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "LastModifiedBy":
            return Optional.ofNullable(clazz.cast(lastModifiedBy()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "LogSubscription":
            return Optional.ofNullable(clazz.cast(logSubscriptionAsString()));
        case "MaxCapacity":
            return Optional.ofNullable(clazz.cast(maxCapacity()));
        case "MaxRetries":
            return Optional.ofNullable(clazz.cast(maxRetries()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "DataCatalogOutputs":
            return Optional.ofNullable(clazz.cast(dataCatalogOutputs()));
        case "DatabaseOutputs":
            return Optional.ofNullable(clazz.cast(databaseOutputs()));
        case "ProjectName":
            return Optional.ofNullable(clazz.cast(projectName()));
        case "RecipeReference":
            return Optional.ofNullable(clazz.cast(recipeReference()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "Timeout":
            return Optional.ofNullable(clazz.cast(timeout()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "JobSample":
            return Optional.ofNullable(clazz.cast(jobSample()));
        case "ValidationConfigurations":
            return Optional.ofNullable(clazz.cast(validationConfigurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Job, T> g) {
        return obj -> g.apply((Job) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Job> {
        /**
         * <p>
         * The ID of the Amazon Web Services account that owns the job.
         * </p>
         * 
         * @param accountId
         *        The ID of the Amazon Web Services account that owns the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the user who created the job.
         * </p>
         * 
         * @param createdBy
         *        The Amazon Resource Name (ARN) of the user who created the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The date and time that the job was created.
         * </p>
         * 
         * @param createDate
         *        The date and time that the job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createDate(Instant createDate);

        /**
         * <p>
         * A dataset that the job is to process.
         * </p>
         * 
         * @param datasetName
         *        A dataset that the job is to process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datasetName(String datasetName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an encryption key that is used to protect the job output. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/databrew/latest/dg/encryption-security-configuration.html">Encrypting data
         * written by DataBrew jobs</a>
         * </p>
         * 
         * @param encryptionKeyArn
         *        The Amazon Resource Name (ARN) of an encryption key that is used to protect the job output. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/databrew/latest/dg/encryption-security-configuration.html"
         *        >Encrypting data written by DataBrew jobs</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKeyArn(String encryptionKeyArn);

        /**
         * <p>
         * The encryption mode for the job, which can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionMode
         *        The encryption mode for the job, which can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
         *        </p>
         *        </li>
         * @see EncryptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMode
         */
        Builder encryptionMode(String encryptionMode);

        /**
         * <p>
         * The encryption mode for the job, which can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
         * </p>
         * </li>
         * </ul>
         * 
         * @param encryptionMode
         *        The encryption mode for the job, which can be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SSE-KMS</code> - Server-side encryption with keys managed by KMS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSE-S3</code> - Server-side encryption with keys managed by Amazon S3.
         *        </p>
         *        </li>
         * @see EncryptionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EncryptionMode
         */
        Builder encryptionMode(EncryptionMode encryptionMode);

        /**
         * <p>
         * The unique name of the job.
         * </p>
         * 
         * @param name
         *        The unique name of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The job type of the job, which must be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data distribution, and
         * more.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The job type of the job, which must be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data
         *        distribution, and more.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
         *        </p>
         *        </li>
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder type(String type);

        /**
         * <p>
         * The job type of the job, which must be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data distribution, and
         * more.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The job type of the job, which must be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PROFILE</code> - A job to analyze a dataset, to determine its size, data types, data
         *        distribution, and more.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RECIPE</code> - A job to apply one or more transformations to a dataset.
         *        </p>
         *        </li>
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder type(JobType type);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the user who last modified the job.
         * </p>
         * 
         * @param lastModifiedBy
         *        The Amazon Resource Name (ARN) of the user who last modified the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedBy(String lastModifiedBy);

        /**
         * <p>
         * The modification date and time of the job.
         * </p>
         * 
         * @param lastModifiedDate
         *        The modification date and time of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * The current status of Amazon CloudWatch logging for the job.
         * </p>
         * 
         * @param logSubscription
         *        The current status of Amazon CloudWatch logging for the job.
         * @see LogSubscription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogSubscription
         */
        Builder logSubscription(String logSubscription);

        /**
         * <p>
         * The current status of Amazon CloudWatch logging for the job.
         * </p>
         * 
         * @param logSubscription
         *        The current status of Amazon CloudWatch logging for the job.
         * @see LogSubscription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogSubscription
         */
        Builder logSubscription(LogSubscription logSubscription);

        /**
         * <p>
         * The maximum number of nodes that can be consumed when the job processes data.
         * </p>
         * 
         * @param maxCapacity
         *        The maximum number of nodes that can be consumed when the job processes data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxCapacity(Integer maxCapacity);

        /**
         * <p>
         * The maximum number of times to retry the job after a job run fails.
         * </p>
         * 
         * @param maxRetries
         *        The maximum number of times to retry the job after a job run fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRetries(Integer maxRetries);

        /**
         * <p>
         * One or more artifacts that represent output from running the job.
         * </p>
         * 
         * @param outputs
         *        One or more artifacts that represent output from running the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Collection<Output> outputs);

        /**
         * <p>
         * One or more artifacts that represent output from running the job.
         * </p>
         * 
         * @param outputs
         *        One or more artifacts that represent output from running the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Output... outputs);

        /**
         * <p>
         * One or more artifacts that represent output from running the job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.databrew.model.Output.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.databrew.model.Output#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.databrew.model.Output.Builder#build()} is called immediately and its
         * result is passed to {@link #outputs(List<Output>)}.
         * 
         * @param outputs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.databrew.model.Output.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputs(java.util.Collection<Output>)
         */
        Builder outputs(Consumer<Output.Builder>... outputs);

        /**
         * <p>
         * One or more artifacts that represent the Glue Data Catalog output from running the job.
         * </p>
         * 
         * @param dataCatalogOutputs
         *        One or more artifacts that represent the Glue Data Catalog output from running the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCatalogOutputs(Collection<DataCatalogOutput> dataCatalogOutputs);

        /**
         * <p>
         * One or more artifacts that represent the Glue Data Catalog output from running the job.
         * </p>
         * 
         * @param dataCatalogOutputs
         *        One or more artifacts that represent the Glue Data Catalog output from running the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataCatalogOutputs(DataCatalogOutput... dataCatalogOutputs);

        /**
         * <p>
         * One or more artifacts that represent the Glue Data Catalog output from running the job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.databrew.model.DataCatalogOutput.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.databrew.model.DataCatalogOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.databrew.model.DataCatalogOutput.Builder#build()} is called
         * immediately and its result is passed to {@link #dataCatalogOutputs(List<DataCatalogOutput>)}.
         * 
         * @param dataCatalogOutputs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.databrew.model.DataCatalogOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataCatalogOutputs(java.util.Collection<DataCatalogOutput>)
         */
        Builder dataCatalogOutputs(Consumer<DataCatalogOutput.Builder>... dataCatalogOutputs);

        /**
         * <p>
         * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe
         * job to write into.
         * </p>
         * 
         * @param databaseOutputs
         *        Represents a list of JDBC database output objects which defines the output destination for a DataBrew
         *        recipe job to write into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseOutputs(Collection<DatabaseOutput> databaseOutputs);

        /**
         * <p>
         * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe
         * job to write into.
         * </p>
         * 
         * @param databaseOutputs
         *        Represents a list of JDBC database output objects which defines the output destination for a DataBrew
         *        recipe job to write into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseOutputs(DatabaseOutput... databaseOutputs);

        /**
         * <p>
         * Represents a list of JDBC database output objects which defines the output destination for a DataBrew recipe
         * job to write into.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.databrew.model.DatabaseOutput.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.databrew.model.DatabaseOutput#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.databrew.model.DatabaseOutput.Builder#build()} is called immediately
         * and its result is passed to {@link #databaseOutputs(List<DatabaseOutput>)}.
         * 
         * @param databaseOutputs
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.databrew.model.DatabaseOutput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #databaseOutputs(java.util.Collection<DatabaseOutput>)
         */
        Builder databaseOutputs(Consumer<DatabaseOutput.Builder>... databaseOutputs);

        /**
         * <p>
         * The name of the project that the job is associated with.
         * </p>
         * 
         * @param projectName
         *        The name of the project that the job is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectName(String projectName);

        /**
         * <p>
         * A set of steps that the job runs.
         * </p>
         * 
         * @param recipeReference
         *        A set of steps that the job runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recipeReference(RecipeReference recipeReference);

        /**
         * <p>
         * A set of steps that the job runs.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecipeReference.Builder} avoiding the
         * need to create one manually via {@link RecipeReference#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecipeReference.Builder#build()} is called immediately and its
         * result is passed to {@link #recipeReference(RecipeReference)}.
         * 
         * @param recipeReference
         *        a consumer that will call methods on {@link RecipeReference.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recipeReference(RecipeReference)
         */
        default Builder recipeReference(Consumer<RecipeReference.Builder> recipeReference) {
            return recipeReference(RecipeReference.builder().applyMutation(recipeReference).build());
        }

        /**
         * <p>
         * The unique Amazon Resource Name (ARN) for the job.
         * </p>
         * 
         * @param resourceArn
         *        The unique Amazon Resource Name (ARN) for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the role to be assumed for this job.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the role to be assumed for this job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The job's timeout in minutes. A job that attempts to run longer than this timeout period ends with a status
         * of <code>TIMEOUT</code>.
         * </p>
         * 
         * @param timeout
         *        The job's timeout in minutes. A job that attempts to run longer than this timeout period ends with a
         *        status of <code>TIMEOUT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeout(Integer timeout);

        /**
         * <p>
         * Metadata tags that have been applied to the job.
         * </p>
         * 
         * @param tags
         *        Metadata tags that have been applied to the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * A sample configuration for profile jobs only, which determines the number of rows on which the profile job is
         * run. If a <code>JobSample</code> value isn't provided, the default value is used. The default value is
         * CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
         * </p>
         * 
         * @param jobSample
         *        A sample configuration for profile jobs only, which determines the number of rows on which the profile
         *        job is run. If a <code>JobSample</code> value isn't provided, the default value is used. The default
         *        value is CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobSample(JobSample jobSample);

        /**
         * <p>
         * A sample configuration for profile jobs only, which determines the number of rows on which the profile job is
         * run. If a <code>JobSample</code> value isn't provided, the default value is used. The default value is
         * CUSTOM_ROWS for the mode parameter and 20,000 for the size parameter.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobSample.Builder} avoiding the need to
         * create one manually via {@link JobSample#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobSample.Builder#build()} is called immediately and its result
         * is passed to {@link #jobSample(JobSample)}.
         * 
         * @param jobSample
         *        a consumer that will call methods on {@link JobSample.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobSample(JobSample)
         */
        default Builder jobSample(Consumer<JobSample.Builder> jobSample) {
            return jobSample(JobSample.builder().applyMutation(jobSample).build());
        }

        /**
         * <p>
         * List of validation configurations that are applied to the profile job.
         * </p>
         * 
         * @param validationConfigurations
         *        List of validation configurations that are applied to the profile job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationConfigurations(Collection<ValidationConfiguration> validationConfigurations);

        /**
         * <p>
         * List of validation configurations that are applied to the profile job.
         * </p>
         * 
         * @param validationConfigurations
         *        List of validation configurations that are applied to the profile job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationConfigurations(ValidationConfiguration... validationConfigurations);

        /**
         * <p>
         * List of validation configurations that are applied to the profile job.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.databrew.model.ValidationConfiguration.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.databrew.model.ValidationConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.databrew.model.ValidationConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #validationConfigurations(List<ValidationConfiguration>)}.
         * 
         * @param validationConfigurations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.databrew.model.ValidationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationConfigurations(java.util.Collection<ValidationConfiguration>)
         */
        Builder validationConfigurations(Consumer<ValidationConfiguration.Builder>... validationConfigurations);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String createdBy;

        private Instant createDate;

        private String datasetName;

        private String encryptionKeyArn;

        private String encryptionMode;

        private String name;

        private String type;

        private String lastModifiedBy;

        private Instant lastModifiedDate;

        private String logSubscription;

        private Integer maxCapacity;

        private Integer maxRetries;

        private List<Output> outputs = DefaultSdkAutoConstructList.getInstance();

        private List<DataCatalogOutput> dataCatalogOutputs = DefaultSdkAutoConstructList.getInstance();

        private List<DatabaseOutput> databaseOutputs = DefaultSdkAutoConstructList.getInstance();

        private String projectName;

        private RecipeReference recipeReference;

        private String resourceArn;

        private String roleArn;

        private Integer timeout;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private JobSample jobSample;

        private List<ValidationConfiguration> validationConfigurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Job model) {
            accountId(model.accountId);
            createdBy(model.createdBy);
            createDate(model.createDate);
            datasetName(model.datasetName);
            encryptionKeyArn(model.encryptionKeyArn);
            encryptionMode(model.encryptionMode);
            name(model.name);
            type(model.type);
            lastModifiedBy(model.lastModifiedBy);
            lastModifiedDate(model.lastModifiedDate);
            logSubscription(model.logSubscription);
            maxCapacity(model.maxCapacity);
            maxRetries(model.maxRetries);
            outputs(model.outputs);
            dataCatalogOutputs(model.dataCatalogOutputs);
            databaseOutputs(model.databaseOutputs);
            projectName(model.projectName);
            recipeReference(model.recipeReference);
            resourceArn(model.resourceArn);
            roleArn(model.roleArn);
            timeout(model.timeout);
            tags(model.tags);
            jobSample(model.jobSample);
            validationConfigurations(model.validationConfigurations);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final Instant getCreateDate() {
            return createDate;
        }

        public final void setCreateDate(Instant createDate) {
            this.createDate = createDate;
        }

        @Override
        public final Builder createDate(Instant createDate) {
            this.createDate = createDate;
            return this;
        }

        public final String getDatasetName() {
            return datasetName;
        }

        public final void setDatasetName(String datasetName) {
            this.datasetName = datasetName;
        }

        @Override
        public final Builder datasetName(String datasetName) {
            this.datasetName = datasetName;
            return this;
        }

        public final String getEncryptionKeyArn() {
            return encryptionKeyArn;
        }

        public final void setEncryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
        }

        @Override
        public final Builder encryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
            return this;
        }

        public final String getEncryptionMode() {
            return encryptionMode;
        }

        public final void setEncryptionMode(String encryptionMode) {
            this.encryptionMode = encryptionMode;
        }

        @Override
        public final Builder encryptionMode(String encryptionMode) {
            this.encryptionMode = encryptionMode;
            return this;
        }

        @Override
        public final Builder encryptionMode(EncryptionMode encryptionMode) {
            this.encryptionMode(encryptionMode == null ? null : encryptionMode.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(JobType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getLastModifiedBy() {
            return lastModifiedBy;
        }

        public final void setLastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
        }

        @Override
        public final Builder lastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
            return this;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final String getLogSubscription() {
            return logSubscription;
        }

        public final void setLogSubscription(String logSubscription) {
            this.logSubscription = logSubscription;
        }

        @Override
        public final Builder logSubscription(String logSubscription) {
            this.logSubscription = logSubscription;
            return this;
        }

        @Override
        public final Builder logSubscription(LogSubscription logSubscription) {
            this.logSubscription(logSubscription == null ? null : logSubscription.toString());
            return this;
        }

        public final Integer getMaxCapacity() {
            return maxCapacity;
        }

        public final void setMaxCapacity(Integer maxCapacity) {
            this.maxCapacity = maxCapacity;
        }

        @Override
        public final Builder maxCapacity(Integer maxCapacity) {
            this.maxCapacity = maxCapacity;
            return this;
        }

        public final Integer getMaxRetries() {
            return maxRetries;
        }

        public final void setMaxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
        }

        @Override
        public final Builder maxRetries(Integer maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public final List<Output.Builder> getOutputs() {
            List<Output.Builder> result = OutputListCopier.copyToBuilder(this.outputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOutputs(Collection<Output.BuilderImpl> outputs) {
            this.outputs = OutputListCopier.copyFromBuilder(outputs);
        }

        @Override
        public final Builder outputs(Collection<Output> outputs) {
            this.outputs = OutputListCopier.copy(outputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputs(Output... outputs) {
            outputs(Arrays.asList(outputs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputs(Consumer<Output.Builder>... outputs) {
            outputs(Stream.of(outputs).map(c -> Output.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<DataCatalogOutput.Builder> getDataCatalogOutputs() {
            List<DataCatalogOutput.Builder> result = DataCatalogOutputListCopier.copyToBuilder(this.dataCatalogOutputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDataCatalogOutputs(Collection<DataCatalogOutput.BuilderImpl> dataCatalogOutputs) {
            this.dataCatalogOutputs = DataCatalogOutputListCopier.copyFromBuilder(dataCatalogOutputs);
        }

        @Override
        public final Builder dataCatalogOutputs(Collection<DataCatalogOutput> dataCatalogOutputs) {
            this.dataCatalogOutputs = DataCatalogOutputListCopier.copy(dataCatalogOutputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataCatalogOutputs(DataCatalogOutput... dataCatalogOutputs) {
            dataCatalogOutputs(Arrays.asList(dataCatalogOutputs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataCatalogOutputs(Consumer<DataCatalogOutput.Builder>... dataCatalogOutputs) {
            dataCatalogOutputs(Stream.of(dataCatalogOutputs).map(c -> DataCatalogOutput.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<DatabaseOutput.Builder> getDatabaseOutputs() {
            List<DatabaseOutput.Builder> result = DatabaseOutputListCopier.copyToBuilder(this.databaseOutputs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDatabaseOutputs(Collection<DatabaseOutput.BuilderImpl> databaseOutputs) {
            this.databaseOutputs = DatabaseOutputListCopier.copyFromBuilder(databaseOutputs);
        }

        @Override
        public final Builder databaseOutputs(Collection<DatabaseOutput> databaseOutputs) {
            this.databaseOutputs = DatabaseOutputListCopier.copy(databaseOutputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder databaseOutputs(DatabaseOutput... databaseOutputs) {
            databaseOutputs(Arrays.asList(databaseOutputs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder databaseOutputs(Consumer<DatabaseOutput.Builder>... databaseOutputs) {
            databaseOutputs(Stream.of(databaseOutputs).map(c -> DatabaseOutput.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getProjectName() {
            return projectName;
        }

        public final void setProjectName(String projectName) {
            this.projectName = projectName;
        }

        @Override
        public final Builder projectName(String projectName) {
            this.projectName = projectName;
            return this;
        }

        public final RecipeReference.Builder getRecipeReference() {
            return recipeReference != null ? recipeReference.toBuilder() : null;
        }

        public final void setRecipeReference(RecipeReference.BuilderImpl recipeReference) {
            this.recipeReference = recipeReference != null ? recipeReference.build() : null;
        }

        @Override
        public final Builder recipeReference(RecipeReference recipeReference) {
            this.recipeReference = recipeReference;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Integer getTimeout() {
            return timeout;
        }

        public final void setTimeout(Integer timeout) {
            this.timeout = timeout;
        }

        @Override
        public final Builder timeout(Integer timeout) {
            this.timeout = timeout;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final JobSample.Builder getJobSample() {
            return jobSample != null ? jobSample.toBuilder() : null;
        }

        public final void setJobSample(JobSample.BuilderImpl jobSample) {
            this.jobSample = jobSample != null ? jobSample.build() : null;
        }

        @Override
        public final Builder jobSample(JobSample jobSample) {
            this.jobSample = jobSample;
            return this;
        }

        public final List<ValidationConfiguration.Builder> getValidationConfigurations() {
            List<ValidationConfiguration.Builder> result = ValidationConfigurationListCopier
                    .copyToBuilder(this.validationConfigurations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidationConfigurations(Collection<ValidationConfiguration.BuilderImpl> validationConfigurations) {
            this.validationConfigurations = ValidationConfigurationListCopier.copyFromBuilder(validationConfigurations);
        }

        @Override
        public final Builder validationConfigurations(Collection<ValidationConfiguration> validationConfigurations) {
            this.validationConfigurations = ValidationConfigurationListCopier.copy(validationConfigurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationConfigurations(ValidationConfiguration... validationConfigurations) {
            validationConfigurations(Arrays.asList(validationConfigurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationConfigurations(Consumer<ValidationConfiguration.Builder>... validationConfigurations) {
            validationConfigurations(Stream.of(validationConfigurations)
                    .map(c -> ValidationConfiguration.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Job build() {
            return new Job(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
