/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databrew.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a set of options that define how DataBrew will interpret a Microsoft Excel file when creating a dataset
 * from that file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExcelOptions implements SdkPojo, Serializable, ToCopyableBuilder<ExcelOptions.Builder, ExcelOptions> {
    private static final SdkField<List<String>> SHEET_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SheetNames")
            .getter(getter(ExcelOptions::sheetNames))
            .setter(setter(Builder::sheetNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SheetNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Integer>> SHEET_INDEXES_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("SheetIndexes")
            .getter(getter(ExcelOptions::sheetIndexes))
            .setter(setter(Builder::sheetIndexes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SheetIndexes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> HEADER_ROW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("HeaderRow").getter(getter(ExcelOptions::headerRow)).setter(setter(Builder::headerRow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeaderRow").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SHEET_NAMES_FIELD,
            SHEET_INDEXES_FIELD, HEADER_ROW_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> sheetNames;

    private final List<Integer> sheetIndexes;

    private final Boolean headerRow;

    private ExcelOptions(BuilderImpl builder) {
        this.sheetNames = builder.sheetNames;
        this.sheetIndexes = builder.sheetIndexes;
        this.headerRow = builder.headerRow;
    }

    /**
     * For responses, this returns true if the service returned a value for the SheetNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSheetNames() {
        return sheetNames != null && !(sheetNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more named sheets in the Excel file that will be included in the dataset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSheetNames} method.
     * </p>
     * 
     * @return One or more named sheets in the Excel file that will be included in the dataset.
     */
    public final List<String> sheetNames() {
        return sheetNames;
    }

    /**
     * For responses, this returns true if the service returned a value for the SheetIndexes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSheetIndexes() {
        return sheetIndexes != null && !(sheetIndexes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more sheet numbers in the Excel file that will be included in the dataset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSheetIndexes} method.
     * </p>
     * 
     * @return One or more sheet numbers in the Excel file that will be included in the dataset.
     */
    public final List<Integer> sheetIndexes() {
        return sheetIndexes;
    }

    /**
     * <p>
     * A variable that specifies whether the first row in the file is parsed as the header. If this value is false,
     * column names are auto-generated.
     * </p>
     * 
     * @return A variable that specifies whether the first row in the file is parsed as the header. If this value is
     *         false, column names are auto-generated.
     */
    public final Boolean headerRow() {
        return headerRow;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasSheetNames() ? sheetNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSheetIndexes() ? sheetIndexes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(headerRow());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExcelOptions)) {
            return false;
        }
        ExcelOptions other = (ExcelOptions) obj;
        return hasSheetNames() == other.hasSheetNames() && Objects.equals(sheetNames(), other.sheetNames())
                && hasSheetIndexes() == other.hasSheetIndexes() && Objects.equals(sheetIndexes(), other.sheetIndexes())
                && Objects.equals(headerRow(), other.headerRow());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExcelOptions").add("SheetNames", hasSheetNames() ? sheetNames() : null)
                .add("SheetIndexes", hasSheetIndexes() ? sheetIndexes() : null).add("HeaderRow", headerRow()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SheetNames":
            return Optional.ofNullable(clazz.cast(sheetNames()));
        case "SheetIndexes":
            return Optional.ofNullable(clazz.cast(sheetIndexes()));
        case "HeaderRow":
            return Optional.ofNullable(clazz.cast(headerRow()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SheetNames", SHEET_NAMES_FIELD);
        map.put("SheetIndexes", SHEET_INDEXES_FIELD);
        map.put("HeaderRow", HEADER_ROW_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ExcelOptions, T> g) {
        return obj -> g.apply((ExcelOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExcelOptions> {
        /**
         * <p>
         * One or more named sheets in the Excel file that will be included in the dataset.
         * </p>
         * 
         * @param sheetNames
         *        One or more named sheets in the Excel file that will be included in the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sheetNames(Collection<String> sheetNames);

        /**
         * <p>
         * One or more named sheets in the Excel file that will be included in the dataset.
         * </p>
         * 
         * @param sheetNames
         *        One or more named sheets in the Excel file that will be included in the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sheetNames(String... sheetNames);

        /**
         * <p>
         * One or more sheet numbers in the Excel file that will be included in the dataset.
         * </p>
         * 
         * @param sheetIndexes
         *        One or more sheet numbers in the Excel file that will be included in the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sheetIndexes(Collection<Integer> sheetIndexes);

        /**
         * <p>
         * One or more sheet numbers in the Excel file that will be included in the dataset.
         * </p>
         * 
         * @param sheetIndexes
         *        One or more sheet numbers in the Excel file that will be included in the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sheetIndexes(Integer... sheetIndexes);

        /**
         * <p>
         * A variable that specifies whether the first row in the file is parsed as the header. If this value is false,
         * column names are auto-generated.
         * </p>
         * 
         * @param headerRow
         *        A variable that specifies whether the first row in the file is parsed as the header. If this value is
         *        false, column names are auto-generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headerRow(Boolean headerRow);
    }

    static final class BuilderImpl implements Builder {
        private List<String> sheetNames = DefaultSdkAutoConstructList.getInstance();

        private List<Integer> sheetIndexes = DefaultSdkAutoConstructList.getInstance();

        private Boolean headerRow;

        private BuilderImpl() {
        }

        private BuilderImpl(ExcelOptions model) {
            sheetNames(model.sheetNames);
            sheetIndexes(model.sheetIndexes);
            headerRow(model.headerRow);
        }

        public final Collection<String> getSheetNames() {
            if (sheetNames instanceof SdkAutoConstructList) {
                return null;
            }
            return sheetNames;
        }

        public final void setSheetNames(Collection<String> sheetNames) {
            this.sheetNames = SheetNameListCopier.copy(sheetNames);
        }

        @Override
        public final Builder sheetNames(Collection<String> sheetNames) {
            this.sheetNames = SheetNameListCopier.copy(sheetNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sheetNames(String... sheetNames) {
            sheetNames(Arrays.asList(sheetNames));
            return this;
        }

        public final Collection<Integer> getSheetIndexes() {
            if (sheetIndexes instanceof SdkAutoConstructList) {
                return null;
            }
            return sheetIndexes;
        }

        public final void setSheetIndexes(Collection<Integer> sheetIndexes) {
            this.sheetIndexes = SheetIndexListCopier.copy(sheetIndexes);
        }

        @Override
        public final Builder sheetIndexes(Collection<Integer> sheetIndexes) {
            this.sheetIndexes = SheetIndexListCopier.copy(sheetIndexes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sheetIndexes(Integer... sheetIndexes) {
            sheetIndexes(Arrays.asList(sheetIndexes));
            return this;
        }

        public final Boolean getHeaderRow() {
            return headerRow;
        }

        public final void setHeaderRow(Boolean headerRow) {
            this.headerRow = headerRow;
        }

        @Override
        public final Builder headerRow(Boolean headerRow) {
            this.headerRow = headerRow;
            return this;
        }

        @Override
        public ExcelOptions build() {
            return new ExcelOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
