/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datapipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about a pipeline task that is assigned to a task runner.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TaskObject implements SdkPojo, Serializable, ToCopyableBuilder<TaskObject.Builder, TaskObject> {
    private static final SdkField<String> TASK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("taskId")
            .getter(getter(TaskObject::taskId)).setter(setter(Builder::taskId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("taskId").build()).build();

    private static final SdkField<String> PIPELINE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pipelineId").getter(getter(TaskObject::pipelineId)).setter(setter(Builder::pipelineId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineId").build()).build();

    private static final SdkField<String> ATTEMPT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("attemptId").getter(getter(TaskObject::attemptId)).setter(setter(Builder::attemptId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attemptId").build()).build();

    private static final SdkField<Map<String, PipelineObject>> OBJECTS_FIELD = SdkField
            .<Map<String, PipelineObject>> builder(MarshallingType.MAP)
            .memberName("objects")
            .getter(getter(TaskObject::objects))
            .setter(setter(Builder::objects))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("objects").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<PipelineObject> builder(MarshallingType.SDK_POJO)
                                            .constructor(PipelineObject::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TASK_ID_FIELD,
            PIPELINE_ID_FIELD, ATTEMPT_ID_FIELD, OBJECTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String taskId;

    private final String pipelineId;

    private final String attemptId;

    private final Map<String, PipelineObject> objects;

    private TaskObject(BuilderImpl builder) {
        this.taskId = builder.taskId;
        this.pipelineId = builder.pipelineId;
        this.attemptId = builder.attemptId;
        this.objects = builder.objects;
    }

    /**
     * <p>
     * An internal identifier for the task. This ID is passed to the <a>SetTaskStatus</a> and <a>ReportTaskProgress</a>
     * actions.
     * </p>
     * 
     * @return An internal identifier for the task. This ID is passed to the <a>SetTaskStatus</a> and
     *         <a>ReportTaskProgress</a> actions.
     */
    public String taskId() {
        return taskId;
    }

    /**
     * <p>
     * The ID of the pipeline that provided the task.
     * </p>
     * 
     * @return The ID of the pipeline that provided the task.
     */
    public String pipelineId() {
        return pipelineId;
    }

    /**
     * <p>
     * The ID of the pipeline task attempt object. AWS Data Pipeline uses this value to track how many times a task is
     * attempted.
     * </p>
     * 
     * @return The ID of the pipeline task attempt object. AWS Data Pipeline uses this value to track how many times a
     *         task is attempted.
     */
    public String attemptId() {
        return attemptId;
    }

    /**
     * Returns true if the Objects property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasObjects() {
        return objects != null && !(objects instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Connection information for the location where the task runner will publish the output of the task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasObjects()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Connection information for the location where the task runner will publish the output of the task.
     */
    public Map<String, PipelineObject> objects() {
        return objects;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(taskId());
        hashCode = 31 * hashCode + Objects.hashCode(pipelineId());
        hashCode = 31 * hashCode + Objects.hashCode(attemptId());
        hashCode = 31 * hashCode + Objects.hashCode(hasObjects() ? objects() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TaskObject)) {
            return false;
        }
        TaskObject other = (TaskObject) obj;
        return Objects.equals(taskId(), other.taskId()) && Objects.equals(pipelineId(), other.pipelineId())
                && Objects.equals(attemptId(), other.attemptId()) && hasObjects() == other.hasObjects()
                && Objects.equals(objects(), other.objects());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("TaskObject").add("TaskId", taskId()).add("PipelineId", pipelineId())
                .add("AttemptId", attemptId()).add("Objects", hasObjects() ? objects() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "taskId":
            return Optional.ofNullable(clazz.cast(taskId()));
        case "pipelineId":
            return Optional.ofNullable(clazz.cast(pipelineId()));
        case "attemptId":
            return Optional.ofNullable(clazz.cast(attemptId()));
        case "objects":
            return Optional.ofNullable(clazz.cast(objects()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<TaskObject, T> g) {
        return obj -> g.apply((TaskObject) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TaskObject> {
        /**
         * <p>
         * An internal identifier for the task. This ID is passed to the <a>SetTaskStatus</a> and
         * <a>ReportTaskProgress</a> actions.
         * </p>
         * 
         * @param taskId
         *        An internal identifier for the task. This ID is passed to the <a>SetTaskStatus</a> and
         *        <a>ReportTaskProgress</a> actions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskId(String taskId);

        /**
         * <p>
         * The ID of the pipeline that provided the task.
         * </p>
         * 
         * @param pipelineId
         *        The ID of the pipeline that provided the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineId(String pipelineId);

        /**
         * <p>
         * The ID of the pipeline task attempt object. AWS Data Pipeline uses this value to track how many times a task
         * is attempted.
         * </p>
         * 
         * @param attemptId
         *        The ID of the pipeline task attempt object. AWS Data Pipeline uses this value to track how many times
         *        a task is attempted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attemptId(String attemptId);

        /**
         * <p>
         * Connection information for the location where the task runner will publish the output of the task.
         * </p>
         * 
         * @param objects
         *        Connection information for the location where the task runner will publish the output of the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objects(Map<String, PipelineObject> objects);
    }

    static final class BuilderImpl implements Builder {
        private String taskId;

        private String pipelineId;

        private String attemptId;

        private Map<String, PipelineObject> objects = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TaskObject model) {
            taskId(model.taskId);
            pipelineId(model.pipelineId);
            attemptId(model.attemptId);
            objects(model.objects);
        }

        public final String getTaskId() {
            return taskId;
        }

        @Override
        public final Builder taskId(String taskId) {
            this.taskId = taskId;
            return this;
        }

        public final void setTaskId(String taskId) {
            this.taskId = taskId;
        }

        public final String getPipelineId() {
            return pipelineId;
        }

        @Override
        public final Builder pipelineId(String pipelineId) {
            this.pipelineId = pipelineId;
            return this;
        }

        public final void setPipelineId(String pipelineId) {
            this.pipelineId = pipelineId;
        }

        public final String getAttemptId() {
            return attemptId;
        }

        @Override
        public final Builder attemptId(String attemptId) {
            this.attemptId = attemptId;
            return this;
        }

        public final void setAttemptId(String attemptId) {
            this.attemptId = attemptId;
        }

        public final Map<String, PipelineObject.Builder> getObjects() {
            if (objects instanceof SdkAutoConstructMap) {
                return null;
            }
            return objects != null ? CollectionUtils.mapValues(objects, PipelineObject::toBuilder) : null;
        }

        @Override
        public final Builder objects(Map<String, PipelineObject> objects) {
            this.objects = PipelineObjectMapCopier.copy(objects);
            return this;
        }

        public final void setObjects(Map<String, PipelineObject.BuilderImpl> objects) {
            this.objects = PipelineObjectMapCopier.copyFromBuilder(objects);
        }

        @Override
        public TaskObject build() {
            return new TaskObject(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
