/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datapipeline;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.datapipeline.model.ActivatePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.ActivatePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.AddTagsRequest;
import software.amazon.awssdk.services.datapipeline.model.AddTagsResponse;
import software.amazon.awssdk.services.datapipeline.model.CreatePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.CreatePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.DataPipelineException;
import software.amazon.awssdk.services.datapipeline.model.DataPipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.DeactivatePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.DeactivatePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.DeletePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.DeletePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.DescribeObjectsRequest;
import software.amazon.awssdk.services.datapipeline.model.DescribeObjectsResponse;
import software.amazon.awssdk.services.datapipeline.model.DescribePipelinesRequest;
import software.amazon.awssdk.services.datapipeline.model.DescribePipelinesResponse;
import software.amazon.awssdk.services.datapipeline.model.EvaluateExpressionRequest;
import software.amazon.awssdk.services.datapipeline.model.EvaluateExpressionResponse;
import software.amazon.awssdk.services.datapipeline.model.GetPipelineDefinitionRequest;
import software.amazon.awssdk.services.datapipeline.model.GetPipelineDefinitionResponse;
import software.amazon.awssdk.services.datapipeline.model.InternalServiceErrorException;
import software.amazon.awssdk.services.datapipeline.model.InvalidRequestException;
import software.amazon.awssdk.services.datapipeline.model.ListPipelinesRequest;
import software.amazon.awssdk.services.datapipeline.model.ListPipelinesResponse;
import software.amazon.awssdk.services.datapipeline.model.PipelineDeletedException;
import software.amazon.awssdk.services.datapipeline.model.PipelineNotFoundException;
import software.amazon.awssdk.services.datapipeline.model.PollForTaskRequest;
import software.amazon.awssdk.services.datapipeline.model.PollForTaskResponse;
import software.amazon.awssdk.services.datapipeline.model.PutPipelineDefinitionRequest;
import software.amazon.awssdk.services.datapipeline.model.PutPipelineDefinitionResponse;
import software.amazon.awssdk.services.datapipeline.model.QueryObjectsRequest;
import software.amazon.awssdk.services.datapipeline.model.QueryObjectsResponse;
import software.amazon.awssdk.services.datapipeline.model.RemoveTagsRequest;
import software.amazon.awssdk.services.datapipeline.model.RemoveTagsResponse;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskProgressRequest;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskProgressResponse;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskRunnerHeartbeatRequest;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskRunnerHeartbeatResponse;
import software.amazon.awssdk.services.datapipeline.model.SetStatusRequest;
import software.amazon.awssdk.services.datapipeline.model.SetStatusResponse;
import software.amazon.awssdk.services.datapipeline.model.SetTaskStatusRequest;
import software.amazon.awssdk.services.datapipeline.model.SetTaskStatusResponse;
import software.amazon.awssdk.services.datapipeline.model.TaskNotFoundException;
import software.amazon.awssdk.services.datapipeline.model.ValidatePipelineDefinitionRequest;
import software.amazon.awssdk.services.datapipeline.model.ValidatePipelineDefinitionResponse;
import software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsPublisher;
import software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesPublisher;
import software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsPublisher;
import software.amazon.awssdk.services.datapipeline.transform.ActivatePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DeactivatePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DescribeObjectsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DescribePipelinesRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.EvaluateExpressionRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.GetPipelineDefinitionRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.PollForTaskRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.PutPipelineDefinitionRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.QueryObjectsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ReportTaskProgressRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ReportTaskRunnerHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.SetStatusRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.SetTaskStatusRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ValidatePipelineDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DataPipelineAsyncClient}.
 *
 * @see DataPipelineAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDataPipelineAsyncClient implements DataPipelineAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDataPipelineAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final DataPipelineServiceClientConfiguration serviceClientConfiguration;

    protected DefaultDataPipelineAsyncClient(DataPipelineServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation,
     * activation fails.
     * </p>
     * <p>
     * If you need to pause the pipeline to investigate an issue with a component, such as a data source or script, call
     * <a>DeactivatePipeline</a>.
     * </p>
     * <p>
     * To activate a finished pipeline, modify the end date for the pipeline and then activate it.
     * </p>
     *
     * @param activatePipelineRequest
     *        Contains the parameters for ActivatePipeline.
     * @return A Java Future containing the result of the ActivatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.ActivatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ActivatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivatePipelineResponse> activatePipeline(ActivatePipelineRequest activatePipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivatePipelineRequest, ActivatePipelineResponse>()
                            .withOperationName("ActivatePipeline")
                            .withMarshaller(new ActivatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activatePipelineRequest));
            CompletableFuture<ActivatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or modifies tags for the specified pipeline.
     * </p>
     *
     * @param addTagsRequest
     *        Contains the parameters for AddTags.
     * @return A Java Future containing the result of the AddTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.AddTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/AddTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddTagsResponse> addTags(AddTagsRequest addTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                            .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(addTagsRequest));
            CompletableFuture<AddTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new, empty pipeline. Use <a>PutPipelineDefinition</a> to populate the pipeline.
     * </p>
     *
     * @param createPipelineRequest
     *        Contains the parameters for CreatePipeline.
     * @return A Java Future containing the result of the CreatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.CreatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/CreatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePipelineResponse> createPipeline(CreatePipelineRequest createPipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                            .withOperationName("CreatePipeline")
                            .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPipelineRequest));
            CompletableFuture<CreatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates the specified running pipeline. The pipeline is set to the <code>DEACTIVATING</code> state until the
     * deactivation process completes.
     * </p>
     * <p>
     * To resume a deactivated pipeline, use <a>ActivatePipeline</a>. By default, the pipeline resumes from the last
     * completed execution. Optionally, you can specify the date and time to resume the pipeline.
     * </p>
     *
     * @param deactivatePipelineRequest
     *        Contains the parameters for DeactivatePipeline.
     * @return A Java Future containing the result of the DeactivatePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.DeactivatePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DeactivatePipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivatePipelineResponse> deactivatePipeline(DeactivatePipelineRequest deactivatePipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivatePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivatePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivatePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivatePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivatePipelineRequest, DeactivatePipelineResponse>()
                            .withOperationName("DeactivatePipeline")
                            .withMarshaller(new DeactivatePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deactivatePipelineRequest));
            CompletableFuture<DeactivatePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a pipeline, its pipeline definition, and its run history. AWS Data Pipeline attempts to cancel instances
     * associated with the pipeline that are currently being processed by task runners.
     * </p>
     * <p>
     * Deleting a pipeline cannot be undone. You cannot query or restore a deleted pipeline. To temporarily pause a
     * pipeline instead of deleting it, call <a>SetStatus</a> with the status set to <code>PAUSE</code> on individual
     * components. Components that are paused by <a>SetStatus</a> can be resumed.
     * </p>
     *
     * @param deletePipelineRequest
     *        Contains the parameters for DeletePipeline.
     * @return A Java Future containing the result of the DeletePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.DeletePipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DeletePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePipelineResponse> deletePipeline(DeletePipelineRequest deletePipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                            .withOperationName("DeletePipeline")
                            .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePipelineRequest));
            CompletableFuture<DeletePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the object definitions for a set of objects associated with the pipeline. Object definitions are composed of
     * a set of fields that define the properties of the object.
     * </p>
     *
     * @param describeObjectsRequest
     *        Contains the parameters for DescribeObjects.
     * @return A Java Future containing the result of the DescribeObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.DescribeObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DescribeObjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeObjectsResponse> describeObjects(DescribeObjectsRequest describeObjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeObjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeObjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeObjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeObjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeObjectsRequest, DescribeObjectsResponse>()
                            .withOperationName("DescribeObjects")
                            .withMarshaller(new DescribeObjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeObjectsRequest));
            CompletableFuture<DescribeObjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the object definitions for a set of objects associated with the pipeline. Object definitions are composed of
     * a set of fields that define the properties of the object.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeObjects(software.amazon.awssdk.services.datapipeline.model.DescribeObjectsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsPublisher publisher = client.describeObjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsPublisher publisher = client.describeObjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datapipeline.model.DescribeObjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datapipeline.model.DescribeObjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeObjects(software.amazon.awssdk.services.datapipeline.model.DescribeObjectsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeObjectsRequest
     *        Contains the parameters for DescribeObjects.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.DescribeObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DescribeObjects" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeObjectsPublisher describeObjectsPaginator(DescribeObjectsRequest describeObjectsRequest) {
        return new DescribeObjectsPublisher(this, applyPaginatorUserAgent(describeObjectsRequest));
    }

    /**
     * <p>
     * Retrieves metadata about one or more pipelines. The information retrieved includes the name of the pipeline, the
     * pipeline identifier, its current state, and the user account that owns the pipeline. Using account credentials,
     * you can retrieve metadata about pipelines that you or your IAM users have created. If you are using an IAM user
     * account, you can retrieve metadata about only those pipelines for which you have read permissions.
     * </p>
     * <p>
     * To retrieve the full pipeline definition instead of metadata about the pipeline, call
     * <a>GetPipelineDefinition</a>.
     * </p>
     *
     * @param describePipelinesRequest
     *        Contains the parameters for DescribePipelines.
     * @return A Java Future containing the result of the DescribePipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.DescribePipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DescribePipelines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePipelinesResponse> describePipelines(DescribePipelinesRequest describePipelinesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePipelinesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribePipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePipelinesRequest, DescribePipelinesResponse>()
                            .withOperationName("DescribePipelines")
                            .withMarshaller(new DescribePipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describePipelinesRequest));
            CompletableFuture<DescribePipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Task runners call <code>EvaluateExpression</code> to evaluate a string in the context of the specified object.
     * For example, a task runner can evaluate SQL queries stored in Amazon S3.
     * </p>
     *
     * @param evaluateExpressionRequest
     *        Contains the parameters for EvaluateExpression.
     * @return A Java Future containing the result of the EvaluateExpression operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>TaskNotFoundException The specified task was not found.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.EvaluateExpression
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/EvaluateExpression"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EvaluateExpressionResponse> evaluateExpression(EvaluateExpressionRequest evaluateExpressionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, evaluateExpressionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EvaluateExpression");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EvaluateExpressionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EvaluateExpressionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EvaluateExpressionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EvaluateExpressionRequest, EvaluateExpressionResponse>()
                            .withOperationName("EvaluateExpression")
                            .withMarshaller(new EvaluateExpressionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(evaluateExpressionRequest));
            CompletableFuture<EvaluateExpressionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the definition of the specified pipeline. You can call <code>GetPipelineDefinition</code> to retrieve the
     * pipeline definition that you provided using <a>PutPipelineDefinition</a>.
     * </p>
     *
     * @param getPipelineDefinitionRequest
     *        Contains the parameters for GetPipelineDefinition.
     * @return A Java Future containing the result of the GetPipelineDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.GetPipelineDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/GetPipelineDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPipelineDefinitionResponse> getPipelineDefinition(
            GetPipelineDefinitionRequest getPipelineDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPipelineDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPipelineDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPipelineDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetPipelineDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPipelineDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPipelineDefinitionRequest, GetPipelineDefinitionResponse>()
                            .withOperationName("GetPipelineDefinition")
                            .withMarshaller(new GetPipelineDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPipelineDefinitionRequest));
            CompletableFuture<GetPipelineDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     * </p>
     *
     * @param listPipelinesRequest
     *        Contains the parameters for ListPipelines.
     * @return A Java Future containing the result of the ListPipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.ListPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ListPipelines" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListPipelinesResponse> listPipelines(ListPipelinesRequest listPipelinesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                            .withOperationName("ListPipelines")
                            .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPipelinesRequest));
            CompletableFuture<ListPipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPipelines(software.amazon.awssdk.services.datapipeline.model.ListPipelinesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesPublisher publisher = client.listPipelinesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesPublisher publisher = client.listPipelinesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datapipeline.model.ListPipelinesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datapipeline.model.ListPipelinesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPipelines(software.amazon.awssdk.services.datapipeline.model.ListPipelinesRequest)} operation.</b>
     * </p>
     *
     * @param listPipelinesRequest
     *        Contains the parameters for ListPipelines.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.ListPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ListPipelines" target="_top">AWS
     *      API Documentation</a>
     */
    public ListPipelinesPublisher listPipelinesPaginator(ListPipelinesRequest listPipelinesRequest) {
        return new ListPipelinesPublisher(this, applyPaginatorUserAgent(listPipelinesRequest));
    }

    /**
     * <p>
     * Task runners call <code>PollForTask</code> to receive a task to perform from AWS Data Pipeline. The task runner
     * specifies which tasks it can perform by setting a value for the <code>workerGroup</code> parameter. The task
     * returned can come from any of the pipelines that match the <code>workerGroup</code> value passed in by the task
     * runner and that was launched using the IAM user credentials specified by the task runner.
     * </p>
     * <p>
     * If tasks are ready in the work queue, <code>PollForTask</code> returns a response immediately. If no tasks are
     * available in the queue, <code>PollForTask</code> uses long-polling and holds on to a poll connection for up to a
     * 90 seconds, during which time the first newly scheduled task is handed to the task runner. To accomodate this,
     * set the socket timeout in your task runner to 90 seconds. The task runner should not call
     * <code>PollForTask</code> again on the same <code>workerGroup</code> until it receives a response, and this can
     * take up to 90 seconds.
     * </p>
     *
     * @param pollForTaskRequest
     *        Contains the parameters for PollForTask.
     * @return A Java Future containing the result of the PollForTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>TaskNotFoundException The specified task was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.PollForTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/PollForTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PollForTaskResponse> pollForTask(PollForTaskRequest pollForTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, pollForTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PollForTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PollForTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PollForTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PollForTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PollForTaskRequest, PollForTaskResponse>()
                            .withOperationName("PollForTask").withMarshaller(new PollForTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(pollForTaskRequest));
            CompletableFuture<PollForTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tasks, schedules, and preconditions to the specified pipeline. You can use
     * <code>PutPipelineDefinition</code> to populate a new pipeline.
     * </p>
     * <p>
     * <code>PutPipelineDefinition</code> also validates the configuration as it adds it to the pipeline. Changes to the
     * pipeline are saved unless one of the following three validation errors exists in the pipeline.
     * </p>
     * <ol>
     * <li>An object is missing a name or identifier field.</li>
     * <li>A string or reference field is empty.</li>
     * <li>The number of objects in the pipeline exceeds the maximum allowed objects.</li>
     * <li>The pipeline is in a FINISHED state.</li>
     * </ol>
     * <p>
     * Pipeline object definitions are passed to the <code>PutPipelineDefinition</code> action and returned by the
     * <a>GetPipelineDefinition</a> action.
     * </p>
     *
     * @param putPipelineDefinitionRequest
     *        Contains the parameters for PutPipelineDefinition.
     * @return A Java Future containing the result of the PutPipelineDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.PutPipelineDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/PutPipelineDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutPipelineDefinitionResponse> putPipelineDefinition(
            PutPipelineDefinitionRequest putPipelineDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPipelineDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPipelineDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPipelineDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutPipelineDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPipelineDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPipelineDefinitionRequest, PutPipelineDefinitionResponse>()
                            .withOperationName("PutPipelineDefinition")
                            .withMarshaller(new PutPipelineDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPipelineDefinitionRequest));
            CompletableFuture<PutPipelineDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     * </p>
     *
     * @param queryObjectsRequest
     *        Contains the parameters for QueryObjects.
     * @return A Java Future containing the result of the QueryObjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.QueryObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/QueryObjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<QueryObjectsResponse> queryObjects(QueryObjectsRequest queryObjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryObjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "QueryObjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<QueryObjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    QueryObjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<QueryObjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<QueryObjectsRequest, QueryObjectsResponse>()
                            .withOperationName("QueryObjects").withMarshaller(new QueryObjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(queryObjectsRequest));
            CompletableFuture<QueryObjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #queryObjects(software.amazon.awssdk.services.datapipeline.model.QueryObjectsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsPublisher publisher = client.queryObjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsPublisher publisher = client.queryObjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datapipeline.model.QueryObjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datapipeline.model.QueryObjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #queryObjects(software.amazon.awssdk.services.datapipeline.model.QueryObjectsRequest)} operation.</b>
     * </p>
     *
     * @param queryObjectsRequest
     *        Contains the parameters for QueryObjects.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.QueryObjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/QueryObjects" target="_top">AWS API
     *      Documentation</a>
     */
    public QueryObjectsPublisher queryObjectsPaginator(QueryObjectsRequest queryObjectsRequest) {
        return new QueryObjectsPublisher(this, applyPaginatorUserAgent(queryObjectsRequest));
    }

    /**
     * <p>
     * Removes existing tags from the specified pipeline.
     * </p>
     *
     * @param removeTagsRequest
     *        Contains the parameters for RemoveTags.
     * @return A Java Future containing the result of the RemoveTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.RemoveTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/RemoveTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveTagsResponse> removeTags(RemoveTagsRequest removeTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RemoveTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>().withOperationName("RemoveTags")
                            .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeTagsRequest));
            CompletableFuture<RemoveTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Task runners call <code>ReportTaskProgress</code> when assigned a task to acknowledge that it has the task. If
     * the web service does not receive this acknowledgement within 2 minutes, it assigns the task in a subsequent
     * <a>PollForTask</a> call. After this initial acknowledgement, the task runner only needs to report progress every
     * 15 minutes to maintain its ownership of the task. You can change this reporting time from 15 minutes by
     * specifying a <code>reportProgressTimeout</code> field in your pipeline.
     * </p>
     * <p>
     * If a task runner does not report its status after 5 minutes, AWS Data Pipeline assumes that the task runner is
     * unable to process the task and reassigns the task in a subsequent response to <a>PollForTask</a>. Task runners
     * should call <code>ReportTaskProgress</code> every 60 seconds.
     * </p>
     *
     * @param reportTaskProgressRequest
     *        Contains the parameters for ReportTaskProgress.
     * @return A Java Future containing the result of the ReportTaskProgress operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>TaskNotFoundException The specified task was not found.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.ReportTaskProgress
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ReportTaskProgress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ReportTaskProgressResponse> reportTaskProgress(ReportTaskProgressRequest reportTaskProgressRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reportTaskProgressRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReportTaskProgress");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReportTaskProgressResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReportTaskProgressResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReportTaskProgressResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReportTaskProgressRequest, ReportTaskProgressResponse>()
                            .withOperationName("ReportTaskProgress")
                            .withMarshaller(new ReportTaskProgressRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(reportTaskProgressRequest));
            CompletableFuture<ReportTaskProgressResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Task runners call <code>ReportTaskRunnerHeartbeat</code> every 15 minutes to indicate that they are operational.
     * If the AWS Data Pipeline Task Runner is launched on a resource managed by AWS Data Pipeline, the web service can
     * use this call to detect when the task runner application has failed and restart a new instance.
     * </p>
     *
     * @param reportTaskRunnerHeartbeatRequest
     *        Contains the parameters for ReportTaskRunnerHeartbeat.
     * @return A Java Future containing the result of the ReportTaskRunnerHeartbeat operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.ReportTaskRunnerHeartbeat
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ReportTaskRunnerHeartbeat"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ReportTaskRunnerHeartbeatResponse> reportTaskRunnerHeartbeat(
            ReportTaskRunnerHeartbeatRequest reportTaskRunnerHeartbeatRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, reportTaskRunnerHeartbeatRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReportTaskRunnerHeartbeat");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ReportTaskRunnerHeartbeatResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ReportTaskRunnerHeartbeatResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ReportTaskRunnerHeartbeatResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ReportTaskRunnerHeartbeatRequest, ReportTaskRunnerHeartbeatResponse>()
                            .withOperationName("ReportTaskRunnerHeartbeat")
                            .withMarshaller(new ReportTaskRunnerHeartbeatRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(reportTaskRunnerHeartbeatRequest));
            CompletableFuture<ReportTaskRunnerHeartbeatResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Requests that the status of the specified physical or logical pipeline objects be updated in the specified
     * pipeline. This update might not occur immediately, but is eventually consistent. The status that can be set
     * depends on the type of object (for example, DataNode or Activity). You cannot perform this operation on
     * <code>FINISHED</code> pipelines and attempting to do so returns <code>InvalidRequestException</code>.
     * </p>
     *
     * @param setStatusRequest
     *        Contains the parameters for SetStatus.
     * @return A Java Future containing the result of the SetStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.SetStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/SetStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SetStatusResponse> setStatus(SetStatusRequest setStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SetStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetStatusRequest, SetStatusResponse>().withOperationName("SetStatus")
                            .withMarshaller(new SetStatusRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(setStatusRequest));
            CompletableFuture<SetStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Task runners call <code>SetTaskStatus</code> to notify AWS Data Pipeline that a task is completed and provide
     * information about the final status. A task runner makes this call regardless of whether the task was sucessful. A
     * task runner does not need to call <code>SetTaskStatus</code> for tasks that are canceled by the web service
     * during a call to <a>ReportTaskProgress</a>.
     * </p>
     *
     * @param setTaskStatusRequest
     *        Contains the parameters for SetTaskStatus.
     * @return A Java Future containing the result of the SetTaskStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>TaskNotFoundException The specified task was not found.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.SetTaskStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/SetTaskStatus" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<SetTaskStatusResponse> setTaskStatus(SetTaskStatusRequest setTaskStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setTaskStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetTaskStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetTaskStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SetTaskStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetTaskStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetTaskStatusRequest, SetTaskStatusResponse>()
                            .withOperationName("SetTaskStatus")
                            .withMarshaller(new SetTaskStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(setTaskStatusRequest));
            CompletableFuture<SetTaskStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Validates the specified pipeline definition to ensure that it is well formed and can be run without error.
     * </p>
     *
     * @param validatePipelineDefinitionRequest
     *        Contains the parameters for ValidatePipelineDefinition.
     * @return A Java Future containing the result of the ValidatePipelineDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServiceErrorException An internal service error occurred.</li>
     *         <li>InvalidRequestException The request was not valid. Verify that your request was properly formatted,
     *         that the signature was generated with the correct credentials, and that you haven't exceeded any of the
     *         service limits for your account.</li>
     *         <li>PipelineNotFoundException The specified pipeline was not found. Verify that you used the correct user
     *         and account identifiers.</li>
     *         <li>PipelineDeletedException The specified pipeline has been deleted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataPipelineException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataPipelineAsyncClient.ValidatePipelineDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ValidatePipelineDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ValidatePipelineDefinitionResponse> validatePipelineDefinition(
            ValidatePipelineDefinitionRequest validatePipelineDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validatePipelineDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Data Pipeline");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidatePipelineDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ValidatePipelineDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ValidatePipelineDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ValidatePipelineDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ValidatePipelineDefinitionRequest, ValidatePipelineDefinitionResponse>()
                            .withOperationName("ValidatePipelineDefinition")
                            .withMarshaller(new ValidatePipelineDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(validatePipelineDefinitionRequest));
            CompletableFuture<ValidatePipelineDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DataPipelineServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DataPipelineException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaskNotFoundException")
                                .exceptionBuilderSupplier(TaskNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineDeletedException")
                                .exceptionBuilderSupplier(PipelineDeletedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineNotFoundException")
                                .exceptionBuilderSupplier(PipelineNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends DataPipelineRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
