/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datapipeline;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.datapipeline.model.ActivatePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.ActivatePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.AddTagsRequest;
import software.amazon.awssdk.services.datapipeline.model.AddTagsResponse;
import software.amazon.awssdk.services.datapipeline.model.CreatePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.CreatePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.DataPipelineException;
import software.amazon.awssdk.services.datapipeline.model.DataPipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.DeactivatePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.DeactivatePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.DeletePipelineRequest;
import software.amazon.awssdk.services.datapipeline.model.DeletePipelineResponse;
import software.amazon.awssdk.services.datapipeline.model.DescribeObjectsRequest;
import software.amazon.awssdk.services.datapipeline.model.DescribeObjectsResponse;
import software.amazon.awssdk.services.datapipeline.model.DescribePipelinesRequest;
import software.amazon.awssdk.services.datapipeline.model.DescribePipelinesResponse;
import software.amazon.awssdk.services.datapipeline.model.EvaluateExpressionRequest;
import software.amazon.awssdk.services.datapipeline.model.EvaluateExpressionResponse;
import software.amazon.awssdk.services.datapipeline.model.GetPipelineDefinitionRequest;
import software.amazon.awssdk.services.datapipeline.model.GetPipelineDefinitionResponse;
import software.amazon.awssdk.services.datapipeline.model.InternalServiceErrorException;
import software.amazon.awssdk.services.datapipeline.model.InvalidRequestException;
import software.amazon.awssdk.services.datapipeline.model.ListPipelinesRequest;
import software.amazon.awssdk.services.datapipeline.model.ListPipelinesResponse;
import software.amazon.awssdk.services.datapipeline.model.PipelineDeletedException;
import software.amazon.awssdk.services.datapipeline.model.PipelineNotFoundException;
import software.amazon.awssdk.services.datapipeline.model.PollForTaskRequest;
import software.amazon.awssdk.services.datapipeline.model.PollForTaskResponse;
import software.amazon.awssdk.services.datapipeline.model.PutPipelineDefinitionRequest;
import software.amazon.awssdk.services.datapipeline.model.PutPipelineDefinitionResponse;
import software.amazon.awssdk.services.datapipeline.model.QueryObjectsRequest;
import software.amazon.awssdk.services.datapipeline.model.QueryObjectsResponse;
import software.amazon.awssdk.services.datapipeline.model.RemoveTagsRequest;
import software.amazon.awssdk.services.datapipeline.model.RemoveTagsResponse;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskProgressRequest;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskProgressResponse;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskRunnerHeartbeatRequest;
import software.amazon.awssdk.services.datapipeline.model.ReportTaskRunnerHeartbeatResponse;
import software.amazon.awssdk.services.datapipeline.model.SetStatusRequest;
import software.amazon.awssdk.services.datapipeline.model.SetStatusResponse;
import software.amazon.awssdk.services.datapipeline.model.SetTaskStatusRequest;
import software.amazon.awssdk.services.datapipeline.model.SetTaskStatusResponse;
import software.amazon.awssdk.services.datapipeline.model.TaskNotFoundException;
import software.amazon.awssdk.services.datapipeline.model.ValidatePipelineDefinitionRequest;
import software.amazon.awssdk.services.datapipeline.model.ValidatePipelineDefinitionResponse;
import software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsIterable;
import software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesIterable;
import software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsIterable;
import software.amazon.awssdk.services.datapipeline.transform.ActivatePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.CreatePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DeactivatePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DeletePipelineRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DescribeObjectsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.DescribePipelinesRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.EvaluateExpressionRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.GetPipelineDefinitionRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ListPipelinesRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.PollForTaskRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.PutPipelineDefinitionRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.QueryObjectsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ReportTaskProgressRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ReportTaskRunnerHeartbeatRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.SetStatusRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.SetTaskStatusRequestMarshaller;
import software.amazon.awssdk.services.datapipeline.transform.ValidatePipelineDefinitionRequestMarshaller;

/**
 * Internal implementation of {@link DataPipelineClient}.
 *
 * @see DataPipelineClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDataPipelineClient implements DataPipelineClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDataPipelineClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation,
     * activation fails.
     * </p>
     * <p>
     * If you need to pause the pipeline to investigate an issue with a component, such as a data source or script, call
     * <a>DeactivatePipeline</a>.
     * </p>
     * <p>
     * To activate a finished pipeline, modify the end date for the pipeline and then activate it.
     * </p>
     *
     * @param activatePipelineRequest
     *        Contains the parameters for ActivatePipeline.
     * @return Result of the ActivatePipeline operation returned by the service.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.ActivatePipeline
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ActivatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ActivatePipelineResponse activatePipeline(ActivatePipelineRequest activatePipelineRequest)
            throws PipelineNotFoundException, PipelineDeletedException, InternalServiceErrorException, InvalidRequestException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ActivatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ActivatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ActivatePipelineRequest, ActivatePipelineResponse>()
                .withOperationName("ActivatePipeline").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(activatePipelineRequest)
                .withMarshaller(new ActivatePipelineRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds or modifies tags for the specified pipeline.
     * </p>
     *
     * @param addTagsRequest
     *        Contains the parameters for AddTags.
     * @return Result of the AddTags operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.AddTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/AddTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddTagsResponse addTags(AddTagsRequest addTagsRequest) throws InternalServiceErrorException, InvalidRequestException,
            PipelineNotFoundException, PipelineDeletedException, AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(addTagsRequest)
                .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates a new, empty pipeline. Use <a>PutPipelineDefinition</a> to populate the pipeline.
     * </p>
     *
     * @param createPipelineRequest
     *        Contains the parameters for CreatePipeline.
     * @return Result of the CreatePipeline operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.CreatePipeline
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/CreatePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreatePipelineResponse createPipeline(CreatePipelineRequest createPipelineRequest)
            throws InternalServiceErrorException, InvalidRequestException, AwsServiceException, SdkClientException,
            DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreatePipelineRequest, CreatePipelineResponse>()
                .withOperationName("CreatePipeline").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createPipelineRequest)
                .withMarshaller(new CreatePipelineRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deactivates the specified running pipeline. The pipeline is set to the <code>DEACTIVATING</code> state until the
     * deactivation process completes.
     * </p>
     * <p>
     * To resume a deactivated pipeline, use <a>ActivatePipeline</a>. By default, the pipeline resumes from the last
     * completed execution. Optionally, you can specify the date and time to resume the pipeline.
     * </p>
     *
     * @param deactivatePipelineRequest
     *        Contains the parameters for DeactivatePipeline.
     * @return Result of the DeactivatePipeline operation returned by the service.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.DeactivatePipeline
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DeactivatePipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeactivatePipelineResponse deactivatePipeline(DeactivatePipelineRequest deactivatePipelineRequest)
            throws PipelineNotFoundException, PipelineDeletedException, InternalServiceErrorException, InvalidRequestException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeactivatePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeactivatePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeactivatePipelineRequest, DeactivatePipelineResponse>()
                .withOperationName("DeactivatePipeline").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deactivatePipelineRequest)
                .withMarshaller(new DeactivatePipelineRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a pipeline, its pipeline definition, and its run history. AWS Data Pipeline attempts to cancel instances
     * associated with the pipeline that are currently being processed by task runners.
     * </p>
     * <p>
     * Deleting a pipeline cannot be undone. You cannot query or restore a deleted pipeline. To temporarily pause a
     * pipeline instead of deleting it, call <a>SetStatus</a> with the status set to <code>PAUSE</code> on individual
     * components. Components that are paused by <a>SetStatus</a> can be resumed.
     * </p>
     *
     * @param deletePipelineRequest
     *        Contains the parameters for DeletePipeline.
     * @return Result of the DeletePipeline operation returned by the service.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.DeletePipeline
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DeletePipeline" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeletePipelineResponse deletePipeline(DeletePipelineRequest deletePipelineRequest) throws PipelineNotFoundException,
            InternalServiceErrorException, InvalidRequestException, AwsServiceException, SdkClientException,
            DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePipelineResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePipelineResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeletePipelineRequest, DeletePipelineResponse>()
                .withOperationName("DeletePipeline").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deletePipelineRequest)
                .withMarshaller(new DeletePipelineRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the object definitions for a set of objects associated with the pipeline. Object definitions are composed of
     * a set of fields that define the properties of the object.
     * </p>
     *
     * @param describeObjectsRequest
     *        Contains the parameters for DescribeObjects.
     * @return Result of the DescribeObjects operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.DescribeObjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DescribeObjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeObjectsResponse describeObjects(DescribeObjectsRequest describeObjectsRequest)
            throws InternalServiceErrorException, InvalidRequestException, PipelineNotFoundException, PipelineDeletedException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeObjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeObjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribeObjectsRequest, DescribeObjectsResponse>()
                .withOperationName("DescribeObjects").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeObjectsRequest)
                .withMarshaller(new DescribeObjectsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the object definitions for a set of objects associated with the pipeline. Object definitions are composed of
     * a set of fields that define the properties of the object.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeObjects(software.amazon.awssdk.services.datapipeline.model.DescribeObjectsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsIterable responses = client.describeObjectsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsIterable responses = client
     *             .describeObjectsPaginator(request);
     *     for (software.amazon.awssdk.services.datapipeline.model.DescribeObjectsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.DescribeObjectsIterable responses = client.describeObjectsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeObjects(software.amazon.awssdk.services.datapipeline.model.DescribeObjectsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeObjectsRequest
     *        Contains the parameters for DescribeObjects.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.DescribeObjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DescribeObjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeObjectsIterable describeObjectsPaginator(DescribeObjectsRequest describeObjectsRequest)
            throws InternalServiceErrorException, InvalidRequestException, PipelineNotFoundException, PipelineDeletedException,
            AwsServiceException, SdkClientException, DataPipelineException {
        return new DescribeObjectsIterable(this, applyPaginatorUserAgent(describeObjectsRequest));
    }

    /**
     * <p>
     * Retrieves metadata about one or more pipelines. The information retrieved includes the name of the pipeline, the
     * pipeline identifier, its current state, and the user account that owns the pipeline. Using account credentials,
     * you can retrieve metadata about pipelines that you or your IAM users have created. If you are using an IAM user
     * account, you can retrieve metadata about only those pipelines for which you have read permissions.
     * </p>
     * <p>
     * To retrieve the full pipeline definition instead of metadata about the pipeline, call
     * <a>GetPipelineDefinition</a>.
     * </p>
     *
     * @param describePipelinesRequest
     *        Contains the parameters for DescribePipelines.
     * @return Result of the DescribePipelines operation returned by the service.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.DescribePipelines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/DescribePipelines" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribePipelinesResponse describePipelines(DescribePipelinesRequest describePipelinesRequest)
            throws PipelineNotFoundException, PipelineDeletedException, InternalServiceErrorException, InvalidRequestException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribePipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribePipelinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DescribePipelinesRequest, DescribePipelinesResponse>()
                .withOperationName("DescribePipelines").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describePipelinesRequest)
                .withMarshaller(new DescribePipelinesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Task runners call <code>EvaluateExpression</code> to evaluate a string in the context of the specified object.
     * For example, a task runner can evaluate SQL queries stored in Amazon S3.
     * </p>
     *
     * @param evaluateExpressionRequest
     *        Contains the parameters for EvaluateExpression.
     * @return Result of the EvaluateExpression operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws TaskNotFoundException
     *         The specified task was not found.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.EvaluateExpression
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/EvaluateExpression"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EvaluateExpressionResponse evaluateExpression(EvaluateExpressionRequest evaluateExpressionRequest)
            throws InternalServiceErrorException, TaskNotFoundException, InvalidRequestException, PipelineNotFoundException,
            PipelineDeletedException, AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EvaluateExpressionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EvaluateExpressionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<EvaluateExpressionRequest, EvaluateExpressionResponse>()
                .withOperationName("EvaluateExpression").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(evaluateExpressionRequest)
                .withMarshaller(new EvaluateExpressionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets the definition of the specified pipeline. You can call <code>GetPipelineDefinition</code> to retrieve the
     * pipeline definition that you provided using <a>PutPipelineDefinition</a>.
     * </p>
     *
     * @param getPipelineDefinitionRequest
     *        Contains the parameters for GetPipelineDefinition.
     * @return Result of the GetPipelineDefinition operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.GetPipelineDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/GetPipelineDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetPipelineDefinitionResponse getPipelineDefinition(GetPipelineDefinitionRequest getPipelineDefinitionRequest)
            throws InternalServiceErrorException, InvalidRequestException, PipelineNotFoundException, PipelineDeletedException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPipelineDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPipelineDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetPipelineDefinitionRequest, GetPipelineDefinitionResponse>()
                .withOperationName("GetPipelineDefinition").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getPipelineDefinitionRequest)
                .withMarshaller(new GetPipelineDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     * </p>
     *
     * @param listPipelinesRequest
     *        Contains the parameters for ListPipelines.
     * @return Result of the ListPipelines operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.ListPipelines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ListPipelines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPipelinesResponse listPipelines(ListPipelinesRequest listPipelinesRequest) throws InternalServiceErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPipelinesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListPipelinesRequest, ListPipelinesResponse>()
                .withOperationName("ListPipelines").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listPipelinesRequest)
                .withMarshaller(new ListPipelinesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPipelines(software.amazon.awssdk.services.datapipeline.model.ListPipelinesRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesIterable responses = client.listPipelinesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesIterable responses = client
     *             .listPipelinesPaginator(request);
     *     for (software.amazon.awssdk.services.datapipeline.model.ListPipelinesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.ListPipelinesIterable responses = client.listPipelinesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPipelines(software.amazon.awssdk.services.datapipeline.model.ListPipelinesRequest)} operation.</b>
     * </p>
     *
     * @param listPipelinesRequest
     *        Contains the parameters for ListPipelines.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.ListPipelines
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ListPipelines" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPipelinesIterable listPipelinesPaginator(ListPipelinesRequest listPipelinesRequest)
            throws InternalServiceErrorException, InvalidRequestException, AwsServiceException, SdkClientException,
            DataPipelineException {
        return new ListPipelinesIterable(this, applyPaginatorUserAgent(listPipelinesRequest));
    }

    /**
     * <p>
     * Task runners call <code>PollForTask</code> to receive a task to perform from AWS Data Pipeline. The task runner
     * specifies which tasks it can perform by setting a value for the <code>workerGroup</code> parameter. The task
     * returned can come from any of the pipelines that match the <code>workerGroup</code> value passed in by the task
     * runner and that was launched using the IAM user credentials specified by the task runner.
     * </p>
     * <p>
     * If tasks are ready in the work queue, <code>PollForTask</code> returns a response immediately. If no tasks are
     * available in the queue, <code>PollForTask</code> uses long-polling and holds on to a poll connection for up to a
     * 90 seconds, during which time the first newly scheduled task is handed to the task runner. To accomodate this,
     * set the socket timeout in your task runner to 90 seconds. The task runner should not call
     * <code>PollForTask</code> again on the same <code>workerGroup</code> until it receives a response, and this can
     * take up to 90 seconds.
     * </p>
     *
     * @param pollForTaskRequest
     *        Contains the parameters for PollForTask.
     * @return Result of the PollForTask operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws TaskNotFoundException
     *         The specified task was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.PollForTask
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/PollForTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PollForTaskResponse pollForTask(PollForTaskRequest pollForTaskRequest) throws InternalServiceErrorException,
            InvalidRequestException, TaskNotFoundException, AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PollForTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PollForTaskResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PollForTaskRequest, PollForTaskResponse>()
                .withOperationName("PollForTask").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(pollForTaskRequest)
                .withMarshaller(new PollForTaskRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Adds tasks, schedules, and preconditions to the specified pipeline. You can use
     * <code>PutPipelineDefinition</code> to populate a new pipeline.
     * </p>
     * <p>
     * <code>PutPipelineDefinition</code> also validates the configuration as it adds it to the pipeline. Changes to the
     * pipeline are saved unless one of the following three validation errors exists in the pipeline.
     * </p>
     * <ol>
     * <li>An object is missing a name or identifier field.</li>
     * <li>A string or reference field is empty.</li>
     * <li>The number of objects in the pipeline exceeds the maximum allowed objects.</li>
     * <li>The pipeline is in a FINISHED state.</li>
     * </ol>
     * <p>
     * Pipeline object definitions are passed to the <code>PutPipelineDefinition</code> action and returned by the
     * <a>GetPipelineDefinition</a> action.
     * </p>
     *
     * @param putPipelineDefinitionRequest
     *        Contains the parameters for PutPipelineDefinition.
     * @return Result of the PutPipelineDefinition operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.PutPipelineDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/PutPipelineDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutPipelineDefinitionResponse putPipelineDefinition(PutPipelineDefinitionRequest putPipelineDefinitionRequest)
            throws InternalServiceErrorException, InvalidRequestException, PipelineNotFoundException, PipelineDeletedException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutPipelineDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutPipelineDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<PutPipelineDefinitionRequest, PutPipelineDefinitionResponse>()
                .withOperationName("PutPipelineDefinition").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putPipelineDefinitionRequest)
                .withMarshaller(new PutPipelineDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     * </p>
     *
     * @param queryObjectsRequest
     *        Contains the parameters for QueryObjects.
     * @return Result of the QueryObjects operation returned by the service.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.QueryObjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/QueryObjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryObjectsResponse queryObjects(QueryObjectsRequest queryObjectsRequest) throws PipelineNotFoundException,
            PipelineDeletedException, InternalServiceErrorException, InvalidRequestException, AwsServiceException,
            SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryObjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryObjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<QueryObjectsRequest, QueryObjectsResponse>()
                .withOperationName("QueryObjects").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(queryObjectsRequest)
                .withMarshaller(new QueryObjectsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #queryObjects(software.amazon.awssdk.services.datapipeline.model.QueryObjectsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsIterable responses = client.queryObjectsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsIterable responses = client
     *             .queryObjectsPaginator(request);
     *     for (software.amazon.awssdk.services.datapipeline.model.QueryObjectsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datapipeline.paginators.QueryObjectsIterable responses = client.queryObjectsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #queryObjects(software.amazon.awssdk.services.datapipeline.model.QueryObjectsRequest)} operation.</b>
     * </p>
     *
     * @param queryObjectsRequest
     *        Contains the parameters for QueryObjects.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.QueryObjects
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/QueryObjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryObjectsIterable queryObjectsPaginator(QueryObjectsRequest queryObjectsRequest) throws PipelineNotFoundException,
            PipelineDeletedException, InternalServiceErrorException, InvalidRequestException, AwsServiceException,
            SdkClientException, DataPipelineException {
        return new QueryObjectsIterable(this, applyPaginatorUserAgent(queryObjectsRequest));
    }

    /**
     * <p>
     * Removes existing tags from the specified pipeline.
     * </p>
     *
     * @param removeTagsRequest
     *        Contains the parameters for RemoveTags.
     * @return Result of the RemoveTags operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.RemoveTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/RemoveTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemoveTagsResponse removeTags(RemoveTagsRequest removeTagsRequest) throws InternalServiceErrorException,
            InvalidRequestException, PipelineNotFoundException, PipelineDeletedException, AwsServiceException,
            SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>()
                .withOperationName("RemoveTags").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(removeTagsRequest)
                .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Task runners call <code>ReportTaskProgress</code> when assigned a task to acknowledge that it has the task. If
     * the web service does not receive this acknowledgement within 2 minutes, it assigns the task in a subsequent
     * <a>PollForTask</a> call. After this initial acknowledgement, the task runner only needs to report progress every
     * 15 minutes to maintain its ownership of the task. You can change this reporting time from 15 minutes by
     * specifying a <code>reportProgressTimeout</code> field in your pipeline.
     * </p>
     * <p>
     * If a task runner does not report its status after 5 minutes, AWS Data Pipeline assumes that the task runner is
     * unable to process the task and reassigns the task in a subsequent response to <a>PollForTask</a>. Task runners
     * should call <code>ReportTaskProgress</code> every 60 seconds.
     * </p>
     *
     * @param reportTaskProgressRequest
     *        Contains the parameters for ReportTaskProgress.
     * @return Result of the ReportTaskProgress operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws TaskNotFoundException
     *         The specified task was not found.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.ReportTaskProgress
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ReportTaskProgress"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ReportTaskProgressResponse reportTaskProgress(ReportTaskProgressRequest reportTaskProgressRequest)
            throws InternalServiceErrorException, InvalidRequestException, TaskNotFoundException, PipelineNotFoundException,
            PipelineDeletedException, AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReportTaskProgressResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ReportTaskProgressResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ReportTaskProgressRequest, ReportTaskProgressResponse>()
                .withOperationName("ReportTaskProgress").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(reportTaskProgressRequest)
                .withMarshaller(new ReportTaskProgressRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Task runners call <code>ReportTaskRunnerHeartbeat</code> every 15 minutes to indicate that they are operational.
     * If the AWS Data Pipeline Task Runner is launched on a resource managed by AWS Data Pipeline, the web service can
     * use this call to detect when the task runner application has failed and restart a new instance.
     * </p>
     *
     * @param reportTaskRunnerHeartbeatRequest
     *        Contains the parameters for ReportTaskRunnerHeartbeat.
     * @return Result of the ReportTaskRunnerHeartbeat operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.ReportTaskRunnerHeartbeat
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ReportTaskRunnerHeartbeat"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ReportTaskRunnerHeartbeatResponse reportTaskRunnerHeartbeat(
            ReportTaskRunnerHeartbeatRequest reportTaskRunnerHeartbeatRequest) throws InternalServiceErrorException,
            InvalidRequestException, AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReportTaskRunnerHeartbeatResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ReportTaskRunnerHeartbeatResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ReportTaskRunnerHeartbeatRequest, ReportTaskRunnerHeartbeatResponse>()
                        .withOperationName("ReportTaskRunnerHeartbeat").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(reportTaskRunnerHeartbeatRequest)
                        .withMarshaller(new ReportTaskRunnerHeartbeatRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Requests that the status of the specified physical or logical pipeline objects be updated in the specified
     * pipeline. This update might not occur immediately, but is eventually consistent. The status that can be set
     * depends on the type of object (for example, DataNode or Activity). You cannot perform this operation on
     * <code>FINISHED</code> pipelines and attempting to do so returns <code>InvalidRequestException</code>.
     * </p>
     *
     * @param setStatusRequest
     *        Contains the parameters for SetStatus.
     * @return Result of the SetStatus operation returned by the service.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.SetStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/SetStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetStatusResponse setStatus(SetStatusRequest setStatusRequest) throws PipelineNotFoundException,
            PipelineDeletedException, InternalServiceErrorException, InvalidRequestException, AwsServiceException,
            SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SetStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SetStatusRequest, SetStatusResponse>()
                .withOperationName("SetStatus").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setStatusRequest)
                .withMarshaller(new SetStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Task runners call <code>SetTaskStatus</code> to notify AWS Data Pipeline that a task is completed and provide
     * information about the final status. A task runner makes this call regardless of whether the task was sucessful. A
     * task runner does not need to call <code>SetTaskStatus</code> for tasks that are canceled by the web service
     * during a call to <a>ReportTaskProgress</a>.
     * </p>
     *
     * @param setTaskStatusRequest
     *        Contains the parameters for SetTaskStatus.
     * @return Result of the SetTaskStatus operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws TaskNotFoundException
     *         The specified task was not found.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.SetTaskStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/SetTaskStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetTaskStatusResponse setTaskStatus(SetTaskStatusRequest setTaskStatusRequest) throws InternalServiceErrorException,
            TaskNotFoundException, InvalidRequestException, PipelineNotFoundException, PipelineDeletedException,
            AwsServiceException, SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SetTaskStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SetTaskStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SetTaskStatusRequest, SetTaskStatusResponse>()
                .withOperationName("SetTaskStatus").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setTaskStatusRequest)
                .withMarshaller(new SetTaskStatusRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Validates the specified pipeline definition to ensure that it is well formed and can be run without error.
     * </p>
     *
     * @param validatePipelineDefinitionRequest
     *        Contains the parameters for ValidatePipelineDefinition.
     * @return Result of the ValidatePipelineDefinition operation returned by the service.
     * @throws InternalServiceErrorException
     *         An internal service error occurred.
     * @throws InvalidRequestException
     *         The request was not valid. Verify that your request was properly formatted, that the signature was
     *         generated with the correct credentials, and that you haven't exceeded any of the service limits for your
     *         account.
     * @throws PipelineNotFoundException
     *         The specified pipeline was not found. Verify that you used the correct user and account identifiers.
     * @throws PipelineDeletedException
     *         The specified pipeline has been deleted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DataPipelineException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DataPipelineClient.ValidatePipelineDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/datapipeline-2012-10-29/ValidatePipelineDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ValidatePipelineDefinitionResponse validatePipelineDefinition(
            ValidatePipelineDefinitionRequest validatePipelineDefinitionRequest) throws InternalServiceErrorException,
            InvalidRequestException, PipelineNotFoundException, PipelineDeletedException, AwsServiceException,
            SdkClientException, DataPipelineException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidatePipelineDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ValidatePipelineDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ValidatePipelineDefinitionRequest, ValidatePipelineDefinitionResponse>()
                        .withOperationName("ValidatePipelineDefinition").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(validatePipelineDefinitionRequest)
                        .withMarshaller(new ValidatePipelineDefinitionRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DataPipelineException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TaskNotFoundException")
                                .exceptionBuilderSupplier(TaskNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineDeletedException")
                                .exceptionBuilderSupplier(PipelineDeletedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PipelineNotFoundException")
                                .exceptionBuilderSupplier(PipelineNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends DataPipelineRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
