/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datapipeline.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for QueryObjects.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryObjectsRequest extends DataPipelineRequest implements
        ToCopyableBuilder<QueryObjectsRequest.Builder, QueryObjectsRequest> {
    private static final SdkField<String> PIPELINE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(QueryObjectsRequest::pipelineId)).setter(setter(Builder::pipelineId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineId").build()).build();

    private static final SdkField<Query> QUERY_FIELD = SdkField.<Query> builder(MarshallingType.SDK_POJO)
            .getter(getter(QueryObjectsRequest::query)).setter(setter(Builder::query)).constructor(Query::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("query").build()).build();

    private static final SdkField<String> SPHERE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(QueryObjectsRequest::sphere)).setter(setter(Builder::sphere))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sphere").build()).build();

    private static final SdkField<String> MARKER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(QueryObjectsRequest::marker)).setter(setter(Builder::marker))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("marker").build()).build();

    private static final SdkField<Integer> LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(QueryObjectsRequest::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("limit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PIPELINE_ID_FIELD,
            QUERY_FIELD, SPHERE_FIELD, MARKER_FIELD, LIMIT_FIELD));

    private final String pipelineId;

    private final Query query;

    private final String sphere;

    private final String marker;

    private final Integer limit;

    private QueryObjectsRequest(BuilderImpl builder) {
        super(builder);
        this.pipelineId = builder.pipelineId;
        this.query = builder.query;
        this.sphere = builder.sphere;
        this.marker = builder.marker;
        this.limit = builder.limit;
    }

    /**
     * <p>
     * The ID of the pipeline.
     * </p>
     * 
     * @return The ID of the pipeline.
     */
    public String pipelineId() {
        return pipelineId;
    }

    /**
     * <p>
     * The query that defines the objects to be returned. The <code>Query</code> object can contain a maximum of ten
     * selectors. The conditions in the query are limited to top-level String fields in the object. These filters can be
     * applied to components, instances, and attempts.
     * </p>
     * 
     * @return The query that defines the objects to be returned. The <code>Query</code> object can contain a maximum of
     *         ten selectors. The conditions in the query are limited to top-level String fields in the object. These
     *         filters can be applied to components, instances, and attempts.
     */
    public Query query() {
        return query;
    }

    /**
     * <p>
     * Indicates whether the query applies to components or instances. The possible values are: <code>COMPONENT</code>,
     * <code>INSTANCE</code>, and <code>ATTEMPT</code>.
     * </p>
     * 
     * @return Indicates whether the query applies to components or instances. The possible values are:
     *         <code>COMPONENT</code>, <code>INSTANCE</code>, and <code>ATTEMPT</code>.
     */
    public String sphere() {
        return sphere;
    }

    /**
     * <p>
     * The starting point for the results to be returned. For the first call, this value should be empty. As long as
     * there are more results, continue to call <code>QueryObjects</code> with the marker value from the previous call
     * to retrieve the next set of results.
     * </p>
     * 
     * @return The starting point for the results to be returned. For the first call, this value should be empty. As
     *         long as there are more results, continue to call <code>QueryObjects</code> with the marker value from the
     *         previous call to retrieve the next set of results.
     */
    public String marker() {
        return marker;
    }

    /**
     * <p>
     * The maximum number of object names that <code>QueryObjects</code> will return in a single call. The default value
     * is 100.
     * </p>
     * 
     * @return The maximum number of object names that <code>QueryObjects</code> will return in a single call. The
     *         default value is 100.
     */
    public Integer limit() {
        return limit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(pipelineId());
        hashCode = 31 * hashCode + Objects.hashCode(query());
        hashCode = 31 * hashCode + Objects.hashCode(sphere());
        hashCode = 31 * hashCode + Objects.hashCode(marker());
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryObjectsRequest)) {
            return false;
        }
        QueryObjectsRequest other = (QueryObjectsRequest) obj;
        return Objects.equals(pipelineId(), other.pipelineId()) && Objects.equals(query(), other.query())
                && Objects.equals(sphere(), other.sphere()) && Objects.equals(marker(), other.marker())
                && Objects.equals(limit(), other.limit());
    }

    @Override
    public String toString() {
        return ToString.builder("QueryObjectsRequest").add("PipelineId", pipelineId()).add("Query", query())
                .add("Sphere", sphere()).add("Marker", marker()).add("Limit", limit()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "pipelineId":
            return Optional.ofNullable(clazz.cast(pipelineId()));
        case "query":
            return Optional.ofNullable(clazz.cast(query()));
        case "sphere":
            return Optional.ofNullable(clazz.cast(sphere()));
        case "marker":
            return Optional.ofNullable(clazz.cast(marker()));
        case "limit":
            return Optional.ofNullable(clazz.cast(limit()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<QueryObjectsRequest, T> g) {
        return obj -> g.apply((QueryObjectsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataPipelineRequest.Builder, SdkPojo, CopyableBuilder<Builder, QueryObjectsRequest> {
        /**
         * <p>
         * The ID of the pipeline.
         * </p>
         * 
         * @param pipelineId
         *        The ID of the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineId(String pipelineId);

        /**
         * <p>
         * The query that defines the objects to be returned. The <code>Query</code> object can contain a maximum of ten
         * selectors. The conditions in the query are limited to top-level String fields in the object. These filters
         * can be applied to components, instances, and attempts.
         * </p>
         * 
         * @param query
         *        The query that defines the objects to be returned. The <code>Query</code> object can contain a maximum
         *        of ten selectors. The conditions in the query are limited to top-level String fields in the object.
         *        These filters can be applied to components, instances, and attempts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder query(Query query);

        /**
         * <p>
         * The query that defines the objects to be returned. The <code>Query</code> object can contain a maximum of ten
         * selectors. The conditions in the query are limited to top-level String fields in the object. These filters
         * can be applied to components, instances, and attempts.
         * </p>
         * This is a convenience that creates an instance of the {@link Query.Builder} avoiding the need to create one
         * manually via {@link Query#builder()}.
         *
         * When the {@link Consumer} completes, {@link Query.Builder#build()} is called immediately and its result is
         * passed to {@link #query(Query)}.
         * 
         * @param query
         *        a consumer that will call methods on {@link Query.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #query(Query)
         */
        default Builder query(Consumer<Query.Builder> query) {
            return query(Query.builder().applyMutation(query).build());
        }

        /**
         * <p>
         * Indicates whether the query applies to components or instances. The possible values are:
         * <code>COMPONENT</code>, <code>INSTANCE</code>, and <code>ATTEMPT</code>.
         * </p>
         * 
         * @param sphere
         *        Indicates whether the query applies to components or instances. The possible values are:
         *        <code>COMPONENT</code>, <code>INSTANCE</code>, and <code>ATTEMPT</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sphere(String sphere);

        /**
         * <p>
         * The starting point for the results to be returned. For the first call, this value should be empty. As long as
         * there are more results, continue to call <code>QueryObjects</code> with the marker value from the previous
         * call to retrieve the next set of results.
         * </p>
         * 
         * @param marker
         *        The starting point for the results to be returned. For the first call, this value should be empty. As
         *        long as there are more results, continue to call <code>QueryObjects</code> with the marker value from
         *        the previous call to retrieve the next set of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder marker(String marker);

        /**
         * <p>
         * The maximum number of object names that <code>QueryObjects</code> will return in a single call. The default
         * value is 100.
         * </p>
         * 
         * @param limit
         *        The maximum number of object names that <code>QueryObjects</code> will return in a single call. The
         *        default value is 100.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Integer limit);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataPipelineRequest.BuilderImpl implements Builder {
        private String pipelineId;

        private Query query;

        private String sphere;

        private String marker;

        private Integer limit;

        private BuilderImpl() {
        }

        private BuilderImpl(QueryObjectsRequest model) {
            super(model);
            pipelineId(model.pipelineId);
            query(model.query);
            sphere(model.sphere);
            marker(model.marker);
            limit(model.limit);
        }

        public final String getPipelineId() {
            return pipelineId;
        }

        @Override
        public final Builder pipelineId(String pipelineId) {
            this.pipelineId = pipelineId;
            return this;
        }

        public final void setPipelineId(String pipelineId) {
            this.pipelineId = pipelineId;
        }

        public final Query.Builder getQuery() {
            return query != null ? query.toBuilder() : null;
        }

        @Override
        public final Builder query(Query query) {
            this.query = query;
            return this;
        }

        public final void setQuery(Query.BuilderImpl query) {
            this.query = query != null ? query.build() : null;
        }

        public final String getSphere() {
            return sphere;
        }

        @Override
        public final Builder sphere(String sphere) {
            this.sphere = sphere;
            return this;
        }

        public final void setSphere(String sphere) {
            this.sphere = sphere;
        }

        public final String getMarker() {
            return marker;
        }

        @Override
        public final Builder marker(String marker) {
            this.marker = marker;
            return this;
        }

        public final void setMarker(String marker) {
            this.marker = marker;
        }

        public final Integer getLimit() {
            return limit;
        }

        @Override
        public final Builder limit(Integer limit) {
            this.limit = limit;
            return this;
        }

        public final void setLimit(Integer limit) {
            this.limit = limit;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public QueryObjectsRequest build() {
            return new QueryObjectsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
