/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * DescribeTaskResponse
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeTaskResponse extends DataSyncResponse implements
        ToCopyableBuilder<DescribeTaskResponse.Builder, DescribeTaskResponse> {
    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::taskArn)).setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> CURRENT_TASK_EXECUTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::currentTaskExecutionArn)).setter(setter(Builder::currentTaskExecutionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CurrentTaskExecutionArn").build())
            .build();

    private static final SdkField<String> SOURCE_LOCATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::sourceLocationArn)).setter(setter(Builder::sourceLocationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceLocationArn").build()).build();

    private static final SdkField<String> DESTINATION_LOCATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::destinationLocationArn)).setter(setter(Builder::destinationLocationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationLocationArn").build())
            .build();

    private static final SdkField<String> CLOUD_WATCH_LOG_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::cloudWatchLogGroupArn)).setter(setter(Builder::cloudWatchLogGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchLogGroupArn").build())
            .build();

    private static final SdkField<List<String>> SOURCE_NETWORK_INTERFACE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DescribeTaskResponse::sourceNetworkInterfaceArns))
            .setter(setter(Builder::sourceNetworkInterfaceArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceNetworkInterfaceArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> DESTINATION_NETWORK_INTERFACE_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DescribeTaskResponse::destinationNetworkInterfaceArns))
            .setter(setter(Builder::destinationNetworkInterfaceArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationNetworkInterfaceArns")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Options> OPTIONS_FIELD = SdkField.<Options> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeTaskResponse::options)).setter(setter(Builder::options)).constructor(Options::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Options").build()).build();

    private static final SdkField<List<FilterRule>> EXCLUDES_FIELD = SdkField
            .<List<FilterRule>> builder(MarshallingType.LIST)
            .getter(getter(DescribeTaskResponse::excludes))
            .setter(setter(Builder::excludes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Excludes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FilterRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(FilterRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TaskSchedule> SCHEDULE_FIELD = SdkField.<TaskSchedule> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeTaskResponse::schedule)).setter(setter(Builder::schedule)).constructor(TaskSchedule::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedule").build()).build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::errorCode)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<String> ERROR_DETAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeTaskResponse::errorDetail)).setter(setter(Builder::errorDetail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorDetail").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DescribeTaskResponse::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TASK_ARN_FIELD, STATUS_FIELD,
            NAME_FIELD, CURRENT_TASK_EXECUTION_ARN_FIELD, SOURCE_LOCATION_ARN_FIELD, DESTINATION_LOCATION_ARN_FIELD,
            CLOUD_WATCH_LOG_GROUP_ARN_FIELD, SOURCE_NETWORK_INTERFACE_ARNS_FIELD, DESTINATION_NETWORK_INTERFACE_ARNS_FIELD,
            OPTIONS_FIELD, EXCLUDES_FIELD, SCHEDULE_FIELD, ERROR_CODE_FIELD, ERROR_DETAIL_FIELD, CREATION_TIME_FIELD));

    private final String taskArn;

    private final String status;

    private final String name;

    private final String currentTaskExecutionArn;

    private final String sourceLocationArn;

    private final String destinationLocationArn;

    private final String cloudWatchLogGroupArn;

    private final List<String> sourceNetworkInterfaceArns;

    private final List<String> destinationNetworkInterfaceArns;

    private final Options options;

    private final List<FilterRule> excludes;

    private final TaskSchedule schedule;

    private final String errorCode;

    private final String errorDetail;

    private final Instant creationTime;

    private DescribeTaskResponse(BuilderImpl builder) {
        super(builder);
        this.taskArn = builder.taskArn;
        this.status = builder.status;
        this.name = builder.name;
        this.currentTaskExecutionArn = builder.currentTaskExecutionArn;
        this.sourceLocationArn = builder.sourceLocationArn;
        this.destinationLocationArn = builder.destinationLocationArn;
        this.cloudWatchLogGroupArn = builder.cloudWatchLogGroupArn;
        this.sourceNetworkInterfaceArns = builder.sourceNetworkInterfaceArns;
        this.destinationNetworkInterfaceArns = builder.destinationNetworkInterfaceArns;
        this.options = builder.options;
        this.excludes = builder.excludes;
        this.schedule = builder.schedule;
        this.errorCode = builder.errorCode;
        this.errorDetail = builder.errorDetail;
        this.creationTime = builder.creationTime;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the task that was described.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the task that was described.
     */
    public String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The status of the task that was described.
     * </p>
     * <p>
     * For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS DataSync
     * User Guide.</i>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TaskStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task that was described.</p>
     *         <p>
     *         For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS
     *         DataSync User Guide.</i>
     * @see TaskStatus
     */
    public TaskStatus status() {
        return TaskStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the task that was described.
     * </p>
     * <p>
     * For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS DataSync
     * User Guide.</i>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link TaskStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task that was described.</p>
     *         <p>
     *         For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS
     *         DataSync User Guide.</i>
     * @see TaskStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The name of the task that was described.
     * </p>
     * 
     * @return The name of the task that was described.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the task execution that is syncing files.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the task execution that is syncing files.
     */
    public String currentTaskExecutionArn() {
        return currentTaskExecutionArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the source file system's location.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the source file system's location.
     */
    public String sourceLocationArn() {
        return sourceLocationArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the AWS storage resource's location.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the AWS storage resource's location.
     */
    public String destinationLocationArn() {
        return destinationLocationArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Amazon CloudWatch log group that was used to monitor and log events in the
     * task.
     * </p>
     * <p>
     * For more information on these groups, see Working with Log Groups and Log Streams in the <i>Amazon CloudWatch
     * User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Amazon CloudWatch log group that was used to monitor and log events
     *         in the task.</p>
     *         <p>
     *         For more information on these groups, see Working with Log Groups and Log Streams in the <i>Amazon
     *         CloudWatch User Guide</i>.
     */
    public String cloudWatchLogGroupArn() {
        return cloudWatchLogGroupArn;
    }

    /**
     * Returns true if the SourceNetworkInterfaceArns property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasSourceNetworkInterfaceArns() {
        return sourceNetworkInterfaceArns != null && !(sourceNetworkInterfaceArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the source ENIs (Elastic Network Interface) that was created for your subnet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSourceNetworkInterfaceArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the source ENIs (Elastic Network Interface) that was created for your
     *         subnet.
     */
    public List<String> sourceNetworkInterfaceArns() {
        return sourceNetworkInterfaceArns;
    }

    /**
     * Returns true if the DestinationNetworkInterfaceArns property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasDestinationNetworkInterfaceArns() {
        return destinationNetworkInterfaceArns != null && !(destinationNetworkInterfaceArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the destination ENIs (Elastic Network Interface) that was created for your
     * subnet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDestinationNetworkInterfaceArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the destination ENIs (Elastic Network Interface) that was created for
     *         your subnet.
     */
    public List<String> destinationNetworkInterfaceArns() {
        return destinationNetworkInterfaceArns;
    }

    /**
     * <p>
     * The set of configuration options that control the behavior of a single execution of the task that occurs when you
     * call <code>StartTaskExecution</code>. You can configure these options to preserve metadata such as user ID (UID)
     * and group (GID), file permissions, data integrity verification, and so on.
     * </p>
     * <p>
     * For each individual task execution, you can override these options by specifying the overriding
     * <code>OverrideOptions</code> value to operation.
     * </p>
     * 
     * @return The set of configuration options that control the behavior of a single execution of the task that occurs
     *         when you call <code>StartTaskExecution</code>. You can configure these options to preserve metadata such
     *         as user ID (UID) and group (GID), file permissions, data integrity verification, and so on.</p>
     *         <p>
     *         For each individual task execution, you can override these options by specifying the overriding
     *         <code>OverrideOptions</code> value to operation.
     */
    public Options options() {
        return options;
    }

    /**
     * Returns true if the Excludes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasExcludes() {
        return excludes != null && !(excludes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of filter rules that determines which files to exclude from a task. The list should contain a single
     * filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that is, a pipe), for
     * example: <code>"/folder1|/folder2"</code>
     * </p>
     * <p>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasExcludes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of filter rules that determines which files to exclude from a task. The list should contain a
     *         single filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that
     *         is, a pipe), for example: <code>"/folder1|/folder2"</code> </p>
     *         <p>
     */
    public List<FilterRule> excludes() {
        return excludes;
    }

    /**
     * <p>
     * The schedule used to periodically transfer files from a source to a destination location.
     * </p>
     * 
     * @return The schedule used to periodically transfer files from a source to a destination location.
     */
    public TaskSchedule schedule() {
        return schedule;
    }

    /**
     * <p>
     * Errors that AWS DataSync encountered during execution of the task. You can use this error code to help
     * troubleshoot issues.
     * </p>
     * 
     * @return Errors that AWS DataSync encountered during execution of the task. You can use this error code to help
     *         troubleshoot issues.
     */
    public String errorCode() {
        return errorCode;
    }

    /**
     * <p>
     * Detailed description of an error that was encountered during the task execution. You can use this information to
     * help troubleshoot issues.
     * </p>
     * 
     * @return Detailed description of an error that was encountered during the task execution. You can use this
     *         information to help troubleshoot issues.
     */
    public String errorDetail() {
        return errorDetail;
    }

    /**
     * <p>
     * The time that the task was created.
     * </p>
     * 
     * @return The time that the task was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(currentTaskExecutionArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceLocationArn());
        hashCode = 31 * hashCode + Objects.hashCode(destinationLocationArn());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchLogGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceNetworkInterfaceArns());
        hashCode = 31 * hashCode + Objects.hashCode(destinationNetworkInterfaceArns());
        hashCode = 31 * hashCode + Objects.hashCode(options());
        hashCode = 31 * hashCode + Objects.hashCode(excludes());
        hashCode = 31 * hashCode + Objects.hashCode(schedule());
        hashCode = 31 * hashCode + Objects.hashCode(errorCode());
        hashCode = 31 * hashCode + Objects.hashCode(errorDetail());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeTaskResponse)) {
            return false;
        }
        DescribeTaskResponse other = (DescribeTaskResponse) obj;
        return Objects.equals(taskArn(), other.taskArn()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(name(), other.name())
                && Objects.equals(currentTaskExecutionArn(), other.currentTaskExecutionArn())
                && Objects.equals(sourceLocationArn(), other.sourceLocationArn())
                && Objects.equals(destinationLocationArn(), other.destinationLocationArn())
                && Objects.equals(cloudWatchLogGroupArn(), other.cloudWatchLogGroupArn())
                && Objects.equals(sourceNetworkInterfaceArns(), other.sourceNetworkInterfaceArns())
                && Objects.equals(destinationNetworkInterfaceArns(), other.destinationNetworkInterfaceArns())
                && Objects.equals(options(), other.options()) && Objects.equals(excludes(), other.excludes())
                && Objects.equals(schedule(), other.schedule()) && Objects.equals(errorCode(), other.errorCode())
                && Objects.equals(errorDetail(), other.errorDetail()) && Objects.equals(creationTime(), other.creationTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DescribeTaskResponse").add("TaskArn", taskArn()).add("Status", statusAsString())
                .add("Name", name()).add("CurrentTaskExecutionArn", currentTaskExecutionArn())
                .add("SourceLocationArn", sourceLocationArn()).add("DestinationLocationArn", destinationLocationArn())
                .add("CloudWatchLogGroupArn", cloudWatchLogGroupArn())
                .add("SourceNetworkInterfaceArns", sourceNetworkInterfaceArns())
                .add("DestinationNetworkInterfaceArns", destinationNetworkInterfaceArns()).add("Options", options())
                .add("Excludes", excludes()).add("Schedule", schedule()).add("ErrorCode", errorCode())
                .add("ErrorDetail", errorDetail()).add("CreationTime", creationTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TaskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "CurrentTaskExecutionArn":
            return Optional.ofNullable(clazz.cast(currentTaskExecutionArn()));
        case "SourceLocationArn":
            return Optional.ofNullable(clazz.cast(sourceLocationArn()));
        case "DestinationLocationArn":
            return Optional.ofNullable(clazz.cast(destinationLocationArn()));
        case "CloudWatchLogGroupArn":
            return Optional.ofNullable(clazz.cast(cloudWatchLogGroupArn()));
        case "SourceNetworkInterfaceArns":
            return Optional.ofNullable(clazz.cast(sourceNetworkInterfaceArns()));
        case "DestinationNetworkInterfaceArns":
            return Optional.ofNullable(clazz.cast(destinationNetworkInterfaceArns()));
        case "Options":
            return Optional.ofNullable(clazz.cast(options()));
        case "Excludes":
            return Optional.ofNullable(clazz.cast(excludes()));
        case "Schedule":
            return Optional.ofNullable(clazz.cast(schedule()));
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCode()));
        case "ErrorDetail":
            return Optional.ofNullable(clazz.cast(errorDetail()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeTaskResponse, T> g) {
        return obj -> g.apply((DescribeTaskResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncResponse.Builder, SdkPojo, CopyableBuilder<Builder, DescribeTaskResponse> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the task that was described.
         * </p>
         * 
         * @param taskArn
         *        The Amazon Resource Name (ARN) of the task that was described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The status of the task that was described.
         * </p>
         * <p>
         * For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS
         * DataSync User Guide.</i>
         * </p>
         * 
         * @param status
         *        The status of the task that was described.</p>
         *        <p>
         *        For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS
         *        DataSync User Guide.</i>
         * @see TaskStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the task that was described.
         * </p>
         * <p>
         * For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS
         * DataSync User Guide.</i>
         * </p>
         * 
         * @param status
         *        The status of the task that was described.</p>
         *        <p>
         *        For detailed information about task execution statuses, see Understanding Task Statuses in the <i>AWS
         *        DataSync User Guide.</i>
         * @see TaskStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskStatus
         */
        Builder status(TaskStatus status);

        /**
         * <p>
         * The name of the task that was described.
         * </p>
         * 
         * @param name
         *        The name of the task that was described.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the task execution that is syncing files.
         * </p>
         * 
         * @param currentTaskExecutionArn
         *        The Amazon Resource Name (ARN) of the task execution that is syncing files.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentTaskExecutionArn(String currentTaskExecutionArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the source file system's location.
         * </p>
         * 
         * @param sourceLocationArn
         *        The Amazon Resource Name (ARN) of the source file system's location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceLocationArn(String sourceLocationArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the AWS storage resource's location.
         * </p>
         * 
         * @param destinationLocationArn
         *        The Amazon Resource Name (ARN) of the AWS storage resource's location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationLocationArn(String destinationLocationArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Amazon CloudWatch log group that was used to monitor and log events in
         * the task.
         * </p>
         * <p>
         * For more information on these groups, see Working with Log Groups and Log Streams in the <i>Amazon CloudWatch
         * User Guide</i>.
         * </p>
         * 
         * @param cloudWatchLogGroupArn
         *        The Amazon Resource Name (ARN) of the Amazon CloudWatch log group that was used to monitor and log
         *        events in the task.</p>
         *        <p>
         *        For more information on these groups, see Working with Log Groups and Log Streams in the <i>Amazon
         *        CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLogGroupArn(String cloudWatchLogGroupArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the source ENIs (Elastic Network Interface) that was created for your
         * subnet.
         * </p>
         * 
         * @param sourceNetworkInterfaceArns
         *        The Amazon Resource Name (ARN) of the source ENIs (Elastic Network Interface) that was created for
         *        your subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceNetworkInterfaceArns(Collection<String> sourceNetworkInterfaceArns);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the source ENIs (Elastic Network Interface) that was created for your
         * subnet.
         * </p>
         * 
         * @param sourceNetworkInterfaceArns
         *        The Amazon Resource Name (ARN) of the source ENIs (Elastic Network Interface) that was created for
         *        your subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceNetworkInterfaceArns(String... sourceNetworkInterfaceArns);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the destination ENIs (Elastic Network Interface) that was created for your
         * subnet.
         * </p>
         * 
         * @param destinationNetworkInterfaceArns
         *        The Amazon Resource Name (ARN) of the destination ENIs (Elastic Network Interface) that was created
         *        for your subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationNetworkInterfaceArns(Collection<String> destinationNetworkInterfaceArns);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the destination ENIs (Elastic Network Interface) that was created for your
         * subnet.
         * </p>
         * 
         * @param destinationNetworkInterfaceArns
         *        The Amazon Resource Name (ARN) of the destination ENIs (Elastic Network Interface) that was created
         *        for your subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationNetworkInterfaceArns(String... destinationNetworkInterfaceArns);

        /**
         * <p>
         * The set of configuration options that control the behavior of a single execution of the task that occurs when
         * you call <code>StartTaskExecution</code>. You can configure these options to preserve metadata such as user
         * ID (UID) and group (GID), file permissions, data integrity verification, and so on.
         * </p>
         * <p>
         * For each individual task execution, you can override these options by specifying the overriding
         * <code>OverrideOptions</code> value to operation.
         * </p>
         * 
         * @param options
         *        The set of configuration options that control the behavior of a single execution of the task that
         *        occurs when you call <code>StartTaskExecution</code>. You can configure these options to preserve
         *        metadata such as user ID (UID) and group (GID), file permissions, data integrity verification, and so
         *        on.</p>
         *        <p>
         *        For each individual task execution, you can override these options by specifying the overriding
         *        <code>OverrideOptions</code> value to operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Options options);

        /**
         * <p>
         * The set of configuration options that control the behavior of a single execution of the task that occurs when
         * you call <code>StartTaskExecution</code>. You can configure these options to preserve metadata such as user
         * ID (UID) and group (GID), file permissions, data integrity verification, and so on.
         * </p>
         * <p>
         * For each individual task execution, you can override these options by specifying the overriding
         * <code>OverrideOptions</code> value to operation.
         * </p>
         * This is a convenience that creates an instance of the {@link Options.Builder} avoiding the need to create one
         * manually via {@link Options#builder()}.
         *
         * When the {@link Consumer} completes, {@link Options.Builder#build()} is called immediately and its result is
         * passed to {@link #options(Options)}.
         * 
         * @param options
         *        a consumer that will call methods on {@link Options.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #options(Options)
         */
        default Builder options(Consumer<Options.Builder> options) {
            return options(Options.builder().applyMutation(options).build());
        }

        /**
         * <p>
         * A list of filter rules that determines which files to exclude from a task. The list should contain a single
         * filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that is, a pipe),
         * for example: <code>"/folder1|/folder2"</code>
         * </p>
         * <p>
         * </p>
         * 
         * @param excludes
         *        A list of filter rules that determines which files to exclude from a task. The list should contain a
         *        single filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that
         *        is, a pipe), for example: <code>"/folder1|/folder2"</code> </p>
         *        <p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludes(Collection<FilterRule> excludes);

        /**
         * <p>
         * A list of filter rules that determines which files to exclude from a task. The list should contain a single
         * filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that is, a pipe),
         * for example: <code>"/folder1|/folder2"</code>
         * </p>
         * <p>
         * </p>
         * 
         * @param excludes
         *        A list of filter rules that determines which files to exclude from a task. The list should contain a
         *        single filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that
         *        is, a pipe), for example: <code>"/folder1|/folder2"</code> </p>
         *        <p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludes(FilterRule... excludes);

        /**
         * <p>
         * A list of filter rules that determines which files to exclude from a task. The list should contain a single
         * filter string that consists of the patterns to exclude. The patterns are delimited by "|" (that is, a pipe),
         * for example: <code>"/folder1|/folder2"</code>
         * </p>
         * <p>
         * </p>
         * This is a convenience that creates an instance of the {@link List<FilterRule>.Builder} avoiding the need to
         * create one manually via {@link List<FilterRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FilterRule>.Builder#build()} is called immediately and its
         * result is passed to {@link #excludes(List<FilterRule>)}.
         * 
         * @param excludes
         *        a consumer that will call methods on {@link List<FilterRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #excludes(List<FilterRule>)
         */
        Builder excludes(Consumer<FilterRule.Builder>... excludes);

        /**
         * <p>
         * The schedule used to periodically transfer files from a source to a destination location.
         * </p>
         * 
         * @param schedule
         *        The schedule used to periodically transfer files from a source to a destination location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedule(TaskSchedule schedule);

        /**
         * <p>
         * The schedule used to periodically transfer files from a source to a destination location.
         * </p>
         * This is a convenience that creates an instance of the {@link TaskSchedule.Builder} avoiding the need to
         * create one manually via {@link TaskSchedule#builder()}.
         *
         * When the {@link Consumer} completes, {@link TaskSchedule.Builder#build()} is called immediately and its
         * result is passed to {@link #schedule(TaskSchedule)}.
         * 
         * @param schedule
         *        a consumer that will call methods on {@link TaskSchedule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schedule(TaskSchedule)
         */
        default Builder schedule(Consumer<TaskSchedule.Builder> schedule) {
            return schedule(TaskSchedule.builder().applyMutation(schedule).build());
        }

        /**
         * <p>
         * Errors that AWS DataSync encountered during execution of the task. You can use this error code to help
         * troubleshoot issues.
         * </p>
         * 
         * @param errorCode
         *        Errors that AWS DataSync encountered during execution of the task. You can use this error code to help
         *        troubleshoot issues.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * Detailed description of an error that was encountered during the task execution. You can use this information
         * to help troubleshoot issues.
         * </p>
         * 
         * @param errorDetail
         *        Detailed description of an error that was encountered during the task execution. You can use this
         *        information to help troubleshoot issues.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorDetail(String errorDetail);

        /**
         * <p>
         * The time that the task was created.
         * </p>
         * 
         * @param creationTime
         *        The time that the task was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);
    }

    static final class BuilderImpl extends DataSyncResponse.BuilderImpl implements Builder {
        private String taskArn;

        private String status;

        private String name;

        private String currentTaskExecutionArn;

        private String sourceLocationArn;

        private String destinationLocationArn;

        private String cloudWatchLogGroupArn;

        private List<String> sourceNetworkInterfaceArns = DefaultSdkAutoConstructList.getInstance();

        private List<String> destinationNetworkInterfaceArns = DefaultSdkAutoConstructList.getInstance();

        private Options options;

        private List<FilterRule> excludes = DefaultSdkAutoConstructList.getInstance();

        private TaskSchedule schedule;

        private String errorCode;

        private String errorDetail;

        private Instant creationTime;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeTaskResponse model) {
            super(model);
            taskArn(model.taskArn);
            status(model.status);
            name(model.name);
            currentTaskExecutionArn(model.currentTaskExecutionArn);
            sourceLocationArn(model.sourceLocationArn);
            destinationLocationArn(model.destinationLocationArn);
            cloudWatchLogGroupArn(model.cloudWatchLogGroupArn);
            sourceNetworkInterfaceArns(model.sourceNetworkInterfaceArns);
            destinationNetworkInterfaceArns(model.destinationNetworkInterfaceArns);
            options(model.options);
            excludes(model.excludes);
            schedule(model.schedule);
            errorCode(model.errorCode);
            errorDetail(model.errorDetail);
            creationTime(model.creationTime);
        }

        public final String getTaskArn() {
            return taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(TaskStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getCurrentTaskExecutionArn() {
            return currentTaskExecutionArn;
        }

        @Override
        public final Builder currentTaskExecutionArn(String currentTaskExecutionArn) {
            this.currentTaskExecutionArn = currentTaskExecutionArn;
            return this;
        }

        public final void setCurrentTaskExecutionArn(String currentTaskExecutionArn) {
            this.currentTaskExecutionArn = currentTaskExecutionArn;
        }

        public final String getSourceLocationArn() {
            return sourceLocationArn;
        }

        @Override
        public final Builder sourceLocationArn(String sourceLocationArn) {
            this.sourceLocationArn = sourceLocationArn;
            return this;
        }

        public final void setSourceLocationArn(String sourceLocationArn) {
            this.sourceLocationArn = sourceLocationArn;
        }

        public final String getDestinationLocationArn() {
            return destinationLocationArn;
        }

        @Override
        public final Builder destinationLocationArn(String destinationLocationArn) {
            this.destinationLocationArn = destinationLocationArn;
            return this;
        }

        public final void setDestinationLocationArn(String destinationLocationArn) {
            this.destinationLocationArn = destinationLocationArn;
        }

        public final String getCloudWatchLogGroupArn() {
            return cloudWatchLogGroupArn;
        }

        @Override
        public final Builder cloudWatchLogGroupArn(String cloudWatchLogGroupArn) {
            this.cloudWatchLogGroupArn = cloudWatchLogGroupArn;
            return this;
        }

        public final void setCloudWatchLogGroupArn(String cloudWatchLogGroupArn) {
            this.cloudWatchLogGroupArn = cloudWatchLogGroupArn;
        }

        public final Collection<String> getSourceNetworkInterfaceArns() {
            return sourceNetworkInterfaceArns;
        }

        @Override
        public final Builder sourceNetworkInterfaceArns(Collection<String> sourceNetworkInterfaceArns) {
            this.sourceNetworkInterfaceArns = SourceNetworkInterfaceArnsCopier.copy(sourceNetworkInterfaceArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceNetworkInterfaceArns(String... sourceNetworkInterfaceArns) {
            sourceNetworkInterfaceArns(Arrays.asList(sourceNetworkInterfaceArns));
            return this;
        }

        public final void setSourceNetworkInterfaceArns(Collection<String> sourceNetworkInterfaceArns) {
            this.sourceNetworkInterfaceArns = SourceNetworkInterfaceArnsCopier.copy(sourceNetworkInterfaceArns);
        }

        public final Collection<String> getDestinationNetworkInterfaceArns() {
            return destinationNetworkInterfaceArns;
        }

        @Override
        public final Builder destinationNetworkInterfaceArns(Collection<String> destinationNetworkInterfaceArns) {
            this.destinationNetworkInterfaceArns = DestinationNetworkInterfaceArnsCopier.copy(destinationNetworkInterfaceArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinationNetworkInterfaceArns(String... destinationNetworkInterfaceArns) {
            destinationNetworkInterfaceArns(Arrays.asList(destinationNetworkInterfaceArns));
            return this;
        }

        public final void setDestinationNetworkInterfaceArns(Collection<String> destinationNetworkInterfaceArns) {
            this.destinationNetworkInterfaceArns = DestinationNetworkInterfaceArnsCopier.copy(destinationNetworkInterfaceArns);
        }

        public final Options.Builder getOptions() {
            return options != null ? options.toBuilder() : null;
        }

        @Override
        public final Builder options(Options options) {
            this.options = options;
            return this;
        }

        public final void setOptions(Options.BuilderImpl options) {
            this.options = options != null ? options.build() : null;
        }

        public final Collection<FilterRule.Builder> getExcludes() {
            return excludes != null ? excludes.stream().map(FilterRule::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder excludes(Collection<FilterRule> excludes) {
            this.excludes = FilterListCopier.copy(excludes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludes(FilterRule... excludes) {
            excludes(Arrays.asList(excludes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludes(Consumer<FilterRule.Builder>... excludes) {
            excludes(Stream.of(excludes).map(c -> FilterRule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setExcludes(Collection<FilterRule.BuilderImpl> excludes) {
            this.excludes = FilterListCopier.copyFromBuilder(excludes);
        }

        public final TaskSchedule.Builder getSchedule() {
            return schedule != null ? schedule.toBuilder() : null;
        }

        @Override
        public final Builder schedule(TaskSchedule schedule) {
            this.schedule = schedule;
            return this;
        }

        public final void setSchedule(TaskSchedule.BuilderImpl schedule) {
            this.schedule = schedule != null ? schedule.build() : null;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        public final String getErrorDetail() {
            return errorDetail;
        }

        @Override
        public final Builder errorDetail(String errorDetail) {
            this.errorDetail = errorDetail;
            return this;
        }

        public final void setErrorDetail(String errorDetail) {
            this.errorDetail = errorDetail;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public DescribeTaskResponse build() {
            return new DescribeTaskResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
