/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CreateLocationSmbRequest
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLocationSmbRequest extends DataSyncRequest implements
        ToCopyableBuilder<CreateLocationSmbRequest.Builder, CreateLocationSmbRequest> {
    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(CreateLocationSmbRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> SERVER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerHostname").getter(getter(CreateLocationSmbRequest::serverHostname))
            .setter(setter(Builder::serverHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerHostname").build()).build();

    private static final SdkField<String> USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("User")
            .getter(getter(CreateLocationSmbRequest::user)).setter(setter(Builder::user))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("User").build()).build();

    private static final SdkField<String> DOMAIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Domain")
            .getter(getter(CreateLocationSmbRequest::domain)).setter(setter(Builder::domain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Domain").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(CreateLocationSmbRequest::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(CreateLocationSmbRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SmbMountOptions> MOUNT_OPTIONS_FIELD = SdkField
            .<SmbMountOptions> builder(MarshallingType.SDK_POJO).memberName("MountOptions")
            .getter(getter(CreateLocationSmbRequest::mountOptions)).setter(setter(Builder::mountOptions))
            .constructor(SmbMountOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MountOptions").build()).build();

    private static final SdkField<List<TagListEntry>> TAGS_FIELD = SdkField
            .<List<TagListEntry>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateLocationSmbRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TagListEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagListEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBDIRECTORY_FIELD,
            SERVER_HOSTNAME_FIELD, USER_FIELD, DOMAIN_FIELD, PASSWORD_FIELD, AGENT_ARNS_FIELD, MOUNT_OPTIONS_FIELD, TAGS_FIELD));

    private final String subdirectory;

    private final String serverHostname;

    private final String user;

    private final String domain;

    private final String password;

    private final List<String> agentArns;

    private final SmbMountOptions mountOptions;

    private final List<TagListEntry> tags;

    private CreateLocationSmbRequest(BuilderImpl builder) {
        super(builder);
        this.subdirectory = builder.subdirectory;
        this.serverHostname = builder.serverHostname;
        this.user = builder.user;
        this.domain = builder.domain;
        this.password = builder.password;
        this.agentArns = builder.agentArns;
        this.mountOptions = builder.mountOptions;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The subdirectory in the SMB file system that is used to read data from the SMB source location or write data to
     * the SMB destination. The SMB path should be a path that's exported by the SMB server, or a subdirectory of that
     * path. The path should be such that it can be mounted by other SMB clients in your network.
     * </p>
     * <note>
     * <p>
     * <code>Subdirectory</code> must be specified with forward slashes. For example, <code>/path/to/folder</code>.
     * </p>
     * </note>
     * <p>
     * To transfer all the data in the folder you specified, DataSync needs to have permissions to mount the SMB share,
     * as well as to access all the data in that share. To ensure this, either ensure that the user/password specified
     * belongs to the user who can mount the share, and who has the appropriate permissions for all of the files and
     * directories that you want DataSync to access, or use credentials of a member of the Backup Operators group to
     * mount the share. Doing either enables the agent to access the data. For the agent to access directories, you must
     * additionally enable all execute access.
     * </p>
     * 
     * @return The subdirectory in the SMB file system that is used to read data from the SMB source location or write
     *         data to the SMB destination. The SMB path should be a path that's exported by the SMB server, or a
     *         subdirectory of that path. The path should be such that it can be mounted by other SMB clients in your
     *         network.</p> <note>
     *         <p>
     *         <code>Subdirectory</code> must be specified with forward slashes. For example,
     *         <code>/path/to/folder</code>.
     *         </p>
     *         </note>
     *         <p>
     *         To transfer all the data in the folder you specified, DataSync needs to have permissions to mount the SMB
     *         share, as well as to access all the data in that share. To ensure this, either ensure that the
     *         user/password specified belongs to the user who can mount the share, and who has the appropriate
     *         permissions for all of the files and directories that you want DataSync to access, or use credentials of
     *         a member of the Backup Operators group to mount the share. Doing either enables the agent to access the
     *         data. For the agent to access directories, you must additionally enable all execute access.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * The name of the SMB server. This value is the IP address or Domain Name Service (DNS) name of the SMB server. An
     * agent that is installed on-premises uses this hostname to mount the SMB server in a network.
     * </p>
     * <note>
     * <p>
     * This name must either be DNS-compliant or must be an IP version 4 (IPv4) address.
     * </p>
     * </note>
     * 
     * @return The name of the SMB server. This value is the IP address or Domain Name Service (DNS) name of the SMB
     *         server. An agent that is installed on-premises uses this hostname to mount the SMB server in a
     *         network.</p> <note>
     *         <p>
     *         This name must either be DNS-compliant or must be an IP version 4 (IPv4) address.
     *         </p>
     */
    public final String serverHostname() {
        return serverHostname;
    }

    /**
     * <p>
     * The user who can mount the share, has the permissions to access files and folders in the SMB share.
     * </p>
     * <p>
     * For information about choosing a user name that ensures sufficient permissions to files, folders, and metadata,
     * see <a href="create-smb-location.html#SMBuser">user</a>.
     * </p>
     * 
     * @return The user who can mount the share, has the permissions to access files and folders in the SMB share.</p>
     *         <p>
     *         For information about choosing a user name that ensures sufficient permissions to files, folders, and
     *         metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
     */
    public final String user() {
        return user;
    }

    /**
     * <p>
     * The name of the Windows domain that the SMB server belongs to.
     * </p>
     * 
     * @return The name of the Windows domain that the SMB server belongs to.
     */
    public final String domain() {
        return domain;
    }

    /**
     * <p>
     * The password of the user who can mount the share, has the permissions to access files and folders in the SMB
     * share.
     * </p>
     * 
     * @return The password of the user who can mount the share, has the permissions to access files and folders in the
     *         SMB share.
     */
    public final String password() {
        return password;
    }

    /**
     * Returns true if the AgentArns property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of agents to use for a Simple Message Block (SMB) location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAgentArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of agents to use for a Simple Message Block (SMB) location.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * <p>
     * The mount options used by DataSync to access the SMB server.
     * </p>
     * 
     * @return The mount options used by DataSync to access the SMB server.
     */
    public final SmbMountOptions mountOptions() {
        return mountOptions;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
     * string. We recommend using tags to name your resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The key-value pair that represents the tag that you want to add to the location. The value can be an
     *         empty string. We recommend using tags to name your resources.
     */
    public final List<TagListEntry> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(serverHostname());
        hashCode = 31 * hashCode + Objects.hashCode(user());
        hashCode = 31 * hashCode + Objects.hashCode(domain());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(mountOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLocationSmbRequest)) {
            return false;
        }
        CreateLocationSmbRequest other = (CreateLocationSmbRequest) obj;
        return Objects.equals(subdirectory(), other.subdirectory()) && Objects.equals(serverHostname(), other.serverHostname())
                && Objects.equals(user(), other.user()) && Objects.equals(domain(), other.domain())
                && Objects.equals(password(), other.password()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns()) && Objects.equals(mountOptions(), other.mountOptions())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLocationSmbRequest").add("Subdirectory", subdirectory())
                .add("ServerHostname", serverHostname()).add("User", user()).add("Domain", domain())
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("MountOptions", mountOptions())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "ServerHostname":
            return Optional.ofNullable(clazz.cast(serverHostname()));
        case "User":
            return Optional.ofNullable(clazz.cast(user()));
        case "Domain":
            return Optional.ofNullable(clazz.cast(domain()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "MountOptions":
            return Optional.ofNullable(clazz.cast(mountOptions()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLocationSmbRequest, T> g) {
        return obj -> g.apply((CreateLocationSmbRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateLocationSmbRequest> {
        /**
         * <p>
         * The subdirectory in the SMB file system that is used to read data from the SMB source location or write data
         * to the SMB destination. The SMB path should be a path that's exported by the SMB server, or a subdirectory of
         * that path. The path should be such that it can be mounted by other SMB clients in your network.
         * </p>
         * <note>
         * <p>
         * <code>Subdirectory</code> must be specified with forward slashes. For example, <code>/path/to/folder</code>.
         * </p>
         * </note>
         * <p>
         * To transfer all the data in the folder you specified, DataSync needs to have permissions to mount the SMB
         * share, as well as to access all the data in that share. To ensure this, either ensure that the user/password
         * specified belongs to the user who can mount the share, and who has the appropriate permissions for all of the
         * files and directories that you want DataSync to access, or use credentials of a member of the Backup
         * Operators group to mount the share. Doing either enables the agent to access the data. For the agent to
         * access directories, you must additionally enable all execute access.
         * </p>
         * 
         * @param subdirectory
         *        The subdirectory in the SMB file system that is used to read data from the SMB source location or
         *        write data to the SMB destination. The SMB path should be a path that's exported by the SMB server, or
         *        a subdirectory of that path. The path should be such that it can be mounted by other SMB clients in
         *        your network.</p> <note>
         *        <p>
         *        <code>Subdirectory</code> must be specified with forward slashes. For example,
         *        <code>/path/to/folder</code>.
         *        </p>
         *        </note>
         *        <p>
         *        To transfer all the data in the folder you specified, DataSync needs to have permissions to mount the
         *        SMB share, as well as to access all the data in that share. To ensure this, either ensure that the
         *        user/password specified belongs to the user who can mount the share, and who has the appropriate
         *        permissions for all of the files and directories that you want DataSync to access, or use credentials
         *        of a member of the Backup Operators group to mount the share. Doing either enables the agent to access
         *        the data. For the agent to access directories, you must additionally enable all execute access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * The name of the SMB server. This value is the IP address or Domain Name Service (DNS) name of the SMB server.
         * An agent that is installed on-premises uses this hostname to mount the SMB server in a network.
         * </p>
         * <note>
         * <p>
         * This name must either be DNS-compliant or must be an IP version 4 (IPv4) address.
         * </p>
         * </note>
         * 
         * @param serverHostname
         *        The name of the SMB server. This value is the IP address or Domain Name Service (DNS) name of the SMB
         *        server. An agent that is installed on-premises uses this hostname to mount the SMB server in a
         *        network.</p> <note>
         *        <p>
         *        This name must either be DNS-compliant or must be an IP version 4 (IPv4) address.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverHostname(String serverHostname);

        /**
         * <p>
         * The user who can mount the share, has the permissions to access files and folders in the SMB share.
         * </p>
         * <p>
         * For information about choosing a user name that ensures sufficient permissions to files, folders, and
         * metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
         * </p>
         * 
         * @param user
         *        The user who can mount the share, has the permissions to access files and folders in the SMB
         *        share.</p>
         *        <p>
         *        For information about choosing a user name that ensures sufficient permissions to files, folders, and
         *        metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder user(String user);

        /**
         * <p>
         * The name of the Windows domain that the SMB server belongs to.
         * </p>
         * 
         * @param domain
         *        The name of the Windows domain that the SMB server belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domain(String domain);

        /**
         * <p>
         * The password of the user who can mount the share, has the permissions to access files and folders in the SMB
         * share.
         * </p>
         * 
         * @param password
         *        The password of the user who can mount the share, has the permissions to access files and folders in
         *        the SMB share.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of agents to use for a Simple Message Block (SMB) location.
         * </p>
         * 
         * @param agentArns
         *        The Amazon Resource Names (ARNs) of agents to use for a Simple Message Block (SMB) location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of agents to use for a Simple Message Block (SMB) location.
         * </p>
         * 
         * @param agentArns
         *        The Amazon Resource Names (ARNs) of agents to use for a Simple Message Block (SMB) location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * The mount options used by DataSync to access the SMB server.
         * </p>
         * 
         * @param mountOptions
         *        The mount options used by DataSync to access the SMB server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mountOptions(SmbMountOptions mountOptions);

        /**
         * <p>
         * The mount options used by DataSync to access the SMB server.
         * </p>
         * This is a convenience that creates an instance of the {@link SmbMountOptions.Builder} avoiding the need to
         * create one manually via {@link SmbMountOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link SmbMountOptions.Builder#build()} is called immediately and its
         * result is passed to {@link #mountOptions(SmbMountOptions)}.
         * 
         * @param mountOptions
         *        a consumer that will call methods on {@link SmbMountOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mountOptions(SmbMountOptions)
         */
        default Builder mountOptions(Consumer<SmbMountOptions.Builder> mountOptions) {
            return mountOptions(SmbMountOptions.builder().applyMutation(mountOptions).build());
        }

        /**
         * <p>
         * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
         * string. We recommend using tags to name your resources.
         * </p>
         * 
         * @param tags
         *        The key-value pair that represents the tag that you want to add to the location. The value can be an
         *        empty string. We recommend using tags to name your resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagListEntry> tags);

        /**
         * <p>
         * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
         * string. We recommend using tags to name your resources.
         * </p>
         * 
         * @param tags
         *        The key-value pair that represents the tag that you want to add to the location. The value can be an
         *        empty string. We recommend using tags to name your resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagListEntry... tags);

        /**
         * <p>
         * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
         * string. We recommend using tags to name your resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TagListEntry>.Builder} avoiding the need to
         * create one manually via {@link List<TagListEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagListEntry>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<TagListEntry>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<TagListEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<TagListEntry>)
         */
        Builder tags(Consumer<TagListEntry.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String subdirectory;

        private String serverHostname;

        private String user;

        private String domain;

        private String password;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private SmbMountOptions mountOptions;

        private List<TagListEntry> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLocationSmbRequest model) {
            super(model);
            subdirectory(model.subdirectory);
            serverHostname(model.serverHostname);
            user(model.user);
            domain(model.domain);
            password(model.password);
            agentArns(model.agentArns);
            mountOptions(model.mountOptions);
            tags(model.tags);
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        public final String getServerHostname() {
            return serverHostname;
        }

        @Override
        public final Builder serverHostname(String serverHostname) {
            this.serverHostname = serverHostname;
            return this;
        }

        public final void setServerHostname(String serverHostname) {
            this.serverHostname = serverHostname;
        }

        public final String getUser() {
            return user;
        }

        @Override
        public final Builder user(String user) {
            this.user = user;
            return this;
        }

        public final void setUser(String user) {
            this.user = user;
        }

        public final String getDomain() {
            return domain;
        }

        @Override
        public final Builder domain(String domain) {
            this.domain = domain;
            return this;
        }

        public final void setDomain(String domain) {
            this.domain = domain;
        }

        public final String getPassword() {
            return password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        public final SmbMountOptions.Builder getMountOptions() {
            return mountOptions != null ? mountOptions.toBuilder() : null;
        }

        @Override
        public final Builder mountOptions(SmbMountOptions mountOptions) {
            this.mountOptions = mountOptions;
            return this;
        }

        public final void setMountOptions(SmbMountOptions.BuilderImpl mountOptions) {
            this.mountOptions = mountOptions != null ? mountOptions.build() : null;
        }

        public final List<TagListEntry.Builder> getTags() {
            List<TagListEntry.Builder> result = InputTagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<TagListEntry> tags) {
            this.tags = InputTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagListEntry... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagListEntry.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagListEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<TagListEntry.BuilderImpl> tags) {
            this.tags = InputTagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLocationSmbRequest build() {
            return new CreateLocationSmbRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
