/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CreateLocationS3Request
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLocationS3Request extends DataSyncRequest implements
        ToCopyableBuilder<CreateLocationS3Request.Builder, CreateLocationS3Request> {
    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(CreateLocationS3Request::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> S3_BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3BucketArn").getter(getter(CreateLocationS3Request::s3BucketArn)).setter(setter(Builder::s3BucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3BucketArn").build()).build();

    private static final SdkField<String> S3_STORAGE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3StorageClass").getter(getter(CreateLocationS3Request::s3StorageClassAsString))
            .setter(setter(Builder::s3StorageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3StorageClass").build()).build();

    private static final SdkField<S3Config> S3_CONFIG_FIELD = SdkField.<S3Config> builder(MarshallingType.SDK_POJO)
            .memberName("S3Config").getter(getter(CreateLocationS3Request::s3Config)).setter(setter(Builder::s3Config))
            .constructor(S3Config::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Config").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(CreateLocationS3Request::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TagListEntry>> TAGS_FIELD = SdkField
            .<List<TagListEntry>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateLocationS3Request::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TagListEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagListEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBDIRECTORY_FIELD,
            S3_BUCKET_ARN_FIELD, S3_STORAGE_CLASS_FIELD, S3_CONFIG_FIELD, AGENT_ARNS_FIELD, TAGS_FIELD));

    private final String subdirectory;

    private final String s3BucketArn;

    private final String s3StorageClass;

    private final S3Config s3Config;

    private final List<String> agentArns;

    private final List<TagListEntry> tags;

    private CreateLocationS3Request(BuilderImpl builder) {
        super(builder);
        this.subdirectory = builder.subdirectory;
        this.s3BucketArn = builder.s3BucketArn;
        this.s3StorageClass = builder.s3StorageClass;
        this.s3Config = builder.s3Config;
        this.agentArns = builder.agentArns;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * A subdirectory in the Amazon S3 bucket. This subdirectory in Amazon S3 is used to read data from the S3 source
     * location or write data to the S3 destination.
     * </p>
     * 
     * @return A subdirectory in the Amazon S3 bucket. This subdirectory in Amazon S3 is used to read data from the S3
     *         source location or write data to the S3 destination.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * The ARN of the Amazon S3 bucket. If the bucket is on an AWS Outpost, this must be an access point ARN.
     * </p>
     * 
     * @return The ARN of the Amazon S3 bucket. If the bucket is on an AWS Outpost, this must be an access point ARN.
     */
    public final String s3BucketArn() {
        return s3BucketArn;
    }

    /**
     * <p>
     * The Amazon S3 storage class that you want to store your files in when this location is used as a task
     * destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS Outposts, the
     * storage class defaults to AWS S3 Outposts.
     * </p>
     * <p>
     * For more information about S3 storage classes, see <a href="http://aws.amazon.com/s3/storage-classes/">Amazon S3
     * Storage Classes</a>. Some storage classes have behaviors that can affect your S3 storage cost. For detailed
     * information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     * >Considerations when working with S3 storage classes in DataSync</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3StorageClass}
     * will return {@link S3StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #s3StorageClassAsString}.
     * </p>
     * 
     * @return The Amazon S3 storage class that you want to store your files in when this location is used as a task
     *         destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS
     *         Outposts, the storage class defaults to AWS S3 Outposts.</p>
     *         <p>
     *         For more information about S3 storage classes, see <a
     *         href="http://aws.amazon.com/s3/storage-classes/">Amazon S3 Storage Classes</a>. Some storage classes have
     *         behaviors that can affect your S3 storage cost. For detailed information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     *         >Considerations when working with S3 storage classes in DataSync</a>.
     * @see S3StorageClass
     */
    public final S3StorageClass s3StorageClass() {
        return S3StorageClass.fromValue(s3StorageClass);
    }

    /**
     * <p>
     * The Amazon S3 storage class that you want to store your files in when this location is used as a task
     * destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS Outposts, the
     * storage class defaults to AWS S3 Outposts.
     * </p>
     * <p>
     * For more information about S3 storage classes, see <a href="http://aws.amazon.com/s3/storage-classes/">Amazon S3
     * Storage Classes</a>. Some storage classes have behaviors that can affect your S3 storage cost. For detailed
     * information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     * >Considerations when working with S3 storage classes in DataSync</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #s3StorageClass}
     * will return {@link S3StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #s3StorageClassAsString}.
     * </p>
     * 
     * @return The Amazon S3 storage class that you want to store your files in when this location is used as a task
     *         destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS
     *         Outposts, the storage class defaults to AWS S3 Outposts.</p>
     *         <p>
     *         For more information about S3 storage classes, see <a
     *         href="http://aws.amazon.com/s3/storage-classes/">Amazon S3 Storage Classes</a>. Some storage classes have
     *         behaviors that can affect your S3 storage cost. For detailed information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     *         >Considerations when working with S3 storage classes in DataSync</a>.
     * @see S3StorageClass
     */
    public final String s3StorageClassAsString() {
        return s3StorageClass;
    }

    /**
     * Returns the value of the S3Config property for this object.
     * 
     * @return The value of the S3Config property for this object.
     */
    public final S3Config s3Config() {
        return s3Config;
    }

    /**
     * Returns true if the AgentArns property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If you are using DataSync on an AWS Outpost, specify the Amazon Resource Names (ARNs) of the DataSync agents
     * deployed on your Outpost. For more information about launching a DataSync agent on an AWS Outpost, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/deploy-agents.html#outposts-agent">Deploy your
     * DataSync agent on AWS Outposts</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAgentArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you are using DataSync on an AWS Outpost, specify the Amazon Resource Names (ARNs) of the DataSync
     *         agents deployed on your Outpost. For more information about launching a DataSync agent on an AWS Outpost,
     *         see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/deploy-agents.html#outposts-agent">Deploy
     *         your DataSync agent on AWS Outposts</a>.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
     * string. We recommend using tags to name your resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The key-value pair that represents the tag that you want to add to the location. The value can be an
     *         empty string. We recommend using tags to name your resources.
     */
    public final List<TagListEntry> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(s3StorageClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3Config());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLocationS3Request)) {
            return false;
        }
        CreateLocationS3Request other = (CreateLocationS3Request) obj;
        return Objects.equals(subdirectory(), other.subdirectory()) && Objects.equals(s3BucketArn(), other.s3BucketArn())
                && Objects.equals(s3StorageClassAsString(), other.s3StorageClassAsString())
                && Objects.equals(s3Config(), other.s3Config()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLocationS3Request").add("Subdirectory", subdirectory()).add("S3BucketArn", s3BucketArn())
                .add("S3StorageClass", s3StorageClassAsString()).add("S3Config", s3Config())
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "S3BucketArn":
            return Optional.ofNullable(clazz.cast(s3BucketArn()));
        case "S3StorageClass":
            return Optional.ofNullable(clazz.cast(s3StorageClassAsString()));
        case "S3Config":
            return Optional.ofNullable(clazz.cast(s3Config()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLocationS3Request, T> g) {
        return obj -> g.apply((CreateLocationS3Request) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateLocationS3Request> {
        /**
         * <p>
         * A subdirectory in the Amazon S3 bucket. This subdirectory in Amazon S3 is used to read data from the S3
         * source location or write data to the S3 destination.
         * </p>
         * 
         * @param subdirectory
         *        A subdirectory in the Amazon S3 bucket. This subdirectory in Amazon S3 is used to read data from the
         *        S3 source location or write data to the S3 destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * The ARN of the Amazon S3 bucket. If the bucket is on an AWS Outpost, this must be an access point ARN.
         * </p>
         * 
         * @param s3BucketArn
         *        The ARN of the Amazon S3 bucket. If the bucket is on an AWS Outpost, this must be an access point ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketArn(String s3BucketArn);

        /**
         * <p>
         * The Amazon S3 storage class that you want to store your files in when this location is used as a task
         * destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS Outposts,
         * the storage class defaults to AWS S3 Outposts.
         * </p>
         * <p>
         * For more information about S3 storage classes, see <a href="http://aws.amazon.com/s3/storage-classes/">Amazon
         * S3 Storage Classes</a>. Some storage classes have behaviors that can affect your S3 storage cost. For
         * detailed information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         * >Considerations when working with S3 storage classes in DataSync</a>.
         * </p>
         * 
         * @param s3StorageClass
         *        The Amazon S3 storage class that you want to store your files in when this location is used as a task
         *        destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS
         *        Outposts, the storage class defaults to AWS S3 Outposts.</p>
         *        <p>
         *        For more information about S3 storage classes, see <a
         *        href="http://aws.amazon.com/s3/storage-classes/">Amazon S3 Storage Classes</a>. Some storage classes
         *        have behaviors that can affect your S3 storage cost. For detailed information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         *        >Considerations when working with S3 storage classes in DataSync</a>.
         * @see S3StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3StorageClass
         */
        Builder s3StorageClass(String s3StorageClass);

        /**
         * <p>
         * The Amazon S3 storage class that you want to store your files in when this location is used as a task
         * destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS Outposts,
         * the storage class defaults to AWS S3 Outposts.
         * </p>
         * <p>
         * For more information about S3 storage classes, see <a href="http://aws.amazon.com/s3/storage-classes/">Amazon
         * S3 Storage Classes</a>. Some storage classes have behaviors that can affect your S3 storage cost. For
         * detailed information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         * >Considerations when working with S3 storage classes in DataSync</a>.
         * </p>
         * 
         * @param s3StorageClass
         *        The Amazon S3 storage class that you want to store your files in when this location is used as a task
         *        destination. For buckets in AWS Regions, the storage class defaults to Standard. For buckets on AWS
         *        Outposts, the storage class defaults to AWS S3 Outposts.</p>
         *        <p>
         *        For more information about S3 storage classes, see <a
         *        href="http://aws.amazon.com/s3/storage-classes/">Amazon S3 Storage Classes</a>. Some storage classes
         *        have behaviors that can affect your S3 storage cost. For detailed information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         *        >Considerations when working with S3 storage classes in DataSync</a>.
         * @see S3StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3StorageClass
         */
        Builder s3StorageClass(S3StorageClass s3StorageClass);

        /**
         * Sets the value of the S3Config property for this object.
         *
         * @param s3Config
         *        The new value for the S3Config property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Config(S3Config s3Config);

        /**
         * Sets the value of the S3Config property for this object.
         *
         * This is a convenience that creates an instance of the {@link S3Config.Builder} avoiding the need to create
         * one manually via {@link S3Config#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Config.Builder#build()} is called immediately and its result is
         * passed to {@link #s3Config(S3Config)}.
         * 
         * @param s3Config
         *        a consumer that will call methods on {@link S3Config.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Config(S3Config)
         */
        default Builder s3Config(Consumer<S3Config.Builder> s3Config) {
            return s3Config(S3Config.builder().applyMutation(s3Config).build());
        }

        /**
         * <p>
         * If you are using DataSync on an AWS Outpost, specify the Amazon Resource Names (ARNs) of the DataSync agents
         * deployed on your Outpost. For more information about launching a DataSync agent on an AWS Outpost, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/deploy-agents.html#outposts-agent">Deploy your
         * DataSync agent on AWS Outposts</a>.
         * </p>
         * 
         * @param agentArns
         *        If you are using DataSync on an AWS Outpost, specify the Amazon Resource Names (ARNs) of the DataSync
         *        agents deployed on your Outpost. For more information about launching a DataSync agent on an AWS
         *        Outpost, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/deploy-agents.html#outposts-agent">Deploy
         *        your DataSync agent on AWS Outposts</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * If you are using DataSync on an AWS Outpost, specify the Amazon Resource Names (ARNs) of the DataSync agents
         * deployed on your Outpost. For more information about launching a DataSync agent on an AWS Outpost, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/deploy-agents.html#outposts-agent">Deploy your
         * DataSync agent on AWS Outposts</a>.
         * </p>
         * 
         * @param agentArns
         *        If you are using DataSync on an AWS Outpost, specify the Amazon Resource Names (ARNs) of the DataSync
         *        agents deployed on your Outpost. For more information about launching a DataSync agent on an AWS
         *        Outpost, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/deploy-agents.html#outposts-agent">Deploy
         *        your DataSync agent on AWS Outposts</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
         * string. We recommend using tags to name your resources.
         * </p>
         * 
         * @param tags
         *        The key-value pair that represents the tag that you want to add to the location. The value can be an
         *        empty string. We recommend using tags to name your resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagListEntry> tags);

        /**
         * <p>
         * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
         * string. We recommend using tags to name your resources.
         * </p>
         * 
         * @param tags
         *        The key-value pair that represents the tag that you want to add to the location. The value can be an
         *        empty string. We recommend using tags to name your resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagListEntry... tags);

        /**
         * <p>
         * The key-value pair that represents the tag that you want to add to the location. The value can be an empty
         * string. We recommend using tags to name your resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TagListEntry>.Builder} avoiding the need to
         * create one manually via {@link List<TagListEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagListEntry>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<TagListEntry>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<TagListEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<TagListEntry>)
         */
        Builder tags(Consumer<TagListEntry.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String subdirectory;

        private String s3BucketArn;

        private String s3StorageClass;

        private S3Config s3Config;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private List<TagListEntry> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLocationS3Request model) {
            super(model);
            subdirectory(model.subdirectory);
            s3BucketArn(model.s3BucketArn);
            s3StorageClass(model.s3StorageClass);
            s3Config(model.s3Config);
            agentArns(model.agentArns);
            tags(model.tags);
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        public final String getS3BucketArn() {
            return s3BucketArn;
        }

        @Override
        public final Builder s3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
            return this;
        }

        public final void setS3BucketArn(String s3BucketArn) {
            this.s3BucketArn = s3BucketArn;
        }

        public final String getS3StorageClass() {
            return s3StorageClass;
        }

        @Override
        public final Builder s3StorageClass(String s3StorageClass) {
            this.s3StorageClass = s3StorageClass;
            return this;
        }

        @Override
        public final Builder s3StorageClass(S3StorageClass s3StorageClass) {
            this.s3StorageClass(s3StorageClass == null ? null : s3StorageClass.toString());
            return this;
        }

        public final void setS3StorageClass(String s3StorageClass) {
            this.s3StorageClass = s3StorageClass;
        }

        public final S3Config.Builder getS3Config() {
            return s3Config != null ? s3Config.toBuilder() : null;
        }

        @Override
        public final Builder s3Config(S3Config s3Config) {
            this.s3Config = s3Config;
            return this;
        }

        public final void setS3Config(S3Config.BuilderImpl s3Config) {
            this.s3Config = s3Config != null ? s3Config.build() : null;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        public final List<TagListEntry.Builder> getTags() {
            List<TagListEntry.Builder> result = InputTagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<TagListEntry> tags) {
            this.tags = InputTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagListEntry... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagListEntry.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagListEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<TagListEntry.BuilderImpl> tags) {
            this.tags = InputTagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLocationS3Request build() {
            return new CreateLocationS3Request(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
