/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The VPC endpoint, subnet, and security group that an agent uses to access IP addresses in a VPC (Virtual Private
 * Cloud).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PrivateLinkConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<PrivateLinkConfig.Builder, PrivateLinkConfig> {
    private static final SdkField<String> VPC_ENDPOINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VpcEndpointId").getter(getter(PrivateLinkConfig::vpcEndpointId)).setter(setter(Builder::vpcEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcEndpointId").build()).build();

    private static final SdkField<String> PRIVATE_LINK_ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PrivateLinkEndpoint").getter(getter(PrivateLinkConfig::privateLinkEndpoint))
            .setter(setter(Builder::privateLinkEndpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateLinkEndpoint").build())
            .build();

    private static final SdkField<List<String>> SUBNET_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SubnetArns")
            .getter(getter(PrivateLinkConfig::subnetArns))
            .setter(setter(Builder::subnetArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupArns")
            .getter(getter(PrivateLinkConfig::securityGroupArns))
            .setter(setter(Builder::securityGroupArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VPC_ENDPOINT_ID_FIELD,
            PRIVATE_LINK_ENDPOINT_FIELD, SUBNET_ARNS_FIELD, SECURITY_GROUP_ARNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String vpcEndpointId;

    private final String privateLinkEndpoint;

    private final List<String> subnetArns;

    private final List<String> securityGroupArns;

    private PrivateLinkConfig(BuilderImpl builder) {
        this.vpcEndpointId = builder.vpcEndpointId;
        this.privateLinkEndpoint = builder.privateLinkEndpoint;
        this.subnetArns = builder.subnetArns;
        this.securityGroupArns = builder.securityGroupArns;
    }

    /**
     * <p>
     * The ID of the VPC endpoint that is configured for an agent. An agent that is configured with a VPC endpoint will
     * not be accessible over the public internet.
     * </p>
     * 
     * @return The ID of the VPC endpoint that is configured for an agent. An agent that is configured with a VPC
     *         endpoint will not be accessible over the public internet.
     */
    public final String vpcEndpointId() {
        return vpcEndpointId;
    }

    /**
     * <p>
     * The private endpoint that is configured for an agent that has access to IP addresses in a <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-service.html">PrivateLink</a>. An agent that is
     * configured with this endpoint will not be accessible over the public internet.
     * </p>
     * 
     * @return The private endpoint that is configured for an agent that has access to IP addresses in a <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-service.html">PrivateLink</a>. An agent
     *         that is configured with this endpoint will not be accessible over the public internet.
     */
    public final String privateLinkEndpoint() {
        return privateLinkEndpoint;
    }

    /**
     * Returns true if the SubnetArns property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSubnetArns() {
        return subnetArns != null && !(subnetArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the subnets that are configured for an agent activated in a VPC or an agent
     * that has access to a VPC endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the subnets that are configured for an agent activated in a VPC or an
     *         agent that has access to a VPC endpoint.
     */
    public final List<String> subnetArns() {
        return subnetArns;
    }

    /**
     * Returns true if the SecurityGroupArns property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSecurityGroupArns() {
        return securityGroupArns != null && !(securityGroupArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the security groups that are configured for the EC2 resource that hosts an
     * agent activated in a VPC or an agent that has access to a VPC endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecurityGroupArns()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the security groups that are configured for the EC2 resource that
     *         hosts an agent activated in a VPC or an agent that has access to a VPC endpoint.
     */
    public final List<String> securityGroupArns() {
        return securityGroupArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vpcEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(privateLinkEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubnetArns() ? subnetArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupArns() ? securityGroupArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PrivateLinkConfig)) {
            return false;
        }
        PrivateLinkConfig other = (PrivateLinkConfig) obj;
        return Objects.equals(vpcEndpointId(), other.vpcEndpointId())
                && Objects.equals(privateLinkEndpoint(), other.privateLinkEndpoint()) && hasSubnetArns() == other.hasSubnetArns()
                && Objects.equals(subnetArns(), other.subnetArns()) && hasSecurityGroupArns() == other.hasSecurityGroupArns()
                && Objects.equals(securityGroupArns(), other.securityGroupArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PrivateLinkConfig").add("VpcEndpointId", vpcEndpointId())
                .add("PrivateLinkEndpoint", privateLinkEndpoint()).add("SubnetArns", hasSubnetArns() ? subnetArns() : null)
                .add("SecurityGroupArns", hasSecurityGroupArns() ? securityGroupArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VpcEndpointId":
            return Optional.ofNullable(clazz.cast(vpcEndpointId()));
        case "PrivateLinkEndpoint":
            return Optional.ofNullable(clazz.cast(privateLinkEndpoint()));
        case "SubnetArns":
            return Optional.ofNullable(clazz.cast(subnetArns()));
        case "SecurityGroupArns":
            return Optional.ofNullable(clazz.cast(securityGroupArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PrivateLinkConfig, T> g) {
        return obj -> g.apply((PrivateLinkConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PrivateLinkConfig> {
        /**
         * <p>
         * The ID of the VPC endpoint that is configured for an agent. An agent that is configured with a VPC endpoint
         * will not be accessible over the public internet.
         * </p>
         * 
         * @param vpcEndpointId
         *        The ID of the VPC endpoint that is configured for an agent. An agent that is configured with a VPC
         *        endpoint will not be accessible over the public internet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcEndpointId(String vpcEndpointId);

        /**
         * <p>
         * The private endpoint that is configured for an agent that has access to IP addresses in a <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-service.html">PrivateLink</a>. An agent that
         * is configured with this endpoint will not be accessible over the public internet.
         * </p>
         * 
         * @param privateLinkEndpoint
         *        The private endpoint that is configured for an agent that has access to IP addresses in a <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-service.html">PrivateLink</a>. An
         *        agent that is configured with this endpoint will not be accessible over the public internet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateLinkEndpoint(String privateLinkEndpoint);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the subnets that are configured for an agent activated in a VPC or an
         * agent that has access to a VPC endpoint.
         * </p>
         * 
         * @param subnetArns
         *        The Amazon Resource Names (ARNs) of the subnets that are configured for an agent activated in a VPC or
         *        an agent that has access to a VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetArns(Collection<String> subnetArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the subnets that are configured for an agent activated in a VPC or an
         * agent that has access to a VPC endpoint.
         * </p>
         * 
         * @param subnetArns
         *        The Amazon Resource Names (ARNs) of the subnets that are configured for an agent activated in a VPC or
         *        an agent that has access to a VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetArns(String... subnetArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the security groups that are configured for the EC2 resource that hosts
         * an agent activated in a VPC or an agent that has access to a VPC endpoint.
         * </p>
         * 
         * @param securityGroupArns
         *        The Amazon Resource Names (ARNs) of the security groups that are configured for the EC2 resource that
         *        hosts an agent activated in a VPC or an agent that has access to a VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(Collection<String> securityGroupArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the security groups that are configured for the EC2 resource that hosts
         * an agent activated in a VPC or an agent that has access to a VPC endpoint.
         * </p>
         * 
         * @param securityGroupArns
         *        The Amazon Resource Names (ARNs) of the security groups that are configured for the EC2 resource that
         *        hosts an agent activated in a VPC or an agent that has access to a VPC endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(String... securityGroupArns);
    }

    static final class BuilderImpl implements Builder {
        private String vpcEndpointId;

        private String privateLinkEndpoint;

        private List<String> subnetArns = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PrivateLinkConfig model) {
            vpcEndpointId(model.vpcEndpointId);
            privateLinkEndpoint(model.privateLinkEndpoint);
            subnetArns(model.subnetArns);
            securityGroupArns(model.securityGroupArns);
        }

        public final String getVpcEndpointId() {
            return vpcEndpointId;
        }

        public final void setVpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
        }

        @Override
        @Transient
        public final Builder vpcEndpointId(String vpcEndpointId) {
            this.vpcEndpointId = vpcEndpointId;
            return this;
        }

        public final String getPrivateLinkEndpoint() {
            return privateLinkEndpoint;
        }

        public final void setPrivateLinkEndpoint(String privateLinkEndpoint) {
            this.privateLinkEndpoint = privateLinkEndpoint;
        }

        @Override
        @Transient
        public final Builder privateLinkEndpoint(String privateLinkEndpoint) {
            this.privateLinkEndpoint = privateLinkEndpoint;
            return this;
        }

        public final Collection<String> getSubnetArns() {
            if (subnetArns instanceof SdkAutoConstructList) {
                return null;
            }
            return subnetArns;
        }

        public final void setSubnetArns(Collection<String> subnetArns) {
            this.subnetArns = PLSubnetArnListCopier.copy(subnetArns);
        }

        @Override
        @Transient
        public final Builder subnetArns(Collection<String> subnetArns) {
            this.subnetArns = PLSubnetArnListCopier.copy(subnetArns);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder subnetArns(String... subnetArns) {
            subnetArns(Arrays.asList(subnetArns));
            return this;
        }

        public final Collection<String> getSecurityGroupArns() {
            if (securityGroupArns instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupArns;
        }

        public final void setSecurityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = PLSecurityGroupArnListCopier.copy(securityGroupArns);
        }

        @Override
        @Transient
        public final Builder securityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = PLSecurityGroupArnListCopier.copy(securityGroupArns);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder securityGroupArns(String... securityGroupArns) {
            securityGroupArns(Arrays.asList(securityGroupArns));
            return this;
        }

        @Override
        public PrivateLinkConfig build() {
            return new PrivateLinkConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
