/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the options that are available to control the behavior of a <a
 * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_StartTaskExecution.html">StartTaskExecution</a>
 * operation. Behavior includes preserving metadata such as user ID (UID), group ID (GID), and file permissions, and
 * also overwriting files in the destination, data integrity verification, and so on.
 * </p>
 * <p>
 * A task has a set of default options associated with it. If you don't specify an option in <a
 * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_StartTaskExecution.html">StartTaskExecution</a>, the
 * default value is used. You can override the defaults options on each task execution by specifying an overriding
 * <code>Options</code> value to <a
 * href="https://docs.aws.amazon.com/datasync/latest/userguide/API_StartTaskExecution.html">StartTaskExecution</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Options implements SdkPojo, Serializable, ToCopyableBuilder<Options.Builder, Options> {
    private static final SdkField<String> VERIFY_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VerifyMode").getter(getter(Options::verifyModeAsString)).setter(setter(Builder::verifyMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerifyMode").build()).build();

    private static final SdkField<String> OVERWRITE_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OverwriteMode").getter(getter(Options::overwriteModeAsString)).setter(setter(Builder::overwriteMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OverwriteMode").build()).build();

    private static final SdkField<String> ATIME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Atime")
            .getter(getter(Options::atimeAsString)).setter(setter(Builder::atime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Atime").build()).build();

    private static final SdkField<String> MTIME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mtime")
            .getter(getter(Options::mtimeAsString)).setter(setter(Builder::mtime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mtime").build()).build();

    private static final SdkField<String> UID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Uid")
            .getter(getter(Options::uidAsString)).setter(setter(Builder::uid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Uid").build()).build();

    private static final SdkField<String> GID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Gid")
            .getter(getter(Options::gidAsString)).setter(setter(Builder::gid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Gid").build()).build();

    private static final SdkField<String> PRESERVE_DELETED_FILES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreserveDeletedFiles").getter(getter(Options::preserveDeletedFilesAsString))
            .setter(setter(Builder::preserveDeletedFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreserveDeletedFiles").build())
            .build();

    private static final SdkField<String> PRESERVE_DEVICES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreserveDevices").getter(getter(Options::preserveDevicesAsString))
            .setter(setter(Builder::preserveDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreserveDevices").build()).build();

    private static final SdkField<String> POSIX_PERMISSIONS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PosixPermissions").getter(getter(Options::posixPermissionsAsString))
            .setter(setter(Builder::posixPermissions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PosixPermissions").build()).build();

    private static final SdkField<Long> BYTES_PER_SECOND_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BytesPerSecond").getter(getter(Options::bytesPerSecond)).setter(setter(Builder::bytesPerSecond))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BytesPerSecond").build()).build();

    private static final SdkField<String> TASK_QUEUEING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskQueueing").getter(getter(Options::taskQueueingAsString)).setter(setter(Builder::taskQueueing))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskQueueing").build()).build();

    private static final SdkField<String> LOG_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogLevel").getter(getter(Options::logLevelAsString)).setter(setter(Builder::logLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogLevel").build()).build();

    private static final SdkField<String> TRANSFER_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TransferMode").getter(getter(Options::transferModeAsString)).setter(setter(Builder::transferMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransferMode").build()).build();

    private static final SdkField<String> SECURITY_DESCRIPTOR_COPY_FLAGS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SecurityDescriptorCopyFlags")
            .getter(getter(Options::securityDescriptorCopyFlagsAsString))
            .setter(setter(Builder::securityDescriptorCopyFlags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityDescriptorCopyFlags")
                    .build()).build();

    private static final SdkField<String> OBJECT_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ObjectTags").getter(getter(Options::objectTagsAsString)).setter(setter(Builder::objectTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectTags").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERIFY_MODE_FIELD,
            OVERWRITE_MODE_FIELD, ATIME_FIELD, MTIME_FIELD, UID_FIELD, GID_FIELD, PRESERVE_DELETED_FILES_FIELD,
            PRESERVE_DEVICES_FIELD, POSIX_PERMISSIONS_FIELD, BYTES_PER_SECOND_FIELD, TASK_QUEUEING_FIELD, LOG_LEVEL_FIELD,
            TRANSFER_MODE_FIELD, SECURITY_DESCRIPTOR_COPY_FLAGS_FIELD, OBJECT_TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String verifyMode;

    private final String overwriteMode;

    private final String atime;

    private final String mtime;

    private final String uid;

    private final String gid;

    private final String preserveDeletedFiles;

    private final String preserveDevices;

    private final String posixPermissions;

    private final Long bytesPerSecond;

    private final String taskQueueing;

    private final String logLevel;

    private final String transferMode;

    private final String securityDescriptorCopyFlags;

    private final String objectTags;

    private Options(BuilderImpl builder) {
        this.verifyMode = builder.verifyMode;
        this.overwriteMode = builder.overwriteMode;
        this.atime = builder.atime;
        this.mtime = builder.mtime;
        this.uid = builder.uid;
        this.gid = builder.gid;
        this.preserveDeletedFiles = builder.preserveDeletedFiles;
        this.preserveDevices = builder.preserveDevices;
        this.posixPermissions = builder.posixPermissions;
        this.bytesPerSecond = builder.bytesPerSecond;
        this.taskQueueing = builder.taskQueueing;
        this.logLevel = builder.logLevel;
        this.transferMode = builder.transferMode;
        this.securityDescriptorCopyFlags = builder.securityDescriptorCopyFlags;
        this.objectTags = builder.objectTags;
    }

    /**
     * <p>
     * A value that determines whether a data integrity verification should be performed at the end of a task execution
     * after all data and metadata have been transferred. For more information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task settings</a>.
     * </p>
     * <p>
     * Default value: POINT_IN_TIME_CONSISTENT.
     * </p>
     * <p>
     * ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
     * </p>
     * <p>
     * POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to verify that
     * source and destination are fully synchronized. This option isn't supported when transferring to S3 Glacier
     * Flexible Retrieval or S3 Glacier Deep Archive storage classes.
     * </p>
     * <p>
     * NONE: No additional verification is done at the end of the transfer, but all data transmissions are
     * integrity-checked with checksum verification during the transfer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #verifyMode} will
     * return {@link VerifyMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #verifyModeAsString}.
     * </p>
     * 
     * @return A value that determines whether a data integrity verification should be performed at the end of a task
     *         execution after all data and metadata have been transferred. For more information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task
     *         settings</a>. </p>
     *         <p>
     *         Default value: POINT_IN_TIME_CONSISTENT.
     *         </p>
     *         <p>
     *         ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
     *         </p>
     *         <p>
     *         POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to
     *         verify that source and destination are fully synchronized. This option isn't supported when transferring
     *         to S3 Glacier Flexible Retrieval or S3 Glacier Deep Archive storage classes.
     *         </p>
     *         <p>
     *         NONE: No additional verification is done at the end of the transfer, but all data transmissions are
     *         integrity-checked with checksum verification during the transfer.
     * @see VerifyMode
     */
    public final VerifyMode verifyMode() {
        return VerifyMode.fromValue(verifyMode);
    }

    /**
     * <p>
     * A value that determines whether a data integrity verification should be performed at the end of a task execution
     * after all data and metadata have been transferred. For more information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task settings</a>.
     * </p>
     * <p>
     * Default value: POINT_IN_TIME_CONSISTENT.
     * </p>
     * <p>
     * ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
     * </p>
     * <p>
     * POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to verify that
     * source and destination are fully synchronized. This option isn't supported when transferring to S3 Glacier
     * Flexible Retrieval or S3 Glacier Deep Archive storage classes.
     * </p>
     * <p>
     * NONE: No additional verification is done at the end of the transfer, but all data transmissions are
     * integrity-checked with checksum verification during the transfer.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #verifyMode} will
     * return {@link VerifyMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #verifyModeAsString}.
     * </p>
     * 
     * @return A value that determines whether a data integrity verification should be performed at the end of a task
     *         execution after all data and metadata have been transferred. For more information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task
     *         settings</a>. </p>
     *         <p>
     *         Default value: POINT_IN_TIME_CONSISTENT.
     *         </p>
     *         <p>
     *         ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
     *         </p>
     *         <p>
     *         POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to
     *         verify that source and destination are fully synchronized. This option isn't supported when transferring
     *         to S3 Glacier Flexible Retrieval or S3 Glacier Deep Archive storage classes.
     *         </p>
     *         <p>
     *         NONE: No additional verification is done at the end of the transfer, but all data transmissions are
     *         integrity-checked with checksum verification during the transfer.
     * @see VerifyMode
     */
    public final String verifyModeAsString() {
        return verifyMode;
    }

    /**
     * <p>
     * A value that determines whether files at the destination should be overwritten or preserved when copying files.
     * If set to <code>NEVER</code> a destination file will not be replaced by a source file, even if the destination
     * file differs from the source file. If you modify files in the destination and you sync the files, you can use
     * this value to protect against overwriting those changes.
     * </p>
     * <p>
     * Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed information, see
     * <a href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes">
     * Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #overwriteMode}
     * will return {@link OverwriteMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #overwriteModeAsString}.
     * </p>
     * 
     * @return A value that determines whether files at the destination should be overwritten or preserved when copying
     *         files. If set to <code>NEVER</code> a destination file will not be replaced by a source file, even if the
     *         destination file differs from the source file. If you modify files in the destination and you sync the
     *         files, you can use this value to protect against overwriting those changes. </p>
     *         <p>
     *         Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     *         >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
     *         Guide</i>.
     * @see OverwriteMode
     */
    public final OverwriteMode overwriteMode() {
        return OverwriteMode.fromValue(overwriteMode);
    }

    /**
     * <p>
     * A value that determines whether files at the destination should be overwritten or preserved when copying files.
     * If set to <code>NEVER</code> a destination file will not be replaced by a source file, even if the destination
     * file differs from the source file. If you modify files in the destination and you sync the files, you can use
     * this value to protect against overwriting those changes.
     * </p>
     * <p>
     * Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed information, see
     * <a href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes">
     * Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #overwriteMode}
     * will return {@link OverwriteMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #overwriteModeAsString}.
     * </p>
     * 
     * @return A value that determines whether files at the destination should be overwritten or preserved when copying
     *         files. If set to <code>NEVER</code> a destination file will not be replaced by a source file, even if the
     *         destination file differs from the source file. If you modify files in the destination and you sync the
     *         files, you can use this value to protect against overwriting those changes. </p>
     *         <p>
     *         Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     *         >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
     *         Guide</i>.
     * @see OverwriteMode
     */
    public final String overwriteModeAsString() {
        return overwriteMode;
    }

    /**
     * <p>
     * A file metadata value that shows the last time a file was accessed (that is, when the file was read or written
     * to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original <code>Atime</code>
     * attribute on all source files (that is, the version before the PREPARING phase). However, <code>Atime</code>'s
     * behavior is not fully standard across platforms, so DataSync can only do this on a best-effort basis.
     * </p>
     * <p>
     * Default value: BEST_EFFORT.
     * </p>
     * <p>
     * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     * </p>
     * <p>
     * NONE: Ignore <code>Atime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     * </p>
     * <p>
     * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #atime} will return
     * {@link Atime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #atimeAsString}.
     * </p>
     * 
     * @return A file metadata value that shows the last time a file was accessed (that is, when the file was read or
     *         written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
     *         <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
     *         However, <code>Atime</code>'s behavior is not fully standard across platforms, so DataSync can only do
     *         this on a best-effort basis. </p>
     *         <p>
     *         Default value: BEST_EFFORT.
     *         </p>
     *         <p>
     *         BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Atime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     *         </p>
     *         <p>
     *         If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     *         </p>
     * @see Atime
     */
    public final Atime atime() {
        return Atime.fromValue(atime);
    }

    /**
     * <p>
     * A file metadata value that shows the last time a file was accessed (that is, when the file was read or written
     * to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original <code>Atime</code>
     * attribute on all source files (that is, the version before the PREPARING phase). However, <code>Atime</code>'s
     * behavior is not fully standard across platforms, so DataSync can only do this on a best-effort basis.
     * </p>
     * <p>
     * Default value: BEST_EFFORT.
     * </p>
     * <p>
     * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     * </p>
     * <p>
     * NONE: Ignore <code>Atime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     * </p>
     * <p>
     * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #atime} will return
     * {@link Atime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #atimeAsString}.
     * </p>
     * 
     * @return A file metadata value that shows the last time a file was accessed (that is, when the file was read or
     *         written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
     *         <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
     *         However, <code>Atime</code>'s behavior is not fully standard across platforms, so DataSync can only do
     *         this on a best-effort basis. </p>
     *         <p>
     *         Default value: BEST_EFFORT.
     *         </p>
     *         <p>
     *         BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Atime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     *         </p>
     *         <p>
     *         If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     *         </p>
     * @see Atime
     */
    public final String atimeAsString() {
        return atime;
    }

    /**
     * <p>
     * A value that indicates the last time that a file was modified (that is, a file was written to) before the
     * PREPARING phase. This option is required for cases when you need to run the same task more than one time.
     * </p>
     * <p>
     * Default Value: <code>PRESERVE</code>
     * </p>
     * <p>
     * PRESERVE: Preserve original <code>Mtime</code> (recommended)
     * </p>
     * <p>
     * NONE: Ignore <code>Mtime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     * </p>
     * <p>
     * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mtime} will return
     * {@link Mtime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mtimeAsString}.
     * </p>
     * 
     * @return A value that indicates the last time that a file was modified (that is, a file was written to) before the
     *         PREPARING phase. This option is required for cases when you need to run the same task more than one time.
     *         </p>
     *         <p>
     *         Default Value: <code>PRESERVE</code>
     *         </p>
     *         <p>
     *         PRESERVE: Preserve original <code>Mtime</code> (recommended)
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Mtime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     *         </p>
     *         <p>
     *         If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     *         </p>
     * @see Mtime
     */
    public final Mtime mtime() {
        return Mtime.fromValue(mtime);
    }

    /**
     * <p>
     * A value that indicates the last time that a file was modified (that is, a file was written to) before the
     * PREPARING phase. This option is required for cases when you need to run the same task more than one time.
     * </p>
     * <p>
     * Default Value: <code>PRESERVE</code>
     * </p>
     * <p>
     * PRESERVE: Preserve original <code>Mtime</code> (recommended)
     * </p>
     * <p>
     * NONE: Ignore <code>Mtime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     * </p>
     * <p>
     * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mtime} will return
     * {@link Mtime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mtimeAsString}.
     * </p>
     * 
     * @return A value that indicates the last time that a file was modified (that is, a file was written to) before the
     *         PREPARING phase. This option is required for cases when you need to run the same task more than one time.
     *         </p>
     *         <p>
     *         Default Value: <code>PRESERVE</code>
     *         </p>
     *         <p>
     *         PRESERVE: Preserve original <code>Mtime</code> (recommended)
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Mtime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     *         </p>
     *         <p>
     *         If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     *         </p>
     * @see Mtime
     */
    public final String mtimeAsString() {
        return mtime;
    }

    /**
     * <p>
     * The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3 locations. To
     * learn more about what metadata is copied by DataSync, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata Copied
     * by DataSync</a>.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #uid} will return
     * {@link Uid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #uidAsString}.
     * </p>
     * 
     * @return The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3
     *         locations. To learn more about what metadata is copied by DataSync, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
     *         Copied by DataSync</a>.</p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Uid
     */
    public final Uid uid() {
        return Uid.fromValue(uid);
    }

    /**
     * <p>
     * The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3 locations. To
     * learn more about what metadata is copied by DataSync, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata Copied
     * by DataSync</a>.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #uid} will return
     * {@link Uid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #uidAsString}.
     * </p>
     * 
     * @return The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3
     *         locations. To learn more about what metadata is copied by DataSync, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
     *         Copied by DataSync</a>.</p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Uid
     */
    public final String uidAsString() {
        return uid;
    }

    /**
     * <p>
     * The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3 locations. For
     * more information about what metadata is copied by DataSync, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata Copied
     * by DataSync</a>.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #gid} will return
     * {@link Gid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #gidAsString}.
     * </p>
     * 
     * @return The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3
     *         locations. For more information about what metadata is copied by DataSync, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
     *         Copied by DataSync</a>. </p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Gid
     */
    public final Gid gid() {
        return Gid.fromValue(gid);
    }

    /**
     * <p>
     * The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3 locations. For
     * more information about what metadata is copied by DataSync, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata Copied
     * by DataSync</a>.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #gid} will return
     * {@link Gid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #gidAsString}.
     * </p>
     * 
     * @return The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3
     *         locations. For more information about what metadata is copied by DataSync, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
     *         Copied by DataSync</a>. </p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Gid
     */
    public final String gidAsString() {
        return gid;
    }

    /**
     * <p>
     * A value that specifies whether files in the destination that don't exist in the source file system should be
     * preserved. This option can affect your storage cost. If your task deletes objects, you might incur minimum
     * storage duration charges for certain storage classes. For detailed information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     * >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User Guide</i>.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Ignore such destination files (recommended).
     * </p>
     * <p>
     * REMOVE: Delete destination files that aren’t present in the source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #preserveDeletedFiles} will return {@link PreserveDeletedFiles#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #preserveDeletedFilesAsString}.
     * </p>
     * 
     * @return A value that specifies whether files in the destination that don't exist in the source file system should
     *         be preserved. This option can affect your storage cost. If your task deletes objects, you might incur
     *         minimum storage duration charges for certain storage classes. For detailed information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     *         >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
     *         Guide</i>.</p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Ignore such destination files (recommended).
     *         </p>
     *         <p>
     *         REMOVE: Delete destination files that aren’t present in the source.
     * @see PreserveDeletedFiles
     */
    public final PreserveDeletedFiles preserveDeletedFiles() {
        return PreserveDeletedFiles.fromValue(preserveDeletedFiles);
    }

    /**
     * <p>
     * A value that specifies whether files in the destination that don't exist in the source file system should be
     * preserved. This option can affect your storage cost. If your task deletes objects, you might incur minimum
     * storage duration charges for certain storage classes. For detailed information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     * >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User Guide</i>.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Ignore such destination files (recommended).
     * </p>
     * <p>
     * REMOVE: Delete destination files that aren’t present in the source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #preserveDeletedFiles} will return {@link PreserveDeletedFiles#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #preserveDeletedFilesAsString}.
     * </p>
     * 
     * @return A value that specifies whether files in the destination that don't exist in the source file system should
     *         be preserved. This option can affect your storage cost. If your task deletes objects, you might incur
     *         minimum storage duration charges for certain storage classes. For detailed information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
     *         >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
     *         Guide</i>.</p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Ignore such destination files (recommended).
     *         </p>
     *         <p>
     *         REMOVE: Delete destination files that aren’t present in the source.
     * @see PreserveDeletedFiles
     */
    public final String preserveDeletedFilesAsString() {
        return preserveDeletedFiles;
    }

    /**
     * <p>
     * A value that determines whether DataSync should preserve the metadata of block and character devices in the
     * source file system, and re-create the files with that device name and metadata on the destination. DataSync does
     * not copy the contents of such devices, only the name and metadata.
     * </p>
     * <note>
     * <p>
     * DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
     * end-of-file (EOF) marker.
     * </p>
     * </note>
     * <p>
     * Default value: NONE.
     * </p>
     * <p>
     * NONE: Ignore special devices (recommended).
     * </p>
     * <p>
     * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preserveDevices}
     * will return {@link PreserveDevices#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #preserveDevicesAsString}.
     * </p>
     * 
     * @return A value that determines whether DataSync should preserve the metadata of block and character devices in
     *         the source file system, and re-create the files with that device name and metadata on the destination.
     *         DataSync does not copy the contents of such devices, only the name and metadata. </p> <note>
     *         <p>
     *         DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
     *         end-of-file (EOF) marker.
     *         </p>
     *         </note>
     *         <p>
     *         Default value: NONE.
     *         </p>
     *         <p>
     *         NONE: Ignore special devices (recommended).
     *         </p>
     *         <p>
     *         PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon
     *         EFS.
     * @see PreserveDevices
     */
    public final PreserveDevices preserveDevices() {
        return PreserveDevices.fromValue(preserveDevices);
    }

    /**
     * <p>
     * A value that determines whether DataSync should preserve the metadata of block and character devices in the
     * source file system, and re-create the files with that device name and metadata on the destination. DataSync does
     * not copy the contents of such devices, only the name and metadata.
     * </p>
     * <note>
     * <p>
     * DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
     * end-of-file (EOF) marker.
     * </p>
     * </note>
     * <p>
     * Default value: NONE.
     * </p>
     * <p>
     * NONE: Ignore special devices (recommended).
     * </p>
     * <p>
     * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preserveDevices}
     * will return {@link PreserveDevices#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #preserveDevicesAsString}.
     * </p>
     * 
     * @return A value that determines whether DataSync should preserve the metadata of block and character devices in
     *         the source file system, and re-create the files with that device name and metadata on the destination.
     *         DataSync does not copy the contents of such devices, only the name and metadata. </p> <note>
     *         <p>
     *         DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
     *         end-of-file (EOF) marker.
     *         </p>
     *         </note>
     *         <p>
     *         Default value: NONE.
     *         </p>
     *         <p>
     *         NONE: Ignore special devices (recommended).
     *         </p>
     *         <p>
     *         PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon
     *         EFS.
     * @see PreserveDevices
     */
    public final String preserveDevicesAsString() {
        return preserveDevices;
    }

    /**
     * <p>
     * A value that determines which users or groups can access a file for a specific purpose such as reading, writing,
     * or execution of the file. This option should only be set for NFS, EFS, and S3 locations. For more information
     * about what metadata is copied by DataSync, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata Copied
     * by DataSync</a>.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Preserve POSIX-style permissions (recommended).
     * </p>
     * <p>
     * NONE: Ignore permissions.
     * </p>
     * <note>
     * <p>
     * DataSync can preserve extant permissions of a source location.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #posixPermissions}
     * will return {@link PosixPermissions#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #posixPermissionsAsString}.
     * </p>
     * 
     * @return A value that determines which users or groups can access a file for a specific purpose such as reading,
     *         writing, or execution of the file. This option should only be set for NFS, EFS, and S3 locations. For
     *         more information about what metadata is copied by DataSync, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
     *         Copied by DataSync</a>. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Preserve POSIX-style permissions (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore permissions.
     *         </p>
     *         <note>
     *         <p>
     *         DataSync can preserve extant permissions of a source location.
     *         </p>
     * @see PosixPermissions
     */
    public final PosixPermissions posixPermissions() {
        return PosixPermissions.fromValue(posixPermissions);
    }

    /**
     * <p>
     * A value that determines which users or groups can access a file for a specific purpose such as reading, writing,
     * or execution of the file. This option should only be set for NFS, EFS, and S3 locations. For more information
     * about what metadata is copied by DataSync, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata Copied
     * by DataSync</a>.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Preserve POSIX-style permissions (recommended).
     * </p>
     * <p>
     * NONE: Ignore permissions.
     * </p>
     * <note>
     * <p>
     * DataSync can preserve extant permissions of a source location.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #posixPermissions}
     * will return {@link PosixPermissions#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #posixPermissionsAsString}.
     * </p>
     * 
     * @return A value that determines which users or groups can access a file for a specific purpose such as reading,
     *         writing, or execution of the file. This option should only be set for NFS, EFS, and S3 locations. For
     *         more information about what metadata is copied by DataSync, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
     *         Copied by DataSync</a>. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Preserve POSIX-style permissions (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore permissions.
     *         </p>
     *         <note>
     *         <p>
     *         DataSync can preserve extant permissions of a source location.
     *         </p>
     * @see PosixPermissions
     */
    public final String posixPermissionsAsString() {
        return posixPermissions;
    }

    /**
     * <p>
     * A value that limits the bandwidth used by DataSync. For example, if you want DataSync to use a maximum of 1 MB,
     * set this value to <code>1048576</code> (<code>=1024*1024</code>).
     * </p>
     * 
     * @return A value that limits the bandwidth used by DataSync. For example, if you want DataSync to use a maximum of
     *         1 MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
     */
    public final Long bytesPerSecond() {
        return bytesPerSecond;
    }

    /**
     * <p>
     * A value that determines whether tasks should be queued before executing the tasks. If set to <code>ENABLED</code>
     * , the tasks will be queued. The default is <code>ENABLED</code>.
     * </p>
     * <p>
     * If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more information,
     * see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution">Queueing
     * task executions</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskQueueing} will
     * return {@link TaskQueueing#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #taskQueueingAsString}.
     * </p>
     * 
     * @return A value that determines whether tasks should be queued before executing the tasks. If set to
     *         <code>ENABLED</code>, the tasks will be queued. The default is <code>ENABLED</code>.</p>
     *         <p>
     *         If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution">Queueing
     *         task executions</a>.
     * @see TaskQueueing
     */
    public final TaskQueueing taskQueueing() {
        return TaskQueueing.fromValue(taskQueueing);
    }

    /**
     * <p>
     * A value that determines whether tasks should be queued before executing the tasks. If set to <code>ENABLED</code>
     * , the tasks will be queued. The default is <code>ENABLED</code>.
     * </p>
     * <p>
     * If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more information,
     * see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution">Queueing
     * task executions</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskQueueing} will
     * return {@link TaskQueueing#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #taskQueueingAsString}.
     * </p>
     * 
     * @return A value that determines whether tasks should be queued before executing the tasks. If set to
     *         <code>ENABLED</code>, the tasks will be queued. The default is <code>ENABLED</code>.</p>
     *         <p>
     *         If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution">Queueing
     *         task executions</a>.
     * @see TaskQueueing
     */
    public final String taskQueueingAsString() {
        return taskQueueing;
    }

    /**
     * <p>
     * A value that determines the type of logs that DataSync publishes to a log stream in the Amazon CloudWatch log
     * group that you provide. For more information about providing a log group for DataSync, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
     * >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code> publishes logs
     * on errors for individual files transferred, and <code>TRANSFER</code> publishes logs for every file or object
     * that is transferred and integrity checked.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logLevel} will
     * return {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logLevelAsString}.
     * </p>
     * 
     * @return A value that determines the type of logs that DataSync publishes to a log stream in the Amazon CloudWatch
     *         log group that you provide. For more information about providing a log group for DataSync, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
     *         >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code>
     *         publishes logs on errors for individual files transferred, and <code>TRANSFER</code> publishes logs for
     *         every file or object that is transferred and integrity checked.
     * @see LogLevel
     */
    public final LogLevel logLevel() {
        return LogLevel.fromValue(logLevel);
    }

    /**
     * <p>
     * A value that determines the type of logs that DataSync publishes to a log stream in the Amazon CloudWatch log
     * group that you provide. For more information about providing a log group for DataSync, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
     * >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code> publishes logs
     * on errors for individual files transferred, and <code>TRANSFER</code> publishes logs for every file or object
     * that is transferred and integrity checked.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logLevel} will
     * return {@link LogLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #logLevelAsString}.
     * </p>
     * 
     * @return A value that determines the type of logs that DataSync publishes to a log stream in the Amazon CloudWatch
     *         log group that you provide. For more information about providing a log group for DataSync, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
     *         >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code>
     *         publishes logs on errors for individual files transferred, and <code>TRANSFER</code> publishes logs for
     *         every file or object that is transferred and integrity checked.
     * @see LogLevel
     */
    public final String logLevelAsString() {
        return logLevel;
    }

    /**
     * <p>
     * A value that determines whether DataSync transfers only the data and metadata that differ between the source and
     * the destination location, or whether DataSync transfers all the content from the source, without comparing to the
     * destination location.
     * </p>
     * <p>
     * CHANGED: DataSync copies only data or metadata that is new or different content from the source location to the
     * destination location.
     * </p>
     * <p>
     * ALL: DataSync copies all source location content to the destination, without comparing to existing content on the
     * destination.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transferMode} will
     * return {@link TransferMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #transferModeAsString}.
     * </p>
     * 
     * @return A value that determines whether DataSync transfers only the data and metadata that differ between the
     *         source and the destination location, or whether DataSync transfers all the content from the source,
     *         without comparing to the destination location. </p>
     *         <p>
     *         CHANGED: DataSync copies only data or metadata that is new or different content from the source location
     *         to the destination location.
     *         </p>
     *         <p>
     *         ALL: DataSync copies all source location content to the destination, without comparing to existing
     *         content on the destination.
     * @see TransferMode
     */
    public final TransferMode transferMode() {
        return TransferMode.fromValue(transferMode);
    }

    /**
     * <p>
     * A value that determines whether DataSync transfers only the data and metadata that differ between the source and
     * the destination location, or whether DataSync transfers all the content from the source, without comparing to the
     * destination location.
     * </p>
     * <p>
     * CHANGED: DataSync copies only data or metadata that is new or different content from the source location to the
     * destination location.
     * </p>
     * <p>
     * ALL: DataSync copies all source location content to the destination, without comparing to existing content on the
     * destination.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transferMode} will
     * return {@link TransferMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #transferModeAsString}.
     * </p>
     * 
     * @return A value that determines whether DataSync transfers only the data and metadata that differ between the
     *         source and the destination location, or whether DataSync transfers all the content from the source,
     *         without comparing to the destination location. </p>
     *         <p>
     *         CHANGED: DataSync copies only data or metadata that is new or different content from the source location
     *         to the destination location.
     *         </p>
     *         <p>
     *         ALL: DataSync copies all source location content to the destination, without comparing to existing
     *         content on the destination.
     * @see TransferMode
     */
    public final String transferModeAsString() {
        return transferMode;
    }

    /**
     * <p>
     * A value that determines which components of the SMB security descriptor are copied from source to destination
     * objects.
     * </p>
     * <p>
     * This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or between
     * two Amazon FSx for Windows File Server locations. For more information about how DataSync handles metadata, see
     * <a href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How DataSync Handles Metadata
     * and Special Files</a>.
     * </p>
     * <p>
     * Default value: OWNER_DACL.
     * </p>
     * <p>
     * <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Object owner.
     * </p>
     * </li>
     * <li>
     * <p>
     * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which are used by
     * administrators to log attempts to access a secured object.
     * </p>
     * <p>
     * <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Object owner.
     * </p>
     * </li>
     * <li>
     * <p>
     * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     * </p>
     * </li>
     * <li>
     * <p>
     * NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a secured
     * object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to access your SMB
     * location. For information about choosing a user that ensures sufficient permissions to files, folders, and
     * metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
     * </p>
     * <p>
     * <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned by the user
     * that was provided for accessing the destination location. DACLs and SACLs are set based on the destination
     * server’s configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #securityDescriptorCopyFlags} will return {@link SmbSecurityDescriptorCopyFlags#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #securityDescriptorCopyFlagsAsString}.
     * </p>
     * 
     * @return A value that determines which components of the SMB security descriptor are copied from source to
     *         destination objects. </p>
     *         <p>
     *         This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or
     *         between two Amazon FSx for Windows File Server locations. For more information about how DataSync handles
     *         metadata, see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How
     *         DataSync Handles Metadata and Special Files</a>.
     *         </p>
     *         <p>
     *         Default value: OWNER_DACL.
     *         </p>
     *         <p>
     *         <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Object owner.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which are
     *         used by administrators to log attempts to access a secured object.
     *         </p>
     *         <p>
     *         <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Object owner.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a
     *         secured object.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to access
     *         your SMB location. For information about choosing a user that ensures sufficient permissions to files,
     *         folders, and metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
     *         </p>
     *         <p>
     *         <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned by
     *         the user that was provided for accessing the destination location. DACLs and SACLs are set based on the
     *         destination server’s configuration.
     * @see SmbSecurityDescriptorCopyFlags
     */
    public final SmbSecurityDescriptorCopyFlags securityDescriptorCopyFlags() {
        return SmbSecurityDescriptorCopyFlags.fromValue(securityDescriptorCopyFlags);
    }

    /**
     * <p>
     * A value that determines which components of the SMB security descriptor are copied from source to destination
     * objects.
     * </p>
     * <p>
     * This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or between
     * two Amazon FSx for Windows File Server locations. For more information about how DataSync handles metadata, see
     * <a href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How DataSync Handles Metadata
     * and Special Files</a>.
     * </p>
     * <p>
     * Default value: OWNER_DACL.
     * </p>
     * <p>
     * <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Object owner.
     * </p>
     * </li>
     * <li>
     * <p>
     * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which are used by
     * administrators to log attempts to access a secured object.
     * </p>
     * <p>
     * <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Object owner.
     * </p>
     * </li>
     * <li>
     * <p>
     * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     * </p>
     * </li>
     * <li>
     * <p>
     * NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a secured
     * object.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to access your SMB
     * location. For information about choosing a user that ensures sufficient permissions to files, folders, and
     * metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
     * </p>
     * <p>
     * <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned by the user
     * that was provided for accessing the destination location. DACLs and SACLs are set based on the destination
     * server’s configuration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #securityDescriptorCopyFlags} will return {@link SmbSecurityDescriptorCopyFlags#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #securityDescriptorCopyFlagsAsString}.
     * </p>
     * 
     * @return A value that determines which components of the SMB security descriptor are copied from source to
     *         destination objects. </p>
     *         <p>
     *         This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or
     *         between two Amazon FSx for Windows File Server locations. For more information about how DataSync handles
     *         metadata, see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How
     *         DataSync Handles Metadata and Special Files</a>.
     *         </p>
     *         <p>
     *         Default value: OWNER_DACL.
     *         </p>
     *         <p>
     *         <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Object owner.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which are
     *         used by administrators to log attempts to access a secured object.
     *         </p>
     *         <p>
     *         <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Object owner.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a
     *         secured object.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to access
     *         your SMB location. For information about choosing a user that ensures sufficient permissions to files,
     *         folders, and metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
     *         </p>
     *         <p>
     *         <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned by
     *         the user that was provided for accessing the destination location. DACLs and SACLs are set based on the
     *         destination server’s configuration.
     * @see SmbSecurityDescriptorCopyFlags
     */
    public final String securityDescriptorCopyFlagsAsString() {
        return securityDescriptorCopyFlags;
    }

    /**
     * <p>
     * Specifies whether object tags are maintained when transferring between object storage systems. If you want your
     * DataSync task to ignore object tags, specify the <code>NONE</code> value.
     * </p>
     * <p>
     * Default Value: <code>PRESERVE</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #objectTags} will
     * return {@link ObjectTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #objectTagsAsString}.
     * </p>
     * 
     * @return Specifies whether object tags are maintained when transferring between object storage systems. If you
     *         want your DataSync task to ignore object tags, specify the <code>NONE</code> value.</p>
     *         <p>
     *         Default Value: <code>PRESERVE</code>
     * @see ObjectTags
     */
    public final ObjectTags objectTags() {
        return ObjectTags.fromValue(objectTags);
    }

    /**
     * <p>
     * Specifies whether object tags are maintained when transferring between object storage systems. If you want your
     * DataSync task to ignore object tags, specify the <code>NONE</code> value.
     * </p>
     * <p>
     * Default Value: <code>PRESERVE</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #objectTags} will
     * return {@link ObjectTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #objectTagsAsString}.
     * </p>
     * 
     * @return Specifies whether object tags are maintained when transferring between object storage systems. If you
     *         want your DataSync task to ignore object tags, specify the <code>NONE</code> value.</p>
     *         <p>
     *         Default Value: <code>PRESERVE</code>
     * @see ObjectTags
     */
    public final String objectTagsAsString() {
        return objectTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(verifyModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(overwriteModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(atimeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mtimeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(uidAsString());
        hashCode = 31 * hashCode + Objects.hashCode(gidAsString());
        hashCode = 31 * hashCode + Objects.hashCode(preserveDeletedFilesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(preserveDevicesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(posixPermissionsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bytesPerSecond());
        hashCode = 31 * hashCode + Objects.hashCode(taskQueueingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transferModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(securityDescriptorCopyFlagsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(objectTagsAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Options)) {
            return false;
        }
        Options other = (Options) obj;
        return Objects.equals(verifyModeAsString(), other.verifyModeAsString())
                && Objects.equals(overwriteModeAsString(), other.overwriteModeAsString())
                && Objects.equals(atimeAsString(), other.atimeAsString())
                && Objects.equals(mtimeAsString(), other.mtimeAsString()) && Objects.equals(uidAsString(), other.uidAsString())
                && Objects.equals(gidAsString(), other.gidAsString())
                && Objects.equals(preserveDeletedFilesAsString(), other.preserveDeletedFilesAsString())
                && Objects.equals(preserveDevicesAsString(), other.preserveDevicesAsString())
                && Objects.equals(posixPermissionsAsString(), other.posixPermissionsAsString())
                && Objects.equals(bytesPerSecond(), other.bytesPerSecond())
                && Objects.equals(taskQueueingAsString(), other.taskQueueingAsString())
                && Objects.equals(logLevelAsString(), other.logLevelAsString())
                && Objects.equals(transferModeAsString(), other.transferModeAsString())
                && Objects.equals(securityDescriptorCopyFlagsAsString(), other.securityDescriptorCopyFlagsAsString())
                && Objects.equals(objectTagsAsString(), other.objectTagsAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Options").add("VerifyMode", verifyModeAsString()).add("OverwriteMode", overwriteModeAsString())
                .add("Atime", atimeAsString()).add("Mtime", mtimeAsString()).add("Uid", uidAsString()).add("Gid", gidAsString())
                .add("PreserveDeletedFiles", preserveDeletedFilesAsString()).add("PreserveDevices", preserveDevicesAsString())
                .add("PosixPermissions", posixPermissionsAsString()).add("BytesPerSecond", bytesPerSecond())
                .add("TaskQueueing", taskQueueingAsString()).add("LogLevel", logLevelAsString())
                .add("TransferMode", transferModeAsString())
                .add("SecurityDescriptorCopyFlags", securityDescriptorCopyFlagsAsString())
                .add("ObjectTags", objectTagsAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VerifyMode":
            return Optional.ofNullable(clazz.cast(verifyModeAsString()));
        case "OverwriteMode":
            return Optional.ofNullable(clazz.cast(overwriteModeAsString()));
        case "Atime":
            return Optional.ofNullable(clazz.cast(atimeAsString()));
        case "Mtime":
            return Optional.ofNullable(clazz.cast(mtimeAsString()));
        case "Uid":
            return Optional.ofNullable(clazz.cast(uidAsString()));
        case "Gid":
            return Optional.ofNullable(clazz.cast(gidAsString()));
        case "PreserveDeletedFiles":
            return Optional.ofNullable(clazz.cast(preserveDeletedFilesAsString()));
        case "PreserveDevices":
            return Optional.ofNullable(clazz.cast(preserveDevicesAsString()));
        case "PosixPermissions":
            return Optional.ofNullable(clazz.cast(posixPermissionsAsString()));
        case "BytesPerSecond":
            return Optional.ofNullable(clazz.cast(bytesPerSecond()));
        case "TaskQueueing":
            return Optional.ofNullable(clazz.cast(taskQueueingAsString()));
        case "LogLevel":
            return Optional.ofNullable(clazz.cast(logLevelAsString()));
        case "TransferMode":
            return Optional.ofNullable(clazz.cast(transferModeAsString()));
        case "SecurityDescriptorCopyFlags":
            return Optional.ofNullable(clazz.cast(securityDescriptorCopyFlagsAsString()));
        case "ObjectTags":
            return Optional.ofNullable(clazz.cast(objectTagsAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Options, T> g) {
        return obj -> g.apply((Options) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Options> {
        /**
         * <p>
         * A value that determines whether a data integrity verification should be performed at the end of a task
         * execution after all data and metadata have been transferred. For more information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task settings</a>.
         * </p>
         * <p>
         * Default value: POINT_IN_TIME_CONSISTENT.
         * </p>
         * <p>
         * ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
         * </p>
         * <p>
         * POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to verify
         * that source and destination are fully synchronized. This option isn't supported when transferring to S3
         * Glacier Flexible Retrieval or S3 Glacier Deep Archive storage classes.
         * </p>
         * <p>
         * NONE: No additional verification is done at the end of the transfer, but all data transmissions are
         * integrity-checked with checksum verification during the transfer.
         * </p>
         * 
         * @param verifyMode
         *        A value that determines whether a data integrity verification should be performed at the end of a task
         *        execution after all data and metadata have been transferred. For more information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task
         *        settings</a>. </p>
         *        <p>
         *        Default value: POINT_IN_TIME_CONSISTENT.
         *        </p>
         *        <p>
         *        ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
         *        </p>
         *        <p>
         *        POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to
         *        verify that source and destination are fully synchronized. This option isn't supported when
         *        transferring to S3 Glacier Flexible Retrieval or S3 Glacier Deep Archive storage classes.
         *        </p>
         *        <p>
         *        NONE: No additional verification is done at the end of the transfer, but all data transmissions are
         *        integrity-checked with checksum verification during the transfer.
         * @see VerifyMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifyMode
         */
        Builder verifyMode(String verifyMode);

        /**
         * <p>
         * A value that determines whether a data integrity verification should be performed at the end of a task
         * execution after all data and metadata have been transferred. For more information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task settings</a>.
         * </p>
         * <p>
         * Default value: POINT_IN_TIME_CONSISTENT.
         * </p>
         * <p>
         * ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
         * </p>
         * <p>
         * POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to verify
         * that source and destination are fully synchronized. This option isn't supported when transferring to S3
         * Glacier Flexible Retrieval or S3 Glacier Deep Archive storage classes.
         * </p>
         * <p>
         * NONE: No additional verification is done at the end of the transfer, but all data transmissions are
         * integrity-checked with checksum verification during the transfer.
         * </p>
         * 
         * @param verifyMode
         *        A value that determines whether a data integrity verification should be performed at the end of a task
         *        execution after all data and metadata have been transferred. For more information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure task
         *        settings</a>. </p>
         *        <p>
         *        Default value: POINT_IN_TIME_CONSISTENT.
         *        </p>
         *        <p>
         *        ONLY_FILES_TRANSFERRED (recommended): Perform verification only on files that were transferred.
         *        </p>
         *        <p>
         *        POINT_IN_TIME_CONSISTENT: Scan the entire source and entire destination at the end of the transfer to
         *        verify that source and destination are fully synchronized. This option isn't supported when
         *        transferring to S3 Glacier Flexible Retrieval or S3 Glacier Deep Archive storage classes.
         *        </p>
         *        <p>
         *        NONE: No additional verification is done at the end of the transfer, but all data transmissions are
         *        integrity-checked with checksum verification during the transfer.
         * @see VerifyMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifyMode
         */
        Builder verifyMode(VerifyMode verifyMode);

        /**
         * <p>
         * A value that determines whether files at the destination should be overwritten or preserved when copying
         * files. If set to <code>NEVER</code> a destination file will not be replaced by a source file, even if the
         * destination file differs from the source file. If you modify files in the destination and you sync the files,
         * you can use this value to protect against overwriting those changes.
         * </p>
         * <p>
         * Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed information,
         * see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         * >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         * Guide</i>.
         * </p>
         * 
         * @param overwriteMode
         *        A value that determines whether files at the destination should be overwritten or preserved when
         *        copying files. If set to <code>NEVER</code> a destination file will not be replaced by a source file,
         *        even if the destination file differs from the source file. If you modify files in the destination and
         *        you sync the files, you can use this value to protect against overwriting those changes. </p>
         *        <p>
         *        Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         *        >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         *        Guide</i>.
         * @see OverwriteMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OverwriteMode
         */
        Builder overwriteMode(String overwriteMode);

        /**
         * <p>
         * A value that determines whether files at the destination should be overwritten or preserved when copying
         * files. If set to <code>NEVER</code> a destination file will not be replaced by a source file, even if the
         * destination file differs from the source file. If you modify files in the destination and you sync the files,
         * you can use this value to protect against overwriting those changes.
         * </p>
         * <p>
         * Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed information,
         * see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         * >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         * Guide</i>.
         * </p>
         * 
         * @param overwriteMode
         *        A value that determines whether files at the destination should be overwritten or preserved when
         *        copying files. If set to <code>NEVER</code> a destination file will not be replaced by a source file,
         *        even if the destination file differs from the source file. If you modify files in the destination and
         *        you sync the files, you can use this value to protect against overwriting those changes. </p>
         *        <p>
         *        Some storage classes have specific behaviors that can affect your S3 storage cost. For detailed
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         *        >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         *        Guide</i>.
         * @see OverwriteMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OverwriteMode
         */
        Builder overwriteMode(OverwriteMode overwriteMode);

        /**
         * <p>
         * A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         * written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         * <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase). However,
         * <code>Atime</code>'s behavior is not fully standard across platforms, so DataSync can only do this on a
         * best-effort basis.
         * </p>
         * <p>
         * Default value: BEST_EFFORT.
         * </p>
         * <p>
         * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         * </p>
         * <p>
         * NONE: Ignore <code>Atime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         * </p>
         * <p>
         * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         * </p>
         * </note>
         * 
         * @param atime
         *        A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         *        written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         *        <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
         *        However, <code>Atime</code>'s behavior is not fully standard across platforms, so DataSync can only do
         *        this on a best-effort basis. </p>
         *        <p>
         *        Default value: BEST_EFFORT.
         *        </p>
         *        <p>
         *        BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Atime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         *        </p>
         *        <p>
         *        If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         *        </p>
         * @see Atime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Atime
         */
        Builder atime(String atime);

        /**
         * <p>
         * A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         * written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         * <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase). However,
         * <code>Atime</code>'s behavior is not fully standard across platforms, so DataSync can only do this on a
         * best-effort basis.
         * </p>
         * <p>
         * Default value: BEST_EFFORT.
         * </p>
         * <p>
         * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         * </p>
         * <p>
         * NONE: Ignore <code>Atime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         * </p>
         * <p>
         * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         * </p>
         * </note>
         * 
         * @param atime
         *        A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         *        written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         *        <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
         *        However, <code>Atime</code>'s behavior is not fully standard across platforms, so DataSync can only do
         *        this on a best-effort basis. </p>
         *        <p>
         *        Default value: BEST_EFFORT.
         *        </p>
         *        <p>
         *        BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Atime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         *        </p>
         *        <p>
         *        If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         *        </p>
         * @see Atime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Atime
         */
        Builder atime(Atime atime);

        /**
         * <p>
         * A value that indicates the last time that a file was modified (that is, a file was written to) before the
         * PREPARING phase. This option is required for cases when you need to run the same task more than one time.
         * </p>
         * <p>
         * Default Value: <code>PRESERVE</code>
         * </p>
         * <p>
         * PRESERVE: Preserve original <code>Mtime</code> (recommended)
         * </p>
         * <p>
         * NONE: Ignore <code>Mtime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         * </p>
         * <p>
         * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         * </p>
         * </note>
         * 
         * @param mtime
         *        A value that indicates the last time that a file was modified (that is, a file was written to) before
         *        the PREPARING phase. This option is required for cases when you need to run the same task more than
         *        one time. </p>
         *        <p>
         *        Default Value: <code>PRESERVE</code>
         *        </p>
         *        <p>
         *        PRESERVE: Preserve original <code>Mtime</code> (recommended)
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Mtime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         *        </p>
         *        <p>
         *        If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         *        </p>
         * @see Mtime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Mtime
         */
        Builder mtime(String mtime);

        /**
         * <p>
         * A value that indicates the last time that a file was modified (that is, a file was written to) before the
         * PREPARING phase. This option is required for cases when you need to run the same task more than one time.
         * </p>
         * <p>
         * Default Value: <code>PRESERVE</code>
         * </p>
         * <p>
         * PRESERVE: Preserve original <code>Mtime</code> (recommended)
         * </p>
         * <p>
         * NONE: Ignore <code>Mtime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         * </p>
         * <p>
         * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         * </p>
         * </note>
         * 
         * @param mtime
         *        A value that indicates the last time that a file was modified (that is, a file was written to) before
         *        the PREPARING phase. This option is required for cases when you need to run the same task more than
         *        one time. </p>
         *        <p>
         *        Default Value: <code>PRESERVE</code>
         *        </p>
         *        <p>
         *        PRESERVE: Preserve original <code>Mtime</code> (recommended)
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Mtime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         *        </p>
         *        <p>
         *        If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         *        </p>
         * @see Mtime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Mtime
         */
        Builder mtime(Mtime mtime);

        /**
         * <p>
         * The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3 locations.
         * To learn more about what metadata is copied by DataSync, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
         * Copied by DataSync</a>.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param uid
         *        The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3
         *        locations. To learn more about what metadata is copied by DataSync, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied"
         *        >Metadata Copied by DataSync</a>.</p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Uid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Uid
         */
        Builder uid(String uid);

        /**
         * <p>
         * The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3 locations.
         * To learn more about what metadata is copied by DataSync, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
         * Copied by DataSync</a>.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param uid
         *        The POSIX user ID (UID) of the file's owner. This option should only be set for NFS, EFS, and S3
         *        locations. To learn more about what metadata is copied by DataSync, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied"
         *        >Metadata Copied by DataSync</a>.</p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Uid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Uid
         */
        Builder uid(Uid uid);

        /**
         * <p>
         * The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3 locations.
         * For more information about what metadata is copied by DataSync, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
         * Copied by DataSync</a>.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param gid
         *        The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3
         *        locations. For more information about what metadata is copied by DataSync, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied"
         *        >Metadata Copied by DataSync</a>. </p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Gid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Gid
         */
        Builder gid(String gid);

        /**
         * <p>
         * The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3 locations.
         * For more information about what metadata is copied by DataSync, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
         * Copied by DataSync</a>.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param gid
         *        The POSIX group ID (GID) of the file's owners. This option should only be set for NFS, EFS, and S3
         *        locations. For more information about what metadata is copied by DataSync, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied"
         *        >Metadata Copied by DataSync</a>. </p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Gid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Gid
         */
        Builder gid(Gid gid);

        /**
         * <p>
         * A value that specifies whether files in the destination that don't exist in the source file system should be
         * preserved. This option can affect your storage cost. If your task deletes objects, you might incur minimum
         * storage duration charges for certain storage classes. For detailed information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         * >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         * Guide</i>.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Ignore such destination files (recommended).
         * </p>
         * <p>
         * REMOVE: Delete destination files that aren’t present in the source.
         * </p>
         * 
         * @param preserveDeletedFiles
         *        A value that specifies whether files in the destination that don't exist in the source file system
         *        should be preserved. This option can affect your storage cost. If your task deletes objects, you might
         *        incur minimum storage duration charges for certain storage classes. For detailed information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         *        >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         *        Guide</i>.</p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Ignore such destination files (recommended).
         *        </p>
         *        <p>
         *        REMOVE: Delete destination files that aren’t present in the source.
         * @see PreserveDeletedFiles
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDeletedFiles
         */
        Builder preserveDeletedFiles(String preserveDeletedFiles);

        /**
         * <p>
         * A value that specifies whether files in the destination that don't exist in the source file system should be
         * preserved. This option can affect your storage cost. If your task deletes objects, you might incur minimum
         * storage duration charges for certain storage classes. For detailed information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         * >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         * Guide</i>.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Ignore such destination files (recommended).
         * </p>
         * <p>
         * REMOVE: Delete destination files that aren’t present in the source.
         * </p>
         * 
         * @param preserveDeletedFiles
         *        A value that specifies whether files in the destination that don't exist in the source file system
         *        should be preserved. This option can affect your storage cost. If your task deletes objects, you might
         *        incur minimum storage duration charges for certain storage classes. For detailed information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/create-s3-location.html#using-storage-classes"
         *        >Considerations when working with Amazon S3 storage classes in DataSync </a> in the <i>DataSync User
         *        Guide</i>.</p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Ignore such destination files (recommended).
         *        </p>
         *        <p>
         *        REMOVE: Delete destination files that aren’t present in the source.
         * @see PreserveDeletedFiles
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDeletedFiles
         */
        Builder preserveDeletedFiles(PreserveDeletedFiles preserveDeletedFiles);

        /**
         * <p>
         * A value that determines whether DataSync should preserve the metadata of block and character devices in the
         * source file system, and re-create the files with that device name and metadata on the destination. DataSync
         * does not copy the contents of such devices, only the name and metadata.
         * </p>
         * <note>
         * <p>
         * DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
         * end-of-file (EOF) marker.
         * </p>
         * </note>
         * <p>
         * Default value: NONE.
         * </p>
         * <p>
         * NONE: Ignore special devices (recommended).
         * </p>
         * <p>
         * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
         * </p>
         * 
         * @param preserveDevices
         *        A value that determines whether DataSync should preserve the metadata of block and character devices
         *        in the source file system, and re-create the files with that device name and metadata on the
         *        destination. DataSync does not copy the contents of such devices, only the name and metadata. </p>
         *        <note>
         *        <p>
         *        DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return
         *        an end-of-file (EOF) marker.
         *        </p>
         *        </note>
         *        <p>
         *        Default value: NONE.
         *        </p>
         *        <p>
         *        NONE: Ignore special devices (recommended).
         *        </p>
         *        <p>
         *        PRESERVE: Preserve character and block device metadata. This option isn't currently supported for
         *        Amazon EFS.
         * @see PreserveDevices
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDevices
         */
        Builder preserveDevices(String preserveDevices);

        /**
         * <p>
         * A value that determines whether DataSync should preserve the metadata of block and character devices in the
         * source file system, and re-create the files with that device name and metadata on the destination. DataSync
         * does not copy the contents of such devices, only the name and metadata.
         * </p>
         * <note>
         * <p>
         * DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
         * end-of-file (EOF) marker.
         * </p>
         * </note>
         * <p>
         * Default value: NONE.
         * </p>
         * <p>
         * NONE: Ignore special devices (recommended).
         * </p>
         * <p>
         * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
         * </p>
         * 
         * @param preserveDevices
         *        A value that determines whether DataSync should preserve the metadata of block and character devices
         *        in the source file system, and re-create the files with that device name and metadata on the
         *        destination. DataSync does not copy the contents of such devices, only the name and metadata. </p>
         *        <note>
         *        <p>
         *        DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return
         *        an end-of-file (EOF) marker.
         *        </p>
         *        </note>
         *        <p>
         *        Default value: NONE.
         *        </p>
         *        <p>
         *        NONE: Ignore special devices (recommended).
         *        </p>
         *        <p>
         *        PRESERVE: Preserve character and block device metadata. This option isn't currently supported for
         *        Amazon EFS.
         * @see PreserveDevices
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDevices
         */
        Builder preserveDevices(PreserveDevices preserveDevices);

        /**
         * <p>
         * A value that determines which users or groups can access a file for a specific purpose such as reading,
         * writing, or execution of the file. This option should only be set for NFS, EFS, and S3 locations. For more
         * information about what metadata is copied by DataSync, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
         * Copied by DataSync</a>.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Preserve POSIX-style permissions (recommended).
         * </p>
         * <p>
         * NONE: Ignore permissions.
         * </p>
         * <note>
         * <p>
         * DataSync can preserve extant permissions of a source location.
         * </p>
         * </note>
         * 
         * @param posixPermissions
         *        A value that determines which users or groups can access a file for a specific purpose such as
         *        reading, writing, or execution of the file. This option should only be set for NFS, EFS, and S3
         *        locations. For more information about what metadata is copied by DataSync, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied"
         *        >Metadata Copied by DataSync</a>. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Preserve POSIX-style permissions (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore permissions.
         *        </p>
         *        <note>
         *        <p>
         *        DataSync can preserve extant permissions of a source location.
         *        </p>
         * @see PosixPermissions
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PosixPermissions
         */
        Builder posixPermissions(String posixPermissions);

        /**
         * <p>
         * A value that determines which users or groups can access a file for a specific purpose such as reading,
         * writing, or execution of the file. This option should only be set for NFS, EFS, and S3 locations. For more
         * information about what metadata is copied by DataSync, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied">Metadata
         * Copied by DataSync</a>.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Preserve POSIX-style permissions (recommended).
         * </p>
         * <p>
         * NONE: Ignore permissions.
         * </p>
         * <note>
         * <p>
         * DataSync can preserve extant permissions of a source location.
         * </p>
         * </note>
         * 
         * @param posixPermissions
         *        A value that determines which users or groups can access a file for a specific purpose such as
         *        reading, writing, or execution of the file. This option should only be set for NFS, EFS, and S3
         *        locations. For more information about what metadata is copied by DataSync, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html#metadata-copied"
         *        >Metadata Copied by DataSync</a>. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Preserve POSIX-style permissions (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore permissions.
         *        </p>
         *        <note>
         *        <p>
         *        DataSync can preserve extant permissions of a source location.
         *        </p>
         * @see PosixPermissions
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PosixPermissions
         */
        Builder posixPermissions(PosixPermissions posixPermissions);

        /**
         * <p>
         * A value that limits the bandwidth used by DataSync. For example, if you want DataSync to use a maximum of 1
         * MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
         * </p>
         * 
         * @param bytesPerSecond
         *        A value that limits the bandwidth used by DataSync. For example, if you want DataSync to use a maximum
         *        of 1 MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytesPerSecond(Long bytesPerSecond);

        /**
         * <p>
         * A value that determines whether tasks should be queued before executing the tasks. If set to
         * <code>ENABLED</code>, the tasks will be queued. The default is <code>ENABLED</code>.
         * </p>
         * <p>
         * If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution">Queueing task
         * executions</a>.
         * </p>
         * 
         * @param taskQueueing
         *        A value that determines whether tasks should be queued before executing the tasks. If set to
         *        <code>ENABLED</code>, the tasks will be queued. The default is <code>ENABLED</code>.</p>
         *        <p>
         *        If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution"
         *        >Queueing task executions</a>.
         * @see TaskQueueing
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskQueueing
         */
        Builder taskQueueing(String taskQueueing);

        /**
         * <p>
         * A value that determines whether tasks should be queued before executing the tasks. If set to
         * <code>ENABLED</code>, the tasks will be queued. The default is <code>ENABLED</code>.
         * </p>
         * <p>
         * If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution">Queueing task
         * executions</a>.
         * </p>
         * 
         * @param taskQueueing
         *        A value that determines whether tasks should be queued before executing the tasks. If set to
         *        <code>ENABLED</code>, the tasks will be queued. The default is <code>ENABLED</code>.</p>
         *        <p>
         *        If you use the same agent to run multiple tasks, you can enable the tasks to run in series. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/run-task.html#queue-task-execution"
         *        >Queueing task executions</a>.
         * @see TaskQueueing
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TaskQueueing
         */
        Builder taskQueueing(TaskQueueing taskQueueing);

        /**
         * <p>
         * A value that determines the type of logs that DataSync publishes to a log stream in the Amazon CloudWatch log
         * group that you provide. For more information about providing a log group for DataSync, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
         * >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code> publishes
         * logs on errors for individual files transferred, and <code>TRANSFER</code> publishes logs for every file or
         * object that is transferred and integrity checked.
         * </p>
         * 
         * @param logLevel
         *        A value that determines the type of logs that DataSync publishes to a log stream in the Amazon
         *        CloudWatch log group that you provide. For more information about providing a log group for DataSync,
         *        see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
         *        >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code>
         *        publishes logs on errors for individual files transferred, and <code>TRANSFER</code> publishes logs
         *        for every file or object that is transferred and integrity checked.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder logLevel(String logLevel);

        /**
         * <p>
         * A value that determines the type of logs that DataSync publishes to a log stream in the Amazon CloudWatch log
         * group that you provide. For more information about providing a log group for DataSync, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
         * >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code> publishes
         * logs on errors for individual files transferred, and <code>TRANSFER</code> publishes logs for every file or
         * object that is transferred and integrity checked.
         * </p>
         * 
         * @param logLevel
         *        A value that determines the type of logs that DataSync publishes to a log stream in the Amazon
         *        CloudWatch log group that you provide. For more information about providing a log group for DataSync,
         *        see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html#DataSync-CreateTask-request-CloudWatchLogGroupArn"
         *        >CloudWatchLogGroupArn</a>. If set to <code>OFF</code>, no logs are published. <code>BASIC</code>
         *        publishes logs on errors for individual files transferred, and <code>TRANSFER</code> publishes logs
         *        for every file or object that is transferred and integrity checked.
         * @see LogLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LogLevel
         */
        Builder logLevel(LogLevel logLevel);

        /**
         * <p>
         * A value that determines whether DataSync transfers only the data and metadata that differ between the source
         * and the destination location, or whether DataSync transfers all the content from the source, without
         * comparing to the destination location.
         * </p>
         * <p>
         * CHANGED: DataSync copies only data or metadata that is new or different content from the source location to
         * the destination location.
         * </p>
         * <p>
         * ALL: DataSync copies all source location content to the destination, without comparing to existing content on
         * the destination.
         * </p>
         * 
         * @param transferMode
         *        A value that determines whether DataSync transfers only the data and metadata that differ between the
         *        source and the destination location, or whether DataSync transfers all the content from the source,
         *        without comparing to the destination location. </p>
         *        <p>
         *        CHANGED: DataSync copies only data or metadata that is new or different content from the source
         *        location to the destination location.
         *        </p>
         *        <p>
         *        ALL: DataSync copies all source location content to the destination, without comparing to existing
         *        content on the destination.
         * @see TransferMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransferMode
         */
        Builder transferMode(String transferMode);

        /**
         * <p>
         * A value that determines whether DataSync transfers only the data and metadata that differ between the source
         * and the destination location, or whether DataSync transfers all the content from the source, without
         * comparing to the destination location.
         * </p>
         * <p>
         * CHANGED: DataSync copies only data or metadata that is new or different content from the source location to
         * the destination location.
         * </p>
         * <p>
         * ALL: DataSync copies all source location content to the destination, without comparing to existing content on
         * the destination.
         * </p>
         * 
         * @param transferMode
         *        A value that determines whether DataSync transfers only the data and metadata that differ between the
         *        source and the destination location, or whether DataSync transfers all the content from the source,
         *        without comparing to the destination location. </p>
         *        <p>
         *        CHANGED: DataSync copies only data or metadata that is new or different content from the source
         *        location to the destination location.
         *        </p>
         *        <p>
         *        ALL: DataSync copies all source location content to the destination, without comparing to existing
         *        content on the destination.
         * @see TransferMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransferMode
         */
        Builder transferMode(TransferMode transferMode);

        /**
         * <p>
         * A value that determines which components of the SMB security descriptor are copied from source to destination
         * objects.
         * </p>
         * <p>
         * This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or
         * between two Amazon FSx for Windows File Server locations. For more information about how DataSync handles
         * metadata, see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How DataSync
         * Handles Metadata and Special Files</a>.
         * </p>
         * <p>
         * Default value: OWNER_DACL.
         * </p>
         * <p>
         * <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Object owner.
         * </p>
         * </li>
         * <li>
         * <p>
         * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which are
         * used by administrators to log attempts to access a secured object.
         * </p>
         * <p>
         * <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Object owner.
         * </p>
         * </li>
         * <li>
         * <p>
         * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         * </p>
         * </li>
         * <li>
         * <p>
         * NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a
         * secured object.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to access your
         * SMB location. For information about choosing a user that ensures sufficient permissions to files, folders,
         * and metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
         * </p>
         * <p>
         * <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned by the
         * user that was provided for accessing the destination location. DACLs and SACLs are set based on the
         * destination server’s configuration.
         * </p>
         * 
         * @param securityDescriptorCopyFlags
         *        A value that determines which components of the SMB security descriptor are copied from source to
         *        destination objects. </p>
         *        <p>
         *        This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or
         *        between two Amazon FSx for Windows File Server locations. For more information about how DataSync
         *        handles metadata, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How DataSync Handles
         *        Metadata and Special Files</a>.
         *        </p>
         *        <p>
         *        Default value: OWNER_DACL.
         *        </p>
         *        <p>
         *        <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Object owner.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which
         *        are used by administrators to log attempts to access a secured object.
         *        </p>
         *        <p>
         *        <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Object owner.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a
         *        secured object.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to
         *        access your SMB location. For information about choosing a user that ensures sufficient permissions to
         *        files, folders, and metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
         *        </p>
         *        <p>
         *        <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned
         *        by the user that was provided for accessing the destination location. DACLs and SACLs are set based on
         *        the destination server’s configuration.
         * @see SmbSecurityDescriptorCopyFlags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SmbSecurityDescriptorCopyFlags
         */
        Builder securityDescriptorCopyFlags(String securityDescriptorCopyFlags);

        /**
         * <p>
         * A value that determines which components of the SMB security descriptor are copied from source to destination
         * objects.
         * </p>
         * <p>
         * This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or
         * between two Amazon FSx for Windows File Server locations. For more information about how DataSync handles
         * metadata, see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How DataSync
         * Handles Metadata and Special Files</a>.
         * </p>
         * <p>
         * Default value: OWNER_DACL.
         * </p>
         * <p>
         * <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Object owner.
         * </p>
         * </li>
         * <li>
         * <p>
         * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         * </p>
         * </li>
         * </ul>
         * <p>
         * When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which are
         * used by administrators to log attempts to access a secured object.
         * </p>
         * <p>
         * <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Object owner.
         * </p>
         * </li>
         * <li>
         * <p>
         * NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         * </p>
         * </li>
         * <li>
         * <p>
         * NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a
         * secured object.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to access your
         * SMB location. For information about choosing a user that ensures sufficient permissions to files, folders,
         * and metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
         * </p>
         * <p>
         * <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned by the
         * user that was provided for accessing the destination location. DACLs and SACLs are set based on the
         * destination server’s configuration.
         * </p>
         * 
         * @param securityDescriptorCopyFlags
         *        A value that determines which components of the SMB security descriptor are copied from source to
         *        destination objects. </p>
         *        <p>
         *        This value is only used for transfers between SMB and Amazon FSx for Windows File Server locations, or
         *        between two Amazon FSx for Windows File Server locations. For more information about how DataSync
         *        handles metadata, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/special-files.html">How DataSync Handles
         *        Metadata and Special Files</a>.
         *        </p>
         *        <p>
         *        Default value: OWNER_DACL.
         *        </p>
         *        <p>
         *        <b>OWNER_DACL</b>: For each copied object, DataSync copies the following metadata:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Object owner.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When choosing this option, DataSync does NOT copy the NTFS system access control lists (SACLs), which
         *        are used by administrators to log attempts to access a secured object.
         *        </p>
         *        <p>
         *        <b>OWNER_DACL_SACL</b>: For each copied object, DataSync copies the following metadata:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Object owner.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NTFS discretionary access control lists (DACLs), which determine whether to grant access to an object.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NTFS system access control lists (SACLs), which are used by administrators to log attempts to access a
         *        secured object.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Copying SACLs requires granting additional permissions to the Windows user that DataSync uses to
         *        access your SMB location. For information about choosing a user that ensures sufficient permissions to
         *        files, folders, and metadata, see <a href="create-smb-location.html#SMBuser">user</a>.
         *        </p>
         *        <p>
         *        <b>NONE</b>: None of the SMB security descriptor components are copied. Destination objects are owned
         *        by the user that was provided for accessing the destination location. DACLs and SACLs are set based on
         *        the destination server’s configuration.
         * @see SmbSecurityDescriptorCopyFlags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SmbSecurityDescriptorCopyFlags
         */
        Builder securityDescriptorCopyFlags(SmbSecurityDescriptorCopyFlags securityDescriptorCopyFlags);

        /**
         * <p>
         * Specifies whether object tags are maintained when transferring between object storage systems. If you want
         * your DataSync task to ignore object tags, specify the <code>NONE</code> value.
         * </p>
         * <p>
         * Default Value: <code>PRESERVE</code>
         * </p>
         * 
         * @param objectTags
         *        Specifies whether object tags are maintained when transferring between object storage systems. If you
         *        want your DataSync task to ignore object tags, specify the <code>NONE</code> value.</p>
         *        <p>
         *        Default Value: <code>PRESERVE</code>
         * @see ObjectTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObjectTags
         */
        Builder objectTags(String objectTags);

        /**
         * <p>
         * Specifies whether object tags are maintained when transferring between object storage systems. If you want
         * your DataSync task to ignore object tags, specify the <code>NONE</code> value.
         * </p>
         * <p>
         * Default Value: <code>PRESERVE</code>
         * </p>
         * 
         * @param objectTags
         *        Specifies whether object tags are maintained when transferring between object storage systems. If you
         *        want your DataSync task to ignore object tags, specify the <code>NONE</code> value.</p>
         *        <p>
         *        Default Value: <code>PRESERVE</code>
         * @see ObjectTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObjectTags
         */
        Builder objectTags(ObjectTags objectTags);
    }

    static final class BuilderImpl implements Builder {
        private String verifyMode;

        private String overwriteMode;

        private String atime;

        private String mtime;

        private String uid;

        private String gid;

        private String preserveDeletedFiles;

        private String preserveDevices;

        private String posixPermissions;

        private Long bytesPerSecond;

        private String taskQueueing;

        private String logLevel;

        private String transferMode;

        private String securityDescriptorCopyFlags;

        private String objectTags;

        private BuilderImpl() {
        }

        private BuilderImpl(Options model) {
            verifyMode(model.verifyMode);
            overwriteMode(model.overwriteMode);
            atime(model.atime);
            mtime(model.mtime);
            uid(model.uid);
            gid(model.gid);
            preserveDeletedFiles(model.preserveDeletedFiles);
            preserveDevices(model.preserveDevices);
            posixPermissions(model.posixPermissions);
            bytesPerSecond(model.bytesPerSecond);
            taskQueueing(model.taskQueueing);
            logLevel(model.logLevel);
            transferMode(model.transferMode);
            securityDescriptorCopyFlags(model.securityDescriptorCopyFlags);
            objectTags(model.objectTags);
        }

        public final String getVerifyMode() {
            return verifyMode;
        }

        public final void setVerifyMode(String verifyMode) {
            this.verifyMode = verifyMode;
        }

        @Override
        public final Builder verifyMode(String verifyMode) {
            this.verifyMode = verifyMode;
            return this;
        }

        @Override
        public final Builder verifyMode(VerifyMode verifyMode) {
            this.verifyMode(verifyMode == null ? null : verifyMode.toString());
            return this;
        }

        public final String getOverwriteMode() {
            return overwriteMode;
        }

        public final void setOverwriteMode(String overwriteMode) {
            this.overwriteMode = overwriteMode;
        }

        @Override
        public final Builder overwriteMode(String overwriteMode) {
            this.overwriteMode = overwriteMode;
            return this;
        }

        @Override
        public final Builder overwriteMode(OverwriteMode overwriteMode) {
            this.overwriteMode(overwriteMode == null ? null : overwriteMode.toString());
            return this;
        }

        public final String getAtime() {
            return atime;
        }

        public final void setAtime(String atime) {
            this.atime = atime;
        }

        @Override
        public final Builder atime(String atime) {
            this.atime = atime;
            return this;
        }

        @Override
        public final Builder atime(Atime atime) {
            this.atime(atime == null ? null : atime.toString());
            return this;
        }

        public final String getMtime() {
            return mtime;
        }

        public final void setMtime(String mtime) {
            this.mtime = mtime;
        }

        @Override
        public final Builder mtime(String mtime) {
            this.mtime = mtime;
            return this;
        }

        @Override
        public final Builder mtime(Mtime mtime) {
            this.mtime(mtime == null ? null : mtime.toString());
            return this;
        }

        public final String getUid() {
            return uid;
        }

        public final void setUid(String uid) {
            this.uid = uid;
        }

        @Override
        public final Builder uid(String uid) {
            this.uid = uid;
            return this;
        }

        @Override
        public final Builder uid(Uid uid) {
            this.uid(uid == null ? null : uid.toString());
            return this;
        }

        public final String getGid() {
            return gid;
        }

        public final void setGid(String gid) {
            this.gid = gid;
        }

        @Override
        public final Builder gid(String gid) {
            this.gid = gid;
            return this;
        }

        @Override
        public final Builder gid(Gid gid) {
            this.gid(gid == null ? null : gid.toString());
            return this;
        }

        public final String getPreserveDeletedFiles() {
            return preserveDeletedFiles;
        }

        public final void setPreserveDeletedFiles(String preserveDeletedFiles) {
            this.preserveDeletedFiles = preserveDeletedFiles;
        }

        @Override
        public final Builder preserveDeletedFiles(String preserveDeletedFiles) {
            this.preserveDeletedFiles = preserveDeletedFiles;
            return this;
        }

        @Override
        public final Builder preserveDeletedFiles(PreserveDeletedFiles preserveDeletedFiles) {
            this.preserveDeletedFiles(preserveDeletedFiles == null ? null : preserveDeletedFiles.toString());
            return this;
        }

        public final String getPreserveDevices() {
            return preserveDevices;
        }

        public final void setPreserveDevices(String preserveDevices) {
            this.preserveDevices = preserveDevices;
        }

        @Override
        public final Builder preserveDevices(String preserveDevices) {
            this.preserveDevices = preserveDevices;
            return this;
        }

        @Override
        public final Builder preserveDevices(PreserveDevices preserveDevices) {
            this.preserveDevices(preserveDevices == null ? null : preserveDevices.toString());
            return this;
        }

        public final String getPosixPermissions() {
            return posixPermissions;
        }

        public final void setPosixPermissions(String posixPermissions) {
            this.posixPermissions = posixPermissions;
        }

        @Override
        public final Builder posixPermissions(String posixPermissions) {
            this.posixPermissions = posixPermissions;
            return this;
        }

        @Override
        public final Builder posixPermissions(PosixPermissions posixPermissions) {
            this.posixPermissions(posixPermissions == null ? null : posixPermissions.toString());
            return this;
        }

        public final Long getBytesPerSecond() {
            return bytesPerSecond;
        }

        public final void setBytesPerSecond(Long bytesPerSecond) {
            this.bytesPerSecond = bytesPerSecond;
        }

        @Override
        public final Builder bytesPerSecond(Long bytesPerSecond) {
            this.bytesPerSecond = bytesPerSecond;
            return this;
        }

        public final String getTaskQueueing() {
            return taskQueueing;
        }

        public final void setTaskQueueing(String taskQueueing) {
            this.taskQueueing = taskQueueing;
        }

        @Override
        public final Builder taskQueueing(String taskQueueing) {
            this.taskQueueing = taskQueueing;
            return this;
        }

        @Override
        public final Builder taskQueueing(TaskQueueing taskQueueing) {
            this.taskQueueing(taskQueueing == null ? null : taskQueueing.toString());
            return this;
        }

        public final String getLogLevel() {
            return logLevel;
        }

        public final void setLogLevel(String logLevel) {
            this.logLevel = logLevel;
        }

        @Override
        public final Builder logLevel(String logLevel) {
            this.logLevel = logLevel;
            return this;
        }

        @Override
        public final Builder logLevel(LogLevel logLevel) {
            this.logLevel(logLevel == null ? null : logLevel.toString());
            return this;
        }

        public final String getTransferMode() {
            return transferMode;
        }

        public final void setTransferMode(String transferMode) {
            this.transferMode = transferMode;
        }

        @Override
        public final Builder transferMode(String transferMode) {
            this.transferMode = transferMode;
            return this;
        }

        @Override
        public final Builder transferMode(TransferMode transferMode) {
            this.transferMode(transferMode == null ? null : transferMode.toString());
            return this;
        }

        public final String getSecurityDescriptorCopyFlags() {
            return securityDescriptorCopyFlags;
        }

        public final void setSecurityDescriptorCopyFlags(String securityDescriptorCopyFlags) {
            this.securityDescriptorCopyFlags = securityDescriptorCopyFlags;
        }

        @Override
        public final Builder securityDescriptorCopyFlags(String securityDescriptorCopyFlags) {
            this.securityDescriptorCopyFlags = securityDescriptorCopyFlags;
            return this;
        }

        @Override
        public final Builder securityDescriptorCopyFlags(SmbSecurityDescriptorCopyFlags securityDescriptorCopyFlags) {
            this.securityDescriptorCopyFlags(securityDescriptorCopyFlags == null ? null : securityDescriptorCopyFlags.toString());
            return this;
        }

        public final String getObjectTags() {
            return objectTags;
        }

        public final void setObjectTags(String objectTags) {
            this.objectTags = objectTags;
        }

        @Override
        public final Builder objectTags(String objectTags) {
            this.objectTags = objectTags;
            return this;
        }

        @Override
        public final Builder objectTags(ObjectTags objectTags) {
            this.objectTags(objectTags == null ? null : objectTags.toString());
            return this;
        }

        @Override
        public Options build() {
            return new Options(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
