/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.datasync.model.CancelTaskExecutionRequest;
import software.amazon.awssdk.services.datasync.model.CancelTaskExecutionResponse;
import software.amazon.awssdk.services.datasync.model.CreateAgentRequest;
import software.amazon.awssdk.services.datasync.model.CreateAgentResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationEfsRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationEfsResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxLustreRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxLustreResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxOntapRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxOntapResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxOpenZfsRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxOpenZfsResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxWindowsRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationFsxWindowsResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationHdfsRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationHdfsResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationNfsRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationNfsResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationObjectStorageRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationObjectStorageResponse;
import software.amazon.awssdk.services.datasync.model.CreateLocationS3Request;
import software.amazon.awssdk.services.datasync.model.CreateLocationS3Response;
import software.amazon.awssdk.services.datasync.model.CreateLocationSmbRequest;
import software.amazon.awssdk.services.datasync.model.CreateLocationSmbResponse;
import software.amazon.awssdk.services.datasync.model.CreateTaskRequest;
import software.amazon.awssdk.services.datasync.model.CreateTaskResponse;
import software.amazon.awssdk.services.datasync.model.DataSyncException;
import software.amazon.awssdk.services.datasync.model.DataSyncRequest;
import software.amazon.awssdk.services.datasync.model.DeleteAgentRequest;
import software.amazon.awssdk.services.datasync.model.DeleteAgentResponse;
import software.amazon.awssdk.services.datasync.model.DeleteLocationRequest;
import software.amazon.awssdk.services.datasync.model.DeleteLocationResponse;
import software.amazon.awssdk.services.datasync.model.DeleteTaskRequest;
import software.amazon.awssdk.services.datasync.model.DeleteTaskResponse;
import software.amazon.awssdk.services.datasync.model.DescribeAgentRequest;
import software.amazon.awssdk.services.datasync.model.DescribeAgentResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationEfsRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationEfsResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxLustreRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxLustreResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxOntapRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxOntapResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxOpenZfsRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxOpenZfsResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxWindowsRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationFsxWindowsResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationHdfsRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationHdfsResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationNfsRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationNfsResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationObjectStorageRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationObjectStorageResponse;
import software.amazon.awssdk.services.datasync.model.DescribeLocationS3Request;
import software.amazon.awssdk.services.datasync.model.DescribeLocationS3Response;
import software.amazon.awssdk.services.datasync.model.DescribeLocationSmbRequest;
import software.amazon.awssdk.services.datasync.model.DescribeLocationSmbResponse;
import software.amazon.awssdk.services.datasync.model.DescribeTaskExecutionRequest;
import software.amazon.awssdk.services.datasync.model.DescribeTaskExecutionResponse;
import software.amazon.awssdk.services.datasync.model.DescribeTaskRequest;
import software.amazon.awssdk.services.datasync.model.DescribeTaskResponse;
import software.amazon.awssdk.services.datasync.model.InternalException;
import software.amazon.awssdk.services.datasync.model.InvalidRequestException;
import software.amazon.awssdk.services.datasync.model.ListAgentsRequest;
import software.amazon.awssdk.services.datasync.model.ListAgentsResponse;
import software.amazon.awssdk.services.datasync.model.ListLocationsRequest;
import software.amazon.awssdk.services.datasync.model.ListLocationsResponse;
import software.amazon.awssdk.services.datasync.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.datasync.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.datasync.model.ListTaskExecutionsRequest;
import software.amazon.awssdk.services.datasync.model.ListTaskExecutionsResponse;
import software.amazon.awssdk.services.datasync.model.ListTasksRequest;
import software.amazon.awssdk.services.datasync.model.ListTasksResponse;
import software.amazon.awssdk.services.datasync.model.StartTaskExecutionRequest;
import software.amazon.awssdk.services.datasync.model.StartTaskExecutionResponse;
import software.amazon.awssdk.services.datasync.model.TagResourceRequest;
import software.amazon.awssdk.services.datasync.model.TagResourceResponse;
import software.amazon.awssdk.services.datasync.model.UntagResourceRequest;
import software.amazon.awssdk.services.datasync.model.UntagResourceResponse;
import software.amazon.awssdk.services.datasync.model.UpdateAgentRequest;
import software.amazon.awssdk.services.datasync.model.UpdateAgentResponse;
import software.amazon.awssdk.services.datasync.model.UpdateLocationHdfsRequest;
import software.amazon.awssdk.services.datasync.model.UpdateLocationHdfsResponse;
import software.amazon.awssdk.services.datasync.model.UpdateLocationNfsRequest;
import software.amazon.awssdk.services.datasync.model.UpdateLocationNfsResponse;
import software.amazon.awssdk.services.datasync.model.UpdateLocationObjectStorageRequest;
import software.amazon.awssdk.services.datasync.model.UpdateLocationObjectStorageResponse;
import software.amazon.awssdk.services.datasync.model.UpdateLocationSmbRequest;
import software.amazon.awssdk.services.datasync.model.UpdateLocationSmbResponse;
import software.amazon.awssdk.services.datasync.model.UpdateTaskExecutionRequest;
import software.amazon.awssdk.services.datasync.model.UpdateTaskExecutionResponse;
import software.amazon.awssdk.services.datasync.model.UpdateTaskRequest;
import software.amazon.awssdk.services.datasync.model.UpdateTaskResponse;
import software.amazon.awssdk.services.datasync.paginators.ListAgentsPublisher;
import software.amazon.awssdk.services.datasync.paginators.ListLocationsPublisher;
import software.amazon.awssdk.services.datasync.paginators.ListTagsForResourcePublisher;
import software.amazon.awssdk.services.datasync.paginators.ListTaskExecutionsPublisher;
import software.amazon.awssdk.services.datasync.paginators.ListTasksPublisher;
import software.amazon.awssdk.services.datasync.transform.CancelTaskExecutionRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateAgentRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationEfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationFsxLustreRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationFsxOntapRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationFsxOpenZfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationFsxWindowsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationHdfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationNfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationObjectStorageRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationS3RequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateLocationSmbRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.CreateTaskRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DeleteAgentRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DeleteLocationRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DeleteTaskRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeAgentRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationEfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationFsxLustreRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationFsxOntapRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationFsxOpenZfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationFsxWindowsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationHdfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationNfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationObjectStorageRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationS3RequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeLocationSmbRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeTaskExecutionRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.DescribeTaskRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.ListAgentsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.ListLocationsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.ListTaskExecutionsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.ListTasksRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.StartTaskExecutionRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateAgentRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateLocationHdfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateLocationNfsRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateLocationObjectStorageRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateLocationSmbRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateTaskExecutionRequestMarshaller;
import software.amazon.awssdk.services.datasync.transform.UpdateTaskRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DataSyncAsyncClient}.
 *
 * @see DataSyncAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDataSyncAsyncClient implements DataSyncAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDataSyncAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDataSyncAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Cancels execution of a task.
     * </p>
     * <p>
     * When you cancel a task execution, the transfer of some files is abruptly interrupted. The contents of files that
     * are transferred to the destination might be incomplete or inconsistent with the source files. However, if you
     * start a new task execution on the same task and you allow the task execution to complete, file content on the
     * destination is complete and consistent. This applies to other unexpected failures that interrupt a task
     * execution. In all of these cases, DataSync successfully complete the transfer when you start the next task
     * execution.
     * </p>
     *
     * @param cancelTaskExecutionRequest
     *        CancelTaskExecutionRequest
     * @return A Java Future containing the result of the CancelTaskExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CancelTaskExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CancelTaskExecution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CancelTaskExecutionResponse> cancelTaskExecution(
            CancelTaskExecutionRequest cancelTaskExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelTaskExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelTaskExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelTaskExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelTaskExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelTaskExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelTaskExecutionRequest, CancelTaskExecutionResponse>()
                            .withOperationName("CancelTaskExecution")
                            .withMarshaller(new CancelTaskExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelTaskExecutionRequest));
            CompletableFuture<CancelTaskExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates an DataSync agent that you have deployed on your host. The activation process associates your agent
     * with your account. In the activation process, you specify information such as the Amazon Web Services Region that
     * you want to activate the agent in. You activate the agent in the Amazon Web Services Region where your target
     * locations (in Amazon S3 or Amazon EFS) reside. Your tasks are created in this Amazon Web Services Region.
     * </p>
     * <p>
     * You can activate the agent in a VPC (virtual private cloud) or provide the agent access to a VPC endpoint so you
     * can run tasks without going over the public internet.
     * </p>
     * <p>
     * You can use an agent for more than one location. If a task uses multiple agents, all of them need to have status
     * AVAILABLE for the task to run. If you use multiple agents for a source location, the status of all the agents
     * must be AVAILABLE for the task to run.
     * </p>
     * <p>
     * Agents are automatically updated by Amazon Web Services on a regular basis, using a mechanism that ensures
     * minimal interruption to your tasks.
     * </p>
     * <p/>
     *
     * @param createAgentRequest
     *        CreateAgentRequest
     * @return A Java Future containing the result of the CreateAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li> <li>
     *         SdkException Base class for all exceptions that can be thrown by the SDK (both service and client). Can
     *         be used for catch all scenarios.</li> <li>SdkClientException If any client side error occurs such as an
     *         IO related failure, failure to get credentials, etc.</li> <li>DataSyncException Base class for all
     *         service exceptions. Unknown exceptions will be thrown as an instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateAgent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAgentResponse> createAgent(CreateAgentRequest createAgentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAgentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAgentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAgentRequest, CreateAgentResponse>()
                            .withOperationName("CreateAgent").withMarshaller(new CreateAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAgentRequest));
            CompletableFuture<CreateAgentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an Amazon EFS file system that DataSync can access for a transfer. For more information,
     * see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/create-efs-location.html">Creating a location
     * for Amazon EFS</a>.
     * </p>
     *
     * @param createLocationEfsRequest
     *        CreateLocationEfsRequest
     * @return A Java Future containing the result of the CreateLocationEfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationEfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationEfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationEfsResponse> createLocationEfs(CreateLocationEfsRequest createLocationEfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationEfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationEfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationEfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationEfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationEfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationEfsRequest, CreateLocationEfsResponse>()
                            .withOperationName("CreateLocationEfs")
                            .withMarshaller(new CreateLocationEfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationEfsRequest));
            CompletableFuture<CreateLocationEfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an Amazon FSx for Lustre file system.
     * </p>
     *
     * @param createLocationFsxLustreRequest
     * @return A Java Future containing the result of the CreateLocationFsxLustre operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationFsxLustre
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationFsxLustre"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationFsxLustreResponse> createLocationFsxLustre(
            CreateLocationFsxLustreRequest createLocationFsxLustreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationFsxLustreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationFsxLustre");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationFsxLustreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationFsxLustreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationFsxLustreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationFsxLustreRequest, CreateLocationFsxLustreResponse>()
                            .withOperationName("CreateLocationFsxLustre")
                            .withMarshaller(new CreateLocationFsxLustreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationFsxLustreRequest));
            CompletableFuture<CreateLocationFsxLustreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an Amazon FSx for NetApp ONTAP file system that DataSync can access for a transfer. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-ontap-location.html">Creating a location for
     * FSx for ONTAP</a>.
     * </p>
     *
     * @param createLocationFsxOntapRequest
     * @return A Java Future containing the result of the CreateLocationFsxOntap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationFsxOntap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationFsxOntap"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationFsxOntapResponse> createLocationFsxOntap(
            CreateLocationFsxOntapRequest createLocationFsxOntapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationFsxOntapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationFsxOntap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationFsxOntapResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationFsxOntapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationFsxOntapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationFsxOntapRequest, CreateLocationFsxOntapResponse>()
                            .withOperationName("CreateLocationFsxOntap")
                            .withMarshaller(new CreateLocationFsxOntapRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationFsxOntapRequest));
            CompletableFuture<CreateLocationFsxOntapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an Amazon FSx for OpenZFS file system.
     * </p>
     *
     * @param createLocationFsxOpenZfsRequest
     * @return A Java Future containing the result of the CreateLocationFsxOpenZfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationFsxOpenZfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationFsxOpenZfs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationFsxOpenZfsResponse> createLocationFsxOpenZfs(
            CreateLocationFsxOpenZfsRequest createLocationFsxOpenZfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationFsxOpenZfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationFsxOpenZfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationFsxOpenZfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationFsxOpenZfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationFsxOpenZfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationFsxOpenZfsRequest, CreateLocationFsxOpenZfsResponse>()
                            .withOperationName("CreateLocationFsxOpenZfs")
                            .withMarshaller(new CreateLocationFsxOpenZfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationFsxOpenZfsRequest));
            CompletableFuture<CreateLocationFsxOpenZfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an Amazon FSx for Windows File Server file system.
     * </p>
     *
     * @param createLocationFsxWindowsRequest
     * @return A Java Future containing the result of the CreateLocationFsxWindows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationFsxWindows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationFsxWindows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationFsxWindowsResponse> createLocationFsxWindows(
            CreateLocationFsxWindowsRequest createLocationFsxWindowsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationFsxWindowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationFsxWindows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationFsxWindowsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationFsxWindowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationFsxWindowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationFsxWindowsRequest, CreateLocationFsxWindowsResponse>()
                            .withOperationName("CreateLocationFsxWindows")
                            .withMarshaller(new CreateLocationFsxWindowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationFsxWindowsRequest));
            CompletableFuture<CreateLocationFsxWindowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for a Hadoop Distributed File System (HDFS).
     * </p>
     *
     * @param createLocationHdfsRequest
     * @return A Java Future containing the result of the CreateLocationHdfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationHdfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationHdfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationHdfsResponse> createLocationHdfs(CreateLocationHdfsRequest createLocationHdfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationHdfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationHdfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationHdfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationHdfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationHdfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationHdfsRequest, CreateLocationHdfsResponse>()
                            .withOperationName("CreateLocationHdfs")
                            .withMarshaller(new CreateLocationHdfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationHdfsRequest));
            CompletableFuture<CreateLocationHdfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Defines a file system on a Network File System (NFS) server that can be read from or written to.
     * </p>
     *
     * @param createLocationNfsRequest
     *        CreateLocationNfsRequest
     * @return A Java Future containing the result of the CreateLocationNfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationNfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationNfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationNfsResponse> createLocationNfs(CreateLocationNfsRequest createLocationNfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationNfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationNfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationNfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationNfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationNfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationNfsRequest, CreateLocationNfsResponse>()
                            .withOperationName("CreateLocationNfs")
                            .withMarshaller(new CreateLocationNfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationNfsRequest));
            CompletableFuture<CreateLocationNfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an object storage system that DataSync can access for a transfer. For more information,
     * see <a href="https://docs.aws.amazon.com/datasync/latest/userguide/create-object-location.html">Creating a
     * location for object storage</a>.
     * </p>
     *
     * @param createLocationObjectStorageRequest
     *        CreateLocationObjectStorageRequest
     * @return A Java Future containing the result of the CreateLocationObjectStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationObjectStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationObjectStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationObjectStorageResponse> createLocationObjectStorage(
            CreateLocationObjectStorageRequest createLocationObjectStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationObjectStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationObjectStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationObjectStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationObjectStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationObjectStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationObjectStorageRequest, CreateLocationObjectStorageResponse>()
                            .withOperationName("CreateLocationObjectStorage")
                            .withMarshaller(new CreateLocationObjectStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationObjectStorageRequest));
            CompletableFuture<CreateLocationObjectStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an endpoint for an Amazon S3 bucket.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-locations-cli.html#create-location-s3-cli"
     * >Create an Amazon S3 location</a> in the <i>DataSync User Guide</i>.
     * </p>
     *
     * @param createLocationS3Request
     *        CreateLocationS3Request
     * @return A Java Future containing the result of the CreateLocationS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationS3" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationS3Response> createLocationS3(CreateLocationS3Request createLocationS3Request) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationS3");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationS3Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationS3Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationS3Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationS3Request, CreateLocationS3Response>()
                            .withOperationName("CreateLocationS3")
                            .withMarshaller(new CreateLocationS3RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationS3Request));
            CompletableFuture<CreateLocationS3Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Defines a file system on a Server Message Block (SMB) server that can be read from or written to.
     * </p>
     *
     * @param createLocationSmbRequest
     *        CreateLocationSmbRequest
     * @return A Java Future containing the result of the CreateLocationSmb operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateLocationSmb
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateLocationSmb" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLocationSmbResponse> createLocationSmb(CreateLocationSmbRequest createLocationSmbRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLocationSmbRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLocationSmb");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLocationSmbResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLocationSmbResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLocationSmbResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLocationSmbRequest, CreateLocationSmbResponse>()
                            .withOperationName("CreateLocationSmb")
                            .withMarshaller(new CreateLocationSmbRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createLocationSmbRequest));
            CompletableFuture<CreateLocationSmbResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Configures a task, which defines where and how DataSync transfers your data.
     * </p>
     * <p>
     * A task includes a source location, a destination location, and the preferences for how and when you want to
     * transfer your data (such as bandwidth limits, scheduling, among other options).
     * </p>
     * <p>
     * When you create a task that transfers data between Amazon Web Services services in different Amazon Web Services
     * Regions, one of your locations must reside in the Region where you're using DataSync.
     * </p>
     * <p>
     * For more information, see the following topics:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/datasync/latest/userguide/working-with-locations.html">Working with DataSync
     * locations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/datasync/latest/userguide/create-task.html">Configure DataSync task
     * settings</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createTaskRequest
     *        CreateTaskRequest
     * @return A Java Future containing the result of the CreateTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.CreateTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CreateTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTaskResponse> createTask(CreateTaskRequest createTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTaskRequest, CreateTaskResponse>().withOperationName("CreateTask")
                            .withMarshaller(new CreateTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTaskRequest));
            CompletableFuture<CreateTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an agent. To specify which agent to delete, use the Amazon Resource Name (ARN) of the agent in your
     * request. The operation disassociates the agent from your Amazon Web Services account. However, it doesn't delete
     * the agent virtual machine (VM) from your on-premises environment.
     * </p>
     *
     * @param deleteAgentRequest
     *        DeleteAgentRequest
     * @return A Java Future containing the result of the DeleteAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DeleteAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DeleteAgent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAgentResponse> deleteAgent(DeleteAgentRequest deleteAgentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAgentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAgentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAgentRequest, DeleteAgentResponse>()
                            .withOperationName("DeleteAgent").withMarshaller(new DeleteAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAgentRequest));
            CompletableFuture<DeleteAgentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the configuration of a location used by DataSync.
     * </p>
     *
     * @param deleteLocationRequest
     *        DeleteLocation
     * @return A Java Future containing the result of the DeleteLocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DeleteLocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DeleteLocation" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLocationResponse> deleteLocation(DeleteLocationRequest deleteLocationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLocationRequest, DeleteLocationResponse>()
                            .withOperationName("DeleteLocation")
                            .withMarshaller(new DeleteLocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteLocationRequest));
            CompletableFuture<DeleteLocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a task.
     * </p>
     *
     * @param deleteTaskRequest
     *        DeleteTask
     * @return A Java Future containing the result of the DeleteTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DeleteTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DeleteTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTaskResponse> deleteTask(DeleteTaskRequest deleteTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTaskRequest, DeleteTaskResponse>().withOperationName("DeleteTask")
                            .withMarshaller(new DeleteTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTaskRequest));
            CompletableFuture<DeleteTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata such as the name, the network interfaces, and the status (that is, whether the agent is running
     * or not) for an agent. To specify which agent to describe, use the Amazon Resource Name (ARN) of the agent in your
     * request.
     * </p>
     *
     * @param describeAgentRequest
     *        DescribeAgent
     * @return A Java Future containing the result of the DescribeAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeAgent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAgentResponse> describeAgent(DescribeAgentRequest describeAgentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAgentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAgentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAgentRequest, DescribeAgentResponse>()
                            .withOperationName("DescribeAgent")
                            .withMarshaller(new DescribeAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAgentRequest));
            CompletableFuture<DescribeAgentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about your DataSync location for an Amazon EFS file system.
     * </p>
     *
     * @param describeLocationEfsRequest
     *        DescribeLocationEfsRequest
     * @return A Java Future containing the result of the DescribeLocationEfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationEfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationEfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationEfsResponse> describeLocationEfs(
            DescribeLocationEfsRequest describeLocationEfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationEfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationEfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationEfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationEfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationEfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationEfsRequest, DescribeLocationEfsResponse>()
                            .withOperationName("DescribeLocationEfs")
                            .withMarshaller(new DescribeLocationEfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationEfsRequest));
            CompletableFuture<DescribeLocationEfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about an Amazon FSx for Lustre location, such as information about its path.
     * </p>
     *
     * @param describeLocationFsxLustreRequest
     * @return A Java Future containing the result of the DescribeLocationFsxLustre operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationFsxLustre
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationFsxLustre"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationFsxLustreResponse> describeLocationFsxLustre(
            DescribeLocationFsxLustreRequest describeLocationFsxLustreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationFsxLustreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationFsxLustre");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationFsxLustreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationFsxLustreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationFsxLustreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationFsxLustreRequest, DescribeLocationFsxLustreResponse>()
                            .withOperationName("DescribeLocationFsxLustre")
                            .withMarshaller(new DescribeLocationFsxLustreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationFsxLustreRequest));
            CompletableFuture<DescribeLocationFsxLustreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides details about how an DataSync location for an Amazon FSx for NetApp ONTAP file system is configured.
     * </p>
     *
     * @param describeLocationFsxOntapRequest
     * @return A Java Future containing the result of the DescribeLocationFsxOntap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationFsxOntap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationFsxOntap"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationFsxOntapResponse> describeLocationFsxOntap(
            DescribeLocationFsxOntapRequest describeLocationFsxOntapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationFsxOntapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationFsxOntap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationFsxOntapResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationFsxOntapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationFsxOntapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationFsxOntapRequest, DescribeLocationFsxOntapResponse>()
                            .withOperationName("DescribeLocationFsxOntap")
                            .withMarshaller(new DescribeLocationFsxOntapRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationFsxOntapRequest));
            CompletableFuture<DescribeLocationFsxOntapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about an Amazon FSx for OpenZFS location, such as information about its path.
     * </p>
     *
     * @param describeLocationFsxOpenZfsRequest
     * @return A Java Future containing the result of the DescribeLocationFsxOpenZfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationFsxOpenZfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationFsxOpenZfs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationFsxOpenZfsResponse> describeLocationFsxOpenZfs(
            DescribeLocationFsxOpenZfsRequest describeLocationFsxOpenZfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationFsxOpenZfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationFsxOpenZfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationFsxOpenZfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationFsxOpenZfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationFsxOpenZfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationFsxOpenZfsRequest, DescribeLocationFsxOpenZfsResponse>()
                            .withOperationName("DescribeLocationFsxOpenZfs")
                            .withMarshaller(new DescribeLocationFsxOpenZfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationFsxOpenZfsRequest));
            CompletableFuture<DescribeLocationFsxOpenZfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about an Amazon FSx for Windows File Server location, such as information about its path.
     * </p>
     *
     * @param describeLocationFsxWindowsRequest
     * @return A Java Future containing the result of the DescribeLocationFsxWindows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationFsxWindows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationFsxWindows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationFsxWindowsResponse> describeLocationFsxWindows(
            DescribeLocationFsxWindowsRequest describeLocationFsxWindowsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationFsxWindowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationFsxWindows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationFsxWindowsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationFsxWindowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationFsxWindowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationFsxWindowsRequest, DescribeLocationFsxWindowsResponse>()
                            .withOperationName("DescribeLocationFsxWindows")
                            .withMarshaller(new DescribeLocationFsxWindowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationFsxWindowsRequest));
            CompletableFuture<DescribeLocationFsxWindowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata, such as the authentication information about the Hadoop Distributed File System (HDFS)
     * location.
     * </p>
     *
     * @param describeLocationHdfsRequest
     * @return A Java Future containing the result of the DescribeLocationHdfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationHdfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationHdfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationHdfsResponse> describeLocationHdfs(
            DescribeLocationHdfsRequest describeLocationHdfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationHdfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationHdfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationHdfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationHdfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationHdfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationHdfsRequest, DescribeLocationHdfsResponse>()
                            .withOperationName("DescribeLocationHdfs")
                            .withMarshaller(new DescribeLocationHdfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationHdfsRequest));
            CompletableFuture<DescribeLocationHdfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata, such as the path information, about an NFS location.
     * </p>
     *
     * @param describeLocationNfsRequest
     *        DescribeLocationNfsRequest
     * @return A Java Future containing the result of the DescribeLocationNfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationNfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationNfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationNfsResponse> describeLocationNfs(
            DescribeLocationNfsRequest describeLocationNfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationNfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationNfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationNfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationNfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationNfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationNfsRequest, DescribeLocationNfsResponse>()
                            .withOperationName("DescribeLocationNfs")
                            .withMarshaller(new DescribeLocationNfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationNfsRequest));
            CompletableFuture<DescribeLocationNfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about your DataSync location for an object storage system.
     * </p>
     *
     * @param describeLocationObjectStorageRequest
     *        DescribeLocationObjectStorageRequest
     * @return A Java Future containing the result of the DescribeLocationObjectStorage operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationObjectStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationObjectStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationObjectStorageResponse> describeLocationObjectStorage(
            DescribeLocationObjectStorageRequest describeLocationObjectStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeLocationObjectStorageRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationObjectStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationObjectStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationObjectStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationObjectStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationObjectStorageRequest, DescribeLocationObjectStorageResponse>()
                            .withOperationName("DescribeLocationObjectStorage")
                            .withMarshaller(new DescribeLocationObjectStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationObjectStorageRequest));
            CompletableFuture<DescribeLocationObjectStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata, such as bucket name, about an Amazon S3 bucket location.
     * </p>
     *
     * @param describeLocationS3Request
     *        DescribeLocationS3Request
     * @return A Java Future containing the result of the DescribeLocationS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationS3
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationS3" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationS3Response> describeLocationS3(DescribeLocationS3Request describeLocationS3Request) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationS3");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationS3Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationS3Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationS3Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationS3Request, DescribeLocationS3Response>()
                            .withOperationName("DescribeLocationS3")
                            .withMarshaller(new DescribeLocationS3RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationS3Request));
            CompletableFuture<DescribeLocationS3Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata, such as the path and user information about an SMB location.
     * </p>
     *
     * @param describeLocationSmbRequest
     *        DescribeLocationSmbRequest
     * @return A Java Future containing the result of the DescribeLocationSmb operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeLocationSmb
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeLocationSmb" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLocationSmbResponse> describeLocationSmb(
            DescribeLocationSmbRequest describeLocationSmbRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLocationSmbRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLocationSmb");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLocationSmbResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLocationSmbResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeLocationSmbResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLocationSmbRequest, DescribeLocationSmbResponse>()
                            .withOperationName("DescribeLocationSmb")
                            .withMarshaller(new DescribeLocationSmbRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeLocationSmbRequest));
            CompletableFuture<DescribeLocationSmbResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about a task.
     * </p>
     *
     * @param describeTaskRequest
     *        DescribeTaskRequest
     * @return A Java Future containing the result of the DescribeTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTaskResponse> describeTask(DescribeTaskRequest describeTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTaskRequest, DescribeTaskResponse>()
                            .withOperationName("DescribeTask").withMarshaller(new DescribeTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTaskRequest));
            CompletableFuture<DescribeTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns detailed metadata about a task that is being executed.
     * </p>
     *
     * @param describeTaskExecutionRequest
     *        DescribeTaskExecutionRequest
     * @return A Java Future containing the result of the DescribeTaskExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.DescribeTaskExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/DescribeTaskExecution"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTaskExecutionResponse> describeTaskExecution(
            DescribeTaskExecutionRequest describeTaskExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTaskExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTaskExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTaskExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTaskExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTaskExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTaskExecutionRequest, DescribeTaskExecutionResponse>()
                            .withOperationName("DescribeTaskExecution")
                            .withMarshaller(new DescribeTaskExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTaskExecutionRequest));
            CompletableFuture<DescribeTaskExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of agents owned by an Amazon Web Services account in the Amazon Web Services Region specified in
     * the request. The returned list is ordered by agent Amazon Resource Name (ARN).
     * </p>
     * <p>
     * By default, this operation returns a maximum of 100 agents. This operation supports pagination that enables you
     * to optionally reduce the number of agents returned in a response.
     * </p>
     * <p>
     * If you have more agents than are returned in a response (that is, the response returns only a truncated list of
     * your agents), the response contains a marker that you can specify in your next request to fetch the next page of
     * agents.
     * </p>
     *
     * @param listAgentsRequest
     *        ListAgentsRequest
     * @return A Java Future containing the result of the ListAgents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListAgents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListAgents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAgentsResponse> listAgents(ListAgentsRequest listAgentsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAgentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAgents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAgentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAgentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAgentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAgentsRequest, ListAgentsResponse>().withOperationName("ListAgents")
                            .withMarshaller(new ListAgentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAgentsRequest));
            CompletableFuture<ListAgentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of agents owned by an Amazon Web Services account in the Amazon Web Services Region specified in
     * the request. The returned list is ordered by agent Amazon Resource Name (ARN).
     * </p>
     * <p>
     * By default, this operation returns a maximum of 100 agents. This operation supports pagination that enables you
     * to optionally reduce the number of agents returned in a response.
     * </p>
     * <p>
     * If you have more agents than are returned in a response (that is, the response returns only a truncated list of
     * your agents), the response contains a marker that you can specify in your next request to fetch the next page of
     * agents.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAgents(software.amazon.awssdk.services.datasync.model.ListAgentsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListAgentsPublisher publisher = client.listAgentsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListAgentsPublisher publisher = client.listAgentsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datasync.model.ListAgentsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datasync.model.ListAgentsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAgents(software.amazon.awssdk.services.datasync.model.ListAgentsRequest)} operation.</b>
     * </p>
     *
     * @param listAgentsRequest
     *        ListAgentsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListAgents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListAgents" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAgentsPublisher listAgentsPaginator(ListAgentsRequest listAgentsRequest) {
        return new ListAgentsPublisher(this, applyPaginatorUserAgent(listAgentsRequest));
    }

    /**
     * <p>
     * Returns a list of source and destination locations.
     * </p>
     * <p>
     * If you have more locations than are returned in a response (that is, the response returns only a truncated list
     * of your agents), the response contains a token that you can specify in your next request to fetch the next page
     * of locations.
     * </p>
     *
     * @param listLocationsRequest
     *        ListLocationsRequest
     * @return A Java Future containing the result of the ListLocations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListLocations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListLocationsResponse> listLocations(ListLocationsRequest listLocationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLocations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLocationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLocationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLocationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLocationsRequest, ListLocationsResponse>()
                            .withOperationName("ListLocations")
                            .withMarshaller(new ListLocationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listLocationsRequest));
            CompletableFuture<ListLocationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of source and destination locations.
     * </p>
     * <p>
     * If you have more locations than are returned in a response (that is, the response returns only a truncated list
     * of your agents), the response contains a token that you can specify in your next request to fetch the next page
     * of locations.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listLocations(software.amazon.awssdk.services.datasync.model.ListLocationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListLocationsPublisher publisher = client.listLocationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListLocationsPublisher publisher = client.listLocationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datasync.model.ListLocationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datasync.model.ListLocationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listLocations(software.amazon.awssdk.services.datasync.model.ListLocationsRequest)} operation.</b>
     * </p>
     *
     * @param listLocationsRequest
     *        ListLocationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListLocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListLocations" target="_top">AWS API
     *      Documentation</a>
     */
    public ListLocationsPublisher listLocationsPaginator(ListLocationsRequest listLocationsRequest) {
        return new ListLocationsPublisher(this, applyPaginatorUserAgent(listLocationsRequest));
    }

    /**
     * <p>
     * Returns all the tags associated with a specified resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     *        ListTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all the tags associated with a specified resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTagsForResource(software.amazon.awssdk.services.datasync.model.ListTagsForResourceRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListTagsForResourcePublisher publisher = client.listTagsForResourcePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datasync.model.ListTagsForResourceResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datasync.model.ListTagsForResourceResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTagsForResource(software.amazon.awssdk.services.datasync.model.ListTagsForResourceRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTagsForResourceRequest
     *        ListTagsForResourceRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    public ListTagsForResourcePublisher listTagsForResourcePaginator(ListTagsForResourceRequest listTagsForResourceRequest) {
        return new ListTagsForResourcePublisher(this, applyPaginatorUserAgent(listTagsForResourceRequest));
    }

    /**
     * <p>
     * Returns a list of executed tasks.
     * </p>
     *
     * @param listTaskExecutionsRequest
     *        ListTaskExecutions
     * @return A Java Future containing the result of the ListTaskExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListTaskExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListTaskExecutions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTaskExecutionsResponse> listTaskExecutions(ListTaskExecutionsRequest listTaskExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTaskExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTaskExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTaskExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTaskExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTaskExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTaskExecutionsRequest, ListTaskExecutionsResponse>()
                            .withOperationName("ListTaskExecutions")
                            .withMarshaller(new ListTaskExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTaskExecutionsRequest));
            CompletableFuture<ListTaskExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of executed tasks.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTaskExecutions(software.amazon.awssdk.services.datasync.model.ListTaskExecutionsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListTaskExecutionsPublisher publisher = client.listTaskExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListTaskExecutionsPublisher publisher = client.listTaskExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datasync.model.ListTaskExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datasync.model.ListTaskExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTaskExecutions(software.amazon.awssdk.services.datasync.model.ListTaskExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTaskExecutionsRequest
     *        ListTaskExecutions
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListTaskExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListTaskExecutions" target="_top">AWS
     *      API Documentation</a>
     */
    public ListTaskExecutionsPublisher listTaskExecutionsPaginator(ListTaskExecutionsRequest listTaskExecutionsRequest) {
        return new ListTaskExecutionsPublisher(this, applyPaginatorUserAgent(listTaskExecutionsRequest));
    }

    /**
     * <p>
     * Returns a list of the DataSync tasks you created.
     * </p>
     *
     * @param listTasksRequest
     *        ListTasksRequest
     * @return A Java Future containing the result of the ListTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTasksResponse> listTasks(ListTasksRequest listTasksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTasksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTasksRequest, ListTasksResponse>().withOperationName("ListTasks")
                            .withMarshaller(new ListTasksRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTasksRequest));
            CompletableFuture<ListTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the DataSync tasks you created.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTasks(software.amazon.awssdk.services.datasync.model.ListTasksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListTasksPublisher publisher = client.listTasksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.datasync.paginators.ListTasksPublisher publisher = client.listTasksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.datasync.model.ListTasksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.datasync.model.ListTasksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTasks(software.amazon.awssdk.services.datasync.model.ListTasksRequest)} operation.</b>
     * </p>
     *
     * @param listTasksRequest
     *        ListTasksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.ListTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/ListTasks" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTasksPublisher listTasksPaginator(ListTasksRequest listTasksRequest) {
        return new ListTasksPublisher(this, applyPaginatorUserAgent(listTasksRequest));
    }

    /**
     * <p>
     * Starts a specific invocation of a task. A <code>TaskExecution</code> value represents an individual run of a
     * task. Each task can have at most one <code>TaskExecution</code> at a time.
     * </p>
     * <p>
     * <code>TaskExecution</code> has the following transition phases: INITIALIZING | PREPARING | TRANSFERRING |
     * VERIFYING | SUCCESS/FAILURE.
     * </p>
     * <p>
     * For detailed information, see the Task Execution section in the Components and Terminology topic in the
     * <i>DataSync User Guide</i>.
     * </p>
     *
     * @param startTaskExecutionRequest
     *        StartTaskExecutionRequest
     * @return A Java Future containing the result of the StartTaskExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.StartTaskExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/StartTaskExecution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<StartTaskExecutionResponse> startTaskExecution(StartTaskExecutionRequest startTaskExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startTaskExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartTaskExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartTaskExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartTaskExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartTaskExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartTaskExecutionRequest, StartTaskExecutionResponse>()
                            .withOperationName("StartTaskExecution")
                            .withMarshaller(new StartTaskExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startTaskExecutionRequest));
            CompletableFuture<StartTaskExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies a key-value pair to an Amazon Web Services resource.
     * </p>
     *
     * @param tagResourceRequest
     *        TagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from an Amazon Web Services resource.
     * </p>
     *
     * @param untagResourceRequest
     *        UntagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the name of an agent.
     * </p>
     *
     * @param updateAgentRequest
     *        UpdateAgentRequest
     * @return A Java Future containing the result of the UpdateAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateAgent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAgentResponse> updateAgent(UpdateAgentRequest updateAgentRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAgentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAgentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAgentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAgentRequest, UpdateAgentResponse>()
                            .withOperationName("UpdateAgent").withMarshaller(new UpdateAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAgentRequest));
            CompletableFuture<UpdateAgentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates some parameters of a previously created location for a Hadoop Distributed File System cluster.
     * </p>
     *
     * @param updateLocationHdfsRequest
     * @return A Java Future containing the result of the UpdateLocationHdfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateLocationHdfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateLocationHdfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLocationHdfsResponse> updateLocationHdfs(UpdateLocationHdfsRequest updateLocationHdfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLocationHdfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLocationHdfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLocationHdfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLocationHdfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLocationHdfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLocationHdfsRequest, UpdateLocationHdfsResponse>()
                            .withOperationName("UpdateLocationHdfs")
                            .withMarshaller(new UpdateLocationHdfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLocationHdfsRequest));
            CompletableFuture<UpdateLocationHdfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates some of the parameters of a previously created location for Network File System (NFS) access. For
     * information about creating an NFS location, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-nfs-location.html">Creating a location for
     * NFS</a>.
     * </p>
     *
     * @param updateLocationNfsRequest
     * @return A Java Future containing the result of the UpdateLocationNfs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateLocationNfs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateLocationNfs" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLocationNfsResponse> updateLocationNfs(UpdateLocationNfsRequest updateLocationNfsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLocationNfsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLocationNfs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLocationNfsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLocationNfsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLocationNfsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLocationNfsRequest, UpdateLocationNfsResponse>()
                            .withOperationName("UpdateLocationNfs")
                            .withMarshaller(new UpdateLocationNfsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLocationNfsRequest));
            CompletableFuture<UpdateLocationNfsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates some of the parameters of a previously created location for self-managed object storage server access.
     * For information about creating a self-managed object storage location, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-object-location.html">Creating a location for
     * object storage</a>.
     * </p>
     *
     * @param updateLocationObjectStorageRequest
     * @return A Java Future containing the result of the UpdateLocationObjectStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateLocationObjectStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateLocationObjectStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLocationObjectStorageResponse> updateLocationObjectStorage(
            UpdateLocationObjectStorageRequest updateLocationObjectStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLocationObjectStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLocationObjectStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLocationObjectStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLocationObjectStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLocationObjectStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLocationObjectStorageRequest, UpdateLocationObjectStorageResponse>()
                            .withOperationName("UpdateLocationObjectStorage")
                            .withMarshaller(new UpdateLocationObjectStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLocationObjectStorageRequest));
            CompletableFuture<UpdateLocationObjectStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates some of the parameters of a previously created location for Server Message Block (SMB) file system
     * access. For information about creating an SMB location, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/create-smb-location.html">Creating a location for
     * SMB</a>.
     * </p>
     *
     * @param updateLocationSmbRequest
     * @return A Java Future containing the result of the UpdateLocationSmb operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateLocationSmb
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateLocationSmb" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLocationSmbResponse> updateLocationSmb(UpdateLocationSmbRequest updateLocationSmbRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLocationSmbRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLocationSmb");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLocationSmbResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLocationSmbResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLocationSmbResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLocationSmbRequest, UpdateLocationSmbResponse>()
                            .withOperationName("UpdateLocationSmb")
                            .withMarshaller(new UpdateLocationSmbRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateLocationSmbRequest));
            CompletableFuture<UpdateLocationSmbResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the metadata associated with a task.
     * </p>
     *
     * @param updateTaskRequest
     *        UpdateTaskResponse
     * @return A Java Future containing the result of the UpdateTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTaskResponse> updateTask(UpdateTaskRequest updateTaskRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTaskResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTaskRequest, UpdateTaskResponse>().withOperationName("UpdateTask")
                            .withMarshaller(new UpdateTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTaskRequest));
            CompletableFuture<UpdateTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates execution of a task.
     * </p>
     * <p>
     * You can modify bandwidth throttling for a task execution that is running or queued. For more information, see <a
     * href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/working-with-task-executions.html#adjust-bandwidth-throttling"
     * >Adjusting Bandwidth Throttling for a Task Execution</a>.
     * </p>
     * <note>
     * <p>
     * The only <code>Option</code> that can be modified by <code>UpdateTaskExecution</code> is
     * <code> <a href="https://docs.aws.amazon.com/datasync/latest/userguide/API_Options.html#DataSync-Type-Options-BytesPerSecond">BytesPerSecond</a> </code>
     * .
     * </p>
     * </note>
     *
     * @param updateTaskExecutionRequest
     * @return A Java Future containing the result of the UpdateTaskExecution operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidRequestException This exception is thrown when the client submits a malformed request.</li>
     *         <li>InternalException This exception is thrown when an error occurs in the DataSync service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DataSyncException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DataSyncAsyncClient.UpdateTaskExecution
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/UpdateTaskExecution" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTaskExecutionResponse> updateTaskExecution(
            UpdateTaskExecutionRequest updateTaskExecutionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTaskExecutionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DataSync");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTaskExecution");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTaskExecutionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTaskExecutionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTaskExecutionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTaskExecutionRequest, UpdateTaskExecutionResponse>()
                            .withOperationName("UpdateTaskExecution")
                            .withMarshaller(new UpdateTaskExecutionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTaskExecutionRequest));
            CompletableFuture<UpdateTaskExecutionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DataSyncException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends DataSyncRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
