/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLocationAzureBlobRequest extends DataSyncRequest implements
        ToCopyableBuilder<CreateLocationAzureBlobRequest.Builder, CreateLocationAzureBlobRequest> {
    private static final SdkField<String> CONTAINER_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContainerUrl").getter(getter(CreateLocationAzureBlobRequest::containerUrl))
            .setter(setter(Builder::containerUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContainerUrl").build()).build();

    private static final SdkField<String> AUTHENTICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthenticationType").getter(getter(CreateLocationAzureBlobRequest::authenticationTypeAsString))
            .setter(setter(Builder::authenticationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationType").build())
            .build();

    private static final SdkField<AzureBlobSasConfiguration> SAS_CONFIGURATION_FIELD = SdkField
            .<AzureBlobSasConfiguration> builder(MarshallingType.SDK_POJO).memberName("SasConfiguration")
            .getter(getter(CreateLocationAzureBlobRequest::sasConfiguration)).setter(setter(Builder::sasConfiguration))
            .constructor(AzureBlobSasConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SasConfiguration").build()).build();

    private static final SdkField<String> BLOB_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BlobType").getter(getter(CreateLocationAzureBlobRequest::blobTypeAsString))
            .setter(setter(Builder::blobType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlobType").build()).build();

    private static final SdkField<String> ACCESS_TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessTier").getter(getter(CreateLocationAzureBlobRequest::accessTierAsString))
            .setter(setter(Builder::accessTier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessTier").build()).build();

    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(CreateLocationAzureBlobRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(CreateLocationAzureBlobRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TagListEntry>> TAGS_FIELD = SdkField
            .<List<TagListEntry>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateLocationAzureBlobRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TagListEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagListEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONTAINER_URL_FIELD,
            AUTHENTICATION_TYPE_FIELD, SAS_CONFIGURATION_FIELD, BLOB_TYPE_FIELD, ACCESS_TIER_FIELD, SUBDIRECTORY_FIELD,
            AGENT_ARNS_FIELD, TAGS_FIELD));

    private final String containerUrl;

    private final String authenticationType;

    private final AzureBlobSasConfiguration sasConfiguration;

    private final String blobType;

    private final String accessTier;

    private final String subdirectory;

    private final List<String> agentArns;

    private final List<TagListEntry> tags;

    private CreateLocationAzureBlobRequest(BuilderImpl builder) {
        super(builder);
        this.containerUrl = builder.containerUrl;
        this.authenticationType = builder.authenticationType;
        this.sasConfiguration = builder.sasConfiguration;
        this.blobType = builder.blobType;
        this.accessTier = builder.accessTier;
        this.subdirectory = builder.subdirectory;
        this.agentArns = builder.agentArns;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Specifies the URL of the Azure Blob Storage container involved in your transfer.
     * </p>
     * 
     * @return Specifies the URL of the Azure Blob Storage container involved in your transfer.
     */
    public final String containerUrl() {
        return containerUrl;
    }

    /**
     * <p>
     * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
     * storage using a shared access signature (SAS).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link AzureBlobAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access
     *         blob storage using a shared access signature (SAS).
     * @see AzureBlobAuthenticationType
     */
    public final AzureBlobAuthenticationType authenticationType() {
        return AzureBlobAuthenticationType.fromValue(authenticationType);
    }

    /**
     * <p>
     * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
     * storage using a shared access signature (SAS).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link AzureBlobAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access
     *         blob storage using a shared access signature (SAS).
     * @see AzureBlobAuthenticationType
     */
    public final String authenticationTypeAsString() {
        return authenticationType;
    }

    /**
     * <p>
     * Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
     * </p>
     * 
     * @return Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
     */
    public final AzureBlobSasConfiguration sasConfiguration() {
        return sasConfiguration;
    }

    /**
     * <p>
     * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
     * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
     * information on blob types, see the <a href=
     * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     * >Azure Blob Storage documentation</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blobType} will
     * return {@link AzureBlobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blobTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of blob that you want your objects or files to be when transferring them into Azure
     *         Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For
     *         more information on blob types, see the <a href=
     *         "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     *         >Azure Blob Storage documentation</a>.
     * @see AzureBlobType
     */
    public final AzureBlobType blobType() {
        return AzureBlobType.fromValue(blobType);
    }

    /**
     * <p>
     * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
     * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
     * information on blob types, see the <a href=
     * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     * >Azure Blob Storage documentation</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blobType} will
     * return {@link AzureBlobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blobTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of blob that you want your objects or files to be when transferring them into Azure
     *         Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For
     *         more information on blob types, see the <a href=
     *         "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     *         >Azure Blob Storage documentation</a>.
     * @see AzureBlobType
     */
    public final String blobTypeAsString() {
        return blobType;
    }

    /**
     * <p>
     * Specifies the access tier that you want your objects or files transferred into. This only applies when using the
     * location as a transfer destination. For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     * >Access tiers</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessTier} will
     * return {@link AzureAccessTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTierAsString}.
     * </p>
     * 
     * @return Specifies the access tier that you want your objects or files transferred into. This only applies when
     *         using the location as a transfer destination. For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     *         >Access tiers</a>.
     * @see AzureAccessTier
     */
    public final AzureAccessTier accessTier() {
        return AzureAccessTier.fromValue(accessTier);
    }

    /**
     * <p>
     * Specifies the access tier that you want your objects or files transferred into. This only applies when using the
     * location as a transfer destination. For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     * >Access tiers</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessTier} will
     * return {@link AzureAccessTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTierAsString}.
     * </p>
     * 
     * @return Specifies the access tier that you want your objects or files transferred into. This only applies when
     *         using the location as a transfer destination. For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     *         >Access tiers</a>.
     * @see AzureAccessTier
     */
    public final String accessTierAsString() {
        return accessTier;
    }

    /**
     * <p>
     * Specifies path segments if you want to limit your transfer to a virtual directory in your container (for example,
     * <code>/my/images</code>).
     * </p>
     * 
     * @return Specifies path segments if you want to limit your transfer to a virtual directory in your container (for
     *         example, <code>/my/images</code>).
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob Storage
     * container.
     * </p>
     * <p>
     * You can specify more than one agent. For more information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents for your
     * transfer</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob
     *         Storage container.</p>
     *         <p>
     *         You can specify more than one agent. For more information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents
     *         for your transfer</a>.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
     * recommend creating at least a name tag for your transfer location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
     *         recommend creating at least a name tag for your transfer location.
     */
    public final List<TagListEntry> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(containerUrl());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sasConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(blobTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(accessTierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLocationAzureBlobRequest)) {
            return false;
        }
        CreateLocationAzureBlobRequest other = (CreateLocationAzureBlobRequest) obj;
        return Objects.equals(containerUrl(), other.containerUrl())
                && Objects.equals(authenticationTypeAsString(), other.authenticationTypeAsString())
                && Objects.equals(sasConfiguration(), other.sasConfiguration())
                && Objects.equals(blobTypeAsString(), other.blobTypeAsString())
                && Objects.equals(accessTierAsString(), other.accessTierAsString())
                && Objects.equals(subdirectory(), other.subdirectory()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLocationAzureBlobRequest").add("ContainerUrl", containerUrl())
                .add("AuthenticationType", authenticationTypeAsString()).add("SasConfiguration", sasConfiguration())
                .add("BlobType", blobTypeAsString()).add("AccessTier", accessTierAsString()).add("Subdirectory", subdirectory())
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ContainerUrl":
            return Optional.ofNullable(clazz.cast(containerUrl()));
        case "AuthenticationType":
            return Optional.ofNullable(clazz.cast(authenticationTypeAsString()));
        case "SasConfiguration":
            return Optional.ofNullable(clazz.cast(sasConfiguration()));
        case "BlobType":
            return Optional.ofNullable(clazz.cast(blobTypeAsString()));
        case "AccessTier":
            return Optional.ofNullable(clazz.cast(accessTierAsString()));
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLocationAzureBlobRequest, T> g) {
        return obj -> g.apply((CreateLocationAzureBlobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateLocationAzureBlobRequest> {
        /**
         * <p>
         * Specifies the URL of the Azure Blob Storage container involved in your transfer.
         * </p>
         * 
         * @param containerUrl
         *        Specifies the URL of the Azure Blob Storage container involved in your transfer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerUrl(String containerUrl);

        /**
         * <p>
         * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
         * storage using a shared access signature (SAS).
         * </p>
         * 
         * @param authenticationType
         *        Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can
         *        access blob storage using a shared access signature (SAS).
         * @see AzureBlobAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobAuthenticationType
         */
        Builder authenticationType(String authenticationType);

        /**
         * <p>
         * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
         * storage using a shared access signature (SAS).
         * </p>
         * 
         * @param authenticationType
         *        Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can
         *        access blob storage using a shared access signature (SAS).
         * @see AzureBlobAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobAuthenticationType
         */
        Builder authenticationType(AzureBlobAuthenticationType authenticationType);

        /**
         * <p>
         * Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
         * </p>
         * 
         * @param sasConfiguration
         *        Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sasConfiguration(AzureBlobSasConfiguration sasConfiguration);

        /**
         * <p>
         * Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
         * </p>
         * This is a convenience method that creates an instance of the {@link AzureBlobSasConfiguration.Builder}
         * avoiding the need to create one manually via {@link AzureBlobSasConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AzureBlobSasConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #sasConfiguration(AzureBlobSasConfiguration)}.
         * 
         * @param sasConfiguration
         *        a consumer that will call methods on {@link AzureBlobSasConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sasConfiguration(AzureBlobSasConfiguration)
         */
        default Builder sasConfiguration(Consumer<AzureBlobSasConfiguration.Builder> sasConfiguration) {
            return sasConfiguration(AzureBlobSasConfiguration.builder().applyMutation(sasConfiguration).build());
        }

        /**
         * <p>
         * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
         * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
         * information on blob types, see the <a href=
         * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         * >Azure Blob Storage documentation</a>.
         * </p>
         * 
         * @param blobType
         *        Specifies the type of blob that you want your objects or files to be when transferring them into Azure
         *        Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs.
         *        For more information on blob types, see the <a href=
         *        "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         *        >Azure Blob Storage documentation</a>.
         * @see AzureBlobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobType
         */
        Builder blobType(String blobType);

        /**
         * <p>
         * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
         * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
         * information on blob types, see the <a href=
         * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         * >Azure Blob Storage documentation</a>.
         * </p>
         * 
         * @param blobType
         *        Specifies the type of blob that you want your objects or files to be when transferring them into Azure
         *        Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs.
         *        For more information on blob types, see the <a href=
         *        "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         *        >Azure Blob Storage documentation</a>.
         * @see AzureBlobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobType
         */
        Builder blobType(AzureBlobType blobType);

        /**
         * <p>
         * Specifies the access tier that you want your objects or files transferred into. This only applies when using
         * the location as a transfer destination. For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         * >Access tiers</a>.
         * </p>
         * 
         * @param accessTier
         *        Specifies the access tier that you want your objects or files transferred into. This only applies when
         *        using the location as a transfer destination. For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         *        >Access tiers</a>.
         * @see AzureAccessTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureAccessTier
         */
        Builder accessTier(String accessTier);

        /**
         * <p>
         * Specifies the access tier that you want your objects or files transferred into. This only applies when using
         * the location as a transfer destination. For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         * >Access tiers</a>.
         * </p>
         * 
         * @param accessTier
         *        Specifies the access tier that you want your objects or files transferred into. This only applies when
         *        using the location as a transfer destination. For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         *        >Access tiers</a>.
         * @see AzureAccessTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureAccessTier
         */
        Builder accessTier(AzureAccessTier accessTier);

        /**
         * <p>
         * Specifies path segments if you want to limit your transfer to a virtual directory in your container (for
         * example, <code>/my/images</code>).
         * </p>
         * 
         * @param subdirectory
         *        Specifies path segments if you want to limit your transfer to a virtual directory in your container
         *        (for example, <code>/my/images</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob Storage
         * container.
         * </p>
         * <p>
         * You can specify more than one agent. For more information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents for
         * your transfer</a>.
         * </p>
         * 
         * @param agentArns
         *        Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob
         *        Storage container.</p>
         *        <p>
         *        You can specify more than one agent. For more information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple
         *        agents for your transfer</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob Storage
         * container.
         * </p>
         * <p>
         * You can specify more than one agent. For more information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents for
         * your transfer</a>.
         * </p>
         * 
         * @param agentArns
         *        Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob
         *        Storage container.</p>
         *        <p>
         *        You can specify more than one agent. For more information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple
         *        agents for your transfer</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
         * recommend creating at least a name tag for your transfer location.
         * </p>
         * 
         * @param tags
         *        Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources.
         *        We recommend creating at least a name tag for your transfer location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagListEntry> tags);

        /**
         * <p>
         * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
         * recommend creating at least a name tag for your transfer location.
         * </p>
         * 
         * @param tags
         *        Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources.
         *        We recommend creating at least a name tag for your transfer location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagListEntry... tags);

        /**
         * <p>
         * Specifies labels that help you categorize, filter, and search for your Amazon Web Services resources. We
         * recommend creating at least a name tag for your transfer location.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.TagListEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<TagListEntry>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<TagListEntry>)
         */
        Builder tags(Consumer<TagListEntry.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String containerUrl;

        private String authenticationType;

        private AzureBlobSasConfiguration sasConfiguration;

        private String blobType;

        private String accessTier;

        private String subdirectory;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private List<TagListEntry> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLocationAzureBlobRequest model) {
            super(model);
            containerUrl(model.containerUrl);
            authenticationType(model.authenticationType);
            sasConfiguration(model.sasConfiguration);
            blobType(model.blobType);
            accessTier(model.accessTier);
            subdirectory(model.subdirectory);
            agentArns(model.agentArns);
            tags(model.tags);
        }

        public final String getContainerUrl() {
            return containerUrl;
        }

        public final void setContainerUrl(String containerUrl) {
            this.containerUrl = containerUrl;
        }

        @Override
        public final Builder containerUrl(String containerUrl) {
            this.containerUrl = containerUrl;
            return this;
        }

        public final String getAuthenticationType() {
            return authenticationType;
        }

        public final void setAuthenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
        }

        @Override
        public final Builder authenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        @Override
        public final Builder authenticationType(AzureBlobAuthenticationType authenticationType) {
            this.authenticationType(authenticationType == null ? null : authenticationType.toString());
            return this;
        }

        public final AzureBlobSasConfiguration.Builder getSasConfiguration() {
            return sasConfiguration != null ? sasConfiguration.toBuilder() : null;
        }

        public final void setSasConfiguration(AzureBlobSasConfiguration.BuilderImpl sasConfiguration) {
            this.sasConfiguration = sasConfiguration != null ? sasConfiguration.build() : null;
        }

        @Override
        public final Builder sasConfiguration(AzureBlobSasConfiguration sasConfiguration) {
            this.sasConfiguration = sasConfiguration;
            return this;
        }

        public final String getBlobType() {
            return blobType;
        }

        public final void setBlobType(String blobType) {
            this.blobType = blobType;
        }

        @Override
        public final Builder blobType(String blobType) {
            this.blobType = blobType;
            return this;
        }

        @Override
        public final Builder blobType(AzureBlobType blobType) {
            this.blobType(blobType == null ? null : blobType.toString());
            return this;
        }

        public final String getAccessTier() {
            return accessTier;
        }

        public final void setAccessTier(String accessTier) {
            this.accessTier = accessTier;
        }

        @Override
        public final Builder accessTier(String accessTier) {
            this.accessTier = accessTier;
            return this;
        }

        @Override
        public final Builder accessTier(AzureAccessTier accessTier) {
            this.accessTier(accessTier == null ? null : accessTier.toString());
            return this;
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final List<TagListEntry.Builder> getTags() {
            List<TagListEntry.Builder> result = InputTagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<TagListEntry.BuilderImpl> tags) {
            this.tags = InputTagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<TagListEntry> tags) {
            this.tags = InputTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagListEntry... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagListEntry.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagListEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLocationAzureBlobRequest build() {
            return new CreateLocationAzureBlobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
