/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CreateAgentRequest
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAgentRequest extends DataSyncRequest implements
        ToCopyableBuilder<CreateAgentRequest.Builder, CreateAgentRequest> {
    private static final SdkField<String> ACTIVATION_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAgentRequest::activationKey)).setter(setter(Builder::activationKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivationKey").build()).build();

    private static final SdkField<String> AGENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAgentRequest::agentName)).setter(setter(Builder::agentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentName").build()).build();

    private static final SdkField<List<TagListEntry>> TAGS_FIELD = SdkField
            .<List<TagListEntry>> builder(MarshallingType.LIST)
            .getter(getter(CreateAgentRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TagListEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagListEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIVATION_KEY_FIELD,
            AGENT_NAME_FIELD, TAGS_FIELD));

    private final String activationKey;

    private final String agentName;

    private final List<TagListEntry> tags;

    private CreateAgentRequest(BuilderImpl builder) {
        super(builder);
        this.activationKey = builder.activationKey;
        this.agentName = builder.agentName;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Your agent activation key. You can get the activation key either by sending an HTTP GET request with redirects
     * that enable you to get the agent IP address (port 80). Alternatively, you can get it from the AWS DataSync
     * console.
     * </p>
     * <p>
     * The redirect URL returned in the response provides you the activation key for your agent in the query string
     * parameter <code>activationKey</code>. It might also include other activation-related parameters; however, these
     * are merely defaults. The arguments you pass to this API call determine the actual configuration of your agent.
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/sync-service/latest/userguide/working-with-sync-agents.html#activating-sync-agent"
     * >Activating a Sync Agent</a> in the <i>AWS DataSync User Guide.</i>
     * </p>
     * 
     * @return Your agent activation key. You can get the activation key either by sending an HTTP GET request with
     *         redirects that enable you to get the agent IP address (port 80). Alternatively, you can get it from the
     *         AWS DataSync console. </p>
     *         <p>
     *         The redirect URL returned in the response provides you the activation key for your agent in the query
     *         string parameter <code>activationKey</code>. It might also include other activation-related parameters;
     *         however, these are merely defaults. The arguments you pass to this API call determine the actual
     *         configuration of your agent. For more information, see <a href=
     *         "https://docs.aws.amazon.com/sync-service/latest/userguide/working-with-sync-agents.html#activating-sync-agent"
     *         >Activating a Sync Agent</a> in the <i>AWS DataSync User Guide.</i>
     */
    public String activationKey() {
        return activationKey;
    }

    /**
     * <p>
     * The name you configured for your agent. This value is a text reference that is used to identify the agent in the
     * console.
     * </p>
     * 
     * @return The name you configured for your agent. This value is a text reference that is used to identify the agent
     *         in the console.
     */
    public String agentName() {
        return agentName;
    }

    /**
     * <p>
     * The key-value pair that represents the tag you want to associate with the agent. The value can be an empty
     * string. This value helps you manage, filter, and search for your agents.
     * </p>
     * <note>
     * <p>
     * Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and the
     * following special characters: + - = . _ : / @.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The key-value pair that represents the tag you want to associate with the agent. The value can be an
     *         empty string. This value helps you manage, filter, and search for your agents.</p> <note>
     *         <p>
     *         Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and
     *         the following special characters: + - = . _ : / @.
     *         </p>
     */
    public List<TagListEntry> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(activationKey());
        hashCode = 31 * hashCode + Objects.hashCode(agentName());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAgentRequest)) {
            return false;
        }
        CreateAgentRequest other = (CreateAgentRequest) obj;
        return Objects.equals(activationKey(), other.activationKey()) && Objects.equals(agentName(), other.agentName())
                && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateAgentRequest").add("ActivationKey", activationKey()).add("AgentName", agentName())
                .add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActivationKey":
            return Optional.ofNullable(clazz.cast(activationKey()));
        case "AgentName":
            return Optional.ofNullable(clazz.cast(agentName()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateAgentRequest, T> g) {
        return obj -> g.apply((CreateAgentRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateAgentRequest> {
        /**
         * <p>
         * Your agent activation key. You can get the activation key either by sending an HTTP GET request with
         * redirects that enable you to get the agent IP address (port 80). Alternatively, you can get it from the AWS
         * DataSync console.
         * </p>
         * <p>
         * The redirect URL returned in the response provides you the activation key for your agent in the query string
         * parameter <code>activationKey</code>. It might also include other activation-related parameters; however,
         * these are merely defaults. The arguments you pass to this API call determine the actual configuration of your
         * agent. For more information, see <a href=
         * "https://docs.aws.amazon.com/sync-service/latest/userguide/working-with-sync-agents.html#activating-sync-agent"
         * >Activating a Sync Agent</a> in the <i>AWS DataSync User Guide.</i>
         * </p>
         * 
         * @param activationKey
         *        Your agent activation key. You can get the activation key either by sending an HTTP GET request with
         *        redirects that enable you to get the agent IP address (port 80). Alternatively, you can get it from
         *        the AWS DataSync console. </p>
         *        <p>
         *        The redirect URL returned in the response provides you the activation key for your agent in the query
         *        string parameter <code>activationKey</code>. It might also include other activation-related
         *        parameters; however, these are merely defaults. The arguments you pass to this API call determine the
         *        actual configuration of your agent. For more information, see <a href=
         *        "https://docs.aws.amazon.com/sync-service/latest/userguide/working-with-sync-agents.html#activating-sync-agent"
         *        >Activating a Sync Agent</a> in the <i>AWS DataSync User Guide.</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activationKey(String activationKey);

        /**
         * <p>
         * The name you configured for your agent. This value is a text reference that is used to identify the agent in
         * the console.
         * </p>
         * 
         * @param agentName
         *        The name you configured for your agent. This value is a text reference that is used to identify the
         *        agent in the console.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentName(String agentName);

        /**
         * <p>
         * The key-value pair that represents the tag you want to associate with the agent. The value can be an empty
         * string. This value helps you manage, filter, and search for your agents.
         * </p>
         * <note>
         * <p>
         * Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and the
         * following special characters: + - = . _ : / @.
         * </p>
         * </note>
         * 
         * @param tags
         *        The key-value pair that represents the tag you want to associate with the agent. The value can be an
         *        empty string. This value helps you manage, filter, and search for your agents.</p> <note>
         *        <p>
         *        Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and
         *        the following special characters: + - = . _ : / @.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagListEntry> tags);

        /**
         * <p>
         * The key-value pair that represents the tag you want to associate with the agent. The value can be an empty
         * string. This value helps you manage, filter, and search for your agents.
         * </p>
         * <note>
         * <p>
         * Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and the
         * following special characters: + - = . _ : / @.
         * </p>
         * </note>
         * 
         * @param tags
         *        The key-value pair that represents the tag you want to associate with the agent. The value can be an
         *        empty string. This value helps you manage, filter, and search for your agents.</p> <note>
         *        <p>
         *        Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and
         *        the following special characters: + - = . _ : / @.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagListEntry... tags);

        /**
         * <p>
         * The key-value pair that represents the tag you want to associate with the agent. The value can be an empty
         * string. This value helps you manage, filter, and search for your agents.
         * </p>
         * <note>
         * <p>
         * Valid characters for key and value are letters, spaces, and numbers representable in UTF-8 format, and the
         * following special characters: + - = . _ : / @.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<TagListEntry>.Builder} avoiding the
         * need to create one manually via {@link List<TagListEntry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagListEntry>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<TagListEntry>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<TagListEntry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<TagListEntry>)
         */
        Builder tags(Consumer<TagListEntry.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String activationKey;

        private String agentName;

        private List<TagListEntry> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAgentRequest model) {
            super(model);
            activationKey(model.activationKey);
            agentName(model.agentName);
            tags(model.tags);
        }

        public final String getActivationKey() {
            return activationKey;
        }

        @Override
        public final Builder activationKey(String activationKey) {
            this.activationKey = activationKey;
            return this;
        }

        public final void setActivationKey(String activationKey) {
            this.activationKey = activationKey;
        }

        public final String getAgentName() {
            return agentName;
        }

        @Override
        public final Builder agentName(String agentName) {
            this.agentName = agentName;
            return this;
        }

        public final void setAgentName(String agentName) {
            this.agentName = agentName;
        }

        public final Collection<TagListEntry.Builder> getTags() {
            return tags != null ? tags.stream().map(TagListEntry::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<TagListEntry> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagListEntry... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagListEntry.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagListEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<TagListEntry.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAgentRequest build() {
            return new CreateAgentRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
