/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The subnet and security groups that DataSync uses to connect to one of your Amazon EFS file system's <a
 * href="https://docs.aws.amazon.com/efs/latest/ug/accessing-fs.html">mount targets</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ec2Config implements SdkPojo, Serializable, ToCopyableBuilder<Ec2Config.Builder, Ec2Config> {
    private static final SdkField<String> SUBNET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SubnetArn").getter(getter(Ec2Config::subnetArn)).setter(setter(Builder::subnetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetArn").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupArns")
            .getter(getter(Ec2Config::securityGroupArns))
            .setter(setter(Builder::securityGroupArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SUBNET_ARN_FIELD,
            SECURITY_GROUP_ARNS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String subnetArn;

    private final List<String> securityGroupArns;

    private Ec2Config(BuilderImpl builder) {
        this.subnetArn = builder.subnetArn;
        this.securityGroupArns = builder.securityGroupArns;
    }

    /**
     * <p>
     * Specifies the ARN of a subnet where DataSync creates the <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/datasync-network.html#required-network-interfaces"
     * >network interfaces</a> for managing traffic during your transfer.
     * </p>
     * <p>
     * The subnet must be located:
     * </p>
     * <ul>
     * <li>
     * <p>
     * In the same virtual private cloud (VPC) as the Amazon EFS file system.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the same Availability Zone as at least one mount target for the Amazon EFS file system.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You don't need to specify a subnet that includes a file system mount target.
     * </p>
     * </note>
     * 
     * @return Specifies the ARN of a subnet where DataSync creates the <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/datasync-network.html#required-network-interfaces"
     *         >network interfaces</a> for managing traffic during your transfer.</p>
     *         <p>
     *         The subnet must be located:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         In the same virtual private cloud (VPC) as the Amazon EFS file system.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         In the same Availability Zone as at least one mount target for the Amazon EFS file system.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You don't need to specify a subnet that includes a file system mount target.
     *         </p>
     */
    public final String subnetArn() {
        return subnetArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupArns() {
        return securityGroupArns != null && !(securityGroupArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the Amazon Resource Names (ARNs) of the security groups associated with an Amazon EFS file system's
     * mount target.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupArns} method.
     * </p>
     * 
     * @return Specifies the Amazon Resource Names (ARNs) of the security groups associated with an Amazon EFS file
     *         system's mount target.
     */
    public final List<String> securityGroupArns() {
        return securityGroupArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(subnetArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupArns() ? securityGroupArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ec2Config)) {
            return false;
        }
        Ec2Config other = (Ec2Config) obj;
        return Objects.equals(subnetArn(), other.subnetArn()) && hasSecurityGroupArns() == other.hasSecurityGroupArns()
                && Objects.equals(securityGroupArns(), other.securityGroupArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Ec2Config").add("SubnetArn", subnetArn())
                .add("SecurityGroupArns", hasSecurityGroupArns() ? securityGroupArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubnetArn":
            return Optional.ofNullable(clazz.cast(subnetArn()));
        case "SecurityGroupArns":
            return Optional.ofNullable(clazz.cast(securityGroupArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SubnetArn", SUBNET_ARN_FIELD);
        map.put("SecurityGroupArns", SECURITY_GROUP_ARNS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Ec2Config, T> g) {
        return obj -> g.apply((Ec2Config) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ec2Config> {
        /**
         * <p>
         * Specifies the ARN of a subnet where DataSync creates the <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/datasync-network.html#required-network-interfaces"
         * >network interfaces</a> for managing traffic during your transfer.
         * </p>
         * <p>
         * The subnet must be located:
         * </p>
         * <ul>
         * <li>
         * <p>
         * In the same virtual private cloud (VPC) as the Amazon EFS file system.
         * </p>
         * </li>
         * <li>
         * <p>
         * In the same Availability Zone as at least one mount target for the Amazon EFS file system.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You don't need to specify a subnet that includes a file system mount target.
         * </p>
         * </note>
         * 
         * @param subnetArn
         *        Specifies the ARN of a subnet where DataSync creates the <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/datasync-network.html#required-network-interfaces"
         *        >network interfaces</a> for managing traffic during your transfer.</p>
         *        <p>
         *        The subnet must be located:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        In the same virtual private cloud (VPC) as the Amazon EFS file system.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        In the same Availability Zone as at least one mount target for the Amazon EFS file system.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You don't need to specify a subnet that includes a file system mount target.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetArn(String subnetArn);

        /**
         * <p>
         * Specifies the Amazon Resource Names (ARNs) of the security groups associated with an Amazon EFS file system's
         * mount target.
         * </p>
         * 
         * @param securityGroupArns
         *        Specifies the Amazon Resource Names (ARNs) of the security groups associated with an Amazon EFS file
         *        system's mount target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(Collection<String> securityGroupArns);

        /**
         * <p>
         * Specifies the Amazon Resource Names (ARNs) of the security groups associated with an Amazon EFS file system's
         * mount target.
         * </p>
         * 
         * @param securityGroupArns
         *        Specifies the Amazon Resource Names (ARNs) of the security groups associated with an Amazon EFS file
         *        system's mount target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupArns(String... securityGroupArns);
    }

    static final class BuilderImpl implements Builder {
        private String subnetArn;

        private List<String> securityGroupArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Ec2Config model) {
            subnetArn(model.subnetArn);
            securityGroupArns(model.securityGroupArns);
        }

        public final String getSubnetArn() {
            return subnetArn;
        }

        public final void setSubnetArn(String subnetArn) {
            this.subnetArn = subnetArn;
        }

        @Override
        public final Builder subnetArn(String subnetArn) {
            this.subnetArn = subnetArn;
            return this;
        }

        public final Collection<String> getSecurityGroupArns() {
            if (securityGroupArns instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupArns;
        }

        public final void setSecurityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = Ec2SecurityGroupArnListCopier.copy(securityGroupArns);
        }

        @Override
        public final Builder securityGroupArns(Collection<String> securityGroupArns) {
            this.securityGroupArns = Ec2SecurityGroupArnListCopier.copy(securityGroupArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupArns(String... securityGroupArns) {
            securityGroupArns(Arrays.asList(securityGroupArns));
            return this;
        }

        @Override
        public Ec2Config build() {
            return new Ec2Config(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
