/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The information that DataSync Discovery collects about an on-premises storage system cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetAppONTAPCluster implements SdkPojo, Serializable,
        ToCopyableBuilder<NetAppONTAPCluster.Builder, NetAppONTAPCluster> {
    private static final SdkField<Long> CIFS_SHARE_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CifsShareCount").getter(getter(NetAppONTAPCluster::cifsShareCount))
            .setter(setter(Builder::cifsShareCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CifsShareCount").build()).build();

    private static final SdkField<Long> NFS_EXPORTED_VOLUMES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("NfsExportedVolumes").getter(getter(NetAppONTAPCluster::nfsExportedVolumes))
            .setter(setter(Builder::nfsExportedVolumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NfsExportedVolumes").build())
            .build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(NetAppONTAPCluster::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClusterName").getter(getter(NetAppONTAPCluster::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterName").build()).build();

    private static final SdkField<MaxP95Performance> MAX_P95_PERFORMANCE_FIELD = SdkField
            .<MaxP95Performance> builder(MarshallingType.SDK_POJO).memberName("MaxP95Performance")
            .getter(getter(NetAppONTAPCluster::maxP95Performance)).setter(setter(Builder::maxP95Performance))
            .constructor(MaxP95Performance::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxP95Performance").build()).build();

    private static final SdkField<Long> CLUSTER_BLOCK_STORAGE_SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ClusterBlockStorageSize").getter(getter(NetAppONTAPCluster::clusterBlockStorageSize))
            .setter(setter(Builder::clusterBlockStorageSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterBlockStorageSize").build())
            .build();

    private static final SdkField<Long> CLUSTER_BLOCK_STORAGE_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ClusterBlockStorageUsed").getter(getter(NetAppONTAPCluster::clusterBlockStorageUsed))
            .setter(setter(Builder::clusterBlockStorageUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterBlockStorageUsed").build())
            .build();

    private static final SdkField<Long> CLUSTER_BLOCK_STORAGE_LOGICAL_USED_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("ClusterBlockStorageLogicalUsed")
            .getter(getter(NetAppONTAPCluster::clusterBlockStorageLogicalUsed))
            .setter(setter(Builder::clusterBlockStorageLogicalUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterBlockStorageLogicalUsed")
                    .build()).build();

    private static final SdkField<List<Recommendation>> RECOMMENDATIONS_FIELD = SdkField
            .<List<Recommendation>> builder(MarshallingType.LIST)
            .memberName("Recommendations")
            .getter(getter(NetAppONTAPCluster::recommendations))
            .setter(setter(Builder::recommendations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Recommendations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Recommendation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Recommendation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RECOMMENDATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecommendationStatus").getter(getter(NetAppONTAPCluster::recommendationStatusAsString))
            .setter(setter(Builder::recommendationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecommendationStatus").build())
            .build();

    private static final SdkField<Long> LUN_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("LunCount")
            .getter(getter(NetAppONTAPCluster::lunCount)).setter(setter(Builder::lunCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LunCount").build()).build();

    private static final SdkField<Long> CLUSTER_CLOUD_STORAGE_USED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ClusterCloudStorageUsed").getter(getter(NetAppONTAPCluster::clusterCloudStorageUsed))
            .setter(setter(Builder::clusterCloudStorageUsed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterCloudStorageUsed").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CIFS_SHARE_COUNT_FIELD,
            NFS_EXPORTED_VOLUMES_FIELD, RESOURCE_ID_FIELD, CLUSTER_NAME_FIELD, MAX_P95_PERFORMANCE_FIELD,
            CLUSTER_BLOCK_STORAGE_SIZE_FIELD, CLUSTER_BLOCK_STORAGE_USED_FIELD, CLUSTER_BLOCK_STORAGE_LOGICAL_USED_FIELD,
            RECOMMENDATIONS_FIELD, RECOMMENDATION_STATUS_FIELD, LUN_COUNT_FIELD, CLUSTER_CLOUD_STORAGE_USED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Long cifsShareCount;

    private final Long nfsExportedVolumes;

    private final String resourceId;

    private final String clusterName;

    private final MaxP95Performance maxP95Performance;

    private final Long clusterBlockStorageSize;

    private final Long clusterBlockStorageUsed;

    private final Long clusterBlockStorageLogicalUsed;

    private final List<Recommendation> recommendations;

    private final String recommendationStatus;

    private final Long lunCount;

    private final Long clusterCloudStorageUsed;

    private NetAppONTAPCluster(BuilderImpl builder) {
        this.cifsShareCount = builder.cifsShareCount;
        this.nfsExportedVolumes = builder.nfsExportedVolumes;
        this.resourceId = builder.resourceId;
        this.clusterName = builder.clusterName;
        this.maxP95Performance = builder.maxP95Performance;
        this.clusterBlockStorageSize = builder.clusterBlockStorageSize;
        this.clusterBlockStorageUsed = builder.clusterBlockStorageUsed;
        this.clusterBlockStorageLogicalUsed = builder.clusterBlockStorageLogicalUsed;
        this.recommendations = builder.recommendations;
        this.recommendationStatus = builder.recommendationStatus;
        this.lunCount = builder.lunCount;
        this.clusterCloudStorageUsed = builder.clusterCloudStorageUsed;
    }

    /**
     * <p>
     * The number of CIFS shares in the cluster.
     * </p>
     * 
     * @return The number of CIFS shares in the cluster.
     */
    public final Long cifsShareCount() {
        return cifsShareCount;
    }

    /**
     * <p>
     * The number of NFS volumes in the cluster.
     * </p>
     * 
     * @return The number of NFS volumes in the cluster.
     */
    public final Long nfsExportedVolumes() {
        return nfsExportedVolumes;
    }

    /**
     * <p>
     * The universally unique identifier (UUID) of the cluster.
     * </p>
     * 
     * @return The universally unique identifier (UUID) of the cluster.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The name of the cluster.
     * </p>
     * 
     * @return The name of the cluster.
     */
    public final String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * The performance data that DataSync Discovery collects about the cluster.
     * </p>
     * 
     * @return The performance data that DataSync Discovery collects about the cluster.
     */
    public final MaxP95Performance maxP95Performance() {
        return maxP95Performance;
    }

    /**
     * <p>
     * The total storage space that's available in the cluster.
     * </p>
     * 
     * @return The total storage space that's available in the cluster.
     */
    public final Long clusterBlockStorageSize() {
        return clusterBlockStorageSize;
    }

    /**
     * <p>
     * The storage space that's being used in a cluster.
     * </p>
     * 
     * @return The storage space that's being used in a cluster.
     */
    public final Long clusterBlockStorageUsed() {
        return clusterBlockStorageUsed;
    }

    /**
     * <p>
     * The storage space that's being used in the cluster without accounting for compression or deduplication.
     * </p>
     * 
     * @return The storage space that's being used in the cluster without accounting for compression or deduplication.
     */
    public final Long clusterBlockStorageLogicalUsed() {
        return clusterBlockStorageLogicalUsed;
    }

    /**
     * For responses, this returns true if the service returned a value for the Recommendations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendations() {
        return recommendations != null && !(recommendations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
     * >Recommendations provided by DataSync Discovery</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendations} method.
     * </p>
     * 
     * @return The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
     *         >Recommendations provided by DataSync Discovery</a>.
     */
    public final List<Recommendation> recommendations() {
        return recommendations;
    }

    /**
     * <p>
     * Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or can't be
     * determined.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     * >Recommendation statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recommendationStatus} will return {@link RecommendationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #recommendationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or
     *         can't be determined.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     *         >Recommendation statuses</a>.
     * @see RecommendationStatus
     */
    public final RecommendationStatus recommendationStatus() {
        return RecommendationStatus.fromValue(recommendationStatus);
    }

    /**
     * <p>
     * Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or can't be
     * determined.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     * >Recommendation statuses</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recommendationStatus} will return {@link RecommendationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #recommendationStatusAsString}.
     * </p>
     * 
     * @return Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or
     *         can't be determined.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
     *         >Recommendation statuses</a>.
     * @see RecommendationStatus
     */
    public final String recommendationStatusAsString() {
        return recommendationStatus;
    }

    /**
     * <p>
     * The number of LUNs (logical unit numbers) in the cluster.
     * </p>
     * 
     * @return The number of LUNs (logical unit numbers) in the cluster.
     */
    public final Long lunCount() {
        return lunCount;
    }

    /**
     * <p>
     * The amount of space in the cluster that's in cloud storage (for example, if you're using data tiering).
     * </p>
     * 
     * @return The amount of space in the cluster that's in cloud storage (for example, if you're using data tiering).
     */
    public final Long clusterCloudStorageUsed() {
        return clusterCloudStorageUsed;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cifsShareCount());
        hashCode = 31 * hashCode + Objects.hashCode(nfsExportedVolumes());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(clusterName());
        hashCode = 31 * hashCode + Objects.hashCode(maxP95Performance());
        hashCode = 31 * hashCode + Objects.hashCode(clusterBlockStorageSize());
        hashCode = 31 * hashCode + Objects.hashCode(clusterBlockStorageUsed());
        hashCode = 31 * hashCode + Objects.hashCode(clusterBlockStorageLogicalUsed());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendations() ? recommendations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recommendationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lunCount());
        hashCode = 31 * hashCode + Objects.hashCode(clusterCloudStorageUsed());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetAppONTAPCluster)) {
            return false;
        }
        NetAppONTAPCluster other = (NetAppONTAPCluster) obj;
        return Objects.equals(cifsShareCount(), other.cifsShareCount())
                && Objects.equals(nfsExportedVolumes(), other.nfsExportedVolumes())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(clusterName(), other.clusterName())
                && Objects.equals(maxP95Performance(), other.maxP95Performance())
                && Objects.equals(clusterBlockStorageSize(), other.clusterBlockStorageSize())
                && Objects.equals(clusterBlockStorageUsed(), other.clusterBlockStorageUsed())
                && Objects.equals(clusterBlockStorageLogicalUsed(), other.clusterBlockStorageLogicalUsed())
                && hasRecommendations() == other.hasRecommendations()
                && Objects.equals(recommendations(), other.recommendations())
                && Objects.equals(recommendationStatusAsString(), other.recommendationStatusAsString())
                && Objects.equals(lunCount(), other.lunCount())
                && Objects.equals(clusterCloudStorageUsed(), other.clusterCloudStorageUsed());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetAppONTAPCluster").add("CifsShareCount", cifsShareCount())
                .add("NfsExportedVolumes", nfsExportedVolumes()).add("ResourceId", resourceId())
                .add("ClusterName", clusterName()).add("MaxP95Performance", maxP95Performance())
                .add("ClusterBlockStorageSize", clusterBlockStorageSize())
                .add("ClusterBlockStorageUsed", clusterBlockStorageUsed())
                .add("ClusterBlockStorageLogicalUsed", clusterBlockStorageLogicalUsed())
                .add("Recommendations", hasRecommendations() ? recommendations() : null)
                .add("RecommendationStatus", recommendationStatusAsString()).add("LunCount", lunCount())
                .add("ClusterCloudStorageUsed", clusterCloudStorageUsed()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CifsShareCount":
            return Optional.ofNullable(clazz.cast(cifsShareCount()));
        case "NfsExportedVolumes":
            return Optional.ofNullable(clazz.cast(nfsExportedVolumes()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ClusterName":
            return Optional.ofNullable(clazz.cast(clusterName()));
        case "MaxP95Performance":
            return Optional.ofNullable(clazz.cast(maxP95Performance()));
        case "ClusterBlockStorageSize":
            return Optional.ofNullable(clazz.cast(clusterBlockStorageSize()));
        case "ClusterBlockStorageUsed":
            return Optional.ofNullable(clazz.cast(clusterBlockStorageUsed()));
        case "ClusterBlockStorageLogicalUsed":
            return Optional.ofNullable(clazz.cast(clusterBlockStorageLogicalUsed()));
        case "Recommendations":
            return Optional.ofNullable(clazz.cast(recommendations()));
        case "RecommendationStatus":
            return Optional.ofNullable(clazz.cast(recommendationStatusAsString()));
        case "LunCount":
            return Optional.ofNullable(clazz.cast(lunCount()));
        case "ClusterCloudStorageUsed":
            return Optional.ofNullable(clazz.cast(clusterCloudStorageUsed()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CifsShareCount", CIFS_SHARE_COUNT_FIELD);
        map.put("NfsExportedVolumes", NFS_EXPORTED_VOLUMES_FIELD);
        map.put("ResourceId", RESOURCE_ID_FIELD);
        map.put("ClusterName", CLUSTER_NAME_FIELD);
        map.put("MaxP95Performance", MAX_P95_PERFORMANCE_FIELD);
        map.put("ClusterBlockStorageSize", CLUSTER_BLOCK_STORAGE_SIZE_FIELD);
        map.put("ClusterBlockStorageUsed", CLUSTER_BLOCK_STORAGE_USED_FIELD);
        map.put("ClusterBlockStorageLogicalUsed", CLUSTER_BLOCK_STORAGE_LOGICAL_USED_FIELD);
        map.put("Recommendations", RECOMMENDATIONS_FIELD);
        map.put("RecommendationStatus", RECOMMENDATION_STATUS_FIELD);
        map.put("LunCount", LUN_COUNT_FIELD);
        map.put("ClusterCloudStorageUsed", CLUSTER_CLOUD_STORAGE_USED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NetAppONTAPCluster, T> g) {
        return obj -> g.apply((NetAppONTAPCluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetAppONTAPCluster> {
        /**
         * <p>
         * The number of CIFS shares in the cluster.
         * </p>
         * 
         * @param cifsShareCount
         *        The number of CIFS shares in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cifsShareCount(Long cifsShareCount);

        /**
         * <p>
         * The number of NFS volumes in the cluster.
         * </p>
         * 
         * @param nfsExportedVolumes
         *        The number of NFS volumes in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExportedVolumes(Long nfsExportedVolumes);

        /**
         * <p>
         * The universally unique identifier (UUID) of the cluster.
         * </p>
         * 
         * @param resourceId
         *        The universally unique identifier (UUID) of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The name of the cluster.
         * </p>
         * 
         * @param clusterName
         *        The name of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * The performance data that DataSync Discovery collects about the cluster.
         * </p>
         * 
         * @param maxP95Performance
         *        The performance data that DataSync Discovery collects about the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxP95Performance(MaxP95Performance maxP95Performance);

        /**
         * <p>
         * The performance data that DataSync Discovery collects about the cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link MaxP95Performance.Builder} avoiding the
         * need to create one manually via {@link MaxP95Performance#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MaxP95Performance.Builder#build()} is called immediately and its
         * result is passed to {@link #maxP95Performance(MaxP95Performance)}.
         * 
         * @param maxP95Performance
         *        a consumer that will call methods on {@link MaxP95Performance.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maxP95Performance(MaxP95Performance)
         */
        default Builder maxP95Performance(Consumer<MaxP95Performance.Builder> maxP95Performance) {
            return maxP95Performance(MaxP95Performance.builder().applyMutation(maxP95Performance).build());
        }

        /**
         * <p>
         * The total storage space that's available in the cluster.
         * </p>
         * 
         * @param clusterBlockStorageSize
         *        The total storage space that's available in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterBlockStorageSize(Long clusterBlockStorageSize);

        /**
         * <p>
         * The storage space that's being used in a cluster.
         * </p>
         * 
         * @param clusterBlockStorageUsed
         *        The storage space that's being used in a cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterBlockStorageUsed(Long clusterBlockStorageUsed);

        /**
         * <p>
         * The storage space that's being used in the cluster without accounting for compression or deduplication.
         * </p>
         * 
         * @param clusterBlockStorageLogicalUsed
         *        The storage space that's being used in the cluster without accounting for compression or
         *        deduplication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterBlockStorageLogicalUsed(Long clusterBlockStorageLogicalUsed);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * 
         * @param recommendations
         *        The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         *        >Recommendations provided by DataSync Discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Collection<Recommendation> recommendations);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * 
         * @param recommendations
         *        The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         *        >Recommendations provided by DataSync Discovery</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendations(Recommendation... recommendations);

        /**
         * <p>
         * The Amazon Web Services storage services that DataSync Discovery recommends for the cluster. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
         * >Recommendations provided by DataSync Discovery</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.Recommendation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder#build()} is called immediately
         * and its result is passed to {@link #recommendations(List<Recommendation>)}.
         * 
         * @param recommendations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.Recommendation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendations(java.util.Collection<Recommendation>)
         */
        Builder recommendations(Consumer<Recommendation.Builder>... recommendations);

        /**
         * <p>
         * Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or can't
         * be determined.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         * >Recommendation statuses</a>.
         * </p>
         * 
         * @param recommendationStatus
         *        Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or
         *        can't be determined.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         *        >Recommendation statuses</a>.
         * @see RecommendationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationStatus
         */
        Builder recommendationStatus(String recommendationStatus);

        /**
         * <p>
         * Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or can't
         * be determined.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         * >Recommendation statuses</a>.
         * </p>
         * 
         * @param recommendationStatus
         *        Indicates whether DataSync Discovery recommendations for the cluster are ready to view, incomplete, or
         *        can't be determined.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-statuses.html#recommendation-statuses-table"
         *        >Recommendation statuses</a>.
         * @see RecommendationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecommendationStatus
         */
        Builder recommendationStatus(RecommendationStatus recommendationStatus);

        /**
         * <p>
         * The number of LUNs (logical unit numbers) in the cluster.
         * </p>
         * 
         * @param lunCount
         *        The number of LUNs (logical unit numbers) in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lunCount(Long lunCount);

        /**
         * <p>
         * The amount of space in the cluster that's in cloud storage (for example, if you're using data tiering).
         * </p>
         * 
         * @param clusterCloudStorageUsed
         *        The amount of space in the cluster that's in cloud storage (for example, if you're using data
         *        tiering).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterCloudStorageUsed(Long clusterCloudStorageUsed);
    }

    static final class BuilderImpl implements Builder {
        private Long cifsShareCount;

        private Long nfsExportedVolumes;

        private String resourceId;

        private String clusterName;

        private MaxP95Performance maxP95Performance;

        private Long clusterBlockStorageSize;

        private Long clusterBlockStorageUsed;

        private Long clusterBlockStorageLogicalUsed;

        private List<Recommendation> recommendations = DefaultSdkAutoConstructList.getInstance();

        private String recommendationStatus;

        private Long lunCount;

        private Long clusterCloudStorageUsed;

        private BuilderImpl() {
        }

        private BuilderImpl(NetAppONTAPCluster model) {
            cifsShareCount(model.cifsShareCount);
            nfsExportedVolumes(model.nfsExportedVolumes);
            resourceId(model.resourceId);
            clusterName(model.clusterName);
            maxP95Performance(model.maxP95Performance);
            clusterBlockStorageSize(model.clusterBlockStorageSize);
            clusterBlockStorageUsed(model.clusterBlockStorageUsed);
            clusterBlockStorageLogicalUsed(model.clusterBlockStorageLogicalUsed);
            recommendations(model.recommendations);
            recommendationStatus(model.recommendationStatus);
            lunCount(model.lunCount);
            clusterCloudStorageUsed(model.clusterCloudStorageUsed);
        }

        public final Long getCifsShareCount() {
            return cifsShareCount;
        }

        public final void setCifsShareCount(Long cifsShareCount) {
            this.cifsShareCount = cifsShareCount;
        }

        @Override
        public final Builder cifsShareCount(Long cifsShareCount) {
            this.cifsShareCount = cifsShareCount;
            return this;
        }

        public final Long getNfsExportedVolumes() {
            return nfsExportedVolumes;
        }

        public final void setNfsExportedVolumes(Long nfsExportedVolumes) {
            this.nfsExportedVolumes = nfsExportedVolumes;
        }

        @Override
        public final Builder nfsExportedVolumes(Long nfsExportedVolumes) {
            this.nfsExportedVolumes = nfsExportedVolumes;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getClusterName() {
            return clusterName;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final MaxP95Performance.Builder getMaxP95Performance() {
            return maxP95Performance != null ? maxP95Performance.toBuilder() : null;
        }

        public final void setMaxP95Performance(MaxP95Performance.BuilderImpl maxP95Performance) {
            this.maxP95Performance = maxP95Performance != null ? maxP95Performance.build() : null;
        }

        @Override
        public final Builder maxP95Performance(MaxP95Performance maxP95Performance) {
            this.maxP95Performance = maxP95Performance;
            return this;
        }

        public final Long getClusterBlockStorageSize() {
            return clusterBlockStorageSize;
        }

        public final void setClusterBlockStorageSize(Long clusterBlockStorageSize) {
            this.clusterBlockStorageSize = clusterBlockStorageSize;
        }

        @Override
        public final Builder clusterBlockStorageSize(Long clusterBlockStorageSize) {
            this.clusterBlockStorageSize = clusterBlockStorageSize;
            return this;
        }

        public final Long getClusterBlockStorageUsed() {
            return clusterBlockStorageUsed;
        }

        public final void setClusterBlockStorageUsed(Long clusterBlockStorageUsed) {
            this.clusterBlockStorageUsed = clusterBlockStorageUsed;
        }

        @Override
        public final Builder clusterBlockStorageUsed(Long clusterBlockStorageUsed) {
            this.clusterBlockStorageUsed = clusterBlockStorageUsed;
            return this;
        }

        public final Long getClusterBlockStorageLogicalUsed() {
            return clusterBlockStorageLogicalUsed;
        }

        public final void setClusterBlockStorageLogicalUsed(Long clusterBlockStorageLogicalUsed) {
            this.clusterBlockStorageLogicalUsed = clusterBlockStorageLogicalUsed;
        }

        @Override
        public final Builder clusterBlockStorageLogicalUsed(Long clusterBlockStorageLogicalUsed) {
            this.clusterBlockStorageLogicalUsed = clusterBlockStorageLogicalUsed;
            return this;
        }

        public final List<Recommendation.Builder> getRecommendations() {
            List<Recommendation.Builder> result = RecommendationsCopier.copyToBuilder(this.recommendations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendations(Collection<Recommendation.BuilderImpl> recommendations) {
            this.recommendations = RecommendationsCopier.copyFromBuilder(recommendations);
        }

        @Override
        public final Builder recommendations(Collection<Recommendation> recommendations) {
            this.recommendations = RecommendationsCopier.copy(recommendations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Recommendation... recommendations) {
            recommendations(Arrays.asList(recommendations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendations(Consumer<Recommendation.Builder>... recommendations) {
            recommendations(Stream.of(recommendations).map(c -> Recommendation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRecommendationStatus() {
            return recommendationStatus;
        }

        public final void setRecommendationStatus(String recommendationStatus) {
            this.recommendationStatus = recommendationStatus;
        }

        @Override
        public final Builder recommendationStatus(String recommendationStatus) {
            this.recommendationStatus = recommendationStatus;
            return this;
        }

        @Override
        public final Builder recommendationStatus(RecommendationStatus recommendationStatus) {
            this.recommendationStatus(recommendationStatus == null ? null : recommendationStatus.toString());
            return this;
        }

        public final Long getLunCount() {
            return lunCount;
        }

        public final void setLunCount(Long lunCount) {
            this.lunCount = lunCount;
        }

        @Override
        public final Builder lunCount(Long lunCount) {
            this.lunCount = lunCount;
            return this;
        }

        public final Long getClusterCloudStorageUsed() {
            return clusterCloudStorageUsed;
        }

        public final void setClusterCloudStorageUsed(Long clusterCloudStorageUsed) {
            this.clusterCloudStorageUsed = clusterCloudStorageUsed;
        }

        @Override
        public final Builder clusterCloudStorageUsed(Long clusterCloudStorageUsed) {
            this.clusterCloudStorageUsed = clusterCloudStorageUsed;
            return this;
        }

        @Override
        public NetAppONTAPCluster build() {
            return new NetAppONTAPCluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
