/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information, including performance data and capacity usage, provided by DataSync Discovery about a resource in your
 * on-premises storage system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceMetrics implements SdkPojo, Serializable, ToCopyableBuilder<ResourceMetrics.Builder, ResourceMetrics> {
    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(ResourceMetrics::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<P95Metrics> P95_METRICS_FIELD = SdkField.<P95Metrics> builder(MarshallingType.SDK_POJO)
            .memberName("P95Metrics").getter(getter(ResourceMetrics::p95Metrics)).setter(setter(Builder::p95Metrics))
            .constructor(P95Metrics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("P95Metrics").build()).build();

    private static final SdkField<Capacity> CAPACITY_FIELD = SdkField.<Capacity> builder(MarshallingType.SDK_POJO)
            .memberName("Capacity").getter(getter(ResourceMetrics::capacity)).setter(setter(Builder::capacity))
            .constructor(Capacity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capacity").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(ResourceMetrics::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ResourceMetrics::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMESTAMP_FIELD,
            P95_METRICS_FIELD, CAPACITY_FIELD, RESOURCE_ID_FIELD, RESOURCE_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant timestamp;

    private final P95Metrics p95Metrics;

    private final Capacity capacity;

    private final String resourceId;

    private final String resourceType;

    private ResourceMetrics(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.p95Metrics = builder.p95Metrics;
        this.capacity = builder.capacity;
        this.resourceId = builder.resourceId;
        this.resourceType = builder.resourceType;
    }

    /**
     * <p>
     * The time when DataSync Discovery collected this information from the resource.
     * </p>
     * 
     * @return The time when DataSync Discovery collected this information from the resource.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The types of performance data that DataSync Discovery collects about the on-premises storage system resource.
     * </p>
     * 
     * @return The types of performance data that DataSync Discovery collects about the on-premises storage system
     *         resource.
     */
    public final P95Metrics p95Metrics() {
        return p95Metrics;
    }

    /**
     * <p>
     * The storage capacity of the on-premises storage system resource.
     * </p>
     * 
     * @return The storage capacity of the on-premises storage system resource.
     */
    public final Capacity capacity() {
        return capacity;
    }

    /**
     * <p>
     * The universally unique identifier (UUID) of the on-premises storage system resource.
     * </p>
     * 
     * @return The universally unique identifier (UUID) of the on-premises storage system resource.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The type of on-premises storage system resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link DiscoveryResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of on-premises storage system resource.
     * @see DiscoveryResourceType
     */
    public final DiscoveryResourceType resourceType() {
        return DiscoveryResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of on-premises storage system resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link DiscoveryResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of on-premises storage system resource.
     * @see DiscoveryResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(p95Metrics());
        hashCode = 31 * hashCode + Objects.hashCode(capacity());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceMetrics)) {
            return false;
        }
        ResourceMetrics other = (ResourceMetrics) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(p95Metrics(), other.p95Metrics())
                && Objects.equals(capacity(), other.capacity()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceMetrics").add("Timestamp", timestamp()).add("P95Metrics", p95Metrics())
                .add("Capacity", capacity()).add("ResourceId", resourceId()).add("ResourceType", resourceTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "P95Metrics":
            return Optional.ofNullable(clazz.cast(p95Metrics()));
        case "Capacity":
            return Optional.ofNullable(clazz.cast(capacity()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Timestamp", TIMESTAMP_FIELD);
        map.put("P95Metrics", P95_METRICS_FIELD);
        map.put("Capacity", CAPACITY_FIELD);
        map.put("ResourceId", RESOURCE_ID_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceMetrics, T> g) {
        return obj -> g.apply((ResourceMetrics) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceMetrics> {
        /**
         * <p>
         * The time when DataSync Discovery collected this information from the resource.
         * </p>
         * 
         * @param timestamp
         *        The time when DataSync Discovery collected this information from the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The types of performance data that DataSync Discovery collects about the on-premises storage system resource.
         * </p>
         * 
         * @param p95Metrics
         *        The types of performance data that DataSync Discovery collects about the on-premises storage system
         *        resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder p95Metrics(P95Metrics p95Metrics);

        /**
         * <p>
         * The types of performance data that DataSync Discovery collects about the on-premises storage system resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link P95Metrics.Builder} avoiding the need to
         * create one manually via {@link P95Metrics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link P95Metrics.Builder#build()} is called immediately and its result
         * is passed to {@link #p95Metrics(P95Metrics)}.
         * 
         * @param p95Metrics
         *        a consumer that will call methods on {@link P95Metrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #p95Metrics(P95Metrics)
         */
        default Builder p95Metrics(Consumer<P95Metrics.Builder> p95Metrics) {
            return p95Metrics(P95Metrics.builder().applyMutation(p95Metrics).build());
        }

        /**
         * <p>
         * The storage capacity of the on-premises storage system resource.
         * </p>
         * 
         * @param capacity
         *        The storage capacity of the on-premises storage system resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacity(Capacity capacity);

        /**
         * <p>
         * The storage capacity of the on-premises storage system resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link Capacity.Builder} avoiding the need to
         * create one manually via {@link Capacity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Capacity.Builder#build()} is called immediately and its result is
         * passed to {@link #capacity(Capacity)}.
         * 
         * @param capacity
         *        a consumer that will call methods on {@link Capacity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capacity(Capacity)
         */
        default Builder capacity(Consumer<Capacity.Builder> capacity) {
            return capacity(Capacity.builder().applyMutation(capacity).build());
        }

        /**
         * <p>
         * The universally unique identifier (UUID) of the on-premises storage system resource.
         * </p>
         * 
         * @param resourceId
         *        The universally unique identifier (UUID) of the on-premises storage system resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The type of on-premises storage system resource.
         * </p>
         * 
         * @param resourceType
         *        The type of on-premises storage system resource.
         * @see DiscoveryResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiscoveryResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of on-premises storage system resource.
         * </p>
         * 
         * @param resourceType
         *        The type of on-premises storage system resource.
         * @see DiscoveryResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiscoveryResourceType
         */
        Builder resourceType(DiscoveryResourceType resourceType);
    }

    static final class BuilderImpl implements Builder {
        private Instant timestamp;

        private P95Metrics p95Metrics;

        private Capacity capacity;

        private String resourceId;

        private String resourceType;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceMetrics model) {
            timestamp(model.timestamp);
            p95Metrics(model.p95Metrics);
            capacity(model.capacity);
            resourceId(model.resourceId);
            resourceType(model.resourceType);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final P95Metrics.Builder getP95Metrics() {
            return p95Metrics != null ? p95Metrics.toBuilder() : null;
        }

        public final void setP95Metrics(P95Metrics.BuilderImpl p95Metrics) {
            this.p95Metrics = p95Metrics != null ? p95Metrics.build() : null;
        }

        @Override
        public final Builder p95Metrics(P95Metrics p95Metrics) {
            this.p95Metrics = p95Metrics;
            return this;
        }

        public final Capacity.Builder getCapacity() {
            return capacity != null ? capacity.toBuilder() : null;
        }

        public final void setCapacity(Capacity.BuilderImpl capacity) {
            this.capacity = capacity != null ? capacity.build() : null;
        }

        @Override
        public final Builder capacity(Capacity capacity) {
            this.capacity = capacity;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(DiscoveryResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        @Override
        public ResourceMetrics build() {
            return new ResourceMetrics(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
