/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateLocationAzureBlobRequest extends DataSyncRequest implements
        ToCopyableBuilder<UpdateLocationAzureBlobRequest.Builder, UpdateLocationAzureBlobRequest> {
    private static final SdkField<String> LOCATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocationArn").getter(getter(UpdateLocationAzureBlobRequest::locationArn))
            .setter(setter(Builder::locationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationArn").build()).build();

    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(UpdateLocationAzureBlobRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> AUTHENTICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthenticationType").getter(getter(UpdateLocationAzureBlobRequest::authenticationTypeAsString))
            .setter(setter(Builder::authenticationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationType").build())
            .build();

    private static final SdkField<AzureBlobSasConfiguration> SAS_CONFIGURATION_FIELD = SdkField
            .<AzureBlobSasConfiguration> builder(MarshallingType.SDK_POJO).memberName("SasConfiguration")
            .getter(getter(UpdateLocationAzureBlobRequest::sasConfiguration)).setter(setter(Builder::sasConfiguration))
            .constructor(AzureBlobSasConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SasConfiguration").build()).build();

    private static final SdkField<String> BLOB_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BlobType").getter(getter(UpdateLocationAzureBlobRequest::blobTypeAsString))
            .setter(setter(Builder::blobType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlobType").build()).build();

    private static final SdkField<String> ACCESS_TIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessTier").getter(getter(UpdateLocationAzureBlobRequest::accessTierAsString))
            .setter(setter(Builder::accessTier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessTier").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(UpdateLocationAzureBlobRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_ARN_FIELD,
            SUBDIRECTORY_FIELD, AUTHENTICATION_TYPE_FIELD, SAS_CONFIGURATION_FIELD, BLOB_TYPE_FIELD, ACCESS_TIER_FIELD,
            AGENT_ARNS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String locationArn;

    private final String subdirectory;

    private final String authenticationType;

    private final AzureBlobSasConfiguration sasConfiguration;

    private final String blobType;

    private final String accessTier;

    private final List<String> agentArns;

    private UpdateLocationAzureBlobRequest(BuilderImpl builder) {
        super(builder);
        this.locationArn = builder.locationArn;
        this.subdirectory = builder.subdirectory;
        this.authenticationType = builder.authenticationType;
        this.sasConfiguration = builder.sasConfiguration;
        this.blobType = builder.blobType;
        this.accessTier = builder.accessTier;
        this.agentArns = builder.agentArns;
    }

    /**
     * <p>
     * Specifies the ARN of the Azure Blob Storage transfer location that you're updating.
     * </p>
     * 
     * @return Specifies the ARN of the Azure Blob Storage transfer location that you're updating.
     */
    public final String locationArn() {
        return locationArn;
    }

    /**
     * <p>
     * Specifies path segments if you want to limit your transfer to a virtual directory in your container (for example,
     * <code>/my/images</code>).
     * </p>
     * 
     * @return Specifies path segments if you want to limit your transfer to a virtual directory in your container (for
     *         example, <code>/my/images</code>).
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
     * storage using a shared access signature (SAS).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link AzureBlobAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access
     *         blob storage using a shared access signature (SAS).
     * @see AzureBlobAuthenticationType
     */
    public final AzureBlobAuthenticationType authenticationType() {
        return AzureBlobAuthenticationType.fromValue(authenticationType);
    }

    /**
     * <p>
     * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
     * storage using a shared access signature (SAS).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link AzureBlobAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access
     *         blob storage using a shared access signature (SAS).
     * @see AzureBlobAuthenticationType
     */
    public final String authenticationTypeAsString() {
        return authenticationType;
    }

    /**
     * <p>
     * Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
     * </p>
     * 
     * @return Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
     */
    public final AzureBlobSasConfiguration sasConfiguration() {
        return sasConfiguration;
    }

    /**
     * <p>
     * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
     * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
     * information on blob types, see the <a href=
     * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     * >Azure Blob Storage documentation</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blobType} will
     * return {@link AzureBlobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blobTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of blob that you want your objects or files to be when transferring them into Azure
     *         Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For
     *         more information on blob types, see the <a href=
     *         "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     *         >Azure Blob Storage documentation</a>.
     * @see AzureBlobType
     */
    public final AzureBlobType blobType() {
        return AzureBlobType.fromValue(blobType);
    }

    /**
     * <p>
     * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
     * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
     * information on blob types, see the <a href=
     * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     * >Azure Blob Storage documentation</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #blobType} will
     * return {@link AzureBlobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #blobTypeAsString}.
     * </p>
     * 
     * @return Specifies the type of blob that you want your objects or files to be when transferring them into Azure
     *         Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For
     *         more information on blob types, see the <a href=
     *         "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
     *         >Azure Blob Storage documentation</a>.
     * @see AzureBlobType
     */
    public final String blobTypeAsString() {
        return blobType;
    }

    /**
     * <p>
     * Specifies the access tier that you want your objects or files transferred into. This only applies when using the
     * location as a transfer destination. For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     * >Access tiers</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessTier} will
     * return {@link AzureAccessTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTierAsString}.
     * </p>
     * 
     * @return Specifies the access tier that you want your objects or files transferred into. This only applies when
     *         using the location as a transfer destination. For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     *         >Access tiers</a>.
     * @see AzureAccessTier
     */
    public final AzureAccessTier accessTier() {
        return AzureAccessTier.fromValue(accessTier);
    }

    /**
     * <p>
     * Specifies the access tier that you want your objects or files transferred into. This only applies when using the
     * location as a transfer destination. For more information, see <a href=
     * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     * >Access tiers</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessTier} will
     * return {@link AzureAccessTier#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTierAsString}.
     * </p>
     * 
     * @return Specifies the access tier that you want your objects or files transferred into. This only applies when
     *         using the location as a transfer destination. For more information, see <a href=
     *         "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
     *         >Access tiers</a>.
     * @see AzureAccessTier
     */
    public final String accessTierAsString() {
        return accessTier;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob Storage
     * container.
     * </p>
     * <p>
     * You can specify more than one agent. For more information, see <a
     * href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents for your
     * transfer</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob
     *         Storage container.</p>
     *         <p>
     *         You can specify more than one agent. For more information, see <a
     *         href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents
     *         for your transfer</a>.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(locationArn());
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sasConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(blobTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(accessTierAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateLocationAzureBlobRequest)) {
            return false;
        }
        UpdateLocationAzureBlobRequest other = (UpdateLocationAzureBlobRequest) obj;
        return Objects.equals(locationArn(), other.locationArn()) && Objects.equals(subdirectory(), other.subdirectory())
                && Objects.equals(authenticationTypeAsString(), other.authenticationTypeAsString())
                && Objects.equals(sasConfiguration(), other.sasConfiguration())
                && Objects.equals(blobTypeAsString(), other.blobTypeAsString())
                && Objects.equals(accessTierAsString(), other.accessTierAsString()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateLocationAzureBlobRequest").add("LocationArn", locationArn())
                .add("Subdirectory", subdirectory()).add("AuthenticationType", authenticationTypeAsString())
                .add("SasConfiguration", sasConfiguration()).add("BlobType", blobTypeAsString())
                .add("AccessTier", accessTierAsString()).add("AgentArns", hasAgentArns() ? agentArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocationArn":
            return Optional.ofNullable(clazz.cast(locationArn()));
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "AuthenticationType":
            return Optional.ofNullable(clazz.cast(authenticationTypeAsString()));
        case "SasConfiguration":
            return Optional.ofNullable(clazz.cast(sasConfiguration()));
        case "BlobType":
            return Optional.ofNullable(clazz.cast(blobTypeAsString()));
        case "AccessTier":
            return Optional.ofNullable(clazz.cast(accessTierAsString()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LocationArn", LOCATION_ARN_FIELD);
        map.put("Subdirectory", SUBDIRECTORY_FIELD);
        map.put("AuthenticationType", AUTHENTICATION_TYPE_FIELD);
        map.put("SasConfiguration", SAS_CONFIGURATION_FIELD);
        map.put("BlobType", BLOB_TYPE_FIELD);
        map.put("AccessTier", ACCESS_TIER_FIELD);
        map.put("AgentArns", AGENT_ARNS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UpdateLocationAzureBlobRequest, T> g) {
        return obj -> g.apply((UpdateLocationAzureBlobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateLocationAzureBlobRequest> {
        /**
         * <p>
         * Specifies the ARN of the Azure Blob Storage transfer location that you're updating.
         * </p>
         * 
         * @param locationArn
         *        Specifies the ARN of the Azure Blob Storage transfer location that you're updating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationArn(String locationArn);

        /**
         * <p>
         * Specifies path segments if you want to limit your transfer to a virtual directory in your container (for
         * example, <code>/my/images</code>).
         * </p>
         * 
         * @param subdirectory
         *        Specifies path segments if you want to limit your transfer to a virtual directory in your container
         *        (for example, <code>/my/images</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
         * storage using a shared access signature (SAS).
         * </p>
         * 
         * @param authenticationType
         *        Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can
         *        access blob storage using a shared access signature (SAS).
         * @see AzureBlobAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobAuthenticationType
         */
        Builder authenticationType(String authenticationType);

        /**
         * <p>
         * Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can access blob
         * storage using a shared access signature (SAS).
         * </p>
         * 
         * @param authenticationType
         *        Specifies the authentication method DataSync uses to access your Azure Blob Storage. DataSync can
         *        access blob storage using a shared access signature (SAS).
         * @see AzureBlobAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobAuthenticationType
         */
        Builder authenticationType(AzureBlobAuthenticationType authenticationType);

        /**
         * <p>
         * Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
         * </p>
         * 
         * @param sasConfiguration
         *        Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sasConfiguration(AzureBlobSasConfiguration sasConfiguration);

        /**
         * <p>
         * Specifies the SAS configuration that allows DataSync to access your Azure Blob Storage.
         * </p>
         * This is a convenience method that creates an instance of the {@link AzureBlobSasConfiguration.Builder}
         * avoiding the need to create one manually via {@link AzureBlobSasConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AzureBlobSasConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #sasConfiguration(AzureBlobSasConfiguration)}.
         * 
         * @param sasConfiguration
         *        a consumer that will call methods on {@link AzureBlobSasConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sasConfiguration(AzureBlobSasConfiguration)
         */
        default Builder sasConfiguration(Consumer<AzureBlobSasConfiguration.Builder> sasConfiguration) {
            return sasConfiguration(AzureBlobSasConfiguration.builder().applyMutation(sasConfiguration).build());
        }

        /**
         * <p>
         * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
         * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
         * information on blob types, see the <a href=
         * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         * >Azure Blob Storage documentation</a>.
         * </p>
         * 
         * @param blobType
         *        Specifies the type of blob that you want your objects or files to be when transferring them into Azure
         *        Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs.
         *        For more information on blob types, see the <a href=
         *        "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         *        >Azure Blob Storage documentation</a>.
         * @see AzureBlobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobType
         */
        Builder blobType(String blobType);

        /**
         * <p>
         * Specifies the type of blob that you want your objects or files to be when transferring them into Azure Blob
         * Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs. For more
         * information on blob types, see the <a href=
         * "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         * >Azure Blob Storage documentation</a>.
         * </p>
         * 
         * @param blobType
         *        Specifies the type of blob that you want your objects or files to be when transferring them into Azure
         *        Blob Storage. Currently, DataSync only supports moving data into Azure Blob Storage as block blobs.
         *        For more information on blob types, see the <a href=
         *        "https://learn.microsoft.com/en-us/rest/api/storageservices/understanding-block-blobs--append-blobs--and-page-blobs"
         *        >Azure Blob Storage documentation</a>.
         * @see AzureBlobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureBlobType
         */
        Builder blobType(AzureBlobType blobType);

        /**
         * <p>
         * Specifies the access tier that you want your objects or files transferred into. This only applies when using
         * the location as a transfer destination. For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         * >Access tiers</a>.
         * </p>
         * 
         * @param accessTier
         *        Specifies the access tier that you want your objects or files transferred into. This only applies when
         *        using the location as a transfer destination. For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         *        >Access tiers</a>.
         * @see AzureAccessTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureAccessTier
         */
        Builder accessTier(String accessTier);

        /**
         * <p>
         * Specifies the access tier that you want your objects or files transferred into. This only applies when using
         * the location as a transfer destination. For more information, see <a href=
         * "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         * >Access tiers</a>.
         * </p>
         * 
         * @param accessTier
         *        Specifies the access tier that you want your objects or files transferred into. This only applies when
         *        using the location as a transfer destination. For more information, see <a href=
         *        "https://docs.aws.amazon.com/datasync/latest/userguide/creating-azure-blob-location.html#azure-blob-access-tiers"
         *        >Access tiers</a>.
         * @see AzureAccessTier
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AzureAccessTier
         */
        Builder accessTier(AzureAccessTier accessTier);

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob Storage
         * container.
         * </p>
         * <p>
         * You can specify more than one agent. For more information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents for
         * your transfer</a>.
         * </p>
         * 
         * @param agentArns
         *        Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob
         *        Storage container.</p>
         *        <p>
         *        You can specify more than one agent. For more information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple
         *        agents for your transfer</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob Storage
         * container.
         * </p>
         * <p>
         * You can specify more than one agent. For more information, see <a
         * href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple agents for
         * your transfer</a>.
         * </p>
         * 
         * @param agentArns
         *        Specifies the Amazon Resource Name (ARN) of the DataSync agent that can connect with your Azure Blob
         *        Storage container.</p>
         *        <p>
         *        You can specify more than one agent. For more information, see <a
         *        href="https://docs.aws.amazon.com/datasync/latest/userguide/multiple-agents.html">Using multiple
         *        agents for your transfer</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String locationArn;

        private String subdirectory;

        private String authenticationType;

        private AzureBlobSasConfiguration sasConfiguration;

        private String blobType;

        private String accessTier;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateLocationAzureBlobRequest model) {
            super(model);
            locationArn(model.locationArn);
            subdirectory(model.subdirectory);
            authenticationType(model.authenticationType);
            sasConfiguration(model.sasConfiguration);
            blobType(model.blobType);
            accessTier(model.accessTier);
            agentArns(model.agentArns);
        }

        public final String getLocationArn() {
            return locationArn;
        }

        public final void setLocationArn(String locationArn) {
            this.locationArn = locationArn;
        }

        @Override
        public final Builder locationArn(String locationArn) {
            this.locationArn = locationArn;
            return this;
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final String getAuthenticationType() {
            return authenticationType;
        }

        public final void setAuthenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
        }

        @Override
        public final Builder authenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        @Override
        public final Builder authenticationType(AzureBlobAuthenticationType authenticationType) {
            this.authenticationType(authenticationType == null ? null : authenticationType.toString());
            return this;
        }

        public final AzureBlobSasConfiguration.Builder getSasConfiguration() {
            return sasConfiguration != null ? sasConfiguration.toBuilder() : null;
        }

        public final void setSasConfiguration(AzureBlobSasConfiguration.BuilderImpl sasConfiguration) {
            this.sasConfiguration = sasConfiguration != null ? sasConfiguration.build() : null;
        }

        @Override
        public final Builder sasConfiguration(AzureBlobSasConfiguration sasConfiguration) {
            this.sasConfiguration = sasConfiguration;
            return this;
        }

        public final String getBlobType() {
            return blobType;
        }

        public final void setBlobType(String blobType) {
            this.blobType = blobType;
        }

        @Override
        public final Builder blobType(String blobType) {
            this.blobType = blobType;
            return this;
        }

        @Override
        public final Builder blobType(AzureBlobType blobType) {
            this.blobType(blobType == null ? null : blobType.toString());
            return this;
        }

        public final String getAccessTier() {
            return accessTier;
        }

        public final void setAccessTier(String accessTier) {
            this.accessTier = accessTier;
        }

        @Override
        public final Builder accessTier(String accessTier) {
            this.accessTier = accessTier;
            return this;
        }

        @Override
        public final Builder accessTier(AzureAccessTier accessTier) {
            this.accessTier(accessTier == null ? null : accessTier.toString());
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateLocationAzureBlobRequest build() {
            return new UpdateLocationAzureBlobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
