/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateLocationHdfsRequest extends DataSyncRequest implements
        ToCopyableBuilder<UpdateLocationHdfsRequest.Builder, UpdateLocationHdfsRequest> {
    private static final SdkField<String> LOCATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocationArn").getter(getter(UpdateLocationHdfsRequest::locationArn))
            .setter(setter(Builder::locationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationArn").build()).build();

    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(UpdateLocationHdfsRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<List<HdfsNameNode>> NAME_NODES_FIELD = SdkField
            .<List<HdfsNameNode>> builder(MarshallingType.LIST)
            .memberName("NameNodes")
            .getter(getter(UpdateLocationHdfsRequest::nameNodes))
            .setter(setter(Builder::nameNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NameNodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HdfsNameNode> builder(MarshallingType.SDK_POJO)
                                            .constructor(HdfsNameNode::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> BLOCK_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BlockSize").getter(getter(UpdateLocationHdfsRequest::blockSize)).setter(setter(Builder::blockSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockSize").build()).build();

    private static final SdkField<Integer> REPLICATION_FACTOR_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ReplicationFactor").getter(getter(UpdateLocationHdfsRequest::replicationFactor))
            .setter(setter(Builder::replicationFactor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationFactor").build()).build();

    private static final SdkField<String> KMS_KEY_PROVIDER_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyProviderUri").getter(getter(UpdateLocationHdfsRequest::kmsKeyProviderUri))
            .setter(setter(Builder::kmsKeyProviderUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyProviderUri").build()).build();

    private static final SdkField<QopConfiguration> QOP_CONFIGURATION_FIELD = SdkField
            .<QopConfiguration> builder(MarshallingType.SDK_POJO).memberName("QopConfiguration")
            .getter(getter(UpdateLocationHdfsRequest::qopConfiguration)).setter(setter(Builder::qopConfiguration))
            .constructor(QopConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QopConfiguration").build()).build();

    private static final SdkField<String> AUTHENTICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthenticationType").getter(getter(UpdateLocationHdfsRequest::authenticationTypeAsString))
            .setter(setter(Builder::authenticationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationType").build())
            .build();

    private static final SdkField<String> SIMPLE_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SimpleUser").getter(getter(UpdateLocationHdfsRequest::simpleUser)).setter(setter(Builder::simpleUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SimpleUser").build()).build();

    private static final SdkField<String> KERBEROS_PRINCIPAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KerberosPrincipal").getter(getter(UpdateLocationHdfsRequest::kerberosPrincipal))
            .setter(setter(Builder::kerberosPrincipal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KerberosPrincipal").build()).build();

    private static final SdkField<SdkBytes> KERBEROS_KEYTAB_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("KerberosKeytab").getter(getter(UpdateLocationHdfsRequest::kerberosKeytab))
            .setter(setter(Builder::kerberosKeytab))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KerberosKeytab").build()).build();

    private static final SdkField<SdkBytes> KERBEROS_KRB5_CONF_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("KerberosKrb5Conf").getter(getter(UpdateLocationHdfsRequest::kerberosKrb5Conf))
            .setter(setter(Builder::kerberosKrb5Conf))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KerberosKrb5Conf").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(UpdateLocationHdfsRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_ARN_FIELD,
            SUBDIRECTORY_FIELD, NAME_NODES_FIELD, BLOCK_SIZE_FIELD, REPLICATION_FACTOR_FIELD, KMS_KEY_PROVIDER_URI_FIELD,
            QOP_CONFIGURATION_FIELD, AUTHENTICATION_TYPE_FIELD, SIMPLE_USER_FIELD, KERBEROS_PRINCIPAL_FIELD,
            KERBEROS_KEYTAB_FIELD, KERBEROS_KRB5_CONF_FIELD, AGENT_ARNS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String locationArn;

    private final String subdirectory;

    private final List<HdfsNameNode> nameNodes;

    private final Integer blockSize;

    private final Integer replicationFactor;

    private final String kmsKeyProviderUri;

    private final QopConfiguration qopConfiguration;

    private final String authenticationType;

    private final String simpleUser;

    private final String kerberosPrincipal;

    private final SdkBytes kerberosKeytab;

    private final SdkBytes kerberosKrb5Conf;

    private final List<String> agentArns;

    private UpdateLocationHdfsRequest(BuilderImpl builder) {
        super(builder);
        this.locationArn = builder.locationArn;
        this.subdirectory = builder.subdirectory;
        this.nameNodes = builder.nameNodes;
        this.blockSize = builder.blockSize;
        this.replicationFactor = builder.replicationFactor;
        this.kmsKeyProviderUri = builder.kmsKeyProviderUri;
        this.qopConfiguration = builder.qopConfiguration;
        this.authenticationType = builder.authenticationType;
        this.simpleUser = builder.simpleUser;
        this.kerberosPrincipal = builder.kerberosPrincipal;
        this.kerberosKeytab = builder.kerberosKeytab;
        this.kerberosKrb5Conf = builder.kerberosKrb5Conf;
        this.agentArns = builder.agentArns;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the source HDFS cluster location.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the source HDFS cluster location.
     */
    public final String locationArn() {
        return locationArn;
    }

    /**
     * <p>
     * A subdirectory in the HDFS cluster. This subdirectory is used to read data from or write data to the HDFS
     * cluster.
     * </p>
     * 
     * @return A subdirectory in the HDFS cluster. This subdirectory is used to read data from or write data to the HDFS
     *         cluster.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * For responses, this returns true if the service returned a value for the NameNodes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNameNodes() {
        return nameNodes != null && !(nameNodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening, closing, and
     * renaming files and directories. The NameNode contains the information to map blocks of data to the DataNodes. You
     * can use only one NameNode.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNameNodes} method.
     * </p>
     * 
     * @return The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening, closing,
     *         and renaming files and directories. The NameNode contains the information to map blocks of data to the
     *         DataNodes. You can use only one NameNode.
     */
    public final List<HdfsNameNode> nameNodes() {
        return nameNodes;
    }

    /**
     * <p>
     * The size of the data blocks to write into the HDFS cluster.
     * </p>
     * 
     * @return The size of the data blocks to write into the HDFS cluster.
     */
    public final Integer blockSize() {
        return blockSize;
    }

    /**
     * <p>
     * The number of DataNodes to replicate the data to when writing to the HDFS cluster.
     * </p>
     * 
     * @return The number of DataNodes to replicate the data to when writing to the HDFS cluster.
     */
    public final Integer replicationFactor() {
        return replicationFactor;
    }

    /**
     * <p>
     * The URI of the HDFS cluster's Key Management Server (KMS).
     * </p>
     * 
     * @return The URI of the HDFS cluster's Key Management Server (KMS).
     */
    public final String kmsKeyProviderUri() {
        return kmsKeyProviderUri;
    }

    /**
     * <p>
     * The Quality of Protection (QOP) configuration specifies the Remote Procedure Call (RPC) and data transfer privacy
     * settings configured on the Hadoop Distributed File System (HDFS) cluster.
     * </p>
     * 
     * @return The Quality of Protection (QOP) configuration specifies the Remote Procedure Call (RPC) and data transfer
     *         privacy settings configured on the Hadoop Distributed File System (HDFS) cluster.
     */
    public final QopConfiguration qopConfiguration() {
        return qopConfiguration;
    }

    /**
     * <p>
     * The type of authentication used to determine the identity of the user.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link HdfsAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return The type of authentication used to determine the identity of the user.
     * @see HdfsAuthenticationType
     */
    public final HdfsAuthenticationType authenticationType() {
        return HdfsAuthenticationType.fromValue(authenticationType);
    }

    /**
     * <p>
     * The type of authentication used to determine the identity of the user.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #authenticationType} will return {@link HdfsAuthenticationType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #authenticationTypeAsString}.
     * </p>
     * 
     * @return The type of authentication used to determine the identity of the user.
     * @see HdfsAuthenticationType
     */
    public final String authenticationTypeAsString() {
        return authenticationType;
    }

    /**
     * <p>
     * The user name used to identify the client on the host operating system.
     * </p>
     * 
     * @return The user name used to identify the client on the host operating system.
     */
    public final String simpleUser() {
        return simpleUser;
    }

    /**
     * <p>
     * The Kerberos principal with access to the files and folders on the HDFS cluster.
     * </p>
     * 
     * @return The Kerberos principal with access to the files and folders on the HDFS cluster.
     */
    public final String kerberosPrincipal() {
        return kerberosPrincipal;
    }

    /**
     * <p>
     * The Kerberos key table (keytab) that contains mappings between the defined Kerberos principal and the encrypted
     * keys. You can load the keytab from a file by providing the file's address. If you use the CLI, it performs base64
     * encoding for you. Otherwise, provide the base64-encoded text.
     * </p>
     * 
     * @return The Kerberos key table (keytab) that contains mappings between the defined Kerberos principal and the
     *         encrypted keys. You can load the keytab from a file by providing the file's address. If you use the CLI,
     *         it performs base64 encoding for you. Otherwise, provide the base64-encoded text.
     */
    public final SdkBytes kerberosKeytab() {
        return kerberosKeytab;
    }

    /**
     * <p>
     * The <code>krb5.conf</code> file that contains the Kerberos configuration information. You can load the
     * <code>krb5.conf</code> file by providing the file's address. If you're using the CLI, it performs the base64
     * encoding for you. Otherwise, provide the base64-encoded text.
     * </p>
     * 
     * @return The <code>krb5.conf</code> file that contains the Kerberos configuration information. You can load the
     *         <code>krb5.conf</code> file by providing the file's address. If you're using the CLI, it performs the
     *         base64 encoding for you. Otherwise, provide the base64-encoded text.
     */
    public final SdkBytes kerberosKrb5Conf() {
        return kerberosKrb5Conf;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of the DataSync agents that can connect to your HDFS cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of the DataSync agents that can connect to your HDFS cluster.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(locationArn());
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(hasNameNodes() ? nameNodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(blockSize());
        hashCode = 31 * hashCode + Objects.hashCode(replicationFactor());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyProviderUri());
        hashCode = 31 * hashCode + Objects.hashCode(qopConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(authenticationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(simpleUser());
        hashCode = 31 * hashCode + Objects.hashCode(kerberosPrincipal());
        hashCode = 31 * hashCode + Objects.hashCode(kerberosKeytab());
        hashCode = 31 * hashCode + Objects.hashCode(kerberosKrb5Conf());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateLocationHdfsRequest)) {
            return false;
        }
        UpdateLocationHdfsRequest other = (UpdateLocationHdfsRequest) obj;
        return Objects.equals(locationArn(), other.locationArn()) && Objects.equals(subdirectory(), other.subdirectory())
                && hasNameNodes() == other.hasNameNodes() && Objects.equals(nameNodes(), other.nameNodes())
                && Objects.equals(blockSize(), other.blockSize())
                && Objects.equals(replicationFactor(), other.replicationFactor())
                && Objects.equals(kmsKeyProviderUri(), other.kmsKeyProviderUri())
                && Objects.equals(qopConfiguration(), other.qopConfiguration())
                && Objects.equals(authenticationTypeAsString(), other.authenticationTypeAsString())
                && Objects.equals(simpleUser(), other.simpleUser())
                && Objects.equals(kerberosPrincipal(), other.kerberosPrincipal())
                && Objects.equals(kerberosKeytab(), other.kerberosKeytab())
                && Objects.equals(kerberosKrb5Conf(), other.kerberosKrb5Conf()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateLocationHdfsRequest").add("LocationArn", locationArn())
                .add("Subdirectory", subdirectory()).add("NameNodes", hasNameNodes() ? nameNodes() : null)
                .add("BlockSize", blockSize()).add("ReplicationFactor", replicationFactor())
                .add("KmsKeyProviderUri", kmsKeyProviderUri()).add("QopConfiguration", qopConfiguration())
                .add("AuthenticationType", authenticationTypeAsString()).add("SimpleUser", simpleUser())
                .add("KerberosPrincipal", kerberosPrincipal()).add("KerberosKeytab", kerberosKeytab())
                .add("KerberosKrb5Conf", kerberosKrb5Conf()).add("AgentArns", hasAgentArns() ? agentArns() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocationArn":
            return Optional.ofNullable(clazz.cast(locationArn()));
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "NameNodes":
            return Optional.ofNullable(clazz.cast(nameNodes()));
        case "BlockSize":
            return Optional.ofNullable(clazz.cast(blockSize()));
        case "ReplicationFactor":
            return Optional.ofNullable(clazz.cast(replicationFactor()));
        case "KmsKeyProviderUri":
            return Optional.ofNullable(clazz.cast(kmsKeyProviderUri()));
        case "QopConfiguration":
            return Optional.ofNullable(clazz.cast(qopConfiguration()));
        case "AuthenticationType":
            return Optional.ofNullable(clazz.cast(authenticationTypeAsString()));
        case "SimpleUser":
            return Optional.ofNullable(clazz.cast(simpleUser()));
        case "KerberosPrincipal":
            return Optional.ofNullable(clazz.cast(kerberosPrincipal()));
        case "KerberosKeytab":
            return Optional.ofNullable(clazz.cast(kerberosKeytab()));
        case "KerberosKrb5Conf":
            return Optional.ofNullable(clazz.cast(kerberosKrb5Conf()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LocationArn", LOCATION_ARN_FIELD);
        map.put("Subdirectory", SUBDIRECTORY_FIELD);
        map.put("NameNodes", NAME_NODES_FIELD);
        map.put("BlockSize", BLOCK_SIZE_FIELD);
        map.put("ReplicationFactor", REPLICATION_FACTOR_FIELD);
        map.put("KmsKeyProviderUri", KMS_KEY_PROVIDER_URI_FIELD);
        map.put("QopConfiguration", QOP_CONFIGURATION_FIELD);
        map.put("AuthenticationType", AUTHENTICATION_TYPE_FIELD);
        map.put("SimpleUser", SIMPLE_USER_FIELD);
        map.put("KerberosPrincipal", KERBEROS_PRINCIPAL_FIELD);
        map.put("KerberosKeytab", KERBEROS_KEYTAB_FIELD);
        map.put("KerberosKrb5Conf", KERBEROS_KRB5_CONF_FIELD);
        map.put("AgentArns", AGENT_ARNS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UpdateLocationHdfsRequest, T> g) {
        return obj -> g.apply((UpdateLocationHdfsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateLocationHdfsRequest> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the source HDFS cluster location.
         * </p>
         * 
         * @param locationArn
         *        The Amazon Resource Name (ARN) of the source HDFS cluster location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationArn(String locationArn);

        /**
         * <p>
         * A subdirectory in the HDFS cluster. This subdirectory is used to read data from or write data to the HDFS
         * cluster.
         * </p>
         * 
         * @param subdirectory
         *        A subdirectory in the HDFS cluster. This subdirectory is used to read data from or write data to the
         *        HDFS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening, closing, and
         * renaming files and directories. The NameNode contains the information to map blocks of data to the DataNodes.
         * You can use only one NameNode.
         * </p>
         * 
         * @param nameNodes
         *        The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening,
         *        closing, and renaming files and directories. The NameNode contains the information to map blocks of
         *        data to the DataNodes. You can use only one NameNode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameNodes(Collection<HdfsNameNode> nameNodes);

        /**
         * <p>
         * The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening, closing, and
         * renaming files and directories. The NameNode contains the information to map blocks of data to the DataNodes.
         * You can use only one NameNode.
         * </p>
         * 
         * @param nameNodes
         *        The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening,
         *        closing, and renaming files and directories. The NameNode contains the information to map blocks of
         *        data to the DataNodes. You can use only one NameNode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nameNodes(HdfsNameNode... nameNodes);

        /**
         * <p>
         * The NameNode that manages the HDFS namespace. The NameNode performs operations such as opening, closing, and
         * renaming files and directories. The NameNode contains the information to map blocks of data to the DataNodes.
         * You can use only one NameNode.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.HdfsNameNode.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.HdfsNameNode#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.HdfsNameNode.Builder#build()} is called immediately and
         * its result is passed to {@link #nameNodes(List<HdfsNameNode>)}.
         * 
         * @param nameNodes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.HdfsNameNode.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nameNodes(java.util.Collection<HdfsNameNode>)
         */
        Builder nameNodes(Consumer<HdfsNameNode.Builder>... nameNodes);

        /**
         * <p>
         * The size of the data blocks to write into the HDFS cluster.
         * </p>
         * 
         * @param blockSize
         *        The size of the data blocks to write into the HDFS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockSize(Integer blockSize);

        /**
         * <p>
         * The number of DataNodes to replicate the data to when writing to the HDFS cluster.
         * </p>
         * 
         * @param replicationFactor
         *        The number of DataNodes to replicate the data to when writing to the HDFS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationFactor(Integer replicationFactor);

        /**
         * <p>
         * The URI of the HDFS cluster's Key Management Server (KMS).
         * </p>
         * 
         * @param kmsKeyProviderUri
         *        The URI of the HDFS cluster's Key Management Server (KMS).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyProviderUri(String kmsKeyProviderUri);

        /**
         * <p>
         * The Quality of Protection (QOP) configuration specifies the Remote Procedure Call (RPC) and data transfer
         * privacy settings configured on the Hadoop Distributed File System (HDFS) cluster.
         * </p>
         * 
         * @param qopConfiguration
         *        The Quality of Protection (QOP) configuration specifies the Remote Procedure Call (RPC) and data
         *        transfer privacy settings configured on the Hadoop Distributed File System (HDFS) cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder qopConfiguration(QopConfiguration qopConfiguration);

        /**
         * <p>
         * The Quality of Protection (QOP) configuration specifies the Remote Procedure Call (RPC) and data transfer
         * privacy settings configured on the Hadoop Distributed File System (HDFS) cluster.
         * </p>
         * This is a convenience method that creates an instance of the {@link QopConfiguration.Builder} avoiding the
         * need to create one manually via {@link QopConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link QopConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #qopConfiguration(QopConfiguration)}.
         * 
         * @param qopConfiguration
         *        a consumer that will call methods on {@link QopConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #qopConfiguration(QopConfiguration)
         */
        default Builder qopConfiguration(Consumer<QopConfiguration.Builder> qopConfiguration) {
            return qopConfiguration(QopConfiguration.builder().applyMutation(qopConfiguration).build());
        }

        /**
         * <p>
         * The type of authentication used to determine the identity of the user.
         * </p>
         * 
         * @param authenticationType
         *        The type of authentication used to determine the identity of the user.
         * @see HdfsAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HdfsAuthenticationType
         */
        Builder authenticationType(String authenticationType);

        /**
         * <p>
         * The type of authentication used to determine the identity of the user.
         * </p>
         * 
         * @param authenticationType
         *        The type of authentication used to determine the identity of the user.
         * @see HdfsAuthenticationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HdfsAuthenticationType
         */
        Builder authenticationType(HdfsAuthenticationType authenticationType);

        /**
         * <p>
         * The user name used to identify the client on the host operating system.
         * </p>
         * 
         * @param simpleUser
         *        The user name used to identify the client on the host operating system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder simpleUser(String simpleUser);

        /**
         * <p>
         * The Kerberos principal with access to the files and folders on the HDFS cluster.
         * </p>
         * 
         * @param kerberosPrincipal
         *        The Kerberos principal with access to the files and folders on the HDFS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kerberosPrincipal(String kerberosPrincipal);

        /**
         * <p>
         * The Kerberos key table (keytab) that contains mappings between the defined Kerberos principal and the
         * encrypted keys. You can load the keytab from a file by providing the file's address. If you use the CLI, it
         * performs base64 encoding for you. Otherwise, provide the base64-encoded text.
         * </p>
         * 
         * @param kerberosKeytab
         *        The Kerberos key table (keytab) that contains mappings between the defined Kerberos principal and the
         *        encrypted keys. You can load the keytab from a file by providing the file's address. If you use the
         *        CLI, it performs base64 encoding for you. Otherwise, provide the base64-encoded text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kerberosKeytab(SdkBytes kerberosKeytab);

        /**
         * <p>
         * The <code>krb5.conf</code> file that contains the Kerberos configuration information. You can load the
         * <code>krb5.conf</code> file by providing the file's address. If you're using the CLI, it performs the base64
         * encoding for you. Otherwise, provide the base64-encoded text.
         * </p>
         * 
         * @param kerberosKrb5Conf
         *        The <code>krb5.conf</code> file that contains the Kerberos configuration information. You can load the
         *        <code>krb5.conf</code> file by providing the file's address. If you're using the CLI, it performs the
         *        base64 encoding for you. Otherwise, provide the base64-encoded text.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kerberosKrb5Conf(SdkBytes kerberosKrb5Conf);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the DataSync agents that can connect to your HDFS cluster.
         * </p>
         * 
         * @param agentArns
         *        The Amazon Resource Names (ARNs) of the DataSync agents that can connect to your HDFS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of the DataSync agents that can connect to your HDFS cluster.
         * </p>
         * 
         * @param agentArns
         *        The Amazon Resource Names (ARNs) of the DataSync agents that can connect to your HDFS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String locationArn;

        private String subdirectory;

        private List<HdfsNameNode> nameNodes = DefaultSdkAutoConstructList.getInstance();

        private Integer blockSize;

        private Integer replicationFactor;

        private String kmsKeyProviderUri;

        private QopConfiguration qopConfiguration;

        private String authenticationType;

        private String simpleUser;

        private String kerberosPrincipal;

        private SdkBytes kerberosKeytab;

        private SdkBytes kerberosKrb5Conf;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateLocationHdfsRequest model) {
            super(model);
            locationArn(model.locationArn);
            subdirectory(model.subdirectory);
            nameNodes(model.nameNodes);
            blockSize(model.blockSize);
            replicationFactor(model.replicationFactor);
            kmsKeyProviderUri(model.kmsKeyProviderUri);
            qopConfiguration(model.qopConfiguration);
            authenticationType(model.authenticationType);
            simpleUser(model.simpleUser);
            kerberosPrincipal(model.kerberosPrincipal);
            kerberosKeytab(model.kerberosKeytab);
            kerberosKrb5Conf(model.kerberosKrb5Conf);
            agentArns(model.agentArns);
        }

        public final String getLocationArn() {
            return locationArn;
        }

        public final void setLocationArn(String locationArn) {
            this.locationArn = locationArn;
        }

        @Override
        public final Builder locationArn(String locationArn) {
            this.locationArn = locationArn;
            return this;
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final List<HdfsNameNode.Builder> getNameNodes() {
            List<HdfsNameNode.Builder> result = HdfsNameNodeListCopier.copyToBuilder(this.nameNodes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNameNodes(Collection<HdfsNameNode.BuilderImpl> nameNodes) {
            this.nameNodes = HdfsNameNodeListCopier.copyFromBuilder(nameNodes);
        }

        @Override
        public final Builder nameNodes(Collection<HdfsNameNode> nameNodes) {
            this.nameNodes = HdfsNameNodeListCopier.copy(nameNodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameNodes(HdfsNameNode... nameNodes) {
            nameNodes(Arrays.asList(nameNodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nameNodes(Consumer<HdfsNameNode.Builder>... nameNodes) {
            nameNodes(Stream.of(nameNodes).map(c -> HdfsNameNode.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getBlockSize() {
            return blockSize;
        }

        public final void setBlockSize(Integer blockSize) {
            this.blockSize = blockSize;
        }

        @Override
        public final Builder blockSize(Integer blockSize) {
            this.blockSize = blockSize;
            return this;
        }

        public final Integer getReplicationFactor() {
            return replicationFactor;
        }

        public final void setReplicationFactor(Integer replicationFactor) {
            this.replicationFactor = replicationFactor;
        }

        @Override
        public final Builder replicationFactor(Integer replicationFactor) {
            this.replicationFactor = replicationFactor;
            return this;
        }

        public final String getKmsKeyProviderUri() {
            return kmsKeyProviderUri;
        }

        public final void setKmsKeyProviderUri(String kmsKeyProviderUri) {
            this.kmsKeyProviderUri = kmsKeyProviderUri;
        }

        @Override
        public final Builder kmsKeyProviderUri(String kmsKeyProviderUri) {
            this.kmsKeyProviderUri = kmsKeyProviderUri;
            return this;
        }

        public final QopConfiguration.Builder getQopConfiguration() {
            return qopConfiguration != null ? qopConfiguration.toBuilder() : null;
        }

        public final void setQopConfiguration(QopConfiguration.BuilderImpl qopConfiguration) {
            this.qopConfiguration = qopConfiguration != null ? qopConfiguration.build() : null;
        }

        @Override
        public final Builder qopConfiguration(QopConfiguration qopConfiguration) {
            this.qopConfiguration = qopConfiguration;
            return this;
        }

        public final String getAuthenticationType() {
            return authenticationType;
        }

        public final void setAuthenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
        }

        @Override
        public final Builder authenticationType(String authenticationType) {
            this.authenticationType = authenticationType;
            return this;
        }

        @Override
        public final Builder authenticationType(HdfsAuthenticationType authenticationType) {
            this.authenticationType(authenticationType == null ? null : authenticationType.toString());
            return this;
        }

        public final String getSimpleUser() {
            return simpleUser;
        }

        public final void setSimpleUser(String simpleUser) {
            this.simpleUser = simpleUser;
        }

        @Override
        public final Builder simpleUser(String simpleUser) {
            this.simpleUser = simpleUser;
            return this;
        }

        public final String getKerberosPrincipal() {
            return kerberosPrincipal;
        }

        public final void setKerberosPrincipal(String kerberosPrincipal) {
            this.kerberosPrincipal = kerberosPrincipal;
        }

        @Override
        public final Builder kerberosPrincipal(String kerberosPrincipal) {
            this.kerberosPrincipal = kerberosPrincipal;
            return this;
        }

        public final ByteBuffer getKerberosKeytab() {
            return kerberosKeytab == null ? null : kerberosKeytab.asByteBuffer();
        }

        public final void setKerberosKeytab(ByteBuffer kerberosKeytab) {
            kerberosKeytab(kerberosKeytab == null ? null : SdkBytes.fromByteBuffer(kerberosKeytab));
        }

        @Override
        public final Builder kerberosKeytab(SdkBytes kerberosKeytab) {
            this.kerberosKeytab = kerberosKeytab;
            return this;
        }

        public final ByteBuffer getKerberosKrb5Conf() {
            return kerberosKrb5Conf == null ? null : kerberosKrb5Conf.asByteBuffer();
        }

        public final void setKerberosKrb5Conf(ByteBuffer kerberosKrb5Conf) {
            kerberosKrb5Conf(kerberosKrb5Conf == null ? null : SdkBytes.fromByteBuffer(kerberosKrb5Conf));
        }

        @Override
        public final Builder kerberosKrb5Conf(SdkBytes kerberosKrb5Conf) {
            this.kerberosKrb5Conf = kerberosKrb5Conf;
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateLocationHdfsRequest build() {
            return new UpdateLocationHdfsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
