/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the options that are available to control the behavior of a <a>StartTaskExecution</a> operation. Behavior
 * includes preserving metadata such as user ID (UID), group ID (GID), and file permissions, and also overwriting files
 * in the destination, data integrity verification, and so on.
 * </p>
 * <p>
 * A task has a set of default options associated with it. If you don't specify an option in <a>StartTaskExecution</a>,
 * the default value is used. You can override the defaults options on each task execution by specifying an overriding
 * <code>Options</code> value to <a>StartTaskExecution</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Options implements SdkPojo, Serializable, ToCopyableBuilder<Options.Builder, Options> {
    private static final SdkField<String> VERIFY_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::verifyModeAsString)).setter(setter(Builder::verifyMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerifyMode").build()).build();

    private static final SdkField<String> ATIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::atimeAsString)).setter(setter(Builder::atime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Atime").build()).build();

    private static final SdkField<String> MTIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::mtimeAsString)).setter(setter(Builder::mtime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mtime").build()).build();

    private static final SdkField<String> UID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::uidAsString)).setter(setter(Builder::uid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Uid").build()).build();

    private static final SdkField<String> GID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::gidAsString)).setter(setter(Builder::gid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Gid").build()).build();

    private static final SdkField<String> PRESERVE_DELETED_FILES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::preserveDeletedFilesAsString)).setter(setter(Builder::preserveDeletedFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreserveDeletedFiles").build())
            .build();

    private static final SdkField<String> PRESERVE_DEVICES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::preserveDevicesAsString)).setter(setter(Builder::preserveDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreserveDevices").build()).build();

    private static final SdkField<String> POSIX_PERMISSIONS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Options::posixPermissionsAsString)).setter(setter(Builder::posixPermissions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PosixPermissions").build()).build();

    private static final SdkField<Long> BYTES_PER_SECOND_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(Options::bytesPerSecond)).setter(setter(Builder::bytesPerSecond))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BytesPerSecond").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERIFY_MODE_FIELD,
            ATIME_FIELD, MTIME_FIELD, UID_FIELD, GID_FIELD, PRESERVE_DELETED_FILES_FIELD, PRESERVE_DEVICES_FIELD,
            POSIX_PERMISSIONS_FIELD, BYTES_PER_SECOND_FIELD));

    private static final long serialVersionUID = 1L;

    private final String verifyMode;

    private final String atime;

    private final String mtime;

    private final String uid;

    private final String gid;

    private final String preserveDeletedFiles;

    private final String preserveDevices;

    private final String posixPermissions;

    private final Long bytesPerSecond;

    private Options(BuilderImpl builder) {
        this.verifyMode = builder.verifyMode;
        this.atime = builder.atime;
        this.mtime = builder.mtime;
        this.uid = builder.uid;
        this.gid = builder.gid;
        this.preserveDeletedFiles = builder.preserveDeletedFiles;
        this.preserveDevices = builder.preserveDevices;
        this.posixPermissions = builder.posixPermissions;
        this.bytesPerSecond = builder.bytesPerSecond;
    }

    /**
     * <p>
     * A value that determines whether a data integrity verification should be performed at the end of a task execution
     * after all data and metadata have been transferred.
     * </p>
     * <p>
     * Default value: POINT_IN_TIME_CONSISTENT.
     * </p>
     * <p>
     * POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
     * </p>
     * <p>
     * NONE: Skip verification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #verifyMode} will
     * return {@link VerifyMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #verifyModeAsString}.
     * </p>
     * 
     * @return A value that determines whether a data integrity verification should be performed at the end of a task
     *         execution after all data and metadata have been transferred. </p>
     *         <p>
     *         Default value: POINT_IN_TIME_CONSISTENT.
     *         </p>
     *         <p>
     *         POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
     *         </p>
     *         <p>
     *         NONE: Skip verification.
     * @see VerifyMode
     */
    public VerifyMode verifyMode() {
        return VerifyMode.fromValue(verifyMode);
    }

    /**
     * <p>
     * A value that determines whether a data integrity verification should be performed at the end of a task execution
     * after all data and metadata have been transferred.
     * </p>
     * <p>
     * Default value: POINT_IN_TIME_CONSISTENT.
     * </p>
     * <p>
     * POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
     * </p>
     * <p>
     * NONE: Skip verification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #verifyMode} will
     * return {@link VerifyMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #verifyModeAsString}.
     * </p>
     * 
     * @return A value that determines whether a data integrity verification should be performed at the end of a task
     *         execution after all data and metadata have been transferred. </p>
     *         <p>
     *         Default value: POINT_IN_TIME_CONSISTENT.
     *         </p>
     *         <p>
     *         POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
     *         </p>
     *         <p>
     *         NONE: Skip verification.
     * @see VerifyMode
     */
    public String verifyModeAsString() {
        return verifyMode;
    }

    /**
     * <p>
     * A file metadata value that shows the last time a file was accessed (that is, when the file was read or written
     * to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original <code>Atime</code>
     * attribute on all source files (that is, the version before the PREPARING phase). However, <code>Atime</code>'s
     * behavior is not fully standard across platforms, so AWS DataSync can only do this on a best-effort basis.
     * </p>
     * <p>
     * Default value: BEST_EFFORT.
     * </p>
     * <p>
     * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     * </p>
     * <p>
     * NONE: Ignore <code>Atime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     * </p>
     * <p>
     * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #atime} will return
     * {@link Atime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #atimeAsString}.
     * </p>
     * 
     * @return A file metadata value that shows the last time a file was accessed (that is, when the file was read or
     *         written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
     *         <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
     *         However, <code>Atime</code>'s behavior is not fully standard across platforms, so AWS DataSync can only
     *         do this on a best-effort basis. </p>
     *         <p>
     *         Default value: BEST_EFFORT.
     *         </p>
     *         <p>
     *         BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Atime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     *         </p>
     *         <p>
     *         If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     *         </p>
     * @see Atime
     */
    public Atime atime() {
        return Atime.fromValue(atime);
    }

    /**
     * <p>
     * A file metadata value that shows the last time a file was accessed (that is, when the file was read or written
     * to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original <code>Atime</code>
     * attribute on all source files (that is, the version before the PREPARING phase). However, <code>Atime</code>'s
     * behavior is not fully standard across platforms, so AWS DataSync can only do this on a best-effort basis.
     * </p>
     * <p>
     * Default value: BEST_EFFORT.
     * </p>
     * <p>
     * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     * </p>
     * <p>
     * NONE: Ignore <code>Atime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     * </p>
     * <p>
     * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #atime} will return
     * {@link Atime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #atimeAsString}.
     * </p>
     * 
     * @return A file metadata value that shows the last time a file was accessed (that is, when the file was read or
     *         written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
     *         <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
     *         However, <code>Atime</code>'s behavior is not fully standard across platforms, so AWS DataSync can only
     *         do this on a best-effort basis. </p>
     *         <p>
     *         Default value: BEST_EFFORT.
     *         </p>
     *         <p>
     *         BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Atime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
     *         </p>
     *         <p>
     *         If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
     *         </p>
     * @see Atime
     */
    public String atimeAsString() {
        return atime;
    }

    /**
     * <p>
     * A value that indicates the last time that a file was modified (that is, a file was written to) before the
     * PREPARING phase.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Preserve original <code>Mtime</code> (recommended)
     * </p>
     * <p>
     * NONE: Ignore <code>Mtime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     * </p>
     * <p>
     * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mtime} will return
     * {@link Mtime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mtimeAsString}.
     * </p>
     * 
     * @return A value that indicates the last time that a file was modified (that is, a file was written to) before the
     *         PREPARING phase. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Preserve original <code>Mtime</code> (recommended)
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Mtime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     *         </p>
     *         <p>
     *         If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     *         </p>
     * @see Mtime
     */
    public Mtime mtime() {
        return Mtime.fromValue(mtime);
    }

    /**
     * <p>
     * A value that indicates the last time that a file was modified (that is, a file was written to) before the
     * PREPARING phase.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Preserve original <code>Mtime</code> (recommended)
     * </p>
     * <p>
     * NONE: Ignore <code>Mtime</code>.
     * </p>
     * <note>
     * <p>
     * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     * </p>
     * <p>
     * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mtime} will return
     * {@link Mtime#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #mtimeAsString}.
     * </p>
     * 
     * @return A value that indicates the last time that a file was modified (that is, a file was written to) before the
     *         PREPARING phase. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Preserve original <code>Mtime</code> (recommended)
     *         </p>
     *         <p>
     *         NONE: Ignore <code>Mtime</code>.
     *         </p>
     *         <note>
     *         <p>
     *         If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
     *         </p>
     *         <p>
     *         If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
     *         </p>
     * @see Mtime
     */
    public String mtimeAsString() {
        return mtime;
    }

    /**
     * <p>
     * The user ID (UID) of the file's owner.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #uid} will return
     * {@link Uid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #uidAsString}.
     * </p>
     * 
     * @return The user ID (UID) of the file's owner. </p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Uid
     */
    public Uid uid() {
        return Uid.fromValue(uid);
    }

    /**
     * <p>
     * The user ID (UID) of the file's owner.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #uid} will return
     * {@link Uid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #uidAsString}.
     * </p>
     * 
     * @return The user ID (UID) of the file's owner. </p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Uid
     */
    public String uidAsString() {
        return uid;
    }

    /**
     * <p>
     * The group ID (GID) of the file's owners.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #gid} will return
     * {@link Gid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #gidAsString}.
     * </p>
     * 
     * @return The group ID (GID) of the file's owners. </p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Gid
     */
    public Gid gid() {
        return Gid.fromValue(gid);
    }

    /**
     * <p>
     * The group ID (GID) of the file's owners.
     * </p>
     * <p>
     * Default value: INT_VALUE. This preserves the integer value of the ID.
     * </p>
     * <p>
     * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     * </p>
     * <p>
     * NONE: Ignore UID and GID.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #gid} will return
     * {@link Gid#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from {@link #gidAsString}.
     * </p>
     * 
     * @return The group ID (GID) of the file's owners. </p>
     *         <p>
     *         Default value: INT_VALUE. This preserves the integer value of the ID.
     *         </p>
     *         <p>
     *         INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore UID and GID.
     * @see Gid
     */
    public String gidAsString() {
        return gid;
    }

    /**
     * <p>
     * A value that specifies whether files in the destination that don't exist in the source file system should be
     * preserved.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Ignore such destination files (recommended).
     * </p>
     * <p>
     * REMOVE: Delete destination files that aren’t present in the source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #preserveDeletedFiles} will return {@link PreserveDeletedFiles#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #preserveDeletedFilesAsString}.
     * </p>
     * 
     * @return A value that specifies whether files in the destination that don't exist in the source file system should
     *         be preserved. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Ignore such destination files (recommended).
     *         </p>
     *         <p>
     *         REMOVE: Delete destination files that aren’t present in the source.
     * @see PreserveDeletedFiles
     */
    public PreserveDeletedFiles preserveDeletedFiles() {
        return PreserveDeletedFiles.fromValue(preserveDeletedFiles);
    }

    /**
     * <p>
     * A value that specifies whether files in the destination that don't exist in the source file system should be
     * preserved.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Ignore such destination files (recommended).
     * </p>
     * <p>
     * REMOVE: Delete destination files that aren’t present in the source.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #preserveDeletedFiles} will return {@link PreserveDeletedFiles#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #preserveDeletedFilesAsString}.
     * </p>
     * 
     * @return A value that specifies whether files in the destination that don't exist in the source file system should
     *         be preserved. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Ignore such destination files (recommended).
     *         </p>
     *         <p>
     *         REMOVE: Delete destination files that aren’t present in the source.
     * @see PreserveDeletedFiles
     */
    public String preserveDeletedFilesAsString() {
        return preserveDeletedFiles;
    }

    /**
     * <p>
     * A value that determines whether AWS DataSync should preserve the metadata of block and character devices in the
     * source file system, and recreate the files with that device name and metadata on the destination.
     * </p>
     * <note>
     * <p>
     * AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
     * end-of-file (EOF) marker.
     * </p>
     * </note>
     * <p>
     * Default value: NONE.
     * </p>
     * <p>
     * NONE: Ignore special devices (recommended).
     * </p>
     * <p>
     * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preserveDevices}
     * will return {@link PreserveDevices#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #preserveDevicesAsString}.
     * </p>
     * 
     * @return A value that determines whether AWS DataSync should preserve the metadata of block and character devices
     *         in the source file system, and recreate the files with that device name and metadata on the
     *         destination.</p> <note>
     *         <p>
     *         AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't
     *         return an end-of-file (EOF) marker.
     *         </p>
     *         </note>
     *         <p>
     *         Default value: NONE.
     *         </p>
     *         <p>
     *         NONE: Ignore special devices (recommended).
     *         </p>
     *         <p>
     *         PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon
     *         EFS.
     * @see PreserveDevices
     */
    public PreserveDevices preserveDevices() {
        return PreserveDevices.fromValue(preserveDevices);
    }

    /**
     * <p>
     * A value that determines whether AWS DataSync should preserve the metadata of block and character devices in the
     * source file system, and recreate the files with that device name and metadata on the destination.
     * </p>
     * <note>
     * <p>
     * AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
     * end-of-file (EOF) marker.
     * </p>
     * </note>
     * <p>
     * Default value: NONE.
     * </p>
     * <p>
     * NONE: Ignore special devices (recommended).
     * </p>
     * <p>
     * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preserveDevices}
     * will return {@link PreserveDevices#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #preserveDevicesAsString}.
     * </p>
     * 
     * @return A value that determines whether AWS DataSync should preserve the metadata of block and character devices
     *         in the source file system, and recreate the files with that device name and metadata on the
     *         destination.</p> <note>
     *         <p>
     *         AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't
     *         return an end-of-file (EOF) marker.
     *         </p>
     *         </note>
     *         <p>
     *         Default value: NONE.
     *         </p>
     *         <p>
     *         NONE: Ignore special devices (recommended).
     *         </p>
     *         <p>
     *         PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon
     *         EFS.
     * @see PreserveDevices
     */
    public String preserveDevicesAsString() {
        return preserveDevices;
    }

    /**
     * <p>
     * A value that determines which users or groups can access a file for a specific purpose such as reading, writing,
     * or execution of the file.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Preserve POSIX-style permissions (recommended).
     * </p>
     * <p>
     * NONE: Ignore permissions.
     * </p>
     * <note>
     * <p>
     * AWS DataSync can preserve extant permissions of a source location.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #posixPermissions}
     * will return {@link PosixPermissions#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #posixPermissionsAsString}.
     * </p>
     * 
     * @return A value that determines which users or groups can access a file for a specific purpose such as reading,
     *         writing, or execution of the file. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Preserve POSIX-style permissions (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore permissions.
     *         </p>
     *         <note>
     *         <p>
     *         AWS DataSync can preserve extant permissions of a source location.
     *         </p>
     * @see PosixPermissions
     */
    public PosixPermissions posixPermissions() {
        return PosixPermissions.fromValue(posixPermissions);
    }

    /**
     * <p>
     * A value that determines which users or groups can access a file for a specific purpose such as reading, writing,
     * or execution of the file.
     * </p>
     * <p>
     * Default value: PRESERVE.
     * </p>
     * <p>
     * PRESERVE: Preserve POSIX-style permissions (recommended).
     * </p>
     * <p>
     * NONE: Ignore permissions.
     * </p>
     * <note>
     * <p>
     * AWS DataSync can preserve extant permissions of a source location.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #posixPermissions}
     * will return {@link PosixPermissions#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #posixPermissionsAsString}.
     * </p>
     * 
     * @return A value that determines which users or groups can access a file for a specific purpose such as reading,
     *         writing, or execution of the file. </p>
     *         <p>
     *         Default value: PRESERVE.
     *         </p>
     *         <p>
     *         PRESERVE: Preserve POSIX-style permissions (recommended).
     *         </p>
     *         <p>
     *         NONE: Ignore permissions.
     *         </p>
     *         <note>
     *         <p>
     *         AWS DataSync can preserve extant permissions of a source location.
     *         </p>
     * @see PosixPermissions
     */
    public String posixPermissionsAsString() {
        return posixPermissions;
    }

    /**
     * <p>
     * A value that limits the bandwidth used by AWS DataSync. For example, if you want AWS DataSync to use a maximum of
     * 1 MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
     * </p>
     * 
     * @return A value that limits the bandwidth used by AWS DataSync. For example, if you want AWS DataSync to use a
     *         maximum of 1 MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
     */
    public Long bytesPerSecond() {
        return bytesPerSecond;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(verifyModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(atimeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(mtimeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(uidAsString());
        hashCode = 31 * hashCode + Objects.hashCode(gidAsString());
        hashCode = 31 * hashCode + Objects.hashCode(preserveDeletedFilesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(preserveDevicesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(posixPermissionsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bytesPerSecond());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Options)) {
            return false;
        }
        Options other = (Options) obj;
        return Objects.equals(verifyModeAsString(), other.verifyModeAsString())
                && Objects.equals(atimeAsString(), other.atimeAsString())
                && Objects.equals(mtimeAsString(), other.mtimeAsString()) && Objects.equals(uidAsString(), other.uidAsString())
                && Objects.equals(gidAsString(), other.gidAsString())
                && Objects.equals(preserveDeletedFilesAsString(), other.preserveDeletedFilesAsString())
                && Objects.equals(preserveDevicesAsString(), other.preserveDevicesAsString())
                && Objects.equals(posixPermissionsAsString(), other.posixPermissionsAsString())
                && Objects.equals(bytesPerSecond(), other.bytesPerSecond());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Options").add("VerifyMode", verifyModeAsString()).add("Atime", atimeAsString())
                .add("Mtime", mtimeAsString()).add("Uid", uidAsString()).add("Gid", gidAsString())
                .add("PreserveDeletedFiles", preserveDeletedFilesAsString()).add("PreserveDevices", preserveDevicesAsString())
                .add("PosixPermissions", posixPermissionsAsString()).add("BytesPerSecond", bytesPerSecond()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VerifyMode":
            return Optional.ofNullable(clazz.cast(verifyModeAsString()));
        case "Atime":
            return Optional.ofNullable(clazz.cast(atimeAsString()));
        case "Mtime":
            return Optional.ofNullable(clazz.cast(mtimeAsString()));
        case "Uid":
            return Optional.ofNullable(clazz.cast(uidAsString()));
        case "Gid":
            return Optional.ofNullable(clazz.cast(gidAsString()));
        case "PreserveDeletedFiles":
            return Optional.ofNullable(clazz.cast(preserveDeletedFilesAsString()));
        case "PreserveDevices":
            return Optional.ofNullable(clazz.cast(preserveDevicesAsString()));
        case "PosixPermissions":
            return Optional.ofNullable(clazz.cast(posixPermissionsAsString()));
        case "BytesPerSecond":
            return Optional.ofNullable(clazz.cast(bytesPerSecond()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Options, T> g) {
        return obj -> g.apply((Options) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Options> {
        /**
         * <p>
         * A value that determines whether a data integrity verification should be performed at the end of a task
         * execution after all data and metadata have been transferred.
         * </p>
         * <p>
         * Default value: POINT_IN_TIME_CONSISTENT.
         * </p>
         * <p>
         * POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
         * </p>
         * <p>
         * NONE: Skip verification.
         * </p>
         * 
         * @param verifyMode
         *        A value that determines whether a data integrity verification should be performed at the end of a task
         *        execution after all data and metadata have been transferred. </p>
         *        <p>
         *        Default value: POINT_IN_TIME_CONSISTENT.
         *        </p>
         *        <p>
         *        POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
         *        </p>
         *        <p>
         *        NONE: Skip verification.
         * @see VerifyMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifyMode
         */
        Builder verifyMode(String verifyMode);

        /**
         * <p>
         * A value that determines whether a data integrity verification should be performed at the end of a task
         * execution after all data and metadata have been transferred.
         * </p>
         * <p>
         * Default value: POINT_IN_TIME_CONSISTENT.
         * </p>
         * <p>
         * POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
         * </p>
         * <p>
         * NONE: Skip verification.
         * </p>
         * 
         * @param verifyMode
         *        A value that determines whether a data integrity verification should be performed at the end of a task
         *        execution after all data and metadata have been transferred. </p>
         *        <p>
         *        Default value: POINT_IN_TIME_CONSISTENT.
         *        </p>
         *        <p>
         *        POINT_IN_TIME_CONSISTENT: Perform verification (recommended).
         *        </p>
         *        <p>
         *        NONE: Skip verification.
         * @see VerifyMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifyMode
         */
        Builder verifyMode(VerifyMode verifyMode);

        /**
         * <p>
         * A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         * written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         * <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase). However,
         * <code>Atime</code>'s behavior is not fully standard across platforms, so AWS DataSync can only do this on a
         * best-effort basis.
         * </p>
         * <p>
         * Default value: BEST_EFFORT.
         * </p>
         * <p>
         * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         * </p>
         * <p>
         * NONE: Ignore <code>Atime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         * </p>
         * <p>
         * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         * </p>
         * </note>
         * 
         * @param atime
         *        A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         *        written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         *        <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
         *        However, <code>Atime</code>'s behavior is not fully standard across platforms, so AWS DataSync can
         *        only do this on a best-effort basis. </p>
         *        <p>
         *        Default value: BEST_EFFORT.
         *        </p>
         *        <p>
         *        BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Atime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         *        </p>
         *        <p>
         *        If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         *        </p>
         * @see Atime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Atime
         */
        Builder atime(String atime);

        /**
         * <p>
         * A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         * written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         * <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase). However,
         * <code>Atime</code>'s behavior is not fully standard across platforms, so AWS DataSync can only do this on a
         * best-effort basis.
         * </p>
         * <p>
         * Default value: BEST_EFFORT.
         * </p>
         * <p>
         * BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         * </p>
         * <p>
         * NONE: Ignore <code>Atime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         * </p>
         * <p>
         * If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         * </p>
         * </note>
         * 
         * @param atime
         *        A file metadata value that shows the last time a file was accessed (that is, when the file was read or
         *        written to). If you set <code>Atime</code> to BEST_EFFORT, DataSync attempts to preserve the original
         *        <code>Atime</code> attribute on all source files (that is, the version before the PREPARING phase).
         *        However, <code>Atime</code>'s behavior is not fully standard across platforms, so AWS DataSync can
         *        only do this on a best-effort basis. </p>
         *        <p>
         *        Default value: BEST_EFFORT.
         *        </p>
         *        <p>
         *        BEST_EFFORT: Attempt to preserve the per-file <code>Atime</code> value (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Atime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Atime</code> is set to BEST_EFFORT, <code>Mtime</code> must be set to PRESERVE.
         *        </p>
         *        <p>
         *        If <code>Atime</code> is set to NONE, <code>Mtime</code> must also be NONE.
         *        </p>
         * @see Atime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Atime
         */
        Builder atime(Atime atime);

        /**
         * <p>
         * A value that indicates the last time that a file was modified (that is, a file was written to) before the
         * PREPARING phase.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Preserve original <code>Mtime</code> (recommended)
         * </p>
         * <p>
         * NONE: Ignore <code>Mtime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         * </p>
         * <p>
         * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         * </p>
         * </note>
         * 
         * @param mtime
         *        A value that indicates the last time that a file was modified (that is, a file was written to) before
         *        the PREPARING phase. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Preserve original <code>Mtime</code> (recommended)
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Mtime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         *        </p>
         *        <p>
         *        If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         *        </p>
         * @see Mtime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Mtime
         */
        Builder mtime(String mtime);

        /**
         * <p>
         * A value that indicates the last time that a file was modified (that is, a file was written to) before the
         * PREPARING phase.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Preserve original <code>Mtime</code> (recommended)
         * </p>
         * <p>
         * NONE: Ignore <code>Mtime</code>.
         * </p>
         * <note>
         * <p>
         * If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         * </p>
         * <p>
         * If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         * </p>
         * </note>
         * 
         * @param mtime
         *        A value that indicates the last time that a file was modified (that is, a file was written to) before
         *        the PREPARING phase. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Preserve original <code>Mtime</code> (recommended)
         *        </p>
         *        <p>
         *        NONE: Ignore <code>Mtime</code>.
         *        </p>
         *        <note>
         *        <p>
         *        If <code>Mtime</code> is set to PRESERVE, <code>Atime</code> must be set to BEST_EFFORT.
         *        </p>
         *        <p>
         *        If <code>Mtime</code> is set to NONE, <code>Atime</code> must also be set to NONE.
         *        </p>
         * @see Mtime
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Mtime
         */
        Builder mtime(Mtime mtime);

        /**
         * <p>
         * The user ID (UID) of the file's owner.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param uid
         *        The user ID (UID) of the file's owner. </p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Uid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Uid
         */
        Builder uid(String uid);

        /**
         * <p>
         * The user ID (UID) of the file's owner.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param uid
         *        The user ID (UID) of the file's owner. </p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of UID and group ID (GID) (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Uid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Uid
         */
        Builder uid(Uid uid);

        /**
         * <p>
         * The group ID (GID) of the file's owners.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param gid
         *        The group ID (GID) of the file's owners. </p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Gid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Gid
         */
        Builder gid(String gid);

        /**
         * <p>
         * The group ID (GID) of the file's owners.
         * </p>
         * <p>
         * Default value: INT_VALUE. This preserves the integer value of the ID.
         * </p>
         * <p>
         * INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         * </p>
         * <p>
         * NONE: Ignore UID and GID.
         * </p>
         * 
         * @param gid
         *        The group ID (GID) of the file's owners. </p>
         *        <p>
         *        Default value: INT_VALUE. This preserves the integer value of the ID.
         *        </p>
         *        <p>
         *        INT_VALUE: Preserve the integer value of user ID (UID) and GID (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore UID and GID.
         * @see Gid
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Gid
         */
        Builder gid(Gid gid);

        /**
         * <p>
         * A value that specifies whether files in the destination that don't exist in the source file system should be
         * preserved.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Ignore such destination files (recommended).
         * </p>
         * <p>
         * REMOVE: Delete destination files that aren’t present in the source.
         * </p>
         * 
         * @param preserveDeletedFiles
         *        A value that specifies whether files in the destination that don't exist in the source file system
         *        should be preserved. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Ignore such destination files (recommended).
         *        </p>
         *        <p>
         *        REMOVE: Delete destination files that aren’t present in the source.
         * @see PreserveDeletedFiles
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDeletedFiles
         */
        Builder preserveDeletedFiles(String preserveDeletedFiles);

        /**
         * <p>
         * A value that specifies whether files in the destination that don't exist in the source file system should be
         * preserved.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Ignore such destination files (recommended).
         * </p>
         * <p>
         * REMOVE: Delete destination files that aren’t present in the source.
         * </p>
         * 
         * @param preserveDeletedFiles
         *        A value that specifies whether files in the destination that don't exist in the source file system
         *        should be preserved. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Ignore such destination files (recommended).
         *        </p>
         *        <p>
         *        REMOVE: Delete destination files that aren’t present in the source.
         * @see PreserveDeletedFiles
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDeletedFiles
         */
        Builder preserveDeletedFiles(PreserveDeletedFiles preserveDeletedFiles);

        /**
         * <p>
         * A value that determines whether AWS DataSync should preserve the metadata of block and character devices in
         * the source file system, and recreate the files with that device name and metadata on the destination.
         * </p>
         * <note>
         * <p>
         * AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
         * end-of-file (EOF) marker.
         * </p>
         * </note>
         * <p>
         * Default value: NONE.
         * </p>
         * <p>
         * NONE: Ignore special devices (recommended).
         * </p>
         * <p>
         * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
         * </p>
         * 
         * @param preserveDevices
         *        A value that determines whether AWS DataSync should preserve the metadata of block and character
         *        devices in the source file system, and recreate the files with that device name and metadata on the
         *        destination.</p> <note>
         *        <p>
         *        AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't
         *        return an end-of-file (EOF) marker.
         *        </p>
         *        </note>
         *        <p>
         *        Default value: NONE.
         *        </p>
         *        <p>
         *        NONE: Ignore special devices (recommended).
         *        </p>
         *        <p>
         *        PRESERVE: Preserve character and block device metadata. This option isn't currently supported for
         *        Amazon EFS.
         * @see PreserveDevices
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDevices
         */
        Builder preserveDevices(String preserveDevices);

        /**
         * <p>
         * A value that determines whether AWS DataSync should preserve the metadata of block and character devices in
         * the source file system, and recreate the files with that device name and metadata on the destination.
         * </p>
         * <note>
         * <p>
         * AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't return an
         * end-of-file (EOF) marker.
         * </p>
         * </note>
         * <p>
         * Default value: NONE.
         * </p>
         * <p>
         * NONE: Ignore special devices (recommended).
         * </p>
         * <p>
         * PRESERVE: Preserve character and block device metadata. This option isn't currently supported for Amazon EFS.
         * </p>
         * 
         * @param preserveDevices
         *        A value that determines whether AWS DataSync should preserve the metadata of block and character
         *        devices in the source file system, and recreate the files with that device name and metadata on the
         *        destination.</p> <note>
         *        <p>
         *        AWS DataSync can't sync the actual contents of such devices, because they are nonterminal and don't
         *        return an end-of-file (EOF) marker.
         *        </p>
         *        </note>
         *        <p>
         *        Default value: NONE.
         *        </p>
         *        <p>
         *        NONE: Ignore special devices (recommended).
         *        </p>
         *        <p>
         *        PRESERVE: Preserve character and block device metadata. This option isn't currently supported for
         *        Amazon EFS.
         * @see PreserveDevices
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreserveDevices
         */
        Builder preserveDevices(PreserveDevices preserveDevices);

        /**
         * <p>
         * A value that determines which users or groups can access a file for a specific purpose such as reading,
         * writing, or execution of the file.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Preserve POSIX-style permissions (recommended).
         * </p>
         * <p>
         * NONE: Ignore permissions.
         * </p>
         * <note>
         * <p>
         * AWS DataSync can preserve extant permissions of a source location.
         * </p>
         * </note>
         * 
         * @param posixPermissions
         *        A value that determines which users or groups can access a file for a specific purpose such as
         *        reading, writing, or execution of the file. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Preserve POSIX-style permissions (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore permissions.
         *        </p>
         *        <note>
         *        <p>
         *        AWS DataSync can preserve extant permissions of a source location.
         *        </p>
         * @see PosixPermissions
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PosixPermissions
         */
        Builder posixPermissions(String posixPermissions);

        /**
         * <p>
         * A value that determines which users or groups can access a file for a specific purpose such as reading,
         * writing, or execution of the file.
         * </p>
         * <p>
         * Default value: PRESERVE.
         * </p>
         * <p>
         * PRESERVE: Preserve POSIX-style permissions (recommended).
         * </p>
         * <p>
         * NONE: Ignore permissions.
         * </p>
         * <note>
         * <p>
         * AWS DataSync can preserve extant permissions of a source location.
         * </p>
         * </note>
         * 
         * @param posixPermissions
         *        A value that determines which users or groups can access a file for a specific purpose such as
         *        reading, writing, or execution of the file. </p>
         *        <p>
         *        Default value: PRESERVE.
         *        </p>
         *        <p>
         *        PRESERVE: Preserve POSIX-style permissions (recommended).
         *        </p>
         *        <p>
         *        NONE: Ignore permissions.
         *        </p>
         *        <note>
         *        <p>
         *        AWS DataSync can preserve extant permissions of a source location.
         *        </p>
         * @see PosixPermissions
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PosixPermissions
         */
        Builder posixPermissions(PosixPermissions posixPermissions);

        /**
         * <p>
         * A value that limits the bandwidth used by AWS DataSync. For example, if you want AWS DataSync to use a
         * maximum of 1 MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
         * </p>
         * 
         * @param bytesPerSecond
         *        A value that limits the bandwidth used by AWS DataSync. For example, if you want AWS DataSync to use a
         *        maximum of 1 MB, set this value to <code>1048576</code> (<code>=1024*1024</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytesPerSecond(Long bytesPerSecond);
    }

    static final class BuilderImpl implements Builder {
        private String verifyMode;

        private String atime;

        private String mtime;

        private String uid;

        private String gid;

        private String preserveDeletedFiles;

        private String preserveDevices;

        private String posixPermissions;

        private Long bytesPerSecond;

        private BuilderImpl() {
        }

        private BuilderImpl(Options model) {
            verifyMode(model.verifyMode);
            atime(model.atime);
            mtime(model.mtime);
            uid(model.uid);
            gid(model.gid);
            preserveDeletedFiles(model.preserveDeletedFiles);
            preserveDevices(model.preserveDevices);
            posixPermissions(model.posixPermissions);
            bytesPerSecond(model.bytesPerSecond);
        }

        public final String getVerifyModeAsString() {
            return verifyMode;
        }

        @Override
        public final Builder verifyMode(String verifyMode) {
            this.verifyMode = verifyMode;
            return this;
        }

        @Override
        public final Builder verifyMode(VerifyMode verifyMode) {
            this.verifyMode(verifyMode.toString());
            return this;
        }

        public final void setVerifyMode(String verifyMode) {
            this.verifyMode = verifyMode;
        }

        public final String getAtimeAsString() {
            return atime;
        }

        @Override
        public final Builder atime(String atime) {
            this.atime = atime;
            return this;
        }

        @Override
        public final Builder atime(Atime atime) {
            this.atime(atime.toString());
            return this;
        }

        public final void setAtime(String atime) {
            this.atime = atime;
        }

        public final String getMtimeAsString() {
            return mtime;
        }

        @Override
        public final Builder mtime(String mtime) {
            this.mtime = mtime;
            return this;
        }

        @Override
        public final Builder mtime(Mtime mtime) {
            this.mtime(mtime.toString());
            return this;
        }

        public final void setMtime(String mtime) {
            this.mtime = mtime;
        }

        public final String getUidAsString() {
            return uid;
        }

        @Override
        public final Builder uid(String uid) {
            this.uid = uid;
            return this;
        }

        @Override
        public final Builder uid(Uid uid) {
            this.uid(uid.toString());
            return this;
        }

        public final void setUid(String uid) {
            this.uid = uid;
        }

        public final String getGidAsString() {
            return gid;
        }

        @Override
        public final Builder gid(String gid) {
            this.gid = gid;
            return this;
        }

        @Override
        public final Builder gid(Gid gid) {
            this.gid(gid.toString());
            return this;
        }

        public final void setGid(String gid) {
            this.gid = gid;
        }

        public final String getPreserveDeletedFilesAsString() {
            return preserveDeletedFiles;
        }

        @Override
        public final Builder preserveDeletedFiles(String preserveDeletedFiles) {
            this.preserveDeletedFiles = preserveDeletedFiles;
            return this;
        }

        @Override
        public final Builder preserveDeletedFiles(PreserveDeletedFiles preserveDeletedFiles) {
            this.preserveDeletedFiles(preserveDeletedFiles.toString());
            return this;
        }

        public final void setPreserveDeletedFiles(String preserveDeletedFiles) {
            this.preserveDeletedFiles = preserveDeletedFiles;
        }

        public final String getPreserveDevicesAsString() {
            return preserveDevices;
        }

        @Override
        public final Builder preserveDevices(String preserveDevices) {
            this.preserveDevices = preserveDevices;
            return this;
        }

        @Override
        public final Builder preserveDevices(PreserveDevices preserveDevices) {
            this.preserveDevices(preserveDevices.toString());
            return this;
        }

        public final void setPreserveDevices(String preserveDevices) {
            this.preserveDevices = preserveDevices;
        }

        public final String getPosixPermissionsAsString() {
            return posixPermissions;
        }

        @Override
        public final Builder posixPermissions(String posixPermissions) {
            this.posixPermissions = posixPermissions;
            return this;
        }

        @Override
        public final Builder posixPermissions(PosixPermissions posixPermissions) {
            this.posixPermissions(posixPermissions.toString());
            return this;
        }

        public final void setPosixPermissions(String posixPermissions) {
            this.posixPermissions = posixPermissions;
        }

        public final Long getBytesPerSecond() {
            return bytesPerSecond;
        }

        @Override
        public final Builder bytesPerSecond(Long bytesPerSecond) {
            this.bytesPerSecond = bytesPerSecond;
            return this;
        }

        public final void setBytesPerSecond(Long bytesPerSecond) {
            this.bytesPerSecond = bytesPerSecond;
        }

        @Override
        public Options build() {
            return new Options(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
