/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datazone.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateProjectRequest extends DataZoneRequest implements
        ToCopyableBuilder<CreateProjectRequest.Builder, CreateProjectRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(CreateProjectRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> DOMAIN_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainIdentifier").getter(getter(CreateProjectRequest::domainIdentifier))
            .setter(setter(Builder::domainIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("domainIdentifier").build()).build();

    private static final SdkField<String> DOMAIN_UNIT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainUnitId").getter(getter(CreateProjectRequest::domainUnitId)).setter(setter(Builder::domainUnitId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainUnitId").build()).build();

    private static final SdkField<List<String>> GLOSSARY_TERMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("glossaryTerms")
            .getter(getter(CreateProjectRequest::glossaryTerms))
            .setter(setter(Builder::glossaryTerms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("glossaryTerms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CreateProjectRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            DOMAIN_IDENTIFIER_FIELD, DOMAIN_UNIT_ID_FIELD, GLOSSARY_TERMS_FIELD, NAME_FIELD));

    private final String description;

    private final String domainIdentifier;

    private final String domainUnitId;

    private final List<String> glossaryTerms;

    private final String name;

    private CreateProjectRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.domainIdentifier = builder.domainIdentifier;
        this.domainUnitId = builder.domainUnitId;
        this.glossaryTerms = builder.glossaryTerms;
        this.name = builder.name;
    }

    /**
     * <p>
     * The description of the Amazon DataZone project.
     * </p>
     * 
     * @return The description of the Amazon DataZone project.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ID of the Amazon DataZone domain in which this project is created.
     * </p>
     * 
     * @return The ID of the Amazon DataZone domain in which this project is created.
     */
    public final String domainIdentifier() {
        return domainIdentifier;
    }

    /**
     * <p>
     * The ID of the domain unit. This parameter is not required and if it is not specified, then the project is created
     * at the root domain unit level.
     * </p>
     * 
     * @return The ID of the domain unit. This parameter is not required and if it is not specified, then the project is
     *         created at the root domain unit level.
     */
    public final String domainUnitId() {
        return domainUnitId;
    }

    /**
     * For responses, this returns true if the service returned a value for the GlossaryTerms property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGlossaryTerms() {
        return glossaryTerms != null && !(glossaryTerms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The glossary terms that can be used in this Amazon DataZone project.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGlossaryTerms} method.
     * </p>
     * 
     * @return The glossary terms that can be used in this Amazon DataZone project.
     */
    public final List<String> glossaryTerms() {
        return glossaryTerms;
    }

    /**
     * <p>
     * The name of the Amazon DataZone project.
     * </p>
     * 
     * @return The name of the Amazon DataZone project.
     */
    public final String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(domainIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(domainUnitId());
        hashCode = 31 * hashCode + Objects.hashCode(hasGlossaryTerms() ? glossaryTerms() : null);
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateProjectRequest)) {
            return false;
        }
        CreateProjectRequest other = (CreateProjectRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(domainIdentifier(), other.domainIdentifier())
                && Objects.equals(domainUnitId(), other.domainUnitId()) && hasGlossaryTerms() == other.hasGlossaryTerms()
                && Objects.equals(glossaryTerms(), other.glossaryTerms()) && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateProjectRequest")
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DomainIdentifier", domainIdentifier()).add("DomainUnitId", domainUnitId())
                .add("GlossaryTerms", hasGlossaryTerms() ? glossaryTerms() : null)
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "domainIdentifier":
            return Optional.ofNullable(clazz.cast(domainIdentifier()));
        case "domainUnitId":
            return Optional.ofNullable(clazz.cast(domainUnitId()));
        case "glossaryTerms":
            return Optional.ofNullable(clazz.cast(glossaryTerms()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateProjectRequest, T> g) {
        return obj -> g.apply((CreateProjectRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataZoneRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateProjectRequest> {
        /**
         * <p>
         * The description of the Amazon DataZone project.
         * </p>
         * 
         * @param description
         *        The description of the Amazon DataZone project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ID of the Amazon DataZone domain in which this project is created.
         * </p>
         * 
         * @param domainIdentifier
         *        The ID of the Amazon DataZone domain in which this project is created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainIdentifier(String domainIdentifier);

        /**
         * <p>
         * The ID of the domain unit. This parameter is not required and if it is not specified, then the project is
         * created at the root domain unit level.
         * </p>
         * 
         * @param domainUnitId
         *        The ID of the domain unit. This parameter is not required and if it is not specified, then the project
         *        is created at the root domain unit level.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainUnitId(String domainUnitId);

        /**
         * <p>
         * The glossary terms that can be used in this Amazon DataZone project.
         * </p>
         * 
         * @param glossaryTerms
         *        The glossary terms that can be used in this Amazon DataZone project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glossaryTerms(Collection<String> glossaryTerms);

        /**
         * <p>
         * The glossary terms that can be used in this Amazon DataZone project.
         * </p>
         * 
         * @param glossaryTerms
         *        The glossary terms that can be used in this Amazon DataZone project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder glossaryTerms(String... glossaryTerms);

        /**
         * <p>
         * The name of the Amazon DataZone project.
         * </p>
         * 
         * @param name
         *        The name of the Amazon DataZone project.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataZoneRequest.BuilderImpl implements Builder {
        private String description;

        private String domainIdentifier;

        private String domainUnitId;

        private List<String> glossaryTerms = DefaultSdkAutoConstructList.getInstance();

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateProjectRequest model) {
            super(model);
            description(model.description);
            domainIdentifier(model.domainIdentifier);
            domainUnitId(model.domainUnitId);
            glossaryTerms(model.glossaryTerms);
            name(model.name);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDomainIdentifier() {
            return domainIdentifier;
        }

        public final void setDomainIdentifier(String domainIdentifier) {
            this.domainIdentifier = domainIdentifier;
        }

        @Override
        public final Builder domainIdentifier(String domainIdentifier) {
            this.domainIdentifier = domainIdentifier;
            return this;
        }

        public final String getDomainUnitId() {
            return domainUnitId;
        }

        public final void setDomainUnitId(String domainUnitId) {
            this.domainUnitId = domainUnitId;
        }

        @Override
        public final Builder domainUnitId(String domainUnitId) {
            this.domainUnitId = domainUnitId;
            return this;
        }

        public final Collection<String> getGlossaryTerms() {
            if (glossaryTerms instanceof SdkAutoConstructList) {
                return null;
            }
            return glossaryTerms;
        }

        public final void setGlossaryTerms(Collection<String> glossaryTerms) {
            this.glossaryTerms = GlossaryTermsCopier.copy(glossaryTerms);
        }

        @Override
        public final Builder glossaryTerms(Collection<String> glossaryTerms) {
            this.glossaryTerms = GlossaryTermsCopier.copy(glossaryTerms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder glossaryTerms(String... glossaryTerms) {
            glossaryTerms(Arrays.asList(glossaryTerms));
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateProjectRequest build() {
            return new CreateProjectRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
