/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dax;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.dax.internal.DaxServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.dax.model.ClusterAlreadyExistsException;
import software.amazon.awssdk.services.dax.model.ClusterNotFoundException;
import software.amazon.awssdk.services.dax.model.ClusterQuotaForCustomerExceededException;
import software.amazon.awssdk.services.dax.model.CreateClusterRequest;
import software.amazon.awssdk.services.dax.model.CreateClusterResponse;
import software.amazon.awssdk.services.dax.model.CreateParameterGroupRequest;
import software.amazon.awssdk.services.dax.model.CreateParameterGroupResponse;
import software.amazon.awssdk.services.dax.model.CreateSubnetGroupRequest;
import software.amazon.awssdk.services.dax.model.CreateSubnetGroupResponse;
import software.amazon.awssdk.services.dax.model.DaxException;
import software.amazon.awssdk.services.dax.model.DecreaseReplicationFactorRequest;
import software.amazon.awssdk.services.dax.model.DecreaseReplicationFactorResponse;
import software.amazon.awssdk.services.dax.model.DeleteClusterRequest;
import software.amazon.awssdk.services.dax.model.DeleteClusterResponse;
import software.amazon.awssdk.services.dax.model.DeleteParameterGroupRequest;
import software.amazon.awssdk.services.dax.model.DeleteParameterGroupResponse;
import software.amazon.awssdk.services.dax.model.DeleteSubnetGroupRequest;
import software.amazon.awssdk.services.dax.model.DeleteSubnetGroupResponse;
import software.amazon.awssdk.services.dax.model.DescribeClustersRequest;
import software.amazon.awssdk.services.dax.model.DescribeClustersResponse;
import software.amazon.awssdk.services.dax.model.DescribeDefaultParametersRequest;
import software.amazon.awssdk.services.dax.model.DescribeDefaultParametersResponse;
import software.amazon.awssdk.services.dax.model.DescribeEventsRequest;
import software.amazon.awssdk.services.dax.model.DescribeEventsResponse;
import software.amazon.awssdk.services.dax.model.DescribeParameterGroupsRequest;
import software.amazon.awssdk.services.dax.model.DescribeParameterGroupsResponse;
import software.amazon.awssdk.services.dax.model.DescribeParametersRequest;
import software.amazon.awssdk.services.dax.model.DescribeParametersResponse;
import software.amazon.awssdk.services.dax.model.DescribeSubnetGroupsRequest;
import software.amazon.awssdk.services.dax.model.DescribeSubnetGroupsResponse;
import software.amazon.awssdk.services.dax.model.IncreaseReplicationFactorRequest;
import software.amazon.awssdk.services.dax.model.IncreaseReplicationFactorResponse;
import software.amazon.awssdk.services.dax.model.InsufficientClusterCapacityException;
import software.amazon.awssdk.services.dax.model.InvalidArnException;
import software.amazon.awssdk.services.dax.model.InvalidClusterStateException;
import software.amazon.awssdk.services.dax.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.dax.model.InvalidParameterGroupStateException;
import software.amazon.awssdk.services.dax.model.InvalidParameterValueException;
import software.amazon.awssdk.services.dax.model.InvalidSubnetException;
import software.amazon.awssdk.services.dax.model.InvalidVpcNetworkStateException;
import software.amazon.awssdk.services.dax.model.ListTagsRequest;
import software.amazon.awssdk.services.dax.model.ListTagsResponse;
import software.amazon.awssdk.services.dax.model.NodeNotFoundException;
import software.amazon.awssdk.services.dax.model.NodeQuotaForClusterExceededException;
import software.amazon.awssdk.services.dax.model.NodeQuotaForCustomerExceededException;
import software.amazon.awssdk.services.dax.model.ParameterGroupAlreadyExistsException;
import software.amazon.awssdk.services.dax.model.ParameterGroupNotFoundException;
import software.amazon.awssdk.services.dax.model.ParameterGroupQuotaExceededException;
import software.amazon.awssdk.services.dax.model.RebootNodeRequest;
import software.amazon.awssdk.services.dax.model.RebootNodeResponse;
import software.amazon.awssdk.services.dax.model.ServiceLinkedRoleNotFoundException;
import software.amazon.awssdk.services.dax.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.dax.model.SubnetGroupAlreadyExistsException;
import software.amazon.awssdk.services.dax.model.SubnetGroupInUseException;
import software.amazon.awssdk.services.dax.model.SubnetGroupNotFoundException;
import software.amazon.awssdk.services.dax.model.SubnetGroupQuotaExceededException;
import software.amazon.awssdk.services.dax.model.SubnetInUseException;
import software.amazon.awssdk.services.dax.model.SubnetQuotaExceededException;
import software.amazon.awssdk.services.dax.model.TagNotFoundException;
import software.amazon.awssdk.services.dax.model.TagQuotaPerResourceExceededException;
import software.amazon.awssdk.services.dax.model.TagResourceRequest;
import software.amazon.awssdk.services.dax.model.TagResourceResponse;
import software.amazon.awssdk.services.dax.model.UntagResourceRequest;
import software.amazon.awssdk.services.dax.model.UntagResourceResponse;
import software.amazon.awssdk.services.dax.model.UpdateClusterRequest;
import software.amazon.awssdk.services.dax.model.UpdateClusterResponse;
import software.amazon.awssdk.services.dax.model.UpdateParameterGroupRequest;
import software.amazon.awssdk.services.dax.model.UpdateParameterGroupResponse;
import software.amazon.awssdk.services.dax.model.UpdateSubnetGroupRequest;
import software.amazon.awssdk.services.dax.model.UpdateSubnetGroupResponse;
import software.amazon.awssdk.services.dax.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.CreateParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.CreateSubnetGroupRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DecreaseReplicationFactorRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DeleteParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DeleteSubnetGroupRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DescribeDefaultParametersRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DescribeEventsRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DescribeParameterGroupsRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DescribeParametersRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.DescribeSubnetGroupsRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.IncreaseReplicationFactorRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.RebootNodeRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.UpdateClusterRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.UpdateParameterGroupRequestMarshaller;
import software.amazon.awssdk.services.dax.transform.UpdateSubnetGroupRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DaxAsyncClient}.
 *
 * @see DaxAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDaxAsyncClient implements DaxAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDaxAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final DaxServiceClientConfiguration serviceClientConfiguration;

    protected DefaultDaxAsyncClient(DaxServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a DAX cluster. All nodes in the cluster run the same DAX caching software.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterAlreadyExistsException You already have a DAX cluster with the given identifier.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>InsufficientClusterCapacityException There are not enough system resources to create the cluster you
     *         requested (or to resize an already-existing cluster).</li>
     *         <li>SubnetGroupNotFoundException The requested subnet group name does not refer to an existing subnet
     *         group.</li>
     *         <li>InvalidParameterGroupStateException One or more parameters in a parameter group are in an invalid
     *         state.</li>
     *         <li>ParameterGroupNotFoundException The specified parameter group does not exist.</li>
     *         <li>ClusterQuotaForCustomerExceededException You have attempted to exceed the maximum number of DAX
     *         clusters for your AWS account.</li>
     *         <li>NodeQuotaForClusterExceededException You have attempted to exceed the maximum number of nodes for a
     *         DAX cluster.</li>
     *         <li>NodeQuotaForCustomerExceededException You have attempted to exceed the maximum number of nodes for
     *         your AWS account.</li>
     *         <li>InvalidVpcNetworkStateException The VPC network is in an invalid state.</li>
     *         <li>TagQuotaPerResourceExceededException You have exceeded the maximum number of tags for this DAX
     *         cluster.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>ServiceQuotaExceededException You have reached the maximum number of x509 certificates that can be
     *         created for encrypted clusters in a 30 day period. Contact AWS customer support to discuss options for
     *         continuing to create encrypted clusters.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster")
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new parameter group. A parameter group is a collection of parameters that you apply to all of the nodes
     * in a DAX cluster.
     * </p>
     *
     * @param createParameterGroupRequest
     * @return A Java Future containing the result of the CreateParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupQuotaExceededException You have attempted to exceed the maximum number of parameter
     *         groups.</li>
     *         <li>ParameterGroupAlreadyExistsException The specified parameter group already exists.</li>
     *         <li>InvalidParameterGroupStateException One or more parameters in a parameter group are in an invalid
     *         state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.CreateParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/CreateParameterGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateParameterGroupResponse> createParameterGroup(
            CreateParameterGroupRequest createParameterGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createParameterGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateParameterGroupRequest, CreateParameterGroupResponse>()
                            .withOperationName("CreateParameterGroup")
                            .withMarshaller(new CreateParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createParameterGroupRequest));
            CompletableFuture<CreateParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new subnet group.
     * </p>
     *
     * @param createSubnetGroupRequest
     * @return A Java Future containing the result of the CreateSubnetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupAlreadyExistsException The specified subnet group already exists.</li>
     *         <li>SubnetGroupQuotaExceededException The request cannot be processed because it would exceed the allowed
     *         number of subnets in a subnet group.</li>
     *         <li>SubnetQuotaExceededException The request cannot be processed because it would exceed the allowed
     *         number of subnets in a subnet group.</li>
     *         <li>InvalidSubnetException An invalid subnet identifier was specified.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.CreateSubnetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/CreateSubnetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubnetGroupResponse> createSubnetGroup(CreateSubnetGroupRequest createSubnetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSubnetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubnetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubnetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubnetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubnetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubnetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubnetGroupRequest, CreateSubnetGroupResponse>()
                            .withOperationName("CreateSubnetGroup")
                            .withMarshaller(new CreateSubnetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSubnetGroupRequest));
            CompletableFuture<CreateSubnetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more nodes from a DAX cluster.
     * </p>
     * <note>
     * <p>
     * You cannot use <code>DecreaseReplicationFactor</code> to remove the last node in a DAX cluster. If you need to do
     * this, use <code>DeleteCluster</code> instead.
     * </p>
     * </note>
     *
     * @param decreaseReplicationFactorRequest
     * @return A Java Future containing the result of the DecreaseReplicationFactor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>NodeNotFoundException None of the nodes in the cluster have the given node ID.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DecreaseReplicationFactor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DecreaseReplicationFactor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DecreaseReplicationFactorResponse> decreaseReplicationFactor(
            DecreaseReplicationFactorRequest decreaseReplicationFactorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(decreaseReplicationFactorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, decreaseReplicationFactorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DecreaseReplicationFactor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DecreaseReplicationFactorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DecreaseReplicationFactorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DecreaseReplicationFactorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DecreaseReplicationFactorRequest, DecreaseReplicationFactorResponse>()
                            .withOperationName("DecreaseReplicationFactor")
                            .withMarshaller(new DecreaseReplicationFactorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(decreaseReplicationFactorRequest));
            CompletableFuture<DecreaseReplicationFactorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a previously provisioned DAX cluster. <i>DeleteCluster</i> deletes all associated nodes, node endpoints
     * and the DAX cluster itself. When you receive a successful response from this action, DAX immediately begins
     * deleting the cluster; you cannot cancel or revert this action.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster")
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified parameter group. You cannot delete a parameter group if it is associated with any DAX
     * clusters.
     * </p>
     *
     * @param deleteParameterGroupRequest
     * @return A Java Future containing the result of the DeleteParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterGroupStateException One or more parameters in a parameter group are in an invalid
     *         state.</li>
     *         <li>ParameterGroupNotFoundException The specified parameter group does not exist.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DeleteParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DeleteParameterGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteParameterGroupResponse> deleteParameterGroup(
            DeleteParameterGroupRequest deleteParameterGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteParameterGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteParameterGroupRequest, DeleteParameterGroupResponse>()
                            .withOperationName("DeleteParameterGroup")
                            .withMarshaller(new DeleteParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteParameterGroupRequest));
            CompletableFuture<DeleteParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a subnet group.
     * </p>
     * <note>
     * <p>
     * You cannot delete a subnet group if it is associated with any DAX clusters.
     * </p>
     * </note>
     *
     * @param deleteSubnetGroupRequest
     * @return A Java Future containing the result of the DeleteSubnetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupInUseException The specified subnet group is currently in use.</li>
     *         <li>SubnetGroupNotFoundException The requested subnet group name does not refer to an existing subnet
     *         group.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DeleteSubnetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DeleteSubnetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSubnetGroupResponse> deleteSubnetGroup(DeleteSubnetGroupRequest deleteSubnetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSubnetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubnetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubnetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSubnetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSubnetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSubnetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSubnetGroupRequest, DeleteSubnetGroupResponse>()
                            .withOperationName("DeleteSubnetGroup")
                            .withMarshaller(new DeleteSubnetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSubnetGroupRequest));
            CompletableFuture<DeleteSubnetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about all provisioned DAX clusters if no cluster identifier is specified, or about a specific
     * DAX cluster if a cluster identifier is supplied.
     * </p>
     * <p>
     * If the cluster is in the CREATING state, only cluster level information will be displayed until all of the nodes
     * are successfully provisioned.
     * </p>
     * <p>
     * If the cluster is in the DELETING state, only cluster level information will be displayed.
     * </p>
     * <p>
     * If nodes are currently being added to the DAX cluster, node endpoint information and creation time for the
     * additional nodes will not be displayed until they are completely provisioned. When the DAX cluster state is
     * <i>available</i>, the cluster is ready for use.
     * </p>
     * <p>
     * If nodes are currently being removed from the DAX cluster, no endpoint information for the removed nodes is
     * displayed.
     * </p>
     *
     * @param describeClustersRequest
     * @return A Java Future containing the result of the DescribeClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DescribeClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClustersResponse> describeClusters(DescribeClustersRequest describeClustersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClustersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                            .withOperationName("DescribeClusters")
                            .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeClustersRequest));
            CompletableFuture<DescribeClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the default system parameter information for the DAX caching software.
     * </p>
     *
     * @param describeDefaultParametersRequest
     * @return A Java Future containing the result of the DescribeDefaultParameters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DescribeDefaultParameters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DescribeDefaultParameters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDefaultParametersResponse> describeDefaultParameters(
            DescribeDefaultParametersRequest describeDefaultParametersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDefaultParametersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDefaultParametersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDefaultParameters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDefaultParametersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDefaultParametersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDefaultParametersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDefaultParametersRequest, DescribeDefaultParametersResponse>()
                            .withOperationName("DescribeDefaultParameters")
                            .withMarshaller(new DescribeDefaultParametersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDefaultParametersRequest));
            CompletableFuture<DescribeDefaultParametersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns events related to DAX clusters and parameter groups. You can obtain events specific to a particular DAX
     * cluster or parameter group by providing the name as a parameter.
     * </p>
     * <p>
     * By default, only the events occurring within the last 24 hours are returned; however, you can retrieve up to 14
     * days' worth of events if necessary.
     * </p>
     *
     * @param describeEventsRequest
     * @return A Java Future containing the result of the DescribeEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DescribeEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DescribeEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEventsResponse> describeEvents(DescribeEventsRequest describeEventsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEventsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEventsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEventsRequest, DescribeEventsResponse>()
                            .withOperationName("DescribeEvents")
                            .withMarshaller(new DescribeEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEventsRequest));
            CompletableFuture<DescribeEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of parameter group descriptions. If a parameter group name is specified, the list will contain
     * only the descriptions for that group.
     * </p>
     *
     * @param describeParameterGroupsRequest
     * @return A Java Future containing the result of the DescribeParameterGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupNotFoundException The specified parameter group does not exist.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DescribeParameterGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DescribeParameterGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeParameterGroupsResponse> describeParameterGroups(
            DescribeParameterGroupsRequest describeParameterGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeParameterGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeParameterGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeParameterGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeParameterGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeParameterGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeParameterGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeParameterGroupsRequest, DescribeParameterGroupsResponse>()
                            .withOperationName("DescribeParameterGroups")
                            .withMarshaller(new DescribeParameterGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeParameterGroupsRequest));
            CompletableFuture<DescribeParameterGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the detailed parameter list for a particular parameter group.
     * </p>
     *
     * @param describeParametersRequest
     * @return A Java Future containing the result of the DescribeParameters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ParameterGroupNotFoundException The specified parameter group does not exist.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DescribeParameters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DescribeParameters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeParametersResponse> describeParameters(DescribeParametersRequest describeParametersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeParametersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeParametersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeParameters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeParametersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeParametersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeParametersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeParametersRequest, DescribeParametersResponse>()
                            .withOperationName("DescribeParameters")
                            .withMarshaller(new DescribeParametersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeParametersRequest));
            CompletableFuture<DescribeParametersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of subnet group descriptions. If a subnet group name is specified, the list will contain only the
     * description of that group.
     * </p>
     *
     * @param describeSubnetGroupsRequest
     * @return A Java Future containing the result of the DescribeSubnetGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupNotFoundException The requested subnet group name does not refer to an existing subnet
     *         group.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.DescribeSubnetGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/DescribeSubnetGroups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSubnetGroupsResponse> describeSubnetGroups(
            DescribeSubnetGroupsRequest describeSubnetGroupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSubnetGroupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSubnetGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSubnetGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSubnetGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSubnetGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSubnetGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSubnetGroupsRequest, DescribeSubnetGroupsResponse>()
                            .withOperationName("DescribeSubnetGroups")
                            .withMarshaller(new DescribeSubnetGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSubnetGroupsRequest));
            CompletableFuture<DescribeSubnetGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more nodes to a DAX cluster.
     * </p>
     *
     * @param increaseReplicationFactorRequest
     * @return A Java Future containing the result of the IncreaseReplicationFactor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>InsufficientClusterCapacityException There are not enough system resources to create the cluster you
     *         requested (or to resize an already-existing cluster).</li>
     *         <li>InvalidVpcNetworkStateException The VPC network is in an invalid state.</li>
     *         <li>NodeQuotaForClusterExceededException You have attempted to exceed the maximum number of nodes for a
     *         DAX cluster.</li>
     *         <li>NodeQuotaForCustomerExceededException You have attempted to exceed the maximum number of nodes for
     *         your AWS account.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.IncreaseReplicationFactor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/IncreaseReplicationFactor" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<IncreaseReplicationFactorResponse> increaseReplicationFactor(
            IncreaseReplicationFactorRequest increaseReplicationFactorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(increaseReplicationFactorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, increaseReplicationFactorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IncreaseReplicationFactor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<IncreaseReplicationFactorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, IncreaseReplicationFactorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<IncreaseReplicationFactorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<IncreaseReplicationFactorRequest, IncreaseReplicationFactorResponse>()
                            .withOperationName("IncreaseReplicationFactor")
                            .withMarshaller(new IncreaseReplicationFactorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(increaseReplicationFactorRequest));
            CompletableFuture<IncreaseReplicationFactorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all of the tags for a DAX cluster. You can call <code>ListTags</code> up to 10 times per second, per
     * account.
     * </p>
     *
     * @param listTagsRequest
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>InvalidArnException The Amazon Resource Name (ARN) supplied in the request is not valid.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reboots a single node of a DAX cluster. The reboot action takes place as soon as possible. During the reboot, the
     * node status is set to REBOOTING.
     * </p>
     * <note>
     * <p>
     * <code>RebootNode</code> restarts the DAX engine process and does not remove the contents of the cache.
     * </p>
     * </note>
     *
     * @param rebootNodeRequest
     * @return A Java Future containing the result of the RebootNode operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>NodeNotFoundException None of the nodes in the cluster have the given node ID.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.RebootNode
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/RebootNode" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RebootNodeResponse> rebootNode(RebootNodeRequest rebootNodeRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rebootNodeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rebootNodeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RebootNode");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RebootNodeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RebootNodeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RebootNodeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RebootNodeRequest, RebootNodeResponse>().withOperationName("RebootNode")
                            .withMarshaller(new RebootNodeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rebootNodeRequest));
            CompletableFuture<RebootNodeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a set of tags with a DAX resource. You can call <code>TagResource</code> up to 5 times per second, per
     * account.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>TagQuotaPerResourceExceededException You have exceeded the maximum number of tags for this DAX
     *         cluster.</li>
     *         <li>InvalidArnException The Amazon Resource Name (ARN) supplied in the request is not valid.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association of tags from a DAX resource. You can call <code>UntagResource</code> up to 5 times per
     * second, per account.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>InvalidArnException The Amazon Resource Name (ARN) supplied in the request is not valid.</li>
     *         <li>TagNotFoundException The tag does not exist.</li>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the settings for a DAX cluster. You can use this action to change one or more cluster configuration
     * parameters by specifying the parameters and the new values.
     * </p>
     *
     * @param updateClusterRequest
     * @return A Java Future containing the result of the UpdateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidClusterStateException The requested DAX cluster is not in the <i>available</i> state.</li>
     *         <li>ClusterNotFoundException The requested cluster ID does not refer to an existing DAX cluster.</li>
     *         <li>InvalidParameterGroupStateException One or more parameters in a parameter group are in an invalid
     *         state.</li>
     *         <li>ParameterGroupNotFoundException The specified parameter group does not exist.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.UpdateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/UpdateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateClusterResponse> updateCluster(UpdateClusterRequest updateClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateClusterRequest, UpdateClusterResponse>()
                            .withOperationName("UpdateCluster")
                            .withMarshaller(new UpdateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateClusterRequest));
            CompletableFuture<UpdateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the parameters of a parameter group. You can modify up to 20 parameters in a single request by
     * submitting a list parameter name and value pairs.
     * </p>
     *
     * @param updateParameterGroupRequest
     * @return A Java Future containing the result of the UpdateParameterGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterGroupStateException One or more parameters in a parameter group are in an invalid
     *         state.</li>
     *         <li>ParameterGroupNotFoundException The specified parameter group does not exist.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>InvalidParameterValueException The value for a parameter is invalid.</li>
     *         <li>InvalidParameterCombinationException Two or more incompatible parameters were specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.UpdateParameterGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/UpdateParameterGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateParameterGroupResponse> updateParameterGroup(
            UpdateParameterGroupRequest updateParameterGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateParameterGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateParameterGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateParameterGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateParameterGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateParameterGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateParameterGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateParameterGroupRequest, UpdateParameterGroupResponse>()
                            .withOperationName("UpdateParameterGroup")
                            .withMarshaller(new UpdateParameterGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateParameterGroupRequest));
            CompletableFuture<UpdateParameterGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies an existing subnet group.
     * </p>
     *
     * @param updateSubnetGroupRequest
     * @return A Java Future containing the result of the UpdateSubnetGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SubnetGroupNotFoundException The requested subnet group name does not refer to an existing subnet
     *         group.</li>
     *         <li>SubnetQuotaExceededException The request cannot be processed because it would exceed the allowed
     *         number of subnets in a subnet group.</li>
     *         <li>SubnetInUseException The requested subnet is being used by another subnet group.</li>
     *         <li>InvalidSubnetException An invalid subnet identifier was specified.</li>
     *         <li>ServiceLinkedRoleNotFoundException The specified service linked role (SLR) was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DaxException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample DaxAsyncClient.UpdateSubnetGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dax-2017-04-19/UpdateSubnetGroup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubnetGroupResponse> updateSubnetGroup(UpdateSubnetGroupRequest updateSubnetGroupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSubnetGroupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubnetGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DAX");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubnetGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubnetGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubnetGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubnetGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubnetGroupRequest, UpdateSubnetGroupResponse>()
                            .withOperationName("UpdateSubnetGroup")
                            .withMarshaller(new UpdateSubnetGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSubnetGroupRequest));
            CompletableFuture<UpdateSubnetGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final DaxServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DaxException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetQuotaExceededFault")
                                .exceptionBuilderSupplier(SubnetQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupNotFoundFault")
                                .exceptionBuilderSupplier(SubnetGroupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupQuotaExceededFault")
                                .exceptionBuilderSupplier(SubnetGroupQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterGroupAlreadyExistsFault")
                                .exceptionBuilderSupplier(ParameterGroupAlreadyExistsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetInUse")
                                .exceptionBuilderSupplier(SubnetInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterNotFoundFault")
                                .exceptionBuilderSupplier(ClusterNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NodeNotFoundFault")
                                .exceptionBuilderSupplier(NodeNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidARNFault")
                                .exceptionBuilderSupplier(InvalidArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombinationException")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NodeQuotaForCustomerExceededFault")
                                .exceptionBuilderSupplier(NodeQuotaForCustomerExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidClusterStateFault")
                                .exceptionBuilderSupplier(InvalidClusterStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientClusterCapacityFault")
                                .exceptionBuilderSupplier(InsufficientClusterCapacityException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagNotFoundFault")
                                .exceptionBuilderSupplier(TagNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidVPCNetworkStateFault")
                                .exceptionBuilderSupplier(InvalidVpcNetworkStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagQuotaPerResourceExceeded")
                                .exceptionBuilderSupplier(TagQuotaPerResourceExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterQuotaForCustomerExceededFault")
                                .exceptionBuilderSupplier(ClusterQuotaForCustomerExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupInUseFault")
                                .exceptionBuilderSupplier(SubnetGroupInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterGroupStateFault")
                                .exceptionBuilderSupplier(InvalidParameterGroupStateException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterGroupNotFoundFault")
                                .exceptionBuilderSupplier(ParameterGroupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSubnet")
                                .exceptionBuilderSupplier(InvalidSubnetException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClusterAlreadyExistsFault")
                                .exceptionBuilderSupplier(ClusterAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLinkedRoleNotFoundFault")
                                .exceptionBuilderSupplier(ServiceLinkedRoleNotFoundException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NodeQuotaForClusterExceededFault")
                                .exceptionBuilderSupplier(NodeQuotaForClusterExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ParameterGroupQuotaExceededFault")
                                .exceptionBuilderSupplier(ParameterGroupQuotaExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("SubnetGroupAlreadyExistsFault")
                                .exceptionBuilderSupplier(SubnetGroupAlreadyExistsException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        DaxServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = DaxServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
