/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dax.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains all of the attributes of a specific DAX cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Cluster implements SdkPojo, Serializable, ToCopyableBuilder<Cluster.Builder, Cluster> {
    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CLUSTER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::clusterArn)).setter(setter(Builder::clusterArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterArn").build()).build();

    private static final SdkField<Integer> TOTAL_NODES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Cluster::totalNodes)).setter(setter(Builder::totalNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalNodes").build()).build();

    private static final SdkField<Integer> ACTIVE_NODES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Cluster::activeNodes)).setter(setter(Builder::activeNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActiveNodes").build()).build();

    private static final SdkField<String> NODE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::nodeType)).setter(setter(Builder::nodeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Endpoint> CLUSTER_DISCOVERY_ENDPOINT_FIELD = SdkField
            .<Endpoint> builder(MarshallingType.SDK_POJO).getter(getter(Cluster::clusterDiscoveryEndpoint))
            .setter(setter(Builder::clusterDiscoveryEndpoint)).constructor(Endpoint::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterDiscoveryEndpoint").build())
            .build();

    private static final SdkField<List<String>> NODE_IDS_TO_REMOVE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Cluster::nodeIdsToRemove))
            .setter(setter(Builder::nodeIdsToRemove))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeIdsToRemove").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Node>> NODES_FIELD = SdkField
            .<List<Node>> builder(MarshallingType.LIST)
            .getter(getter(Cluster::nodes))
            .setter(setter(Builder::nodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Nodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Node> builder(MarshallingType.SDK_POJO)
                                            .constructor(Node::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PREFERRED_MAINTENANCE_WINDOW_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::preferredMaintenanceWindow))
            .setter(setter(Builder::preferredMaintenanceWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredMaintenanceWindow").build())
            .build();

    private static final SdkField<NotificationConfiguration> NOTIFICATION_CONFIGURATION_FIELD = SdkField
            .<NotificationConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(Cluster::notificationConfiguration))
            .setter(setter(Builder::notificationConfiguration)).constructor(NotificationConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationConfiguration").build())
            .build();

    private static final SdkField<String> SUBNET_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::subnetGroup)).setter(setter(Builder::subnetGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetGroup").build()).build();

    private static final SdkField<List<SecurityGroupMembership>> SECURITY_GROUPS_FIELD = SdkField
            .<List<SecurityGroupMembership>> builder(MarshallingType.LIST)
            .getter(getter(Cluster::securityGroups))
            .setter(setter(Builder::securityGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SecurityGroupMembership> builder(MarshallingType.SDK_POJO)
                                            .constructor(SecurityGroupMembership::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<ParameterGroupStatus> PARAMETER_GROUP_FIELD = SdkField
            .<ParameterGroupStatus> builder(MarshallingType.SDK_POJO).getter(getter(Cluster::parameterGroup))
            .setter(setter(Builder::parameterGroup)).constructor(ParameterGroupStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterGroup").build()).build();

    private static final SdkField<SSEDescription> SSE_DESCRIPTION_FIELD = SdkField
            .<SSEDescription> builder(MarshallingType.SDK_POJO).getter(getter(Cluster::sseDescription))
            .setter(setter(Builder::sseDescription)).constructor(SSEDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SSEDescription").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_NAME_FIELD,
            DESCRIPTION_FIELD, CLUSTER_ARN_FIELD, TOTAL_NODES_FIELD, ACTIVE_NODES_FIELD, NODE_TYPE_FIELD, STATUS_FIELD,
            CLUSTER_DISCOVERY_ENDPOINT_FIELD, NODE_IDS_TO_REMOVE_FIELD, NODES_FIELD, PREFERRED_MAINTENANCE_WINDOW_FIELD,
            NOTIFICATION_CONFIGURATION_FIELD, SUBNET_GROUP_FIELD, SECURITY_GROUPS_FIELD, IAM_ROLE_ARN_FIELD,
            PARAMETER_GROUP_FIELD, SSE_DESCRIPTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clusterName;

    private final String description;

    private final String clusterArn;

    private final Integer totalNodes;

    private final Integer activeNodes;

    private final String nodeType;

    private final String status;

    private final Endpoint clusterDiscoveryEndpoint;

    private final List<String> nodeIdsToRemove;

    private final List<Node> nodes;

    private final String preferredMaintenanceWindow;

    private final NotificationConfiguration notificationConfiguration;

    private final String subnetGroup;

    private final List<SecurityGroupMembership> securityGroups;

    private final String iamRoleArn;

    private final ParameterGroupStatus parameterGroup;

    private final SSEDescription sseDescription;

    private Cluster(BuilderImpl builder) {
        this.clusterName = builder.clusterName;
        this.description = builder.description;
        this.clusterArn = builder.clusterArn;
        this.totalNodes = builder.totalNodes;
        this.activeNodes = builder.activeNodes;
        this.nodeType = builder.nodeType;
        this.status = builder.status;
        this.clusterDiscoveryEndpoint = builder.clusterDiscoveryEndpoint;
        this.nodeIdsToRemove = builder.nodeIdsToRemove;
        this.nodes = builder.nodes;
        this.preferredMaintenanceWindow = builder.preferredMaintenanceWindow;
        this.notificationConfiguration = builder.notificationConfiguration;
        this.subnetGroup = builder.subnetGroup;
        this.securityGroups = builder.securityGroups;
        this.iamRoleArn = builder.iamRoleArn;
        this.parameterGroup = builder.parameterGroup;
        this.sseDescription = builder.sseDescription;
    }

    /**
     * <p>
     * The name of the DAX cluster.
     * </p>
     * 
     * @return The name of the DAX cluster.
     */
    public String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * The description of the cluster.
     * </p>
     * 
     * @return The description of the cluster.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the cluster.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the cluster.
     */
    public String clusterArn() {
        return clusterArn;
    }

    /**
     * <p>
     * The total number of nodes in the cluster.
     * </p>
     * 
     * @return The total number of nodes in the cluster.
     */
    public Integer totalNodes() {
        return totalNodes;
    }

    /**
     * <p>
     * The number of nodes in the cluster that are active (i.e., capable of serving requests).
     * </p>
     * 
     * @return The number of nodes in the cluster that are active (i.e., capable of serving requests).
     */
    public Integer activeNodes() {
        return activeNodes;
    }

    /**
     * <p>
     * The node type for the nodes in the cluster. (All nodes in a DAX cluster are of the same type.)
     * </p>
     * 
     * @return The node type for the nodes in the cluster. (All nodes in a DAX cluster are of the same type.)
     */
    public String nodeType() {
        return nodeType;
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * 
     * @return The current status of the cluster.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * The configuration endpoint for this DAX cluster, consisting of a DNS name and a port number. Client applications
     * can specify this endpoint, rather than an individual node endpoint, and allow the DAX client software to
     * intelligently route requests and responses to nodes in the DAX cluster.
     * </p>
     * 
     * @return The configuration endpoint for this DAX cluster, consisting of a DNS name and a port number. Client
     *         applications can specify this endpoint, rather than an individual node endpoint, and allow the DAX client
     *         software to intelligently route requests and responses to nodes in the DAX cluster.
     */
    public Endpoint clusterDiscoveryEndpoint() {
        return clusterDiscoveryEndpoint;
    }

    /**
     * <p>
     * A list of nodes to be removed from the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of nodes to be removed from the cluster.
     */
    public List<String> nodeIdsToRemove() {
        return nodeIdsToRemove;
    }

    /**
     * <p>
     * A list of nodes that are currently in the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of nodes that are currently in the cluster.
     */
    public List<Node> nodes() {
        return nodes;
    }

    /**
     * <p>
     * A range of time when maintenance of DAX cluster software will be performed. For example:
     * <code>sun:01:00-sun:09:00</code>. Cluster maintenance normally takes less than 30 minutes, and is performed
     * automatically within the maintenance window.
     * </p>
     * 
     * @return A range of time when maintenance of DAX cluster software will be performed. For example:
     *         <code>sun:01:00-sun:09:00</code>. Cluster maintenance normally takes less than 30 minutes, and is
     *         performed automatically within the maintenance window.
     */
    public String preferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }

    /**
     * <p>
     * Describes a notification topic and its status. Notification topics are used for publishing DAX events to
     * subscribers using Amazon Simple Notification Service (SNS).
     * </p>
     * 
     * @return Describes a notification topic and its status. Notification topics are used for publishing DAX events to
     *         subscribers using Amazon Simple Notification Service (SNS).
     */
    public NotificationConfiguration notificationConfiguration() {
        return notificationConfiguration;
    }

    /**
     * <p>
     * The subnet group where the DAX cluster is running.
     * </p>
     * 
     * @return The subnet group where the DAX cluster is running.
     */
    public String subnetGroup() {
        return subnetGroup;
    }

    /**
     * <p>
     * A list of security groups, and the status of each, for the nodes in the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of security groups, and the status of each, for the nodes in the cluster.
     */
    public List<SecurityGroupMembership> securityGroups() {
        return securityGroups;
    }

    /**
     * <p>
     * A valid Amazon Resource Name (ARN) that identifies an IAM role. At runtime, DAX will assume this role and use the
     * role's permissions to access DynamoDB on your behalf.
     * </p>
     * 
     * @return A valid Amazon Resource Name (ARN) that identifies an IAM role. At runtime, DAX will assume this role and
     *         use the role's permissions to access DynamoDB on your behalf.
     */
    public String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * The parameter group being used by nodes in the cluster.
     * </p>
     * 
     * @return The parameter group being used by nodes in the cluster.
     */
    public ParameterGroupStatus parameterGroup() {
        return parameterGroup;
    }

    /**
     * <p>
     * The description of the server-side encryption status on the specified DAX cluster.
     * </p>
     * 
     * @return The description of the server-side encryption status on the specified DAX cluster.
     */
    public SSEDescription sseDescription() {
        return sseDescription;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(clusterArn());
        hashCode = 31 * hashCode + Objects.hashCode(totalNodes());
        hashCode = 31 * hashCode + Objects.hashCode(activeNodes());
        hashCode = 31 * hashCode + Objects.hashCode(nodeType());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(clusterDiscoveryEndpoint());
        hashCode = 31 * hashCode + Objects.hashCode(nodeIdsToRemove());
        hashCode = 31 * hashCode + Objects.hashCode(nodes());
        hashCode = 31 * hashCode + Objects.hashCode(preferredMaintenanceWindow());
        hashCode = 31 * hashCode + Objects.hashCode(notificationConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(subnetGroup());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroups());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(parameterGroup());
        hashCode = 31 * hashCode + Objects.hashCode(sseDescription());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Cluster)) {
            return false;
        }
        Cluster other = (Cluster) obj;
        return Objects.equals(clusterName(), other.clusterName()) && Objects.equals(description(), other.description())
                && Objects.equals(clusterArn(), other.clusterArn()) && Objects.equals(totalNodes(), other.totalNodes())
                && Objects.equals(activeNodes(), other.activeNodes()) && Objects.equals(nodeType(), other.nodeType())
                && Objects.equals(status(), other.status())
                && Objects.equals(clusterDiscoveryEndpoint(), other.clusterDiscoveryEndpoint())
                && Objects.equals(nodeIdsToRemove(), other.nodeIdsToRemove()) && Objects.equals(nodes(), other.nodes())
                && Objects.equals(preferredMaintenanceWindow(), other.preferredMaintenanceWindow())
                && Objects.equals(notificationConfiguration(), other.notificationConfiguration())
                && Objects.equals(subnetGroup(), other.subnetGroup()) && Objects.equals(securityGroups(), other.securityGroups())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(parameterGroup(), other.parameterGroup())
                && Objects.equals(sseDescription(), other.sseDescription());
    }

    @Override
    public String toString() {
        return ToString.builder("Cluster").add("ClusterName", clusterName()).add("Description", description())
                .add("ClusterArn", clusterArn()).add("TotalNodes", totalNodes()).add("ActiveNodes", activeNodes())
                .add("NodeType", nodeType()).add("Status", status()).add("ClusterDiscoveryEndpoint", clusterDiscoveryEndpoint())
                .add("NodeIdsToRemove", nodeIdsToRemove()).add("Nodes", nodes())
                .add("PreferredMaintenanceWindow", preferredMaintenanceWindow())
                .add("NotificationConfiguration", notificationConfiguration()).add("SubnetGroup", subnetGroup())
                .add("SecurityGroups", securityGroups()).add("IamRoleArn", iamRoleArn()).add("ParameterGroup", parameterGroup())
                .add("SSEDescription", sseDescription()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClusterName":
            return Optional.ofNullable(clazz.cast(clusterName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ClusterArn":
            return Optional.ofNullable(clazz.cast(clusterArn()));
        case "TotalNodes":
            return Optional.ofNullable(clazz.cast(totalNodes()));
        case "ActiveNodes":
            return Optional.ofNullable(clazz.cast(activeNodes()));
        case "NodeType":
            return Optional.ofNullable(clazz.cast(nodeType()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "ClusterDiscoveryEndpoint":
            return Optional.ofNullable(clazz.cast(clusterDiscoveryEndpoint()));
        case "NodeIdsToRemove":
            return Optional.ofNullable(clazz.cast(nodeIdsToRemove()));
        case "Nodes":
            return Optional.ofNullable(clazz.cast(nodes()));
        case "PreferredMaintenanceWindow":
            return Optional.ofNullable(clazz.cast(preferredMaintenanceWindow()));
        case "NotificationConfiguration":
            return Optional.ofNullable(clazz.cast(notificationConfiguration()));
        case "SubnetGroup":
            return Optional.ofNullable(clazz.cast(subnetGroup()));
        case "SecurityGroups":
            return Optional.ofNullable(clazz.cast(securityGroups()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "ParameterGroup":
            return Optional.ofNullable(clazz.cast(parameterGroup()));
        case "SSEDescription":
            return Optional.ofNullable(clazz.cast(sseDescription()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Cluster, T> g) {
        return obj -> g.apply((Cluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Cluster> {
        /**
         * <p>
         * The name of the DAX cluster.
         * </p>
         * 
         * @param clusterName
         *        The name of the DAX cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * The description of the cluster.
         * </p>
         * 
         * @param description
         *        The description of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the cluster.
         * </p>
         * 
         * @param clusterArn
         *        The Amazon Resource Name (ARN) that uniquely identifies the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterArn(String clusterArn);

        /**
         * <p>
         * The total number of nodes in the cluster.
         * </p>
         * 
         * @param totalNodes
         *        The total number of nodes in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalNodes(Integer totalNodes);

        /**
         * <p>
         * The number of nodes in the cluster that are active (i.e., capable of serving requests).
         * </p>
         * 
         * @param activeNodes
         *        The number of nodes in the cluster that are active (i.e., capable of serving requests).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeNodes(Integer activeNodes);

        /**
         * <p>
         * The node type for the nodes in the cluster. (All nodes in a DAX cluster are of the same type.)
         * </p>
         * 
         * @param nodeType
         *        The node type for the nodes in the cluster. (All nodes in a DAX cluster are of the same type.)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeType(String nodeType);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param status
         *        The current status of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * The configuration endpoint for this DAX cluster, consisting of a DNS name and a port number. Client
         * applications can specify this endpoint, rather than an individual node endpoint, and allow the DAX client
         * software to intelligently route requests and responses to nodes in the DAX cluster.
         * </p>
         * 
         * @param clusterDiscoveryEndpoint
         *        The configuration endpoint for this DAX cluster, consisting of a DNS name and a port number. Client
         *        applications can specify this endpoint, rather than an individual node endpoint, and allow the DAX
         *        client software to intelligently route requests and responses to nodes in the DAX cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterDiscoveryEndpoint(Endpoint clusterDiscoveryEndpoint);

        /**
         * <p>
         * The configuration endpoint for this DAX cluster, consisting of a DNS name and a port number. Client
         * applications can specify this endpoint, rather than an individual node endpoint, and allow the DAX client
         * software to intelligently route requests and responses to nodes in the DAX cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link Endpoint.Builder} avoiding the need to create
         * one manually via {@link Endpoint#builder()}.
         *
         * When the {@link Consumer} completes, {@link Endpoint.Builder#build()} is called immediately and its result is
         * passed to {@link #clusterDiscoveryEndpoint(Endpoint)}.
         * 
         * @param clusterDiscoveryEndpoint
         *        a consumer that will call methods on {@link Endpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clusterDiscoveryEndpoint(Endpoint)
         */
        default Builder clusterDiscoveryEndpoint(Consumer<Endpoint.Builder> clusterDiscoveryEndpoint) {
            return clusterDiscoveryEndpoint(Endpoint.builder().applyMutation(clusterDiscoveryEndpoint).build());
        }

        /**
         * <p>
         * A list of nodes to be removed from the cluster.
         * </p>
         * 
         * @param nodeIdsToRemove
         *        A list of nodes to be removed from the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeIdsToRemove(Collection<String> nodeIdsToRemove);

        /**
         * <p>
         * A list of nodes to be removed from the cluster.
         * </p>
         * 
         * @param nodeIdsToRemove
         *        A list of nodes to be removed from the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeIdsToRemove(String... nodeIdsToRemove);

        /**
         * <p>
         * A list of nodes that are currently in the cluster.
         * </p>
         * 
         * @param nodes
         *        A list of nodes that are currently in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodes(Collection<Node> nodes);

        /**
         * <p>
         * A list of nodes that are currently in the cluster.
         * </p>
         * 
         * @param nodes
         *        A list of nodes that are currently in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodes(Node... nodes);

        /**
         * <p>
         * A list of nodes that are currently in the cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Node>.Builder} avoiding the need to create
         * one manually via {@link List<Node>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Node>.Builder#build()} is called immediately and its result
         * is passed to {@link #nodes(List<Node>)}.
         * 
         * @param nodes
         *        a consumer that will call methods on {@link List<Node>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nodes(List<Node>)
         */
        Builder nodes(Consumer<Node.Builder>... nodes);

        /**
         * <p>
         * A range of time when maintenance of DAX cluster software will be performed. For example:
         * <code>sun:01:00-sun:09:00</code>. Cluster maintenance normally takes less than 30 minutes, and is performed
         * automatically within the maintenance window.
         * </p>
         * 
         * @param preferredMaintenanceWindow
         *        A range of time when maintenance of DAX cluster software will be performed. For example:
         *        <code>sun:01:00-sun:09:00</code>. Cluster maintenance normally takes less than 30 minutes, and is
         *        performed automatically within the maintenance window.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredMaintenanceWindow(String preferredMaintenanceWindow);

        /**
         * <p>
         * Describes a notification topic and its status. Notification topics are used for publishing DAX events to
         * subscribers using Amazon Simple Notification Service (SNS).
         * </p>
         * 
         * @param notificationConfiguration
         *        Describes a notification topic and its status. Notification topics are used for publishing DAX events
         *        to subscribers using Amazon Simple Notification Service (SNS).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationConfiguration(NotificationConfiguration notificationConfiguration);

        /**
         * <p>
         * Describes a notification topic and its status. Notification topics are used for publishing DAX events to
         * subscribers using Amazon Simple Notification Service (SNS).
         * </p>
         * This is a convenience that creates an instance of the {@link NotificationConfiguration.Builder} avoiding the
         * need to create one manually via {@link NotificationConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link NotificationConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #notificationConfiguration(NotificationConfiguration)}.
         * 
         * @param notificationConfiguration
         *        a consumer that will call methods on {@link NotificationConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notificationConfiguration(NotificationConfiguration)
         */
        default Builder notificationConfiguration(Consumer<NotificationConfiguration.Builder> notificationConfiguration) {
            return notificationConfiguration(NotificationConfiguration.builder().applyMutation(notificationConfiguration).build());
        }

        /**
         * <p>
         * The subnet group where the DAX cluster is running.
         * </p>
         * 
         * @param subnetGroup
         *        The subnet group where the DAX cluster is running.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetGroup(String subnetGroup);

        /**
         * <p>
         * A list of security groups, and the status of each, for the nodes in the cluster.
         * </p>
         * 
         * @param securityGroups
         *        A list of security groups, and the status of each, for the nodes in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(Collection<SecurityGroupMembership> securityGroups);

        /**
         * <p>
         * A list of security groups, and the status of each, for the nodes in the cluster.
         * </p>
         * 
         * @param securityGroups
         *        A list of security groups, and the status of each, for the nodes in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroups(SecurityGroupMembership... securityGroups);

        /**
         * <p>
         * A list of security groups, and the status of each, for the nodes in the cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SecurityGroupMembership>.Builder} avoiding
         * the need to create one manually via {@link List<SecurityGroupMembership>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SecurityGroupMembership>.Builder#build()} is called
         * immediately and its result is passed to {@link #securityGroups(List<SecurityGroupMembership>)}.
         * 
         * @param securityGroups
         *        a consumer that will call methods on {@link List<SecurityGroupMembership>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #securityGroups(List<SecurityGroupMembership>)
         */
        Builder securityGroups(Consumer<SecurityGroupMembership.Builder>... securityGroups);

        /**
         * <p>
         * A valid Amazon Resource Name (ARN) that identifies an IAM role. At runtime, DAX will assume this role and use
         * the role's permissions to access DynamoDB on your behalf.
         * </p>
         * 
         * @param iamRoleArn
         *        A valid Amazon Resource Name (ARN) that identifies an IAM role. At runtime, DAX will assume this role
         *        and use the role's permissions to access DynamoDB on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * The parameter group being used by nodes in the cluster.
         * </p>
         * 
         * @param parameterGroup
         *        The parameter group being used by nodes in the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterGroup(ParameterGroupStatus parameterGroup);

        /**
         * <p>
         * The parameter group being used by nodes in the cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link ParameterGroupStatus.Builder} avoiding the need
         * to create one manually via {@link ParameterGroupStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link ParameterGroupStatus.Builder#build()} is called immediately and
         * its result is passed to {@link #parameterGroup(ParameterGroupStatus)}.
         * 
         * @param parameterGroup
         *        a consumer that will call methods on {@link ParameterGroupStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterGroup(ParameterGroupStatus)
         */
        default Builder parameterGroup(Consumer<ParameterGroupStatus.Builder> parameterGroup) {
            return parameterGroup(ParameterGroupStatus.builder().applyMutation(parameterGroup).build());
        }

        /**
         * <p>
         * The description of the server-side encryption status on the specified DAX cluster.
         * </p>
         * 
         * @param sseDescription
         *        The description of the server-side encryption status on the specified DAX cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sseDescription(SSEDescription sseDescription);

        /**
         * <p>
         * The description of the server-side encryption status on the specified DAX cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link SSEDescription.Builder} avoiding the need to
         * create one manually via {@link SSEDescription#builder()}.
         *
         * When the {@link Consumer} completes, {@link SSEDescription.Builder#build()} is called immediately and its
         * result is passed to {@link #sseDescription(SSEDescription)}.
         * 
         * @param sseDescription
         *        a consumer that will call methods on {@link SSEDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sseDescription(SSEDescription)
         */
        default Builder sseDescription(Consumer<SSEDescription.Builder> sseDescription) {
            return sseDescription(SSEDescription.builder().applyMutation(sseDescription).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String clusterName;

        private String description;

        private String clusterArn;

        private Integer totalNodes;

        private Integer activeNodes;

        private String nodeType;

        private String status;

        private Endpoint clusterDiscoveryEndpoint;

        private List<String> nodeIdsToRemove = DefaultSdkAutoConstructList.getInstance();

        private List<Node> nodes = DefaultSdkAutoConstructList.getInstance();

        private String preferredMaintenanceWindow;

        private NotificationConfiguration notificationConfiguration;

        private String subnetGroup;

        private List<SecurityGroupMembership> securityGroups = DefaultSdkAutoConstructList.getInstance();

        private String iamRoleArn;

        private ParameterGroupStatus parameterGroup;

        private SSEDescription sseDescription;

        private BuilderImpl() {
        }

        private BuilderImpl(Cluster model) {
            clusterName(model.clusterName);
            description(model.description);
            clusterArn(model.clusterArn);
            totalNodes(model.totalNodes);
            activeNodes(model.activeNodes);
            nodeType(model.nodeType);
            status(model.status);
            clusterDiscoveryEndpoint(model.clusterDiscoveryEndpoint);
            nodeIdsToRemove(model.nodeIdsToRemove);
            nodes(model.nodes);
            preferredMaintenanceWindow(model.preferredMaintenanceWindow);
            notificationConfiguration(model.notificationConfiguration);
            subnetGroup(model.subnetGroup);
            securityGroups(model.securityGroups);
            iamRoleArn(model.iamRoleArn);
            parameterGroup(model.parameterGroup);
            sseDescription(model.sseDescription);
        }

        public final String getClusterName() {
            return clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getClusterArn() {
            return clusterArn;
        }

        @Override
        public final Builder clusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
            return this;
        }

        public final void setClusterArn(String clusterArn) {
            this.clusterArn = clusterArn;
        }

        public final Integer getTotalNodes() {
            return totalNodes;
        }

        @Override
        public final Builder totalNodes(Integer totalNodes) {
            this.totalNodes = totalNodes;
            return this;
        }

        public final void setTotalNodes(Integer totalNodes) {
            this.totalNodes = totalNodes;
        }

        public final Integer getActiveNodes() {
            return activeNodes;
        }

        @Override
        public final Builder activeNodes(Integer activeNodes) {
            this.activeNodes = activeNodes;
            return this;
        }

        public final void setActiveNodes(Integer activeNodes) {
            this.activeNodes = activeNodes;
        }

        public final String getNodeType() {
            return nodeType;
        }

        @Override
        public final Builder nodeType(String nodeType) {
            this.nodeType = nodeType;
            return this;
        }

        public final void setNodeType(String nodeType) {
            this.nodeType = nodeType;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Endpoint.Builder getClusterDiscoveryEndpoint() {
            return clusterDiscoveryEndpoint != null ? clusterDiscoveryEndpoint.toBuilder() : null;
        }

        @Override
        public final Builder clusterDiscoveryEndpoint(Endpoint clusterDiscoveryEndpoint) {
            this.clusterDiscoveryEndpoint = clusterDiscoveryEndpoint;
            return this;
        }

        public final void setClusterDiscoveryEndpoint(Endpoint.BuilderImpl clusterDiscoveryEndpoint) {
            this.clusterDiscoveryEndpoint = clusterDiscoveryEndpoint != null ? clusterDiscoveryEndpoint.build() : null;
        }

        public final Collection<String> getNodeIdsToRemove() {
            return nodeIdsToRemove;
        }

        @Override
        public final Builder nodeIdsToRemove(Collection<String> nodeIdsToRemove) {
            this.nodeIdsToRemove = NodeIdentifierListCopier.copy(nodeIdsToRemove);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodeIdsToRemove(String... nodeIdsToRemove) {
            nodeIdsToRemove(Arrays.asList(nodeIdsToRemove));
            return this;
        }

        public final void setNodeIdsToRemove(Collection<String> nodeIdsToRemove) {
            this.nodeIdsToRemove = NodeIdentifierListCopier.copy(nodeIdsToRemove);
        }

        public final Collection<Node.Builder> getNodes() {
            return nodes != null ? nodes.stream().map(Node::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder nodes(Collection<Node> nodes) {
            this.nodes = NodeListCopier.copy(nodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodes(Node... nodes) {
            nodes(Arrays.asList(nodes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nodes(Consumer<Node.Builder>... nodes) {
            nodes(Stream.of(nodes).map(c -> Node.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setNodes(Collection<Node.BuilderImpl> nodes) {
            this.nodes = NodeListCopier.copyFromBuilder(nodes);
        }

        public final String getPreferredMaintenanceWindow() {
            return preferredMaintenanceWindow;
        }

        @Override
        public final Builder preferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
            return this;
        }

        public final void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        }

        public final NotificationConfiguration.Builder getNotificationConfiguration() {
            return notificationConfiguration != null ? notificationConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder notificationConfiguration(NotificationConfiguration notificationConfiguration) {
            this.notificationConfiguration = notificationConfiguration;
            return this;
        }

        public final void setNotificationConfiguration(NotificationConfiguration.BuilderImpl notificationConfiguration) {
            this.notificationConfiguration = notificationConfiguration != null ? notificationConfiguration.build() : null;
        }

        public final String getSubnetGroup() {
            return subnetGroup;
        }

        @Override
        public final Builder subnetGroup(String subnetGroup) {
            this.subnetGroup = subnetGroup;
            return this;
        }

        public final void setSubnetGroup(String subnetGroup) {
            this.subnetGroup = subnetGroup;
        }

        public final Collection<SecurityGroupMembership.Builder> getSecurityGroups() {
            return securityGroups != null ? securityGroups.stream().map(SecurityGroupMembership::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder securityGroups(Collection<SecurityGroupMembership> securityGroups) {
            this.securityGroups = SecurityGroupMembershipListCopier.copy(securityGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(SecurityGroupMembership... securityGroups) {
            securityGroups(Arrays.asList(securityGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroups(Consumer<SecurityGroupMembership.Builder>... securityGroups) {
            securityGroups(Stream.of(securityGroups).map(c -> SecurityGroupMembership.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSecurityGroups(Collection<SecurityGroupMembership.BuilderImpl> securityGroups) {
            this.securityGroups = SecurityGroupMembershipListCopier.copyFromBuilder(securityGroups);
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        public final ParameterGroupStatus.Builder getParameterGroup() {
            return parameterGroup != null ? parameterGroup.toBuilder() : null;
        }

        @Override
        public final Builder parameterGroup(ParameterGroupStatus parameterGroup) {
            this.parameterGroup = parameterGroup;
            return this;
        }

        public final void setParameterGroup(ParameterGroupStatus.BuilderImpl parameterGroup) {
            this.parameterGroup = parameterGroup != null ? parameterGroup.build() : null;
        }

        public final SSEDescription.Builder getSseDescription() {
            return sseDescription != null ? sseDescription.toBuilder() : null;
        }

        @Override
        public final Builder sseDescription(SSEDescription sseDescription) {
            this.sseDescription = sseDescription;
            return this;
        }

        public final void setSseDescription(SSEDescription.BuilderImpl sseDescription) {
            this.sseDescription = sseDescription != null ? sseDescription.build() : null;
        }

        @Override
        public Cluster build() {
            return new Cluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
